/*
	msettingsdialog.cpp - settings dialog
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

// TODO: 0.7: stats: use system colors
// TODO: 0.7: create "Advanced" page (stats, custom message)

#include "configuration.h"
#include "extras.h"
#include "links.h"
#include "mactioneditdialog.h"
#include "miscutils.h"
#include "mmessagedialog.h"
#include "msettingsdialog.h"
#include "msystemtray.h"
#include "mtip.h"

#include <qcheckbox.h>
#include <qfile.h>
#include <qhbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <qvgroupbox.h>
#include <qwhatsthis.h>

#include <kconfig.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klineedit.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <knuminput.h>
#include <kpushbutton.h>
#include <krun.h>
#include <kstandarddirs.h>
#include <kurllabel.h>

/* constructor */

MSettingsDialog::MSettingsDialog(QWidget *parent, const char *name)
	: KDialogBase(
		IconList,
		i18n("Settings"),
		Apply | Cancel | Default | Help | Ok,
		Ok, // default button
		parent,
		name,
		TRUE, // modal
		TRUE // separator
	)
{
	initGeneralPage();
	initNotifyPage();
	initAutoPage();
	initMessagesPage();
	initActionsPage();
	initStatsPage();

	connect(
		this, SIGNAL(applyClicked()),
		this, SLOT(okClickedSlot()));
	connect(
		this, SIGNAL(okClicked()),
		this, SLOT(okClickedSlot()));

	// general
	c_timeRemember->setChecked(kshutdownrc->timeRemember);
	c_schedulerEnabled->setChecked(kshutdownrc->schedEnabled);

	// notifications
	_notifyWidget->clear();
	KNotify::Application *app = _notifyWidget->addApplicationEvents("kshutdown/eventsrc");
	if (app)
		_notifyWidget->addVisibleApp(app);

	// automation
	c_autoLock->setChecked(QFile::exists(KGlobalSettings::autostartPath() +
		"/" + AUTO_LOCK_FILE_NAME));
	c_cdTrayClose->setChecked(kshutdownrc->cdTrayClose);
	in_cdTrayCloseCommand->setText(kshutdownrc->cdTrayCloseCommand);

	// messages
	c_confirmAction->setChecked(kshutdownrc->confirmAction);
	c_popupMessages->setChecked(kshutdownrc->popupMessages);
	in_popupTimeout->setValue(kshutdownrc->popupTimeout);
	setWarningMessage(
		kshutdownrc->warningMessageEnabled,
		kshutdownrc->warningMessageDelay
	);
	c_customMessageEnabled->setChecked(kshutdownrc->customMessageEnabled);
	in_customMessageCommand->setText(kshutdownrc->customMessageCommand);

	// statistics
	c_statsLongFormat->setChecked(kshutdownrc->statsLongFormat);
	c_statsToggleFromField->setChecked(kshutdownrc->statsToggleFromField);

	setConfigChanged(FALSE);
	setInitialSize(configDialogSize("Settings"));
}

/* destructor */

MSettingsDialog::~MSettingsDialog()
{
}

/* private */

QFrame *MSettingsDialog::addPage(const QString &itemName, const QString &iconName)
{
	return KDialogBase::addPage(itemName, QString::null, MainBarIcon(iconName, 32));
}

bool MSettingsDialog::disabledByAdmin(const QString &key, QFrame *page,
	QBoxLayout *pageLayout) const
{
	if (mainWindow->isRestricted("kshutdown_settings_" + key))
	{
		// add tip
		if (page)
		{
			MTip *t_admin =
				new MTip(MTip::TipType_Info, page, "MTip::t_admin");
			t_admin->setTipText(
				i18n(
					"This page has been disabled " \
					"by the Administator."
				)
			);
			pageLayout->addWidget(t_admin);
		}

		return TRUE; // disabled
	}

	return FALSE; // enabled
}

void MSettingsDialog::initActionsPage()
{
	_actionsPage = addPage(i18n("Actions & Extras Menu"), "exit");
	QVBoxLayout *l = new QVBoxLayout(_actionsPage, 2);

	if (disabledByAdmin("actions", _actionsPage, l))
		return;

	QVGroupBox *gb_general = new QVGroupBox(i18n("Ac&tions"), _actionsPage,
		"QVGroupBox::gb_general");

	_actions = new QListView(gb_general, "QListView::_actions");
	_actions->setAllColumnsShowFocus(TRUE);
	_actions->setItemMargin(5);
	_actions->setMaximumHeight(200);
	_actions->setSorting(-1); // no sort
	_actions->addColumn(i18n("Action"));
	_actions->addColumn(i18n("Method"));
	_actions->addColumn(i18n("Command before action"));

	_logoutItem = new QListViewItem(_actions,
		mainWindow->getActionName(MMainWindow::What_Logout));
	_logoutItem->setPixmap(
		0, mainWindow->getActionIcon(MMainWindow::What_Logout)
	);

	_lockScreenItem = new QListViewItem(_actions,
		mainWindow->getActionName(MMainWindow::What_LockScreen));
	_lockScreenItem->setPixmap(
		0, mainWindow->getActionIcon(MMainWindow::What_LockScreen)
	);

	_rebootItem = new QListViewItem(_actions,
		mainWindow->getActionName(MMainWindow::What_Reboot));
	_rebootItem->setPixmap(
		0, mainWindow->getActionIcon(MMainWindow::What_Reboot)
	);

	_shutDownItem = new QListViewItem(_actions,
		mainWindow->getActionName(MMainWindow::What_ShutDown));
	_shutDownItem->setPixmap(
		0, mainWindow->getActionIcon(MMainWindow::What_ShutDown)
	);

	connect(
		_actions, SIGNAL(doubleClicked(QListViewItem *, const QPoint &, int)),
		this, SLOT(editActionSlot()));
	connect(
		_actions, SIGNAL(spacePressed(QListViewItem *)),
		this, SLOT(editActionSlot()));
	// edit action
	KPushButton *b_editAction = new KPushButton(SmallIcon("edit"),
		i18n("&Edit..."), gb_general, "KPushButton::b_editAction");
	connect(
		b_editAction, SIGNAL(clicked()),
		this, SLOT(editActionSlot()));
	// tip
	MTip *t_actions =
		new MTip(MTip::TipType_Normal, gb_general, "MTip::t_actions");
	t_actions->setTipText(
		"<ul>" \
		"<li>" + i18n("If you are using <b>KDE</b> and <b>KDM</b> (KDE Display Manager), then set all methods to <i>KDE</i>") + "</li>" \
		"<li>" + i18n("If you are using <b>KDE</b> and display manager different than <b>KDM</b>, then set <i>Turn Off Computer</i> and <i>Restart Computer</i> methods to <i>/sbin/...</i>") + "</li>" \
		"<li>" + i18n("If you are running <b>KShutDown</b> from the non-KDE session (e.g. <b>GNOME</b>), then change all methods...") + "</li>" \
		"<li>" + i18n("See FAQ for more details") + "</li>" \
		"</ul>"
	);
	l->addWidget(gb_general);

	QVGroupBox *gb_extras = new QVGroupBox(i18n("E&xtras Menu"), _actionsPage,
		"QVGroupBox::gb_extras");
	KPushButton *b_modifyExtras = new KPushButton(SmallIcon("edit"),
		i18n("&Modify..."), gb_extras, "KPushButton::b_modifyExtras");
	b_modifyExtras->setEnabled(
		MiscUtils::isActionEnabled(MMainWindow::What_Extras)
	);
	connect(
		b_modifyExtras, SIGNAL(clicked()),
		this, SLOT(modifyExtrasSlot()));
	l->addWidget(gb_extras);

	// update items after widgets init
	updateActions();
}

void MSettingsDialog::initAutoPage()
{
	_autoPage = addPage(i18n("Automation"), "reload");
	QVBoxLayout *l = new QVBoxLayout(_autoPage, 2);

	// after login
	QVGroupBox *gb_afterLogin = new QVGroupBox(i18n("A&fter Login"), _autoPage,
		"QVGroupBox::gb_afterLogin");
	c_autoLock = new QCheckBox(i18n("&Lock screen"), gb_afterLogin,
		"QCheckBox::c_autoLock");
	connect(
		c_autoLock, SIGNAL(clicked()),
		this, SLOT(configChangeSlot()));
	l->addWidget(gb_afterLogin);

	// cd tray close
	QVGroupBox *gb_cdTrayClose = new QVGroupBox(i18n("&Before Logout"), _autoPage, "QVGroupBox::gb_cdTrayClose");
	c_cdTrayClose = new QCheckBox(i18n("Close CD-ROM &Tray"), gb_cdTrayClose,
		"QCheckBox::c_cdTrayClose");
	connect(
		c_cdTrayClose, SIGNAL(clicked()),
		this, SLOT(configChangeSlot()));
	QHBox *cdTrayCloseBox = new QHBox(gb_cdTrayClose);
	QLabel *l_cdTrayCloseCommand = new QLabel(i18n("Co&mmand:"), cdTrayCloseBox, "QLabel::l_cdTrayCloseCommand");
	in_cdTrayCloseCommand = new KLineEdit(cdTrayCloseBox, "KLineEdit::in_cdTrayCloseCommand");
	l_cdTrayCloseCommand->setBuddy(in_cdTrayCloseCommand);
	connect(
		in_cdTrayCloseCommand, SIGNAL(textChanged(const QString &)),
		this, SLOT(configChangeSlot()));
	l->addWidget(gb_cdTrayClose);

	KURLLabel *seeAlso = new KURLLabel(_autoPage);
	seeAlso->setText(i18n("More..."));
	connect(
		seeAlso, SIGNAL(leftClickedURL()),
		this, SLOT(seeAlsoClickedSlot()));
	l->addWidget(seeAlso);

	// -
	l->addStretch();

	connect(
		c_cdTrayClose, SIGNAL(toggled(bool)),
		in_cdTrayCloseCommand, SLOT(setEnabled(bool)));
}

void MSettingsDialog::initGeneralPage()
{
	_generalPage = addPage(i18n("General"), "misc");
	QVBoxLayout *l = new QVBoxLayout(_generalPage, 2);

	// remember time settings
	c_timeRemember = new QCheckBox(i18n("Remember time &settings"),
		_generalPage, "QCheckBox::c_timeRemember");
	connect(
		c_timeRemember, SIGNAL(clicked()),
		this, SLOT(configChangeSlot()));
	l->addWidget(c_timeRemember);

	// scheduler
	c_schedulerEnabled = new QCheckBox(i18n("Enable &Scheduler"), _generalPage, "QCheckBox::c_schedulerEnabled");
	connect(
		c_schedulerEnabled, SIGNAL(clicked()),
		this, SLOT(configChangeSlot()));
	l->addWidget(c_schedulerEnabled);

	// kde settings
	QVGroupBox *gb_kde = new QVGroupBox(i18n("&KDE Settings"), _generalPage,
		"QVGroupBox::gb_kde");
	// lock screen settings
	b_lockScreenSettings = new KPushButton(SmallIcon("kscreensaver"),
		i18n("Screen Sa&ver..."), gb_kde, "KPushButton::b_lockScreenSettings");
	connect(
		b_lockScreenSettings, SIGNAL(clicked()),
		this, SLOT(lockSettingsSlot()));
	// logout settings
	b_logoutSettings = new KPushButton(i18n("Session &Manager..."), gb_kde,
		"KPushButton::b_logoutSettings");
	connect(
		b_logoutSettings, SIGNAL(clicked()),
		this, SLOT(logoutSettingsSlot()));
	l->addWidget(gb_kde);

// TODO: 0.7: "Add/remove link"
	// links
	QVGroupBox *gb_links = new QVGroupBox(i18n("&Links"), _generalPage);
	new Links(gb_links, "Links");
	l->addWidget(gb_links);

	// -
	l->addStretch();

	QString karambaThemes = locate("appdata", "karamba/kshutdown-32x32.png");
	if (!karambaThemes.isNull())
	{
		int i = karambaThemes.findRev("/");
		if (i != -1 && i > 0)
		{
			MTip *t_karamba = new MTip(MTip::TipType_Info, _generalPage);
			karambaThemes = karambaThemes.left(i);
			t_karamba->setTipText(
				// title
				"<h2>SuperKaramba</h2>" \
				// themes
				"<a href=\"" + karambaThemes + "\">" + i18n("Themes") + "</a> (KShutDown)<br>" \
				// home page
				"<a href=\"http://www.superkaramba.com/index.php\">" + i18n("Home Page") + "</a>" \
			);
			l->addWidget(t_karamba);
		}
	}
}

void MSettingsDialog::initMessagesPage()
{
	_messagesPage = addPage(i18n("Confirmations & Messages"), "messagebox_warning");
	QVBoxLayout *l = new QVBoxLayout(_messagesPage, 2);

	// confirm action
	c_confirmAction = new QCheckBox(i18n("Confirm action (&recommended)"), _messagesPage, "QCheckBox::c_confirmAction");
	connect(
		c_confirmAction, SIGNAL(clicked()),
		this, SLOT(configChangeSlot()));
	l->addWidget(c_confirmAction);

	// re-enable messages
	b_enableAllMessages = new KPushButton(i18n("Re-enable All \"Don't ask again\" Me&ssages"), _messagesPage, "KPushButton::b_enableAllMessages");
	QWhatsThis::add(
		b_enableAllMessages,
		"<qt>" +
		i18n(
		"Enable all messages which have been turned off with the " \
		"<b>Do not show this message again</b> feature."
		 ) +
		"</qt>"
	);
	connect(
		b_enableAllMessages, SIGNAL(clicked()),
		this, SLOT(enableAllMessagesSlot()));
	l->addWidget(b_enableAllMessages);

	// -
	l->addStretch();

	// popup messages:
	QVGroupBox *gb_popup = new QVGroupBox(i18n("&Popup Messages (Passive)"), _messagesPage, "QVGroupBox::gb_popup");
	c_popupMessages = new QCheckBox(i18n("&Enabled"), gb_popup, "QCheckBox::c_popupMessages");
	connect(
		c_popupMessages, SIGNAL(clicked()),
		this, SLOT(configChangeSlot()));
	// popup timeout
	in_popupTimeout = new KIntNumInput(gb_popup, "KIntNumInput::in_popupTimeout");
	in_popupTimeout->setLabel(i18n("Hide &message after:"));
	in_popupTimeout->setRange(2, 300, 1, TRUE);
	in_popupTimeout->setSuffix(" " + i18n("second(s)"));
	connect(
		in_popupTimeout, SIGNAL(valueChanged(int)),
		this, SLOT(configChangeSlot()));
	KPushButton *popupMessagesTest = new KPushButton(i18n("Test"), gb_popup, "KPushButton::popupMessagesTest");
	connect(
		popupMessagesTest, SIGNAL(clicked()),
		this, SLOT(popupMessagesTestSlot()));
	l->addWidget(gb_popup);

	// warning message:
	QVGroupBox *gb_warning = new QVGroupBox(i18n("&Warning Message (recommended)"), _messagesPage, "QVGroupBox::gb_warning");
	// enabled/info
	c_warningMessageEnabled = new QCheckBox("", gb_warning, "QCheckBox::c_warningMessageEnabled");
	connect(
		c_warningMessageEnabled, SIGNAL(clicked()),
		this, SLOT(configChangeSlot()));
	// slider
	in_warningMessageDelay = new KIntNumInput(gb_warning, "KIntNumInput::in_warningMessageDelay");
	in_warningMessageDelay->setSuffix(" " + i18n("minute(s)"));
	in_warningMessageDelay->setRange(1, 60, 1, TRUE);
	connect(
		in_warningMessageDelay, SIGNAL(valueChanged(int)),
		this, SLOT(configChangeSlot()));
	connect(
		in_warningMessageDelay, SIGNAL(valueChanged(int)),
		this, SLOT(updateWarningMessageSlot()));
	KPushButton *warningMessageTest = new KPushButton(i18n("Test"), gb_warning, "KPushButton::warningMessageTest");
	connect(
		warningMessageTest, SIGNAL(clicked()),
		this, SLOT(warningMessageTestSlot()));
	l->addWidget(gb_warning);

	// custom message:
	QVGroupBox *gb_customMessage = new QVGroupBox(i18n("C&ustom Message"), _messagesPage, "QVGroupBox::gb_customMessage");
	c_customMessageEnabled = new QCheckBox(i18n("&Enabled"), gb_customMessage, "QCheckBox::c_customMessageEnabled");
	connect(
		c_customMessageEnabled, SIGNAL(clicked()),
		this, SLOT(configChangeSlot()));
	in_customMessageCommand = new KLineEdit(gb_customMessage, "KLineEdit::in_customMessageCommand");
	connect(
		in_customMessageCommand, SIGNAL(textChanged(const QString &)),
		this, SLOT(configChangeSlot()));
	QToolTip::add(in_customMessageCommand, i18n("Enter a command."));
	QWhatsThis::add(in_customMessageCommand,
		"<qt>" +
		i18n("Enter a command.") +
		"<ul>" \
		"<li><b>%appname</b> KShutDown</li>" \
		"<li><b>%text</b> " + i18n("A message text") + "</li>" \
		"<li><b>%title</b> " + i18n("The current main window title") + "</li>" \
		"</ul>"
		"</qt>"
	);
	KPushButton *b_testCustomMessage = new KPushButton(i18n("Test"), gb_customMessage, "KPushButton::b_testCustomMessage");
	connect(
		b_testCustomMessage, SIGNAL(clicked()),
		this, SLOT(customMessageTestSlot()));
	l->addWidget(gb_customMessage);

	connect(
		c_popupMessages, SIGNAL(toggled(bool)),
		in_popupTimeout, SLOT(setEnabled(bool)));
	connect(
		c_popupMessages, SIGNAL(toggled(bool)),
		popupMessagesTest, SLOT(setEnabled(bool)));
	connect(
		c_warningMessageEnabled, SIGNAL(toggled(bool)),
		in_warningMessageDelay, SLOT(setEnabled(bool)));
	connect(
		c_warningMessageEnabled, SIGNAL(toggled(bool)),
		warningMessageTest, SLOT(setEnabled(bool)));
	connect(
		c_customMessageEnabled, SIGNAL(toggled(bool)),
		in_customMessageCommand, SLOT(setEnabled(bool)));
	connect(
		c_customMessageEnabled, SIGNAL(toggled(bool)),
		b_testCustomMessage, SLOT(setEnabled(bool)));

	updateWarningMessageSlot();
}

void MSettingsDialog::initNotifyPage()
{
	_notifyPage = addPage(i18n("Notifications"), "knotify");
	QVBoxLayout *l = new QVBoxLayout(_notifyPage);
	_notifyWidget = new KNotify::KNotifyWidget(_notifyPage);
	connect(
		_notifyWidget, SIGNAL(changed(bool)),
		this, SLOT(configChangeSlot()));
	l->addWidget(_notifyWidget);
}

void MSettingsDialog::initStatsPage()
{
	_statsPage = addPage(i18n("Statistics"), "misc");
	QVBoxLayout *l = new QVBoxLayout(_statsPage, 2);

	// long format
	c_statsLongFormat =
		new QCheckBox(i18n("&Show login time, JCPU and PCPU times"), _statsPage,
		"QCheckBox::c_statsLongFormat");
	connect(
		c_statsLongFormat, SIGNAL(clicked()),
		this, SLOT(configChangeSlot()));
	l->addWidget(c_statsLongFormat);

	// toggle from field
	c_statsToggleFromField = new QCheckBox(i18n("Toggle the \"&from\" (remote hostname) field"), _statsPage, "QCheckBox::c_statsToggleFromField");
	connect(
		c_statsToggleFromField, SIGNAL(clicked()),
		this, SLOT(configChangeSlot()));
	l->addWidget(c_statsToggleFromField);

	// -
	l->addStretch();
}

void MSettingsDialog::setActionMethod(const QString &configEntry,
	const MMainWindow::Method method, const QString &command) const
{
	KConfig *conf = kapp->config();
	conf->setGroup(configEntry);
	if (method == MMainWindow::Method_UserCommand)
		conf->writeEntry("Command", command);
	conf->writeEntry("Method", method);
}

void MSettingsDialog::setConfigChanged(const bool value)
{
	actionButton(Apply)->setEnabled(value);
}

void MSettingsDialog::setWarningMessage(const bool on, const int delay)
{
	in_warningMessageDelay->setValue(delay);
	c_warningMessageEnabled->setChecked(on);
}

void MSettingsDialog::updateActions()
{
	updateItem(_shutDownItem);
	updateItem(_rebootItem);
	updateItem(_lockScreenItem);
	updateItem(_logoutItem);
	_actions->setCurrentItem(_shutDownItem);
}

void MSettingsDialog::updateItem(QListViewItem *item)
{
	if (!item)
		return;

	MMainWindow::Method m = MMainWindow::Method_KDE;
	QString c, group;

	if (item == _shutDownItem)
	{
		group = "Shut Down";
		mainWindow->getMethod(group, m, c, DEFAULT_SHUT_DOWN_COMMAND);
	}
	else if (item == _rebootItem)
	{
		group = "Reboot";
		mainWindow->getMethod(group, m, c, DEFAULT_REBOOT_COMMAND);
	}
	else if (item == _lockScreenItem)
	{
		group = "Lock Screen";
		mainWindow->getMethod(group, m, c, DEFAULT_LOCK_SCREEN_COMMAND);
	}
	else if (item == _logoutItem)
	{
		group = "Logout";
		mainWindow->getMethod(group, m, c, DEFAULT_LOGOUT_COMMAND);
	}
	else
	{
		group = QString::null;
	}

	// method
	if (m == MMainWindow::Method_KDE)
		item->setText(1, "KDE");
	else
		item->setText(1, c);

	// command before action
	if (!group.isNull())
	{
		KConfig *conf = kapp->config();
		conf->setGroup(group);
		if (conf->readBoolEntry("RunCommandBeforeAction", FALSE))
			item->setText(2, conf->readEntry("CommandBeforeAction", ""));
		else
			item->setText(2, "");
	}
}

/* private slots */

void MSettingsDialog::configChangeSlot()
{
	setConfigChanged(TRUE);
}

void MSettingsDialog::customMessageTestSlot()
{
	MiscUtils::customMessage(i18n("Test"), in_customMessageCommand->text());
}

void MSettingsDialog::editActionSlot()
{
	QListViewItem *item = _actions->currentItem();

	if (!item)
		return;

	KConfig *conf = kapp->config();
	MMainWindow::Method m = MMainWindow::Method_KDE;
	QString c;
	QString configEntry;

	MActionEditDialog *dialog =
		new MActionEditDialog(this, "MActionEditDialog::dialog");

	Q_CHECK_PTR(dialog);

	dialog->methods->insertItem("KDE");

	if (item == _shutDownItem)
	{
		configEntry = "Shut Down";
		mainWindow->getMethod(configEntry, m, c, DEFAULT_SHUT_DOWN_COMMAND);
		dialog->methods->insertItem(DEFAULT_SHUT_DOWN_COMMAND);
		dialog->setAction(MMainWindow::What_ShutDown);
	}
	else if (item == _rebootItem)
	{
		configEntry = "Reboot";
		mainWindow->getMethod(configEntry, m, c, DEFAULT_REBOOT_COMMAND);
		dialog->methods->insertItem(DEFAULT_REBOOT_COMMAND);
		dialog->setAction(MMainWindow::What_Reboot);
	}
	else if (item == _lockScreenItem)
	{
		configEntry = "Lock Screen";
		mainWindow->getMethod(configEntry, m, c, DEFAULT_LOCK_SCREEN_COMMAND);
		dialog->methods->insertItem(DEFAULT_LOCK_SCREEN_COMMAND);
		dialog->setAction(MMainWindow::What_LockScreen);
	}
	else if (item == _logoutItem)
	{
		configEntry = "Logout";
		mainWindow->getMethod(configEntry, m, c, DEFAULT_LOGOUT_COMMAND);
		dialog->methods->insertItem(DEFAULT_LOGOUT_COMMAND);
		dialog->setAction(MMainWindow::What_Logout);
	}
	dialog->methods->insertItem(i18n("User Command"));

	// method
	dialog->userCommand = c;
	dialog->setMethod(m);

	// command before action
	conf->setGroup(configEntry);
	dialog->runCommandBeforeAction->setChecked(conf->readBoolEntry("RunCommandBeforeAction", FALSE));
	dialog->commandBeforeAction->setText(conf->readEntry("CommandBeforeAction", ""));
	dialog->commandBeforeActionPause->setValue(conf->readNumEntry("CommandBeforeActionPause", 10));

	bool runCommand = dialog->runCommandBeforeAction->isChecked();
	dialog->commandBeforeAction->setEnabled(runCommand);
	dialog->commandBeforeActionPause->setEnabled(runCommand);
	dialog->testCommandBeforeAction->setEnabled(runCommand);

	if (dialog->exec() == QDialog::Accepted)
	{
		// method
		m = (MMainWindow::Method)dialog->methods->currentItem();
		if (m == MMainWindow::Method_UserCommand)
			c = dialog->command->text();
		setActionMethod(configEntry, m, c);

		// command before action
		conf->setGroup(configEntry);
		conf->writeEntry("RunCommandBeforeAction", dialog->runCommandBeforeAction->isChecked());
		conf->writeEntry("CommandBeforeAction", dialog->commandBeforeAction->text());
		conf->writeEntry("CommandBeforeActionPause", dialog->commandBeforeActionPause->value());

		updateItem(item);
		setConfigChanged(TRUE);
	}

	delete dialog;
}

void MSettingsDialog::enableAllMessagesSlot()
{
	KMessageBox::enableAllMessages();
	b_enableAllMessages->setEnabled(FALSE);
}

void MSettingsDialog::enableLockSettingsButtonSlot()
{
	b_lockScreenSettings->setEnabled(TRUE);
}

void MSettingsDialog::enableLogoutSettingsButtonSlot()
{
	b_logoutSettings->setEnabled(TRUE);
}

void MSettingsDialog::lockSettingsSlot()
{
	b_lockScreenSettings->setEnabled(FALSE);
	QTimer::singleShot(5000, this, SLOT(enableLockSettingsButtonSlot()));
	KRun::run("kcmshell screensaver", KURL::List());
}

void MSettingsDialog::logoutSettingsSlot()
{
	b_logoutSettings->setEnabled(FALSE);
	QTimer::singleShot(5000, this, SLOT(enableLogoutSettingsButtonSlot()));
	KRun::run("kcmshell kcmsmserver", KURL::List());
}

void MSettingsDialog::modifyExtrasSlot()
{
	mainWindow->extras()->modify();
}

void MSettingsDialog::okClickedSlot()
{
	// general
	kshutdownrc->timeRemember = c_timeRemember->isChecked();
	kshutdownrc->schedEnabled = c_schedulerEnabled->isChecked();

	// notifications
	_notifyWidget->save();

	// automation
	if (c_autoLock->isChecked())
	{
		Links::createLink(
			KGlobalSettings::autostartPath() + "/" + AUTO_LOCK_FILE_NAME,
			"kshutdown --init --lock",
			"kshutdown"
		);
	}
	else
	{
		Links::removeLink(
			KGlobalSettings::autostartPath() + "/" + AUTO_LOCK_FILE_NAME
		);
	}

	kshutdownrc->cdTrayClose = c_cdTrayClose->isChecked();
	kshutdownrc->cdTrayCloseCommand = in_cdTrayCloseCommand->text();

	// messages
	kshutdownrc->confirmAction = c_confirmAction->isChecked();
	KConfig *conf = kapp->config();
	conf->setGroup("Notification Messages");
	if (kshutdownrc->confirmAction)
		conf->deleteEntry("ShutDown");
	else
		conf->writeEntry("ShutDown", "yes");
	kshutdownrc->popupMessages = c_popupMessages->isChecked();
	kshutdownrc->popupTimeout = in_popupTimeout->value();
	kshutdownrc->warningMessageEnabled = c_warningMessageEnabled->isChecked();
	kshutdownrc->warningMessageDelay = in_warningMessageDelay->value();
	kshutdownrc->customMessageEnabled = c_customMessageEnabled->isChecked();
	kshutdownrc->customMessageCommand = in_customMessageCommand->text();

	// statistics
	kshutdownrc->statsLongFormat = c_statsLongFormat->isChecked();
	kshutdownrc->statsToggleFromField = c_statsToggleFromField->isChecked();

	// save config to disk
	saveDialogSize("Settings");
	kshutdownrc->write();

	setConfigChanged(FALSE);
}

void MSettingsDialog::popupMessagesTestSlot()
{
	mainWindow->tray()->testMessage(in_popupTimeout->value());
}

void MSettingsDialog::seeAlsoClickedSlot()
{
	showPage(pageIndex(_actionsPage));
	_actions->setFocus();
}

void MSettingsDialog::updateWarningMessageSlot()
{
	QString s =
		QString(i18n("Show warning message %1 minute(s) &before action"))
			.arg(in_warningMessageDelay->value());
	c_warningMessageEnabled->setText(s);
}

void MSettingsDialog::warningMessageTestSlot()
{
	mainWindow->setCancelled(FALSE);
	MMessageDialog *messageDialog = new MMessageDialog(
		TRUE, in_warningMessageDelay->value(),
		MMainWindow::What_Nothing, i18n("Test"),
		this, "MMessageDialog::messageDialog"
	);

	Q_CHECK_PTR(messageDialog);

	messageDialog->exec();
	delete messageDialog;
}

/* protected slots */

void MSettingsDialog::slotDefault()
{
	int index = activePageIndex();

	if (index == -1)
		return;

	// general
	if (index == pageIndex(_generalPage))
	{
		c_timeRemember->setChecked(DEFAULT_TIME_REMEMBER);
		c_schedulerEnabled->setChecked(DEFAULT_SCHED_ENABLED);
		setConfigChanged(TRUE);
	}
	// notifications
	else if (index == pageIndex(_notifyPage))
	{
		_notifyWidget->resetDefaults(TRUE);
		setConfigChanged(TRUE);
	}
	// automation
	else if (index == pageIndex(_autoPage))
	{
		c_autoLock->setChecked(DEFAULT_AUTO_LOCK);
		c_cdTrayClose->setChecked(DEFAULT_CD_TRAY_CLOSE);
		in_cdTrayCloseCommand->setText(DEFAULT_CD_TRAY_CLOSE_COMMAND);
		setConfigChanged(TRUE);
	}
	// messages
	else if (index == pageIndex(_messagesPage))
	{
		c_confirmAction->setChecked(DEFAULT_CONFIRM_ACTION);
		c_popupMessages->setChecked(DEFAULT_POPUP_MESSAGES);
		in_popupTimeout->setValue(DEFAULT_POPUP_TIMEOUT);
		setWarningMessage(
			DEFAULT_WARNING_MESSAGE_ENABLED,
			DEFAULT_WARNING_MESSAGE_DELAY
		);
		c_customMessageEnabled->setChecked(DEFAULT_CUSTOM_MESSAGE_ENABLED);
		in_customMessageCommand->setText(DEFAULT_CUSTOM_MESSAGE_COMMAND);
		setConfigChanged(TRUE);
	}
	// actions
	else if (index == pageIndex(_actionsPage))
	{
		if (disabledByAdmin("actions"))
			return;

		if (KMessageBox::warningYesNo(
			this,
			i18n("Restore default settings for this page?"),
			caption()) != KMessageBox::Yes
		)
			return;

		KConfig *conf = kapp->config();
		conf->deleteGroup("Shut Down");
		conf->deleteGroup("Reboot");
		conf->deleteGroup("Lock Screen");
		conf->deleteGroup("Logout");
		updateActions();
		setConfigChanged(TRUE);
	}
	// stats
	else if (index == pageIndex(_statsPage))
	{
		c_statsLongFormat->setChecked(DEFAULT_STATS_LONG_FORMAT);
		c_statsToggleFromField->setChecked(DEFAULT_STATS_TOGGLE_FROM_FIELD);
		setConfigChanged(TRUE);
	}
}
