/*
	msystemtray.cpp - system tray
	Copyright (C) 2003  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "configuration.h"
#include "miscutils.h"
#include "msystemtray.h"
#include "mwizard.h"

#include <qlistbox.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <qvbox.h>

#include <kiconloader.h>
#include <klocale.h>
#include <kpassivepopup.h>
#include <kpixmap.h>
#include <kpixmapeffect.h>
#include <kpopupmenu.h>
#include <kpushbutton.h>
#include <kstdguiitem.h>
#include <kstringhandler.h>

/* constructor */

MSystemTray::MSystemTray(QWidget *parent, const char *name)
	: KSystemTray(parent, name),
	// init
	_idle(FALSE), // setIdle(TRUE) will toggle it
	_test(FALSE),
	_flashCount(0),
	_popup(0), // initialized in popup()
	_saveIcon(0)
{
// TODO: 0.7: mouse wheel: decrease/increase end time by 5min.

	// init flash timer
	_flashTimer = new QTimer(this);
	connect(
		_flashTimer, SIGNAL(timeout()),
		this, SLOT(flashTimeoutSlot()));

	// init icons
	KPixmap pm(kapp->miniIcon());
	_activeIcon =
		new KPixmap(KPixmapEffect::fade(pm, 0.7f, QColor(255, 255, 0)));

	_flashIcon =
		new KPixmap(KPixmapEffect::fade(pm, 0.5f, QColor(255, 255, 255)));

	setIdle(TRUE);

	show();

	// add some items to the context menu
	KPopupMenu *cm = contextMenu();
// TODO: 0.7: add "Extras" menu

	_actionsMenu = new KPopupMenu(cm, "KPopupMenu::group");
	_actionsMenu->insertTitle(SmallIcon("kshutdown"), i18n("Actions (no delay!)"));
	connect(
		_actionsMenu, SIGNAL(aboutToShow()),
		this, SLOT(actionsMenuAboutToShowSlot()));

	// shut down
	_shutDownItem = _actionsMenu->insertItem(
		mainWindow->getActionIcon(MMainWindow::What_ShutDown),
		i18n("&Turn Off Computer"),
		this, SLOT(shutDownSlot())
	);
	// reboot
	_rebootItem = _actionsMenu->insertItem(
		mainWindow->getActionIcon(MMainWindow::What_Reboot),
		i18n("&Restart Computer"),
		this, SLOT(rebootSlot())
	);
	// -
	_actionsMenu->insertSeparator();
	// lock screen
	_lockScreenItem = _actionsMenu->insertItem(
		mainWindow->getActionIcon(MMainWindow::What_LockScreen),
		i18n("&Lock Session"),
		this, SLOT(lockScreenSlot())
	);
	_actionsMenu->insertSeparator();
	// logout
	_logoutItem = _actionsMenu->insertItem(
		mainWindow->getActionIcon(MMainWindow::What_Logout),
		i18n("&End Current Session"),
		this, SLOT(logoutSlot())
	);

	_actionsMenu->insertTearOffHandle();

	// 0 - title
	cm->insertItem(i18n("&Immediate Action"), _actionsMenu, -1, 1);
	// Wizard...
	_wizardItem = cm->insertItem(
		SmallIcon("wizard"),
		i18n("&Wizard..."),
		this, SLOT(wizardSlot()), 0, -1, 2
	);
	// Cancel
	_cancelItem = cm->insertItem(
		SmallIcon("button_cancel"),
		i18n("&Cancel"),
		mainWindow, SLOT(cancelSlot()), 0, -1, 3
	);
	// -
	cm->insertSeparator(4);

	// Help
	cm->insertItem(
		SmallIcon("help"),
		i18n("&Help"),
		mainWindow->helpMenu(), -1, 5
	);

	// Settings
	cm->insertItem(
		SmallIcon("configure"),
		i18n("C&onfigure..."),
		mainWindow, SLOT(settingsSlot()), 0, -1, 6
	);

	cm->insertTearOffHandle();

	// quit signal
	connect(
		this, SIGNAL(quitSelected()),
		mainWindow, SLOT(closeSlot()));
}

/* destructor */

MSystemTray::~MSystemTray()
{
	if (_activeIcon)
		delete _activeIcon;
	if (_flashIcon)
		delete _flashIcon;
	if (_popup)
		delete _popup;
	if (_saveIcon)
		delete _saveIcon;
}

/* public */

void MSystemTray::flashIcon()
{
	if (_saveIcon)
		delete _saveIcon;
	_saveIcon = new QPixmap;
	*_saveIcon = *pixmap();
	_flashCount = 0;
	setPixmap(*_flashIcon);
	_flashTimer->start(200);
}

void MSystemTray::hidePopupMessages()
{
	if (_popup)
		_popup->hide();
}

KPassivePopup *MSystemTray::popup(const QString &caption)
{
	if (_popup == 0)
	{
		_popup = new KPassivePopup(this, "KPassivePopup::_popup");
		QToolTip::add(_popup, i18n("Click to close"));
		QVBox *popupView = new QVBox(_popup, "QVBox::popupView");
		popupView->setSpacing(2);

		// icon and caption
		QHBox *iconAndCaptionBox = new QHBox(popupView, "QHBox::iconAndCaptionBox");
		// icon
		QLabel *popupIcon = new QLabel(iconAndCaptionBox, "QLabel::popupIcon");
		popupIcon->setPixmap(kapp->miniIcon());
		// caption
		l_popupCaption = new QLabel(iconAndCaptionBox, "QLabel::l_popupCaption");

		QHBox *listAndButtonsBox = new QHBox(popupView, "QHBox::listAndButtonsBox");
		// messages list
		lb_popupMessagesList =
			new QListBox(listAndButtonsBox, "QListBox::lb_popupMessagesList");
		lb_popupMessagesList->setMinimumSize(320, 64);
		QToolTip::add(lb_popupMessagesList, i18n("Messages"));
		// cancel button
		b_cancel = new KPushButton(listAndButtonsBox, "KPushButton::b_cancel");
		b_cancel->setGuiItem(KStdGuiItem::cancel());
		b_cancel->setFlat(TRUE);
		connect(
			b_cancel, SIGNAL(clicked()),
			this, SLOT(cancelSlot()));
		QToolTip::add(b_cancel, i18n("Cancel an active action"));

		_popup->setView(popupView);
	}

	// set popup message caption
	QString s =  caption.isEmpty() ? mainWindow->caption() : caption;
	l_popupCaption->setText(KStringHandler::csqueeze(s, 100));
	QToolTip::add(l_popupCaption, s);

	return _popup;
}

void MSystemTray::popupMessage(const Reason reason, const QString &caption,
	const QString &text, const int timeout, const bool test)
{
	if (!kshutdownrc->popupMessages && !test && (reason != Reason_Error))
		return;

	_test = test;

	popup(caption); // init popup
	if (reason == Reason_Error)
		popup()->setTimeout(60 * 1000); // 1m
	else if (timeout == -1)
		popup()->setTimeout(kshutdownrc->popupTimeout * 1000);
	else
		popup()->setTimeout(timeout * 1000);
	QString t = QTime::currentTime().toString() + ": " + text;
	switch (reason)
	{
		case Reason_Info:
			lb_popupMessagesList->insertItem(SmallIcon("messagebox_info"), t);
			break;
		case Reason_Warning:
			lb_popupMessagesList->insertItem(
				SmallIcon("messagebox_warning"), t
			);
			break;
		case Reason_Error:
			lb_popupMessagesList->insertItem(
				SmallIcon("messagebox_critical"), t
			);
			break;
		default:
			lb_popupMessagesList->insertItem(t);
	}
	// focus last item
	lb_popupMessagesList->setCurrentItem(lb_popupMessagesList->count() - 1);
	// enable/disable "cancel" button
	b_cancel->setEnabled(mainWindow->isActive());

	popup()->show();
}

void MSystemTray::popupMessage(const Reason reason, const QString &text)
{
	popupMessage(reason, mainWindow->caption(), text);
}

void MSystemTray::setIdle(const bool yes)
{
	if (_idle == yes)
		return;

	_idle = yes;
	// idle
	if (_idle)
		setPixmap(kapp->miniIcon());
	// busy
	else
		setPixmap(*_activeIcon);
}

void MSystemTray::testMessage(const int timeout)
{
	popupMessage(
		Reason_Info,
		i18n("Test"),
		i18n("Test"),
		timeout,
		TRUE // test
	);
}

/* protected */

void MSystemTray::contextMenuAboutToShow(KPopupMenu *menu)
{
	menu->setItemEnabled(_wizardItem, MWizard::canRun());
	if (mainWindow->isActive())
	{
		menu->changeItem(
			_cancelItem,
			i18n("&Cancel") + ": " + mainWindow->getCurrentActionName()
		);
		menu->setItemEnabled(_cancelItem, TRUE);
	}
	else
	{
		menu->changeItem(_cancelItem, i18n("&Cancel"));
		menu->setItemEnabled(_cancelItem, FALSE);
	}
}

void MSystemTray::mousePressEvent(QMouseEvent *e)
{
	// middle mouse button - display actions menu
// TODO: 0.7: configure middle button action
	if (e->button() == MidButton)
		_actionsMenu->popup(e->globalPos());
	else
		KSystemTray::mousePressEvent(e);
}

void MSystemTray::doExecAction(const MMainWindow::What action)
{
	if (MiscUtils::confirmAction(action))
	{
		bool stopTimer = action != MMainWindow::What_LockScreen;
		mainWindow->execAction(action, stopTimer);
	}
}

/* private slots */

void MSystemTray::actionsMenuAboutToShowSlot()
{
	_actionsMenu->setItemEnabled(_shutDownItem,
		MiscUtils::isActionEnabled(MMainWindow::What_ShutDown));
	_actionsMenu->setItemEnabled(_rebootItem,
		MiscUtils::isActionEnabled(MMainWindow::What_Reboot));
	_actionsMenu->setItemEnabled(_lockScreenItem,
		MiscUtils::isActionEnabled(MMainWindow::What_LockScreen));
	_actionsMenu->setItemEnabled(_logoutItem,
		MiscUtils::isActionEnabled(MMainWindow::What_Logout));
}

void MSystemTray::cancelSlot()
{
	if (!_test)
		mainWindow->cancel();
}

void MSystemTray::flashTimeoutSlot()
{
	_flashCount++;

	if ((_flashCount % 2) == 0)
		setPixmap(*_saveIcon);
	else
		setPixmap(*_flashIcon);

	if (_flashCount == 6)
		_flashTimer->stop();
}

void MSystemTray::lockScreenSlot()
{
	doExecAction(MMainWindow::What_LockScreen);
}

void MSystemTray::logoutSlot()
{
	doExecAction(MMainWindow::What_Logout);
}

void MSystemTray::rebootSlot()
{
	doExecAction(MMainWindow::What_Reboot);
}

void MSystemTray::shutDownSlot()
{
	doExecAction(MMainWindow::What_ShutDown);
}

void MSystemTray::wizardSlot()
{
	if (mainWindow->runWizard())
		mainWindow->makeVisible();
}
