/*
	links.cpp - A link creator/remover
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "actions.h"
#include "links.h"
#include "miscutils.h"

#include <qcombobox.h>
#include <qfile.h>
#include <qlabel.h>

#include <kapplication.h>
#include <kdesktopfile.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpushbutton.h>
#include <kstandarddirs.h>

// public

Links::Links(QWidget *parent)
	: QVBox(parent, "Links")
{
	setSpacing(5);
	int alignment = AlignVCenter;
	alignment |= kapp->reverseLayout() ? AlignLeft : AlignRight;


	// location
	QHBox *w_location = new QHBox(this);
	w_location->setSpacing(5);
	QLabel *l_location = new QLabel(i18n("Location where to create the link:"), w_location);
	l_location->setAlignment(alignment);
	cb_location = new QComboBox(w_location, "QComboBox::cb_location");
	cb_location->setFocusPolicy(StrongFocus);
	cb_location->insertItem(SmallIcon("desktop"), i18n("Desktop"));
	cb_location->insertItem(SmallIcon("kmenu"), i18n("K Menu"));
	l_location->setBuddy(cb_location);
	connect(cb_location, SIGNAL(activated(int)), SLOT(slotActivated(int)));

	// type
	QHBox *w_type = new QHBox(this);
	w_type->setSpacing(5);
	QLabel *l_type = new QLabel(i18n("Type of the link:"), w_type);
	l_type->setAlignment(alignment);
	cb_type = new QComboBox(w_type, "QComboBox::cb_type");
	cb_type->setFocusPolicy(StrongFocus);
	cb_type->insertItem(kapp->miniIcon(), "KShutDown");

	// NOTE: slotAddRemoveLink()
	cb_type->insertItem(SmallIcon("exit"), i18n("Standard Logout Dialog"));

	cb_type->insertItem(ks_actions->getIcon(Action::ShutDown), ks_actions->getName(Action::ShutDown));
	cb_type->insertItem(ks_actions->getIcon(Action::Reboot), ks_actions->getName(Action::Reboot));
	cb_type->insertItem(ks_actions->getIcon(Action::LockScreen), ks_actions->getName(Action::LockScreen));
	cb_type->insertItem(ks_actions->getIcon(Action::Logout), ks_actions->getName(Action::Logout));
	l_type->setBuddy(cb_type);
	connect(cb_type, SIGNAL(activated(int)), SLOT(slotActivated(int)));

	// add/remove link
	b_addRemoveLink = new KPushButton(this, "KPushButton::b_addRemoveLink");
	connect(b_addRemoveLink, SIGNAL(clicked()), SLOT(slotAddRemoveLink()));

	updateAddRemoveButton();
}

void Links::createLink(const QString &path, const QString &command, const QString &icon, const QString &name, const QString &comment)
{
	if (path.isNull())
		return;

	KDesktopFile *df = new KDesktopFile(path);
	df->setGroup("Desktop Entry");
	df->writeEntry("Comment", comment);
	df->writeEntry("Encoding", "UTF-8");
	df->writeEntry("Exec", command);
	df->writeEntry("GenericName", i18n("System Shut Down Utility"));
	df->writeEntry("Icon", icon);
	df->writeEntry("Name", name);
	df->writeEntry("StartupNotify", "false");
	df->writeEntry("Type", "Application");
	delete df;

	if (!QFile::exists(path))
	{
		KMessageBox::error(
			0,
			MiscUtils::HTML(i18n("Could not create file <b>%1</b>!").arg(path))
		);
	}
}

void Links::removeLink(const QString &path)
{
	if (QFile::exists(path) && !QFile::remove(path))
	{
		KMessageBox::error(
			0,
			MiscUtils::HTML(i18n("Could not remove file <b>%1</b>!").arg(path))
		);
	}
}

// private

QString Links::getCurrentLocationPath() const
{
	QString path;
	switch (cb_location->currentItem())
	{
		case 0:
			path = KGlobalSettings::desktopPath() + "/";
			break;
		case 1:
			path = locateLocal("apps", "") + "/";
			break;
		default:
			return QString::null;
	}
	switch (cb_type->currentItem())
	{
		case 0: return path += "kshutdown-classic.desktop";
		case 1: return path += "kshutdown-standard.desktop";
		case 2: return path += "kshutdown-shutdown.desktop";
		case 3: return path += "kshutdown-reboot.desktop";
		case 4: return path += "kshutdown-lock.desktop";
		case 5: return path += "kshutdown-logout.desktop";
		default: return QString::null;
	}
}

QString Links::getCurrentTypeCommand() const
{
	switch (cb_type->currentItem())
	{
		case 0: return "kshutdown";
		case 1: return "kshutdown --standard";
		case 2: return "kshutdown --confirm --shutdown";
		case 3: return "kshutdown --confirm --reboot";
		case 4: return "kshutdown --confirm --lock";
		case 5: return "kshutdown --confirm --logout";
		default: return QString::null;
	}
}

QString Links::getCurrentTypeIcon() const
{
	switch (cb_type->currentItem())
	{
		case 0: return "kshutdown";
		case 1: return "exit";
		// sync. with Action::getIcon
		case 2: return "exit";
		case 3: return "reload";
		case 4: return "lock";
		case 5: return "undo";
		default: return QString::null;
	}
}

void Links::updateAddRemoveButton() {
	if (QFile::exists(getCurrentLocationPath())) {
		b_addRemoveLink->setIconSet(SmallIcon("editdelete"));
		b_addRemoveLink->setText(i18n("Remove Link"));
	}
	else {
		b_addRemoveLink->setIconSet(SmallIcon("filenew"));
		b_addRemoveLink->setText(i18n("Add Link"));
	}
}

// private slots

void Links::slotActivated(int/* index*/) {
	updateAddRemoveButton();
}

void Links::slotAddRemoveLink() {
	if (QFile::exists(getCurrentLocationPath())) {
		removeLink(getCurrentLocationPath());
	}
	else {
		createLink(
			getCurrentLocationPath(),
			getCurrentTypeCommand(),
			getCurrentTypeIcon(),
			(cb_type->currentItem() == 1) ? i18n("Logout") : cb_type->currentText(),
			cb_type->currentText()
		);
	}
	updateAddRemoveButton();
}
