/***************************************************************************
 *   Copyright (C) 2004-2005 by Giovanni Venturi                           *
 *   gventuri73@tiscali.it                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include "capturedialog.h"

#include <qcombobox.h>
#include <qgroupbox.h>
#include <qcheckbox.h> 
#include <qlabel.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qwhatsthis.h>

#include <kdebug.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kpushbutton.h>
#include <kmessagebox.h>

CaptureDialog::CaptureDialog(QWidget *parent)
 : KDialogBase( Plain, i18n("New Capture"), User1|Cancel, Cancel, parent, "newcapture",
                /*modal*/true, /*separator*/false, KGuiItem (i18n("&Start"), "ok") )
{
    KIconLoader* il = KGlobal::iconLoader();

    QVBoxLayout* mainLay = new QVBoxLayout( plainPage() );
    mainLay->activate();

    // the groupbox to select the interface to sniff
    QGroupBox* groupBox = new QGroupBox( plainPage(), "groupBox" );
    groupBox->setTitle( i18n( "Select Interface" ) );
    mainLay->addWidget( groupBox );

    mainLay->addItem( new QSpacerItem( 10, 10,  QSizePolicy::Minimum, QSizePolicy::Minimum ) );

    // TODO: use a 'Other options >>' button to configure other options
    m_checkBox = new QCheckBox( i18n("Show all options"), plainPage() );
    QWhatsThis::add( m_checkBox,
                     i18n( "<p>Open the Preference Dialog where you can "
                           "choose the settings about the capture.<br><br>"
                           "In case you don't request all the options, the "
                           "capture will start using the default "
                           "settings.</p>" ) );
    mainLay->addWidget( m_checkBox );
    // disabling this checbox since it's - for the moment - no useful to us
    m_checkBox->setEnabled( false );

    QVBoxLayout* groupLay = new QVBoxLayout( groupBox, 15, 6 );
    //groupLay->activate();
    groupLay->addItem( new QSpacerItem( 10,  10,  QSizePolicy::Minimum,  QSizePolicy::Minimum ) );

    QHBoxLayout* ifaceSelectLay = new QHBoxLayout();
    ifaceSelectLay->activate();
    groupLay->addLayout( ifaceSelectLay );

    QLabel* textLabel = new QLabel( groupBox, "textLabel" );
    textLabel->setText( i18n( "I&nterface:" ) );

    m_interface = new QComboBox( FALSE, groupBox, "m_interface" );
    textLabel->setBuddy( m_interface );
    QWhatsThis::add( m_interface,
                     i18n( "<p>Contains the list of network interfaces "
                           "detected on your system.<br><br>"
                           "The special interface <b>any</b> captures network "
                           "packets from all network interfaces of your "
                           "system.</p>" ) );

    KPushButton* m_redetect = new KPushButton( groupBox, "m_rescan" );
    m_redetect->setText( i18n( "Re&detect Interfaces" ) );
    m_redetect->setIconSet( il->loadIconSet( "reload", KIcon::Small ) );
    QWhatsThis::add( m_redetect, i18n( "Update the Interface ComboBox scanning again your system about its network interfaces." ) );

    ifaceSelectLay->addWidget( textLabel );
    ifaceSelectLay->addItem( new QSpacerItem( 10,  10,  QSizePolicy::Expanding,  QSizePolicy::Minimum ) );
    ifaceSelectLay->addWidget( m_interface );
    ifaceSelectLay->addItem( new QSpacerItem( 10,  10,  QSizePolicy::Minimum,  QSizePolicy::Minimum ) );
    ifaceSelectLay->addWidget( m_redetect );

    QGridLayout* detailLay = new QGridLayout( 2, 2, 5 );
    detailLay->activate();
    groupLay->addLayout( detailLay );

    QLabel* label = new QLabel( groupBox, "label" );
    label->setText( i18n( "IP address:" ) );
    detailLay->addWidget( label, 0, 0 );

    m_ipaddress = new QLabel( groupBox, "m_ipaddress" );
    QWhatsThis::add( m_ipaddress, i18n( "Shows the IP address of the selected interface, if there is one." ) );
    detailLay->addWidget( m_ipaddress, 0, 1 );

    label = new QLabel( groupBox, "label" );
    label->setText( i18n( "Netmask:" ) );
    detailLay->addWidget( label, 1, 0 );

    m_netmask = new QLabel( groupBox, "m_netmask" );
    QWhatsThis::add( m_netmask, i18n( "Shows the netmask of the selected interface, if there is one." ) );
    detailLay->addWidget( m_netmask, 1, 1 );

    groupLay->addItem( new QSpacerItem( 10, 50, QSizePolicy::Minimum, QSizePolicy::Expanding ) );    

    setButtonWhatsThis (User1, i18n( "Start a new network packet-capture session." ));

    // fill the Interface ComboBox
    m_interface->insertStringList(m_netDevice.interfaceList());
    setData( m_interface->currentText() );

    resize( QSize( 350, 130 ) );

    connect( m_redetect, SIGNAL( clicked() ), this, SLOT( slotRedetect() ) );
    connect( m_interface, SIGNAL( activated(const QString&) ), this, SLOT( setData(const QString&) ) );
    connect( m_checkBox, SIGNAL ( clicked () ), this, SLOT( slotOption() ) );

    m_strUsedInterface = QString::null;  // no selection at startup
}


CaptureDialog::~CaptureDialog()
{
    m_interface->clear();
}


/*!
    \fn CaptureDialog::slotUser1()
 */
void CaptureDialog::slotUser1()
{
    m_strUsedInterface = m_interface->currentText();  // set the selected interface just when click Start button
    slotOk();
}


void CaptureDialog::slotOption()
{
    //if (m_checkBox->isChecked())
    KMessageBox::sorry(this, i18n("This feature is not yet available.") );
    m_checkBox->setChecked(FALSE);
}


void CaptureDialog::slotRedetect()
{
    m_netDevice.detect();
    m_interface->clear();
    m_interface->insertStringList(m_netDevice.interfaceList());
    setData( m_interface->currentText() );
}


void CaptureDialog::setData( const QString & strInterface )
{
    DeviceData data = m_netDevice.interfaceData(strInterface);

    // updating IP address
    QString str = data.getAddress();
    if ( str.isEmpty() )
        str = i18n("not available");
    m_ipaddress->setText( str );

    // updating netmask
    str = data.getNetmask();
    if ( str.isEmpty() )
        str = i18n("not available");
    m_netmask->setText( str );
}

#include "capturedialog.moc"
