/***************************************************************************
 *   Copyright (C) 2004-2005 by Giovanni Venturi                           *
 *   gventuri73@tiscali.it                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <unistd.h>

#include <qdragobject.h>
#include <kprinter.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qtooltip.h>
#include <qpixmap.h>
#include <qfile.h>

#include <kuniqueapplication.h>
#include <ksystemtray.h>
#include <kpopupmenu.h>
#include <kglobal.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kdeversion.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <kkeydialog.h>
#include <kaccel.h>
#include <kio/netaccess.h>
#include <kfiledialog.h>
#include <kconfig.h>
#include <kurl.h>
#include <kurldrag.h>
#include <kio/job.h>

#include <kedittoolbar.h>

#include <kmainwindow.h>
#include <kstdaccel.h>
#include <kstdaction.h>

#include <kmessagebox.h>
#include <kpassivepopup.h>
#include <kactionclasses.h>
#include <kaction.h>
#include <ktempfile.h>

#include <kdebug.h>

#include "ksniffer.h"
#include "version.h"
#include "capturedialog.h"
#include "pref.h"
#include "sniffer.h"
#include "packetmanager.h"
#include "packet.h"
#include "ksniffermainwidget.h"
//#include "ksnifferiface.h"

KSniffer::KSniffer()
  : /*KSnifferIface("KSniffer-Interface"),*/ KMainWindow( 0, "KSniffer", 0 )
{
  // accept dnd
  setAcceptDrops(true);

  m_mainWidgetUI = new KSnifferMainWidget( this );

  // tell the KMainWindow that this is indeed the main widget
  setCentralWidget(m_mainWidgetUI);

  // then, setup our actions
  setupActions();

  // writing application settings
  readSettings();

  // and a status bar
  statusBar()->show();

  setPlainCaption( i18n("KSniffer - The KDE Network Sniffer") );

  resize(780,520);  // default optimized for 800x600

  changeStatusbar(i18n( "Ready." ));

  // apply the saved mainwindow settings, if any, and ask the mainwindow
  // to automatically save settings if changed: window size, toolbar
  // position, icon size, etc.
  setAutoSaveSettings();

  // allow the view to change the statusbar and caption
  connect(m_mainWidgetUI, SIGNAL(signalChangeStatusbar(const QString&)),
          this,   SLOT(changeStatusbar(const QString&)));
  connect(m_mainWidgetUI, SIGNAL(signalChangeCaption(const QString&)),
          this,   SLOT(changeCaption(const QString&)));

  // we need a PacketManager to store packets and manage them for the View
  m_packets = new PacketManager();

  m_options = new CaptureOptions();
  m_sniffer = new Sniffer();

  // we define how to manage new captured packets:
    //  1 - store the packet in a PacketList (calling savePacket)
    connect( m_sniffer, SIGNAL(gotPacket(ptrPacketType, struct pcap_pkthdr, long, int, int)),
             m_packets, SLOT(savePacket(ptrPacketType, struct pcap_pkthdr, long, int, int)) );
    //  2 - display the packet into the View
    connect( m_packets, SIGNAL(savedPacket(int, PacketManager *)), m_mainWidgetUI, SLOT(displayPacket(int, PacketManager *)) );

  // Setting System tray icon
  m_sysTray = new KSystemTray( this );
  m_sysTray->setTextFormat( Qt::RichText );
  m_sysTray->setPixmap( m_sysTray->loadIcon( "ksniffer" ) );
  QToolTip::add( m_sysTray, QString( "KSniffer " ) + APPVER );
  connect( m_sysTray, SIGNAL(quitSelected()), SLOT(slotQuit()) );
  m_sysTray->show();

  // Create popup menu for tray system
  KPopupMenu *m_popupMenu = new KPopupMenu ( this );

  // Sub menu: "Capture" item
  KPopupMenu *m_popupCapture = new KPopupMenu ( this );
  // the subitems...
  m_actNew->plug( m_popupCapture );
  m_actOpen->plug( m_popupCapture );
  m_actSave->plug( m_popupCapture );
  m_actSaveAs->plug( m_popupCapture );
  m_actStop->plug( m_popupCapture );
  m_actPrint->plug( m_popupCapture );

  // Sub menu: "Help" item
  KPopupMenu *m_popupHelp = helpMenu();
  m_popupMenu->insertItem ( i18n ("&Help"), m_popupHelp );

  // put items as LIFO
  m_sysTray->contextMenu()->insertItem( i18n ("&Help"), m_popupHelp, 0,1 );
  m_sysTray->contextMenu()->insertItem( i18n("&Capture"), m_popupCapture, 0, 1 );

  m_action = DO_NOTHING;  // no action when saving job finished
  m_strFilename = "";     // not defined file name to save, it has to choose by user at first save
  m_strSavedFilename = "";
  m_needToSave = false;   // no captured-packets available yet
}


KSniffer::~KSniffer()
{
  m_packets->clearList();

  delete m_sysTray;
  delete m_sniffer;
  delete m_options;
  delete m_packets;
  delete m_actStop;
  delete m_mainWidgetUI;
  delete m_actOpenRecent;
}


void KSniffer::load(const KURL& url)
{
  QString target;
  // the below code is what you should normally do.  in this
  // example case, we want the url to our own.  you probably
  // want to use this code instead for your app

  #if 0
  // download the contents
  if (KIO::NetAccess::download(url, target))
  {
    // set our caption
    setCaption(url);

    // load in the file (target is always local)
    loadFile(target);

    // and remove the temp file
    KIO::NetAccess::removeTempFile(target);
  }
  #endif

  //m_mainWidgetUI->openURL( url );

  // maybe a pcap file want to be loaded
  if ( m_sniffer->checkFile( url.path() ) == Sniffer::RIGHT_FORMAT_FILE )
  {
    if (m_packets->packetAvailable())
    {
      // ask if continue loading if you have packets available
      if ( KMessageBox::warningContinueCancel( this,
             i18n("<p>Packet-capture session data available. If you want to drop the" \
                  " dragged file you are going to lose all captured data.</p>" \
                  "<p>Do you really want to continue?</p>" ) ) == KMessageBox::Cancel )
        return;
      else
      {
        // it appends the file captured data to the displayed data
        // without clean view and remove packets from the Packet Manager
        setCaption( url.prettyURL() );

        // before to insert packets into the view reset stored packets and clean the view...

        m_mainWidgetUI->clearView();

        // no need to save data: they're from file
        m_needToSave = false;

        // reset the frame number
        m_sniffer->resetFrameNumber();

        // we have to remove the captured data
        m_packets->clearList();

        // we need to store packets again
        m_packets->acceptPacket(true);

        // adding URL to the recent opened files
        m_actOpenRecent->addURL( url );

        // to avoid crash disable sorting
        m_mainWidgetUI->setSortList(false);

        // getting data
        m_sniffer->startSniffingFromFile( url.path() );

        // ok you can enable sorting now
        m_mainWidgetUI->setSortList(true);
      }
    }
    else
    {
      // ok, no packets available
      setCaption( url.prettyURL() );

      // no need to save data: they're from file
      m_needToSave = false;

      // we need to store packets
      m_packets->acceptPacket(true);

      // adding URL to the recent opened files
      m_actOpenRecent->addURL( url );

      // getting data
      m_sniffer->startSniffingFromFile( url.path() );
    }
  }
  else
    KMessageBox::information(
        this,
        i18n( "libpcap format is a standard format used by a lot of sniffer programs",
              "The file you choose to drag & drop is not in libpcap format!" ),
        i18n( "Format Error" ) );
}


void KSniffer::setupActions()
{
  // the action New Capture
  m_actNew = KStdAction::openNew( this, SLOT( slotNewCapture()), actionCollection(), "capt_new" );
  m_actNew->setWhatsThis( i18n( "<b>New Capture</b><br>Start a new packet-capture session." ) );
  m_actNew->setText( i18n( "New Capture..." ) );

  // the action Open Captured-Packet File
  m_actOpen = KStdAction::open( this, SLOT( slotOpen()), actionCollection(), "capt_open" );
  m_actOpen->setWhatsThis( i18n("<b>Open</b><br>Open a previous packet-capture session file.") );

  // the action Open Recent Captured-Packet File
  m_actOpenRecent = KStdAction::openRecent(this, SLOT( slotOpen( const KURL& ) ), actionCollection(), "capt_open_recent" );
  //connect( m_actOpenRecent, SIGNAL( activated() ), m_actOpen, SLOT( activate() ) );

  // the action Save Captured-Packet File
  m_actSave = KStdAction::save( this, SLOT( slotSave()), actionCollection(), "capt_save" );
  m_actSave->setWhatsThis( i18n( "<b>Save</b><br>Save the current packet-capture session." ) );
  m_actSave->setEnabled(FALSE);

  // the action Save Captured-Packet File As
  m_actSaveAs = KStdAction::saveAs( this, SLOT( slotSaveAs()), actionCollection(), "capt_save_as" );
  m_actSaveAs->setWhatsThis( i18n( "<b>Save As...</b><br>Save into a file the current packet-capture session." ) );
  m_actSaveAs->setEnabled(FALSE);

  // the action Stop Capture
  m_actStop = new KAction(i18n("S&top"), "stop",
                    KStdAccel::shortcut(KStdAccel::End),
                    this, SLOT(slotStop()),
                    actionCollection(), "capt_stop");
  m_actStop->setWhatsThis(i18n("<b>Stop</b><br>Stop the packet-capture session."));
  m_actStop->setEnabled(FALSE);

  // the action Print Captured-Packet
  m_actPrint = KStdAction::print(this, SLOT(slotPrint()), actionCollection(), "capt_print");
  m_actPrint->setWhatsThis(i18n("<b>Print...</b><br>Print the packet-capture session."));
  m_actPrint->setEnabled(FALSE);

  // the action Quit
  KAction *m_actQuit = KStdAction::quit(this, SLOT(slotQuit()), actionCollection(), "capt_quit");
  m_actQuit->setWhatsThis(i18n("<b>Quit</b><br>Stop the packet-capture session and quit from KSniffer."));
  m_actQuit->setText( i18n("Quit") );

  KStdAction::keyBindings(this, SLOT(optionsConfigureKeys()), actionCollection());
  KStdAction::configureToolbars(this, SLOT(optionsConfigureToolbars()), actionCollection());
  //KStdAction::preferences(this, SLOT(optionsPreferences()), actionCollection());

  // make sure the tooltips are used for the menu
  //actionCollection()->setHighlightingEnabled( true );

  // setup Settings menu
#if KDE_IS_VERSION(3,1,90)
  createStandardStatusBarAction();
  setStandardToolBarMenuEnabled(true);
#else
  m_toolbarAction = KStdAction::showToolbar(this, SLOT(optionsShowToolbar()), actionCollection());
  m_statusbarAction = KStdAction::showStatusbar(this, SLOT(optionsShowStatusbar()), actionCollection());
#endif

  createGUI("ksnifferui.rc");
}


void KSniffer::saveProperties(KConfig *config)
{
  // the 'config' object points to the session managed
  // config file.  anything you write here will be available
  // later when this app is restored

  //config->setGroup("General Options");

  if (!m_mainWidgetUI->currentURL().isEmpty())
  {
#if KDE_IS_VERSION(3,1,3)
    config->writePathEntry("lastURL", m_mainWidgetUI->currentURL());
#else
    config->writeEntry("lastURL", m_mainWidgetUI->currentURL());
#endif
  }
}


void KSniffer::readProperties(KConfig *config)
{
  // the 'config' object points to the session managed
  // config file.  this function is automatically called whenever
  // the app is being restored.  read in here whatever you wrote
  // in 'saveProperties'


  QString url = config->readPathEntry("lastURL");

  if (!url.isEmpty())
    m_mainWidgetUI->openURL(KURL(url));
}


void KSniffer::readSettings()
{
  m_actOpenRecent->loadEntries( KGlobal::config() );
  m_actOpenRecent->setEnabled( true ); // force enabling
  //m_actOpenRecent->setToolTip( i18n("Click to open a file\nClick and hold to open a recent file") );
}


void KSniffer::writeSettings()
{
  m_actOpenRecent->saveEntries( KGlobal::config() );
  KGlobal::config()->sync();
}


void KSniffer::dragEnterEvent(QDragEnterEvent *event)
{
  // accept uri drops only
  event->accept(KURLDrag::canDecode(event));
}


void KSniffer::dropEvent(QDropEvent *event)
{
  // this is a very simplistic implementation of a drop event.  we
  // will only accept a dropped URL.  the Qt dnd code can do *much*
  // much more, so please read the docs there
  KURL::List urls;

  // see if we can decode a URI.. if not, just ignore it
  if (KURLDrag::decode(event, urls) && !urls.isEmpty())
  {
    // okay, we have a URI... process it
    const KURL &url = urls.first();

    // load in the file
    load(url);
  }
}


// let close the application just from the system tray icon
bool KSniffer::queryClose()
{
  if( kapp->sessionSaving())
    return true;
  else
  {
    if ( isShown() && m_action != QUIT_APPL )
    {
      KMessageBox::information( this,
                                i18n( "<qt>Closing the main window will keep KSniffer running in the system " \
                                      "tray. Use 'Quit' from the system tray menu to quit the application.</qt>" ),
                                i18n( "Docking in System Tray" ), "hideOnCloseInfo");
      hide();
    }
    return false;
  }
}


// start a new Capture section if no available
void KSniffer::slotNewCapture()
{
  // this slot is called whenever the Capture->New menu is selected,
  // the New shortcut is pressed (usually CTRL+N) or the New toolbar
  // button is clicked

  // create a new window
  //(new KSniffer)->show();
  changeStatusbar( i18n("New Capture: select the network interface to start sniffing packets.") );

  CaptureDialog *cod = new CaptureDialog( this );

  if (cod->exec())
  {
    if (m_packets->packetAvailable())
      if ( KMessageBox::warningContinueCancel( this,
              i18n("<p>Packet-capture session data available. If you want to start a " \
                   "new capture session you are going to lose all captured data.</p>" \
                   "<p>Do you really want to continue?</p>" ) ) == KMessageBox::Cancel )
        return;
      else
      {
        // need to save data when available
        m_needToSave = true;

        // reset the frame number
        m_sniffer->resetFrameNumber();

        // we have to remove the captured data
        m_packets->clearList();

        // we need to store packets again
        m_packets->acceptPacket(true);

        // reset main widget title
        setPlainCaption( i18n("KSniffer - The KDE Network Sniffer") );
      }
      else
      {
        // need to save data when available
        m_needToSave = true;

        // reset the frame number
        m_sniffer->resetFrameNumber();

        // we need to store packets: no packets available and capture requested
        m_packets->acceptPacket(true);

        // reset main widget title
        setPlainCaption( i18n("KSniffer - The KDE Network Sniffer") );
      }
      changeStatusbar( i18n("Sniffing network by \"%1\" interface.").arg(cod->selectedInterface()) );

      // Disable New, Open, Save, Save As, Print and Stop Actions
      m_actNew->setEnabled(FALSE);
      m_actOpen->setEnabled(FALSE);
      m_actSave->setEnabled(FALSE);
      m_actSaveAs->setEnabled(FALSE);
      m_actPrint->setEnabled(FALSE);
      m_actStop->setEnabled(TRUE);

      // starting capture variable
      m_doingCapture = true;

      QPixmap icon = BarIcon("ksniffer");
      KPassivePopup::message( i18n("Sniffing"),
                              i18n("<p>You selected <b>%1</b> interface.<br>Now starting to sniff.</p>")
                                  .arg(cod->selectedInterface()), icon, this );

      setPlainCaption( i18n("KSniffer - Capturing on %1 Interface").arg(cod->selectedInterface()) );
      // create CaptureOptions object to give to the Sniffer's constuctor
      m_options->setStrInterface(cod->selectedInterface());
      m_sniffer->setCaptureOptions(m_options);

      // clearing the view before to start sniffing
      m_mainWidgetUI->clearView();

      // to avoid crash disable sorting
      m_mainWidgetUI->setSortList(false);

      // starting sniffer
      m_sniffer->startSniffing();
  }
  else
    changeStatusbar( i18n("Ready.") );

  delete cod;
}


// this slot is called whenever the File->Open menu is selected,
// the Open shortcut is pressed (usually CTRL+O) or the Open toolbar
// button is clicked
void KSniffer::slotOpen( const KURL& urlParam)
{
  bool isLocal;  // true if the file to open is local
  KURL url0;     // URL to open: can be network resource

  changeStatusbar( i18n("libpcap format is a standard format used by a lot "
                        "of sniffer programs",
                        "Select the libpcap format file to open." ) );

  int result = Sniffer::NO_RESULT_AVAILABLE;
  do
  {
    if (urlParam.isEmpty())
      url0 = KFileDialog::getOpenURL(QString::null, QString::null, this, i18n("Open File"));
    else
      url0 = urlParam;
    KURL url = url0;  // real URL on which operate: to have always a local file
    kdDebug() << "File to open: " << url.prettyURL() << endl;
    if ( !url.isEmpty() )
    {
      if ( !(isLocal = url.isLocalFile()) )
      {
        m_tmpFile = new KTempFile;
        m_tmpFile->setAutoDelete(true);
        if ( m_tmpFile->status() == 0 )
        {
          // temporary file created succesfully
          QFile file( m_tmpFile->name() );
          if ( file.open( IO_ReadWrite ) )
          {
            file.close();
            QString name = m_tmpFile->name();
            if ( !KIO::NetAccess::download( url, name, 0 ) )
            {
              KMessageBox::sorry( this, i18n(
                  "Unable to access to the temporary file '%1': %2.").arg( m_tmpFile->name() ).arg( KIO::NetAccess::lastErrorString() ),
                  i18n("Load Failed") );
            }
          }
          else
            KMessageBox::error(this,
                               i18n("Error during loading. Could not open '%1' temporary file for writing.").arg ( m_tmpFile->name() ),
                               i18n("Load Failed"));
        }
        else
          KMessageBox::sorry( this, i18n(
              "Loading error. Could not create temporary file: %1.").arg( strerror( m_tmpFile->status() ) ),
              i18n("Load Failed") );
        url = m_tmpFile->name();
      }
      switch ( result = m_sniffer->checkFile( url.path() ) )
      {
        case Sniffer::BAD_FORMAT_FILE:
          KMessageBox::sorry(this,
                         i18n( "libpcap format is a standard format used by a "
                               "lot of sniffer programs",
                               "<b>%1</b> is not a libpcap format file." ).arg(url0.prettyURL()) );
          if (!isLocal)
            // temporary file has to be deleted
            delete m_tmpFile;
          break;
        case Sniffer::RIGHT_FORMAT_FILE:
          // standard filedialog
          m_actOpenRecent->addURL( url );
          if ( m_packets->packetAvailable() )
            if ( m_needToSave )
              // just if we get some packets we advice the user about
              // the possibilty to lost data is not yet saved
              if ( KMessageBox::warningContinueCancel( this,
                      i18n("<p>Packet-capture session data available. If you want to load a" \
                           "file captured session you are going to lose all captured data.</p>" \
                           "<p>Do you really want to continue?</p>" ) ) == KMessageBox::Cancel)
                return;
              else
              {
                // Packets available, need to save them, continue losting data

                // no need to save already saved file: data gets from a file
                m_needToSave = false;

                // ok, now you can reset Frame Number
                m_sniffer->resetFrameNumber();

                // we have to remove the captured data
                m_packets->clearList();

                // clearing the view before to start sniffing
                m_mainWidgetUI->clearView();

               // we need to store packets again
               m_packets->acceptPacket(true);
              }
            else
            {
              // Packets available, don't need to save them

              // ok, now you can reset Frame Number
              m_sniffer->resetFrameNumber();

              // we have to remove the captured data
              m_packets->clearList();

              // clearing the view before to start sniffing
              m_mainWidgetUI->clearView();

              // we need to store packets again
              m_packets->acceptPacket(true);
            }
          else
          {
            // No packets available

            // we need to store new packets: no previous packets available and capture requested
            m_packets->acceptPacket(true);
          }

          // set the title as the filename: the original choosed, for network trasparency to the user
          setCaption( url0.prettyURL() );
          break;
      }
    }
    else
      result = Sniffer::NO_FILE_SELECTED;
  } while ((result != Sniffer::RIGHT_FORMAT_FILE) && (result != Sniffer::NO_FILE_SELECTED));
  changeStatusbar( i18n("Ready.") );

  if ( (result != Sniffer::NO_FILE_SELECTED) && (result == Sniffer::RIGHT_FORMAT_FILE) )
  {
    // to avoid crash disable sorting
    m_mainWidgetUI->setSortList(false);

    // selected a libpcap file
    m_sniffer->startSniffingFromFile();

    kdDebug() << "ok the file was entirely loaded" << endl;

    // ok you can enable sorting now
    m_mainWidgetUI->setSortList(true);

    // no need to save a loaded file...
    m_actSave->setEnabled(FALSE);
    m_actSaveAs->setEnabled(FALSE);

    // ...but you can print it
    // print is not implemented better set it not enabled
    //m_actPrint->setEnabled(TRUE);

    if (!isLocal)
      delete m_tmpFile;
  }
}


void KSniffer::slotSave()
{
  // this slot is called whenever the File->Save menu is selected,
  // the Save shortcut is pressed (usually CTRL+S) or the Save toolbar
  // button is clicked

  // save the current file
  if ( m_strFilename.isEmpty() )
    slotSaveAs();
  else
  {
    // copy file from /tmp
    KURL src(TEMPFILE);
    KURL dest(m_strFilename);
    KIO::Job *job = KIO::copy( src, dest );
    connect( job, SIGNAL( result(KIO::Job *) ),
             this, SLOT( copyingJobFinished(KIO::Job *) ) );

    // set the title for the saved filename
    m_strSavedFilename = dest;
  }
}


// this slot is called whenever the File->Save As menu is selected,
void KSniffer::slotSaveAs()
{
  changeStatusbar( i18n("Save captured-packets as...") );

  KURL dest = KFileDialog::getSaveURL();
  KURL src(TEMPFILE);
  if (!dest.isEmpty())
    if (dest.isValid())
    {
      m_strFilename = dest.path();

      // copy file from /tmp
      KIO::Job *job = KIO::copy( src, dest );
      connect( job, SIGNAL( result(KIO::Job *) ),
               this, SLOT( copyingJobFinished(KIO::Job *) ) );

      // set the title for the saved filename
      m_strSavedFilename = dest;
      m_actOpenRecent->addURL( dest );
    }
    else
      KMessageBox::information( this, i18n( "%1 not saved!" ).arg(m_strFilename), i18n( "Save Failed" ) );
  else
    // user aborted saving... it was quitting, so remove temporary file
    if (m_action == QUIT_APPL)
      unlink(TEMPFILE);
}


void KSniffer::slotPrint()
{
  // this slot is called whenever the File->Print menu is selected,
  // the Print shortcut is pressed (usually CTRL+P) or the Print toolbar
  // button is clicked
  /*KPrinter printer;
  if (printer.setup(this))
  {
    // setup the printer.  with Qt, you always "print" to a
    // QPainter.. whether the output medium is a pixmap, a screen,
    // or paper
    QPainter p;
    p.begin(&printer);

    // we let our view do the actual printing
    QPaintDeviceMetrics metrics(&printer);
    m_mainWidgetUI->print(&p, metrics.height(), metrics.width());

    // and send the result to the printer
    p.end();
  }*/
  //KMessageBox::sorry(this, i18n("This feature is not yet available.") );
  //changeStatusbar("filePrint(): not working yet!");
}


// with this member we stop packets capture
void KSniffer::slotStop()
{
  // it doesn't need to store packets anymore
  m_packets->acceptPacket(false);

  m_sniffer->stopSniffing();

  // captured data needs to be saved before to quit...
  // ...if there will be packets available
  m_needToSave = true;

  if (!m_packets->packetAvailable())
  {
    setPlainCaption( i18n("KSniffer - The KDE Network Sniffer") );
    KMessageBox::information(this, i18n("You got no packets."));
  }
  else
  {
    setPlainCaption( i18n("KSniffer - Unsaved Packets") );

    // saving icons can be enabled just if you got packets
    m_actSave->setEnabled(TRUE);
    m_actSaveAs->setEnabled(TRUE);
  }
  m_actNew->setEnabled(TRUE);
  m_actOpen->setEnabled(TRUE);
  // print is not implemented better set it not enabled
  //m_actPrint->setEnabled(TRUE);
  m_actStop->setEnabled(FALSE);

  // ok you can enable sorting now
  m_mainWidgetUI->setSortList(true);

  changeStatusbar( i18n("Stopped sniffing. Now, ready again.") );
}


void KSniffer::slotQuit()
{
  // user wants to close KSniffer
  m_action = QUIT_APPL;

  // stop the Sniffing Thread: we are going to quit
  if ( m_sniffer->isSniffing() )
    m_sniffer->stopSniffing();

  // we don't need to store packets anymore
  m_packets->acceptPacket(false);

  if (m_packets->packetAvailable() && m_needToSave)
  {
    changeStatusbar( i18n("You have got data, maybe you want to save them.") );

    // we have some data that maybe we want to save
    if ( KMessageBox::questionYesNo( this,
         i18n( "<p>Quitting KSniffer will cause the current packets to be lost.<br>Do you want to save them now?</p>" ),
           i18n( "Unsaved Captured Packets" ), KStdGuiItem::saveAs(), KStdGuiItem::discard() ) == KMessageBox::Yes)
    {
      slotSave();
    }
    else
      // quitting without saving data: remove the temporary file
      unlink(TEMPFILE);
  }

  // now we can remove the captured data
  m_packets->clearList();

  // writing the application settings
  writeSettings();

  // remove temporary file
  unlink(tmpfile);

  // delay exit from KSniffer if Job's not finished
  kapp->deref();
}


void KSniffer::optionsShowToolbar()
{
  // this is all very cut and paste code for showing/hiding the
  // toolbar
  if (m_toolbarAction->isChecked())
    toolBar()->show();
  else
    toolBar()->hide();
}


void KSniffer::optionsShowStatusbar()
{
  // this is all very cut and paste code for showing/hiding the
  // statusbar
  if (m_statusbarAction->isChecked())
    statusBar()->show();
  else
    statusBar()->hide();
}


void KSniffer::optionsConfigureKeys()
{
  KKeyDialog::configure(actionCollection());
}


void KSniffer::optionsConfigureToolbars()
{
  // use the standard toolbar editor
#if defined(KDE_MAKE_VERSION)
# if KDE_VERSION >= KDE_MAKE_VERSION(3,1,0)
    saveMainWindowSettings(KGlobal::config(), autoSaveGroup());
# else
    saveMainWindowSettings(KGlobal::config());
# endif
#else
    saveMainWindowSettings(KGlobal::config());
#endif

  // use the standard toolbar editor
  KEditToolbar dlg(factory());
  connect(&dlg, SIGNAL(newToolbarConfig()), this, SLOT(applyNewToolbarConfig()));
  dlg.exec();
}


void KSniffer::applyNewToolbarConfig()
{
  // this slot is called when user clicks "Ok" or "Apply" in the toolbar editor.
  // recreate our GUI, and re-apply the settings (e.g. "text under icons", etc.)
  createGUI();

#if defined(KDE_MAKE_VERSION)
# if KDE_VERSION >= KDE_MAKE_VERSION(3,1,0)
    applyMainWindowSettings(KGlobal::config(), autoSaveGroup());
# else
    applyMainWindowSettings(KGlobal::config());
# endif
#else
    applyMainWindowSettings(KGlobal::config());
#endif
}


void KSniffer::optionsPreferences()
{
  // popup some sort of preference dialog, here
  /* KSnifferPreferences dlg;
  if (dlg.exec())
  {
    // redo your settings
  }*/
}


void KSniffer::changeStatusbar(const QString& text)
{
  // display the text on the statusbar
  statusBar()->message(text);
}


void KSniffer::changeCaption(const QString& text)
{
  // display the text on the caption
  setCaption(text);
}


// decide what to do when the Job is finished
void KSniffer::copyingJobFinished(KIO::Job *job)
{
  changeStatusbar( i18n("Saved captured-packets.") );

  // set the main widget new title
  setCaption( m_strSavedFilename.prettyURL() );

  // no need to save: we just completed save
  m_needToSave = false;

  // no need to save it again: just done
  m_actSave->setEnabled(FALSE);
  m_actSaveAs->setEnabled(FALSE);
  m_strFilename = "";

  // remove the temporary file: not more usefull
  unlink(TEMPFILE);
}

#include "ksniffer.moc"
