/***************************************************************************
 *   Copyright (C) 2004-2005 by Giovanni Venturi                           *
 *   gventuri73@tiscali.it                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <qwidget.h>
#include <qlayout.h>
#include <qsplitter.h>
#include <qlistview.h>
#include <qheader.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qstring.h>
#include <qstringlist.h>

#include <qgroupbox.h>
#include <qlabel.h>

#include <kurl.h>
#include <kiconloader.h>
#include <kicontheme.h>

#include <klocale.h>
#include <kdebug.h>

#include "ksniffermainwidget.h"
#include "ksnifferviewitem.h"
#include "packetmanager.h"
#include "frames/frameheader.h"
#include "frames/frameprotocolmanager.h"
#include "protocols/protocolmanager.h"

#include "ksniffermainwidget.moc"

KSnifferMainWidget::KSnifferMainWidget( QWidget* parent, const char* name )
    : QWidget( parent, name )
{
  if ( !name )
      setName( "MainWidget" );
  m_mainViewLayout = new QVBoxLayout( this, 11, 6, "MainWidgetLayout");
  m_mainViewLayout->setAlignment( Qt::AlignTop );

  // splitter widget
  m_splitter = new QSplitter( this, "splitter" );
  m_splitter->setOrientation( QSplitter::Vertical );

  // packets listview widget
  m_list = new QListView( m_splitter, "m_list" );
  m_list->addColumn( i18n( "frame number", "Frame No." ) );
  m_list->addColumn( i18n( "Time" ) );
  m_list->addColumn( i18n( "Source" ) );
  m_list->addColumn( i18n( "Destination" ) );
  m_list->addColumn( i18n( "Protocol" ) );
  m_list->addColumn( i18n( "Information" ) );
  m_list->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)5, (QSizePolicy::SizeType)7, 0, 0, m_list->sizePolicy().hasHeightForWidth() ) );
  m_list->setAllColumnsShowFocus( TRUE );

  // let sort column
  m_list->setShowSortIndicator( TRUE );

  // disable clicking on "Information"
  m_list->header()->setClickEnabled( FALSE, 5 );

  // packets details listview widget
  m_detail = new QListView( m_splitter, "m_detail" );
  KIconLoader kil;
  m_detail->addColumn( kil.loadIconSet("ksniffer", KIcon::Small), i18n( "Packet Details" ) );

  // disable "packet details"
  m_detail->header()->setClickEnabled( FALSE, 0 );
  m_detail->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)5, (QSizePolicy::SizeType)3, 0, 0, m_detail->sizePolicy().hasHeightForWidth() ) );
  m_mainViewLayout->addWidget( m_splitter );

  // we don't need sorting detail
  m_detail->setSortColumn(-1);

  QWhatsThis::add( m_list, i18n("In this area you get all the captured packets.") );
  QToolTip::add( m_list, i18n("The list of all captured packets") );

  QWhatsThis::add( m_detail, i18n("In this area you can see the details about a selected captured packet.") );
  QToolTip::add( m_detail, i18n("More details about the captured packets") );

  m_firstInsert = true;

  m_list->setColumnAlignment(0, Qt::AlignRight);
  m_list->setColumnAlignment(1, Qt::AlignRight);
  m_list->setColumnAlignment(2, Qt::AlignCenter);
  m_list->setColumnAlignment(3, Qt::AlignCenter);
  m_list->setColumnAlignment(4, Qt::AlignCenter);
  m_list->setColumnWidth(1, 100);
  m_list->setColumnWidth(2, 100);
  m_list->setColumnWidth(3, 100);
  m_list->setColumnWidth(4, 80);
  m_list->setColumnWidth(5, 550);

  m_detail->setColumnWidth(0, 550);
  m_detail->setRootIsDecorated(true);

  connect( m_list, SIGNAL( selectionChanged(QListViewItem*) ), this, SLOT( showDetails(QListViewItem*) ) );

  connect( m_detail, SIGNAL( expanded ( QListViewItem * ) ), this, SLOT( setExpand() ) );
  connect( m_detail, SIGNAL( collapsed ( QListViewItem * ) ), this, SLOT( setCollapse() ) );

  resize( QSize(414, 265).expandedTo(minimumSizeHint()) );
}


KSnifferMainWidget::~KSnifferMainWidget()
{
  // no need to delete child widgets, Qt does it all for us
}


void KSnifferMainWidget::print(QPainter */*p*/, int /*height*/, int /*width*/)
{
  // do the actual printing, here
  // p->drawText(etc..)
}


QString KSnifferMainWidget::currentURL()
{
  return "";
}


void KSnifferMainWidget::openURL(const QString& url)
{
  openURL(KURL(url));
}


void KSnifferMainWidget::openURL(const KURL& url)
{
  //
}


void KSnifferMainWidget::slotOnURL(const QString& url)
{
  emit signalChangeStatusbar(url);
}


void KSnifferMainWidget::slotSetTitle(const QString& title)
{
  emit signalChangeCaption(title);
}


void KSnifferMainWidget::setExpand()
{
  m_bExpanding0 = true;
  m_bExpanding1 = true;
}


void KSnifferMainWidget::setCollapse()
{
  m_bExpanding0 = false;
  m_bExpanding1 = false;
}


void KSnifferMainWidget::addListViewItem( const QString& item1, const QString& item2,
                                   const QString& item3, const QString& item4,
                                   const QString& item5, const QString& item6 )
{
  if (m_firstInsert)
  {
    m_lastElem = new KSnifferViewItem( m_list, item1, item2, item3, item4, item5, item6 );
    m_firstInsert = false;
  }
  else
    m_lastElem = new KSnifferViewItem( m_list, m_lastElem, item1, item2, item3, item4, item5, item6 );
}


void KSnifferMainWidget::clearView()
{
  m_list->clear();
  m_detail->clear();
}


void KSnifferMainWidget::setSortList(bool value)
{
  if (value)
    m_list->setSortColumn(0);
  else
    // to avoid crash you should not sort data when sniffing
    m_list->setSortColumn(-1);
}


void KSnifferMainWidget::displayPacket(int ord, PacketManager *pktMan)
{
  if (pktMan->getFrameNumber() != ord)
    pktMan->setFrameNumber( ord );
  addListViewItem(QString::number(ord), pktMan->strTimeStamp(), pktMan->strSourceAddress(),
  pktMan->strDestinationAddress(), pktMan->strProtocol(),
                  /*pktMan->strInfo()*/""
                 );
  // we need to save PacketManager class reference first time
  if (ord == 1)
    m_packetManager = pktMan;
}


void KSnifferMainWidget::showDetails( QListViewItem * item )
{
  if (item == NULL)
    return;

  // getting frame number
  int frameNumber = item->text(0).toInt();

  // clear the detail view
  m_detail->clear();

  m_packetManager->setFrameNumber( frameNumber );
  switch (m_packetManager->getFrameHType())
  {
    case DLT_EN10MB:
      break;
    case DLT_LINUX_SLL:
      break;
    default:
      kdDebug() << "Frame: decoding non yet implemented" << endl;
  }

  // "first" line in the detail view
  QListViewItem *itemFrame0 = new QListViewItem( m_detail, i18n("Frame number: %1").arg(frameNumber) );
  QListViewItem *subitemFrame0 = new QListViewItem( itemFrame0, i18n("general field for the frame header", "Capture time: %1")
      .arg(m_packetManager->strDetailTimeStamp()) );
  subitemFrame0 = new QListViewItem( itemFrame0, subitemFrame0,
                                     i18n("general field for the frame header", "Delay from the previous packet: %1 s")
                                         .arg(m_packetManager->strDiffTimePrevious()) );
  subitemFrame0 = new QListViewItem( itemFrame0, subitemFrame0,
                                     i18n("general field for the frame header", "Delay from the first packet: %1 s")
                                         .arg(m_packetManager->strDiffTimeFirst()) );
  subitemFrame0 = new QListViewItem( itemFrame0, subitemFrame0, i18n("general field for the frame header", "Packet length: %1 bytes")
      .arg(m_packetManager->frameLength()) );
  subitemFrame0 = new QListViewItem( itemFrame0, subitemFrame0,
                                     i18n("general field for the frame header", "Captured packet length: %1 bytes")
                                         .arg(m_packetManager->frameCapturedLength()) );

  FrameHeaderManager fhm(m_packetManager->getPacket());
  QStringList list = fhm.headerLines();
  if (!list.isEmpty())
  {
    // "second" line in the detail view
    QListViewItem *itemFrame1 = new QListViewItem( m_detail, itemFrame0, i18n("Frame header: %1").arg(fhm.strType()) );
    QListViewItem *subitemFrame1 = new QListViewItem ( itemFrame1, *list.begin() );

    for( QStringList::Iterator it = ++list.begin(); it != list.end(); ++it)
      subitemFrame1 = new QListViewItem( itemFrame1, subitemFrame1, *it );


    FrameProtocolManager fpm(m_packetManager->getPacket(), fhm.protocol());
    list = fpm.headerLines();
    if (!list.isEmpty())
    {
      // "third" line in the detail view
      QListViewItem *itemFrame2 = new QListViewItem( m_detail, itemFrame1, i18n("Protocol information: %1").arg( fpm.getProtocol()) );
      QListViewItem *subitemFrame2 = new QListViewItem ( itemFrame2, *list.begin() );

      for( QStringList::Iterator it = ++list.begin(); it != list.end(); ++it)
      {
        if (*it == "*open")
        {
          subitemFrame2 = new QListViewItem( itemFrame2, subitemFrame2, *(++it) );
          QListViewItem *subSubitemFrame2 = new QListViewItem( subitemFrame2, *(++it) );
          while( *(++it) != "*close" )
            subSubitemFrame2 = new QListViewItem( subitemFrame2, subSubitemFrame2, *it );
        }
        else
          subitemFrame2 = new QListViewItem( itemFrame2, subitemFrame2, *it );
      }

      ProtocolManager pm(m_packetManager->getPacket(), fpm.getSubProtocol());
      list = pm.headerLines();
      if (!list.isEmpty())
      {
        // "forth" line in the detail view
        QListViewItem *itemFrame3 = new QListViewItem( m_detail, itemFrame2, i18n("specific protocol information", "%1 information:")
            .arg( fpm.strSubProtocol()) );
        QListViewItem *subitemFrame3 = new QListViewItem ( itemFrame3, *list.begin() );

        for( QStringList::Iterator it = ++list.begin(); it != list.end(); ++it)
        {
            viewSubtree(it, &itemFrame3, &subitemFrame3);
          //subitemFrame3 = new QListViewItem( itemFrame3, subitemFrame3, *it );
        }

        if (m_bExpanding1)
        {
          itemFrame1->setOpen(true);
          itemFrame2->setOpen(true);
          itemFrame3->setOpen(true);
        }
      }
    }
  }

  // expand item lines if needed
  if (m_bExpanding0)
    itemFrame0->setOpen(true);
}


void KSnifferMainWidget::viewSubtree(QStringList::Iterator &it, QListViewItem **itemFrame, QListViewItem **subitemFrame)
{
  if (*it == "*open")
  {
    *subitemFrame = new QListViewItem( *itemFrame, *subitemFrame, *(++it) );
    QListViewItem *subSubitemFrame = new QListViewItem( *subitemFrame, *(++it) );
    while( *(++it) != "*close" )
    {
      if (*it == "*open")
      {
        subSubitemFrame = new QListViewItem( *subitemFrame, subSubitemFrame, *(++it) );
        QListViewItem *subSubSubitemFrame = new QListViewItem( subSubitemFrame, *(++it) );
        ++it;
        while (*it != "*close")
          subSubSubitemFrame = new QListViewItem( subSubitemFrame, subSubSubitemFrame, *(it++) );
      }
      else
        subSubitemFrame = new QListViewItem( *subitemFrame, subSubitemFrame, *it );
    }
  }
  else if (*it != "*close")
    *subitemFrame = new QListViewItem( *itemFrame, *subitemFrame, *it );
}
