/***************************************************************************
 *   Copyright (C) 2004-2005 by Giovanni Venturi                           *
 *   gventuri73@tiscali.it                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <stdio.h>
#include <unistd.h> 
#include <sys/types.h>
#include <netinet/in.h>
#include <pcap.h>
#include <time.h>
#include <linux/if_ether.h>

#include <qdatetime.h>
#include <qstring.h>

#include <kdebug.h>
#include <kglobal.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdeversion.h>

#include "packetmanager.h"
#include "packet.h"
#include "frames/frameheader.h"
#include "protocols/ip-protocol.h"
#include "protocols/arp-protocol.h"

PacketManager::PacketManager(QObject *parent, const char *name)
 : QObject(parent, name)
{
  m_acceptingPacket = true;
  m_tmpFile = fopen(tmpfile, "w+");
  fclose(m_tmpFile);
  m_bPacketAvailable = false;
  m_frameNumber = 0;
}


PacketManager::~PacketManager()
{
}


void PacketManager::setFrameNumber(const long num)
{
  struct pcap_pkthdr packetHeader;
  long frameNumber;
  int frameType;
  int frameLength;
  packetType packet[PKTMAX];
  ptrPacketType pPacket = packet;

  m_mutex.lock();
  m_frameNumber = num;
  m_tmpFile = fopen(tmpfile, "r");
  m_lastPacket = m_packet;

  do
  {
    // packet header (ts, caplen, len)
    fread(&packetHeader, sizeof(struct pcap_pkthdr), 1, m_tmpFile);

    // packet frame number
    fread(&frameNumber, sizeof(long), 1, m_tmpFile);

    // packet frame type
    fread(&frameType, sizeof(int), 1, m_tmpFile);

    // packet frame length
    fread(&frameLength, sizeof(int), 1, m_tmpFile);

    // packet itself
    fread(&packet, packetHeader.caplen, 1, m_tmpFile);

    if (frameNumber + 1 == num)
    {
      // prepare previous packet
      m_lastPacket.setPacketHeader( packetHeader );
      m_lastPacket.setFrameNumber( frameNumber );
      m_lastPacket.setFrameType( frameType );
      m_lastPacket.setFrameHeaderLength( frameLength );
      m_lastPacket.setPacket( pPacket, packetHeader );
    }
  } while ( frameNumber < m_frameNumber );
  fclose(m_tmpFile);
  m_mutex.unlock();

  m_packet.setPacketHeader( packetHeader );
  m_packet.setFrameNumber( frameNumber );
  m_packet.setFrameType( frameType );
  m_packet.setFrameHeaderLength( frameLength );
  m_packet.setPacket( pPacket, packetHeader );

  if (num == 1)
    m_firstPacket = m_packet;
}


long PacketManager::getFrameNumber()
{
  return m_frameNumber;
}


bpf_u_int32 PacketManager::frameCapturedLength()
{
  return m_packet.headerCapturedLength();
}


bpf_u_int32 PacketManager::frameLength()
{
  return m_packet.headerLength();
}


QString PacketManager::strTimeStamp() const
{
  // we assume that the standard time stamp is a relative time stamp
  return strRelativeTimeStamp();
}


// we get the relative time in seconds and 1/1'000'000 seconds when the packet capture started
// first packet has relative time 0.0
QString PacketManager::strRelativeTimeStamp() const
{
  return strDiffTimeFirst();
}


QString PacketManager::strDiffTimePrevious() const
{
  // remember that the first vectore element has
  // index 0, so we need 'num - 1' to get the
  // element number num
  time_t sec = 0;
  suseconds_t usec = 0;
  double diffTime;

  if (m_frameNumber > 1)
  {
    sec = -m_lastPacket.secTimeStamp() + m_packet.secTimeStamp();
    usec = -m_lastPacket.usecTimeStamp() + m_packet.usecTimeStamp();
  }

  diffTime = (double) sec + (double) usec / 1000000;

  return QString::number(diffTime, 'f', 6);
}


QString PacketManager::strDiffTimeFirst() const
{
  time_t sec = 0;
  suseconds_t usec = 0;
  double diffTime;

  if (m_frameNumber > 1)
  {
    sec = -m_firstPacket.secTimeStamp() + m_packet.secTimeStamp();
    usec = -m_firstPacket.usecTimeStamp() + m_packet.usecTimeStamp();
  }

  diffTime = (double) sec + (double) usec / 1000000;

  return QString::number(diffTime, 'f', 6);
}


QString PacketManager::strDetailTimeStamp() const
{
  /**
    * localtime_r(const time_t *timep, struct tm *result):
    * @param timep is time in seconds
    * @param result is a tm structure
    * localtime_r converts the seconds into a tm struct
    */
  struct tm
  /* as defined in <time.h>:
  {
    int tm_sec;    // seconds
    int tm_min;    // minutes
    int tm_hour;   // hours
    int tm_mday;   // day of the month
    int tm_mon;    // month
    int tm_year;   // year
    int tm_wday;   // day of the week
    int tm_yday;   // day in the year
    int tm_isdst;  // daylight saving time
  } */localTime;
  time_t sec = m_packet.secTimeStamp();
  localtime_r(&sec, &localTime);

  QDate d( localTime.tm_year + 1900, localTime.tm_mon + 1, localTime.tm_mday );
  QString date = KGlobal::locale()->formatDate( d , false );

  QTime qTime( localTime.tm_hour, localTime.tm_min, localTime.tm_sec );

#if KDE_VERSION >= KDE_MAKE_VERSION(3,3,0)
  QString time = KGlobal::locale()->formatTime( qTime, true, true) + "."
                 + QString::number( m_packet.usecTimeStamp() );
#else
  QString time = KGlobal::locale()->formatTime( qTime, true) + "."
                 + QString::number( m_packet.usecTimeStamp() );
#endif

  return i18n( "Concatenation of date and time. Translate it according to "
               "your kdelibs.po.", "%1, %2" ).arg( date ).arg( time );
               return "";
}


QString PacketManager::strSourceAddress()
{
  ptrPacketType pPacket = getPacket();
  FrameHeaderManager fhm(pPacket);
  IpProtocol ip(pPacket);
  QString srcAddress;

  switch (fhm.protocol())
  {
    case ETH_P_IP:
      srcAddress = ip.sourceAddress();
      break;
    case ETH_P_ARP:
      srcAddress = fhm.sorceAddress();
      break;
    default:
      srcAddress = i18n("not available", "n/a");
  }

  return srcAddress;
}


QString PacketManager::strDestinationAddress()
{
  ptrPacketType pPacket = getPacket();
  FrameHeaderManager fhm(pPacket);
  IpProtocol ip(pPacket);
  QString dstAddress;

  switch (fhm.protocol())
  {
    case ETH_P_IP:
      dstAddress = ip.destinationAddress();
      break;
    case ETH_P_ARP:
      dstAddress = fhm.destinationAddress();
      break;
    default:
      dstAddress = i18n("not available", "n/a");
  }

  return dstAddress;
}


QString PacketManager::strProtocol()
{
  ptrPacketType pPacket = getPacket();
  FrameHeaderManager fhm(pPacket);
  IpProtocol ip(pPacket);
  QString proto;
  switch (fhm.protocol())
  {
    case ETH_P_IP:
      proto = ip.strProtocol();
      break;
    case ETH_P_ARP:
      proto = "ARP";
      break;
    default:
      proto = i18n("not available", "n/a");
  }

  return proto;
}


int PacketManager::getFrameHType()
{
  return m_packet.getFrameType();
}


int PacketManager::getFrameHLength() const
{
  return m_packet.frameHeaderLength();
}


ptrPacketType PacketManager::getPacket()
{
  return m_packet.getPacket();
}


void PacketManager::clearList()
{
  // reset file
  m_tmpFile = fopen(tmpfile, "w+");
  m_bPacketAvailable = false;
}


// set the policy to store or not packets
void PacketManager::acceptPacket(bool acceptCond)
{
  // the packet manager store received packet with savePacket(...)
  // if m_acceptingPacket is true
  // with this acceptPacket(bool) we decide if the packets have
  // to be stored (true) or not (false)
  m_acceptingPacket = acceptCond;
}


bool PacketManager::packetAvailable()
{
  return m_bPacketAvailable;
}


// store the raw data into the Vector: into m_packetVector
void PacketManager::savePacket(ptrPacketType p, struct pcap_pkthdr packetHeader, long ord, int frameType, int frameLen)
{
  m_mutex.lock();
  m_tmpFile = fopen(tmpfile, "a");
  if (!m_acceptingPacket)
    // the packet cannot be stored
    return;

  // add the captured packet to the list...

  // packet header (ts, caplen, len)
  fwrite(&packetHeader, sizeof(struct pcap_pkthdr), 1, m_tmpFile);

  // packet frame number
  fwrite(&ord, sizeof(long), 1, m_tmpFile);

  // packet frame type
  fwrite(&frameType, sizeof(int), 1, m_tmpFile);

  // packet frame length
  fwrite(&frameLen, sizeof(int), 1, m_tmpFile);

  // packet itself
  fwrite(p, packetHeader.caplen, 1, m_tmpFile);

  // packet available
  m_bPacketAvailable = true;

  // so the writing is done
  fflush(m_tmpFile);
  fclose(m_tmpFile);
  m_mutex.unlock();

  m_lastPacket = m_packet;
  m_packet.setPacketHeader( packetHeader );
  m_packet.setFrameNumber( ord );
  m_frameNumber = ord;
  m_packet.setFrameType( frameType );
  m_packet.setFrameHeaderLength( frameLen );
  m_packet.setPacket( p, packetHeader );

  if (ord == 1)
    m_firstPacket = m_packet;

  // now the packet is saved
  emit savedPacket(ord, this);
}

#include "packetmanager.moc"
