/***************************************************************************
 *   Copyright (C) 2004-2007 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <time.h>
#include <net/ethernet.h>

#include <qdatetime.h>
#include <qstring.h>

#include <kdebug.h>
#include <kglobal.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kdeversion.h>

#include <kuniqueapplication.h>
#include <kconfig.h>
#include <kstandarddirs.h>

#include "packetmanager.h"
#include "frames/frameheader.h"
#include "protocols/ip-protocol.h"
#include "protocols/arp-protocol.h"
#include "sniffer.h"

//FIXME: remove m_counts totally
PacketManager::PacketManager( QObject *parent, const char *name )
 : QObject(parent, name)
{
  m_acceptingPacket = true;
  m_bPacketAvailable = false;
  m_frameNumber = 0;
//  m_counts = 0;
}


PacketManager::~PacketManager()
{
}


void PacketManager::setHeaderData(pcap_t *pHandler, int &frameType, int &hdrLen)
{
  frameType = pcap_datalink(pHandler);

  switch(frameType)
  {
    case DLT_EN10MB:
      hdrLen = 14;
      break;
    case DLT_LINUX_SLL:
      hdrLen = 16;
      break;
    case DLT_LOOP:
      hdrLen = 4;
      break;
    case DLT_NULL:
      hdrLen = 4;
      break;
    case DLT_RAW:
      hdrLen = 0;
      break;
    default:
      hdrLen = -1;
      break;
  }
}


void PacketManager::setFrameNumber( const long& num )
{
  struct pcap_pkthdr packetHeader;
  packetType packet[PKTMAX];
  ptrPacketType pPacket = packet;
  unsigned char *p;
  int frameType, frameLength;
  long frameNumber = 1;
  char errbuf[PCAP_ERRBUF_SIZE];
  kdDebug() << "+++++ going to move into '" << m_strFilePath << "' file" << endl;
  pcap_t *m_pcapfp = pcap_open_offline( m_strFilePath, errbuf );

  if ( m_pcapfp == NULL )
  {
    kdDebug() << "Error opening libpcap: " << errbuf << endl;
    pcap_close( m_pcapfp );
    return;
  }

  m_frameNumber = num;
  m_lastPacket = m_packet;

  while (frameNumber <= m_frameNumber)
  {
    p = (unsigned char*) pcap_next(m_pcapfp, &packetHeader);

    if (p == NULL)
      break;     // file is end

    // we got a packet: raw bytes (p), the timestamp and the captured packet length and the real packet length (hdr)
    /*
     *  hdr is a struct pcap_pkthdr:
     *    ts - a struct timeval containing the time when the packet was captured
     *    caplen - a bpf_u_int32 giving the number of bytes of the packet that are available from the capture
     *    len - a bpf_u_int32 giving the length of the packet, in bytes (which might be more than the number
     *      of bytes available from the capture, if the length of the packet is larger than the maximum
     *      number of bytes to capture)
     */
    setHeaderData(m_pcapfp, frameType, frameLength);
    if (frameLength < 0)
    {
      kdDebug() << "unknown packet" << endl;
      continue;
    }
    else
    {
      // recover packet data
      pPacket = p;
    }
    if ((frameNumber + 1) == num)
    {
      // prepare previous packet
      m_lastPacket.setPacketHeader( packetHeader );
      m_lastPacket.setFrameNumber( frameNumber );
      m_lastPacket.setFrameType( frameType );
      m_lastPacket.setFrameHeaderLength( frameLength );
      m_lastPacket.setPacket( pPacket, packetHeader );
    }
    frameNumber++;
  }

  m_packet.setPacketHeader( packetHeader );
  m_packet.setFrameNumber( frameNumber );
  m_packet.setFrameType( frameType );
  m_packet.setFrameHeaderLength( frameLength );
  m_packet.setPacket( pPacket, packetHeader );

  // assign the packet length
  m_packetSize = packetHeader.caplen;
}


long PacketManager::getFrameNumber() const
{
  return m_frameNumber;
}


// if you set the temporary directory than the temporary file will be the pcap default one
void PacketManager::setFilePath( const QString& tmp )
{
  m_strFilePath = tmp;
}


// if you set the temporary path filename it is the complete path and you have to use this one
void PacketManager::setFilename( const QString& tmpFile )
{
  m_strFilePath = tmpFile;
}


QString PacketManager::tmpFilePath() const
{
  return m_strFilePath;
}


bpf_u_int32 PacketManager::frameCapturedLength()
{
  return m_packet.headerCapturedLength();
}


bpf_u_int32 PacketManager::frameLength()
{
  return m_packet.headerLength();
}


QString PacketManager::strTimeStamp() const
{
  // we assume that the standard time stamp is a relative time stamp
  return strRelativeTimeStamp();
}


// we get the relative time in seconds and 1/1'000'000 seconds when the packet capture started
// first packet has relative time 0.0
QString PacketManager::strRelativeTimeStamp() const
{
  return strDiffTimeFirst();
}


QString PacketManager::strDiffTimePrevious() const
{
  // remember that the first vectore element has
  // index 0, so we need 'num - 1' to get the
  // element number num
  time_t sec = 0;
  suseconds_t usec = 0;
  double diffTime;

  if (m_frameNumber > 1)
  {
    sec = -m_lastPacket.secTimeStamp() + m_packet.secTimeStamp();
    usec = -m_lastPacket.usecTimeStamp() + m_packet.usecTimeStamp();
  }

  diffTime = (double) sec + (double) usec / 1000000;

  return QString::number(diffTime, 'f', 6);
}


QString PacketManager::strDiffTimeFirst() const
{
  time_t sec = 0;
  suseconds_t usec = 0;
  double diffTime;

  if (m_frameNumber > 1)
  {
    sec = -m_firstPacket.secTimeStamp() + m_packet.secTimeStamp();
    usec = -m_firstPacket.usecTimeStamp() + m_packet.usecTimeStamp();
  }

  diffTime = (double) sec + (double) usec / 1000000;

  return QString::number(diffTime, 'f', 6);
}


QString PacketManager::strDetailTimeStamp() const
{
  /**
    * localtime_r(const time_t *timep, struct tm *result):
    * @param timep is time in seconds
    * @param result is a tm structure
    * localtime_r converts the seconds into a tm struct
    */
  struct tm
  /* as defined in <time.h>:
  {
    int tm_sec;    // seconds
    int tm_min;    // minutes
    int tm_hour;   // hours
    int tm_mday;   // day of the month
    int tm_mon;    // month
    int tm_year;   // year
    int tm_wday;   // day of the week
    int tm_yday;   // day in the year
    int tm_isdst;  // daylight saving time
  } */localTime;
  time_t sec = m_packet.secTimeStamp();
  localtime_r(&sec, &localTime);

  QDate d( localTime.tm_year + 1900, localTime.tm_mon + 1, localTime.tm_mday );
  QString date = KGlobal::locale()->formatDate( d , false );

  QTime qTime( localTime.tm_hour, localTime.tm_min, localTime.tm_sec );

  QString time = KGlobal::locale()->formatTime( qTime, true, true) + "."
                 + QString::number( m_packet.usecTimeStamp() );

  return i18n( "Concatenation of date and time. Translate it according to "
               "your kdelibs.po.", "%1, %2" ).arg( date ).arg( time );
}


QString PacketManager::strSourceAddress()
{
  ptrPacketType pPacket = getPacket();
  FrameHeaderManager fhm(pPacket);
  IpProtocol ip(pPacket);
  QString srcAddress;

  switch (fhm.protocol())
  {
    case ETHERTYPE_IP:
      srcAddress = ip.sourceAddress();
      break;
    case ETHERTYPE_ARP:
      srcAddress = fhm.sorceAddress();
      break;
    default:
      srcAddress = i18n( "not available", "n/a" );
  }

  return srcAddress;
}


QString PacketManager::strDestinationAddress()
{
  ptrPacketType pPacket = getPacket();
  FrameHeaderManager fhm(pPacket);
  IpProtocol ip(pPacket);
  QString dstAddress;

  switch (fhm.protocol())
  {
    case ETHERTYPE_IP:
      dstAddress = ip.destinationAddress();
      break;
    case ETHERTYPE_ARP:
      dstAddress = fhm.destinationAddress();
      break;
    default:
      dstAddress = i18n( "not available", "n/a" );
  }

  return dstAddress;
}


QString PacketManager::strProtocol()
{
  ptrPacketType pPacket = getPacket();
  FrameHeaderManager fhm(pPacket);
  IpProtocol ip(pPacket);
  QString proto;

  switch (fhm.protocol())
  {
    case ETHERTYPE_IP:
      proto = ip.strProtocol();
      break;
    case ETHERTYPE_ARP:
      proto = "ARP";
      break;
    default:
      proto = i18n( "not available", "n/a" );
  }

  return proto;
}


QString PacketManager::strInfo()
{
  ptrPacketType pPacket = getPacket();
  FrameHeaderManager fhm(pPacket);
  IpProtocol ip(pPacket);
  QString info;

  switch (fhm.protocol())
  {
    case ETHERTYPE_IP:
      info = ip.strInfo();
      break;
    case ETHERTYPE_ARP:
      info = "ARP";
      break;
    default:
      info = i18n( "not available", "n/a" );
  }

  return info;
}


int PacketManager::getFrameHType()
{
  return m_packet.getFrameType();
}


int PacketManager::getFrameHLength() const
{
  return m_packet.frameHeaderLength();
}


ptrPacketType PacketManager::getPacket()
{
  return m_packet.getPacket();
}


int PacketManager::packetSize() const
{
  return m_packetSize;
}

void PacketManager::clearList()
{
  m_bPacketAvailable = false;
//  m_counts = 0;
}


// set the policy to store or not packets
void PacketManager::acceptPacket( bool acceptCond )
{
  // the packet manager store received packet with savePacket(...)
  // if m_acceptingPacket is true
  // with this acceptPacket(bool) we decide if the packets have
  // to be stored (true) or not (false)
  m_acceptingPacket = acceptCond;
}


bool PacketManager::acceptingPacket()
{
  return m_acceptingPacket;
}


bool PacketManager::packetAvailable()
{
  return m_bPacketAvailable;
}


long PacketManager::numberOfPackets()
{
  return 0/*m_counts*/;
}


// store the raw data into the Vector: into m_packetVector
void PacketManager::savePacket( ptrPacketType p, struct pcap_pkthdr packetHeader, long ord, int frameType, int frameLen )
{
  if (!m_acceptingPacket)
    // the packet cannot be stored
    return;

  // not need to save really a packet because it is contained into the libpcap file
  // packet available
  m_bPacketAvailable = true;

  m_lastPacket = m_packet;
  m_packet.setPacketHeader( packetHeader );
  m_packet.setFrameNumber( ord );
  m_frameNumber = ord;
  m_packet.setFrameType( frameType );
  m_packet.setFrameHeaderLength( frameLen );
  m_packet.setPacket( p, packetHeader );

  if (ord == 1)
  {
    m_firstPacket = m_packet;
    kdDebug() << "++++++++ got first packet: m_bPacketAvailable: " << m_bPacketAvailable << endl;
  }

  // now the packet is saved
  //m_counts++;
  emit savedPacket(ord, this);
}

#include "packetmanager.moc"
