#include "propagator.h"


#include <string.h>
#include <time.h>
#include <sys/time.h>

Propagator::Propagator() {
  // reset some values
	aoslos=0.0;
	velocity=0.0;
	azimuth=0.0;
  elevation=0.0;
  range=0.0;
  range_rate=0.0;
  latitude=0.0;
  longitude=0.0;
  height=0.0;
	ma=0.0;
	footprint=0.0;
	aoslos=0.0;
  squint=0.0;
  orbnum=0;
}

Propagator::~Propagator() {
}

char *Propagator::daynum2String(double daynum)
{
	extern long timezone;
	tzset();
	struct tm TM;
  bzero(&TM, sizeof(tm));
	Calendar_Date(daynum, &TM);
	Time_of_Day(daynum, &TM);
	TM.tm_year-=1900;
	TM.tm_mon-=1;
	time_t t = mktime(&TM);
	t=t-timezone;
	char* r = ctime(&t);
	r[strlen(r)-1]=0;
	return r;
}


void Propagator::setQTH(double lon, double lat, double height) {
	obs_geodetic.lon=lon*de2ra;
	obs_geodetic.lat=lat*de2ra;
	obs_geodetic.alt=height/1000.0;
}

void Propagator::setTLE(char* line1, char* line2) {
	// create 3 lines to use with the Get_Next_Tle_Set function
	strcpy(tle_lines[0], "DUMMY");
	strcpy(tle_lines[1], line1);
	strcpy(tle_lines[2], line2);
}

void Propagator::calc() {
	struct tm utc;
  struct timeval tmval;
	UTC_Calendar_Now(&utc);
  gettimeofday(&tmval, NULL);
  double daynum=Julian_Date(&utc);
  daynum=daynum+(double)tmval.tv_usec/8.64e+10;
  calc(daynum);
}

void Propagator::calc(struct tm* utc, bool calcAosLos) {
  double daynum=Julian_Date(utc);
  calc(daynum, calcAosLos);
}

void Propagator::calc(double daynum, bool calcAosLos) {
  ClearFlag(ALL_FLAGS);
	Get_Next_Tle_Set(tle_lines, &tle);
  // copy the just created data to the local copy.
  // NOTE: We need this, as the following functions will
  //       overwrite some of the data!
  memcpy(&localtle, &tle, sizeof(tle_t));


	select_ephemeris(&tle);
	double jul_epoch, jul_utc, tsince, phase, age;
	vector_t vel = {0,0,0,0};
	vector_t pos = {0,0,0,0};
	vector_t obs_set;
	geodetic_t sat_geodetic;

	jul_utc = daynum;
	jul_epoch = Julian_Date_of_Epoch(tle.epoch);
	tsince = (jul_utc - jul_epoch) * xmnpda;

	// call the norad routines according to the deep-space flag
	if (isFlagSet(DEEP_SPACE_EPHEM_FLAG))
		SDP4(tsince, &tle, &pos, &vel, &phase);
	else
		SGP4(tsince, &tle, &pos, &vel, &phase);
	// scale position and velocity to km and km/sec
	Convert_Sat_State(&pos, &vel);

	// get the velocity of the satellite

	Magnitude(&vel);
	velocity=vel.w;

	Calculate_Obs(jul_utc, &pos, &vel, &obs_geodetic, &obs_set);
	Calculate_LatLonAlt(jul_utc, &pos, &sat_geodetic);

	azimuth=Degrees(obs_set.x);
	elevation=Degrees(obs_set.y);
	range=obs_set.z;
	range_rate=obs_set.w;
	latitude=Degrees(sat_geodetic.lat);
	longitude=Degrees(sat_geodetic.lon);
	height=sat_geodetic.alt;
	ma=Degrees(phase);
	ma=(256.0/360.0)*ma;
	Magnitude(&pos);
	footprint=2.0 * xkmper * acos(xkmper / pos.w);
  age=jul_utc - jul_epoch;
  orbnum=(long)floor((tle.xno*xmnpda/twopi+age*tle.bstar*ae)*age+tle.xmo/twopi)+tle.revnum-1;

	// get the aos / los only if needed
  hasaos=aosHappens();
	if(jul_utc>aoslos && calcAosLos && hasaos)
    aoslos=nextAosLos(jul_utc);

  /////////////////////////////////////////////////////////////////////////////////////////////////
  ////////////// SQUINT ///////////////////////////////////////////////////////////////////////////
  /////////////////////////////////////////////////////////////////////////////////////////////////

  if(squinttype==2) {


   double ax,ay,az;
   double bx,by,bz;
   double cx,cy,cz;
   double rx,ry,rz;
   double _alon,_alat;
   vector_t obs_pos,obs_vel;
   /* calculate antenna angle with line-of-nodes instead of with major axis, and convert to radians */
     _alon = alon*M_PI/180 + tle.omegao1;
     _alat = alat*M_PI/180;
   /* calculate unit vector in orbit plane coordinates: X to asc.node, Z perp.to orbit plane */
     bx = cos(_alat)*cos(_alon);
     by = cos(_alat)*sin(_alon);
     bz = sin(_alat);

   /* transform to coordinate system with X still to asc.node, Z along earth's rotation axis */
     cx = bx;
     cy = by*cos(tle.xincl1) - bz*sin(tle.xincl1);
     cz = by*sin(tle.xincl1) + bz*cos(tle.xincl1);
   /* transform to coordinate system with X to "First Point in Aries", i.e., the standard celestial coordinates ijk */
     ax = cx*cos(tle.xnodeo1) - cy*sin(tle.xnodeo1);
     ay = cx*sin(tle.xnodeo1) + cy*cos(tle.xnodeo1);
     az = cz;

   /* get the range vector */

   Calculate_User_PosVel(jul_utc, &obs_geodetic, &obs_pos, &obs_vel);
   rx = pos.x - obs_pos.x;
   ry = pos.y - obs_pos.y;
   rz = pos.z - obs_pos.z;

   /* cos(squint) is now just minus the inner product of (ax,ay,az) and (rx,ry,rz) normalised */
   squint=Degrees( acos ( - (ax*rx+ay*ry+az*rz) / sqrt(rx*rx+ry*ry+rz*rz) ));
  }

    if(squinttype==0) squint=-1000.0;    // no squint
    if(squinttype==1) {
      double r = height + xkmper;
      squint=Degrees(acos((range*range + r*r + - xkmper*xkmper) / (2*range*r)));
  }

}

long long Propagator::getDoppler(unsigned long long freq) {
  double result;
  double f = freq;
  result = -f*((range_rate*1000.0)/Cl);
  return (long long)result;
}

double Propagator::nextAosLos(double jul_utc) {
  double result;
  if (elevation < 0.00)
    result=FindAOS(jul_utc);
  else
    result=FindLOS(jul_utc);
  return result;
}

int Propagator::getOrbitNum(){
  return orbnum;
}
/** reset the aos los time - needed for the propagation lists as these change that time! */
void Propagator::resetAosLos(){
  aoslos=0.0;
}

bool Propagator::hasAos() {
  return hasaos;
}

// this code is stolen from predict.c. Thanks, John, KD2BD

bool Propagator::aosHappens() {
  bool geo, canreach;
  double lin, sma, apogee;
  // first test if the satellite is geostationary
  if (fabs(localtle.xno-1.0027)<0.0002)
    geo=1; else geo=0;

  // does the sat appear on our qth ?

  if (localtle.xno==0.0)
    canreach=0;
  else
  {
    lin=localtle.xincl;

    if (lin>=90.0)
      lin=180.0-lin;

    sma=331.25*exp(log(1440.0/localtle.xno)*(2.0/3.0));
    apogee=sma*(1.0+tle.eo)-xkmper;

    if ((acos(xkmper/(apogee+xkmper))+(lin*de2ra)) > fabs(obs_geodetic.lat))
      canreach=1;
    else
      canreach=0;
  }
  if (!geo && canreach) return true; else return false;
}

double Propagator::FindAOS(double daynum)
{
	/* This function finds and returns the time of AOS (aostime). */
  double _daynum=daynum;
	double aostime=0.0;

	if (hasAos())
	{
		calc(daynum, false);

		/* Get the satellite in range */

		while (getElevation()<-1.0)
		{
			daynum-=0.00035*(getElevation()*((getHeight()/8400.0)+0.46)-2.0);
			calc(daynum, false);
		}

		/* Find AOS */

		while (aostime==0.0)
		{
			if (fabs(getElevation())<0.03)
				aostime=daynum;
			else
			{
				daynum-=getElevation()*sqrt(getHeight())/530000.0;
        calc(daynum, false);
			}
		}
	}
  calc(_daynum, false);
	return aostime;
}

double Propagator::FindLOS(double daynum)
{
  double _daynum=daynum;
  double lostime=0.0;

	if (hasAos())
	{
		calc(daynum, false);

		do
		{
			daynum+=getElevation()*sqrt(getHeight())/502500.0;
			calc(daynum, false);

			if (fabs(getElevation()) < 0.03)
				lostime=daynum;

		} while (lostime==0.0);
	}

  calc(_daynum, false);
  return lostime;
}
unsigned int Propagator::catnr() {
  return tle.catnr;
}
