#--
# Copyright (C) 2008-2009 Harald Sitter <apachelogger@ubuntu.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of
# the License or (at your option) version 3 or any later version
# accepted by the membership of KDE e.V. (or its successor approved
# by the membership of KDE e.V.), which shall act as a proxy
# defined in Section 14 of version 3 of the license.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
#++

SCRIPT = File.basename($0)
KEY    = ENV['DEBFULLNAME'] # -k'name' to use for signing

# config
cfgfile   = "#{ENV['HOME']}/.batrc"
globalcfg = "/etc/batrc"
if File.exist?(cfgfile+"_path") and not File.exist?(cfgfile)
    oldpwd = Dir.pwd
    Dir.chdir(ENV['HOME'])
    %x[wget #{IO.readlines(cfgfile+"_path")[0]}]
    if $? == 0
        require 'kreadwriteconfig'
        $cfg = KConfig.new(cfgfile)
    else
        exit 1
    end
    Dir.chdir(oldpwd)
elsif File.exist?(cfgfile)
    require 'kreadwriteconfig'
    $cfg = KConfig.new(cfgfile)
elsif File.exist?(globalcfg)
    require 'kreadwriteconfig'
    $cfg = KConfig.new(globalcfg)
    if SCRIPT.include?("bat") and not SCRIPT.include?("batpaste") and not SCRIPT.include?("batpull")
        puts "YOU DON'T HAVE A .BATRC, NOR A .BATRC_PATH, IF YOU WANT TO EMBRACE
        THE FULL POWER OF THE BATSCRIPTS YOU SHOULD CONTACT A NINJA TO GET ONE OF THOSE"
    end
else
    puts "ERRRRRRRRRRROR: File #{cfgfile} not found.\n#{SCRIPT} uses the bat library, which needs this config for most features to work properly."
    exit 1
end

COORDINATOR   = $cfg.read("Ubuntu","COORDINATOR")
TD            = $cfg.read("Ubuntu","TD")
BD            = $cfg.read("Ubuntu","BD")
SV            = $cfg.read("Ubuntu","STANDARDSVERSION")
ARCHIVE       = $cfg.read("Ubuntu","ARCHIVE")
PPA           = $cfg.read("Ubuntu","PPA")

UBUVER        = $cfg.read("Ubuntu","VERSION")
KDEVER        = $cfg.read("Bat","KDEVERSION")
JOINEDVER     = KDEVER.split(".").join if KDEVER

HOST          = $cfg.read("Bat","HOST")
REMOTEPATH    = $cfg.read("Bat","REMOTEPATH")
TARPATH       = $cfg.read("Bat","TARPATH")
REMOTETARPATH = $cfg.read("Bat","REMOTETARPATH")
TARURL        = $cfg.read("Bat","TARURL")

# allow people to use custom wrapper scripts named != pbuilder
@pbuilder = "pbuilder"
if ENV['PBUILDER']
    @pbuilder = ENV['PBUILDER']
end

require 'fileutils'

include FileUtils

module Bat
    # Information if require can't load the required file, It can tell the user
    # what package to install in order to fix a failed require.
    def package_require(requiree,package)
        begin
            require requiree
        rescue LoadError
            err "Pardon mon ami, but you really should install #{package}.\nk, thx, bye"
        end
    end

    # Output script synopsis.
    # mandatory contains a comma seperated list of mandatory arguments, while
    # optional contains a comma seperated list of optional arguments.
    #
    #     synopsis("PACKAGENAME,PATH")       #=> ./batget.rb PACKAGENAME|PATH
    #     synopsis("PATH","download,upload") #=> ./batget.rb PATH [download|upload]
    #     synopsis(,"download,upload")       #=> ./batget.rb [download|upload]
    #
    # Bat::Optparser provides a more powerful implementation of option parsing,
    # in a lot of cases you would only need a simple approach however. For
    # example when you only parse one argument anyway.
    def synopsis(mandatory='',optional='')
        string = ""
        for mandatory in mandatory.split(",")
            string += " #{mandatory}"
        end
        for optional in optional.split(",")
            string += " [#{optional}"
        end
        string += "]" unless optional.empty?
        puts "#{SCRIPT}#{string}"
        exit 1
    end

    # Parse changelog. It reads the very first line of debian/changelog and
    # splits it into application, version, distribution and urgency.
    def parseCh()
        chline   = IO.readlines("debian/changelog")[0]
        chline   = chline.split(" ") # ["kdepim","(4:4.1.1a-0ubuntu1)","intrepid;","urgency=low"]
        @app     = chline[0]
        @version = chline[1].gsub("(","").gsub(")","")
        # remove the epoch if there is one
        if @version.include?(":")
            @version = @version.split(":")
            @epoch   = @version[0]
            @version = @version[1]
        end
        @distro  = chline[2].delete(";")
        @urgency = chline[3]
    end

    # Returns a changelog entry line.
    def composeCh(app, version, distro, epoch=nil, urgency="urgency=low")
        return "#{app} (#{epoch + ":" if epoch}#{version}) #{distro}; #{urgency}"
    end

    # Sanity checks changelog. It expects to be executed in ../ from the
    # changelog (i.e. it will read debian/changelog and fail if it is not
    # readable).
    #
    # Depending on your changelog entry it will prevent unwanted backports or
    # uploads with wrong version number to target distribution (TD)
    # If the suffix indicates BD but the distribution indacates TD it exists
    # with code 1 after explaining the situtation using a KDialog. If the suffix
    # doesn't indicate TD but the version does it will query for user input and
    # possibly exit with code 1.
    def checkCh()
        pt "sanity check changelog"
        parseCh()
        if @version.include?(BD) and not @distro.include?(BD)
            %x[kdialog --error "Version (#{@version}) includes '#{BD}' but series is #{@distro}\n
            ABORTING" > /dev/null 2>&1]
            exit 1
        end
        if not @distro.include?(TD) and not @version.end_with?("-0ubuntu1")
            %x[kdialog --warningcontinuecancel "Version (#{@version}) should end with -0ubuntu1.\n
            Do you want to continue?" > /dev/null 2>&1]
            exitchecker($?,"bye")
        end
        if @distro.downcase == "unreleased"
            %x[kdialog --warningcontinuecancel "Version (#{@version}) should end with -0ubuntu1.\n
            Do you want to continue?" > /dev/null 2>&1]
            exitchecker($?,"bye")
        end
    end
    # Nuke given directory and recreate it. If enter is set to true it will
    # also enter the directory right away.
    def dir_revive(dir,enter=false)
        FileUtils.rm_rf(dir)
        Dir.mkdir(dir)
        Dir.chdir(dir) if enter
    end

    # Evaluates error code and puts a string as warning (using wrn) if
    # keeprunning is true and code != 0. If keeprunning is false it calls err.
    def exitchecker(exitcode,string="The most recent system call didn't exit properly.",keeprunning=false)
        if exitcode != 0 and keeprunning
            wrn(string)
        elsif exitcode != 0 and not keeprunning
            err(string)
        end
        return exitcode
    end

    # Gets latest available source from Launchpad, using pull-lp-source from
    # the ubuntu-dev-tools package.
    # It expects at least the application name and will use the default target
    # distribution unless something else gets defined. If the used prev
    # directory is already available it will use Bat::dir_revive to nuke it and
    # recreate it.
    def get_prev(app,distro=TD,keeprunning=false)
        pt "creating and entering previous #{app} version dir"
        dir_revive(dir="prev",enter=true)

        pt "getting old source for #{app} in #{distro}"
        lp = PullLP.new(app,distro)

        if not lp.response
            return false
        end

        lp.dget

        string = "b0rked. dgetting #{app} from #{distro} didn't work very well :-("
        exitchecker($?,string,keeprunning)
    end

    # Gets latest bzr branch from a specified kubuntu-members application branch.
    def get_branch(app,branch=k_m_branch_url_constructor(app),keeprunning=false)
        pt "getting packaging for #{app} from #{branch}"
        bzr = Bzr.new(branch,"./batbranch")
        bzr.branch

        string = "b0rked. bzr branching #{app} from #{branch} didn't work very well :-("
        Dir.chdir("./batbranch"); bzr.push; Dir.chdir("..") if exitchecker($?,string,keeprunning) == 0
    end

    # Parses a given application or language name according to a predefined
    # conversation list.
    # The default conversion would be from Kubuntu names to upstream names (e.g.
    # kde4libs => kdelibs).
    def name_converter(name,lang=false,invert=false)
        unless lang
            _apps = {
            "kde4libs"       => "kdelibs",
            "kdewebdev-kde4" => "kdewebdev"
            }
            _apps = _apps.invert if invert
            _apps.each_pair{ | x, y |
                if name == x
                    name = y
                end
            }
            return name
        else
            subs = false
            _langs = {
                "bn_IN"     => "bnin",
                "en_GB"     => "engb",
                "pt_BR"     => "ptbr",
                "sr@latin"  => "srlatin",
                "zh_CN"     => "zhcn",
                "zh_TW"     => "zhtw"
                }
            _langs = _langs.invert if invert
            _langs.each_pair{ | x, y |
                if name.include?(x)
                    subs = x
                    name = y
                end
            }
            return [name,subs]
        end
    end

    # Debuild -S -sa -k"KEY". This method expects to be used within the main
    # source tree directory.
    def debuild
        pt "dpkg-buildpackage"
        log = %x[dpkg-buildpackage -S -sa -k\"#{KEY}\" > /dev/stdout 2>&1]
        if $? != 0
            puts log
            err("dpkg-buildpackage didn't exit properly")
        end
        return log
    end

    # Dput to defined location. If no location is defined it will query for
    # input via gets (thus this won't work with scripts that require an argument
    # by default, not yet anyway).
    def dput(location=nil)
        if location == nil
            pt("Please set the location to dput to [kubuntu-ninjas/ubuntu/...]")
            location = gets
        end
        system("dput #{location} *dsc")
    end

    # Wrapper for wget.
    #
    # Before running wget it will try to remove the file passed as argument url.
    # That said Bat::wget should only be used if this behavior is expected (e.g
    # when replacing files with an online stored more up-to-date version).
    #
    #     Dir.glob("*")                             #=> ["example.rb"]
    #     Bat::wget("http://example.tld/examle.rb")
    #     Dir.glob("*")                             #=> ["example.rb"]
    #
    #     Dir.glob("*")                             #=> ["example.rb"]
    #     %x[wget "http://example.tld/examle.rb"]
    #     Dir.glob("*")                             #=> ["example.rb", "example.rb.1"]
    def wget(url)
        FileUtils.rm_f(File.basename(url))
        system("wget #{url}")
    end

    # Wrapper around IO#puts. It will add custom formating to ensure batmessages
    # are more visible than regular terminal output. It expects @batpart to be
    # set. Thus you can influence it at runtime, eventually this will change to
    # harddetection of the part using the script name.
    #
    #     pt("Reading your emails")               #=> .!!!~~~~>Bat batget: Reading your emails
    #     pt("Reading your emails", " seriously") #=> .!!!~~~~>Bat batget seriously: Reading your emails
    def pt(string, add='')
        bat = "Bat " if SCRIPT.downcase.start_with?("bat")
        @batpart = SCRIPT.sub(/.*bat/,"") unless @batpart
        add = " #{add}" if not add.start_with?(" ") and not add.empty?
        puts(".!!!~~~~>#{bat}#{@batpart}#{add}: #{string}")
    end

    # Wrapper around Bat::pt. It will add the word WARNING to the output.
    def wrn(string)
        pt(string," WARNING")
    end

    # Wrapper around Bat::pt. It will add the word ERROR to the output.
    # And exit with code 1.
    def err(string)
        pt(string," ERROR")
        exit 1
    end

    # Opens SSL connection to launchpad.net and tries to get url. If the HTTP
    # response is not HTTPOK, it will invoke a warning and return without value.
    # Otherwise it returns the response object.
    def get_lp_page(url)
        require 'net/https'
        h = Net::HTTP.new("edge.launchpad.net", 443)
        h.use_ssl = true
        h.verify_mode = OpenSSL::SSL::VERIFY_NONE
        response = h.get(url)
        unless response.code == "200"
            wrn("HTTP Error: #{response.message}")
            return
        end
        return response
    rescue SocketError
        wrn("something went wrong :-P")
    end

    # Construct URL for a package owned by the kubuntu-members team on LP
    def k_m_branch_url_constructor(package)
        return "lp:~kubuntu-members/#{package}/ubuntu"
    end

    # Returns true if File is available and the only one of it's kind (necessary
    # if a wildcard is used).
    def avail_n_single(file)
        avail = false
        if (fs = Dir.glob(file)).count == 1
            file = fs[0]
        elsif not fs or fs.count == 0
            wrn "#{file} not available :-("
            file = nil
        else
            p fs
            err "Too many versions of #{file} available (expected one)!"
        end
        return file
    end

    # Constructs very basic option parser object. You still will have to spend a
    # fair amount of time on creating the actual options. If I knew how to pass the
    # desired variable name as exactly that, a variable name, to be used for
    # enhancing the OpenStruct one could reduce adding an item to one method call
    # with 4 arguments. So imagine the following being actually 3 lines if I was a
    # better haxx0r (or ultimately I had intarwebs access right now ;-)
    #
    # I imagine something like Optparser::add_bool() to add new options in a
    # comfortable way...
    #
    #     Optparser.new()
    #     $options.build = true
    #     $opts.on("-b", "--no-build", "Don't testbuild") do |nobuild|
    #         $options.build = nobuild
    #     end
    #     $opts.parse!(ARGV)
    #
    # For more information please consult the OptionParser documentatin. The basic
    # method structure is short option | long option | description | do something.
    # You can also describe mandatory arguments by writing them in upper case, or
    # optional arguments using [ARG].
    #      $opts.on("-s", "--build-slave SLAVE", "SLAVE makes it a mandatory argument for --build-slave")
    class Optparser
        require 'optparse'
        require 'ostruct'
        def initialize()
            $options = OpenStruct.new

            $opts = OptionParser.new{ |$opts|
                $opts.banner = "Usage: #{SCRIPT} [options]"

                $opts.on_tail("-h", "--help", "Show this message"){
                    puts $opts
                    exit
                }
            }
        end
    end

    # Download the dsc, diff, tarball or all of them of a specific package in a
    # specific pocket from Launchpad.
    class PullLP
        attr_reader :response

        # Try to get the page for package in release using get_lp_page from the Bat module.
        # If no release is specified the target distribution set in the Bat
        # module is used.
        def initialize(package,release=TD)
            @response = get_lp_page("/ubuntu/#{release.downcase}/+source/#{package.downcase}")
        end

        def url_constructor(ext)
            url = @response.body.match(/a href=\"(.*\.#{ext})\"/)[1]
            return "https://launchpad.net/#{url}" unless url.empty?
        end
        private :url_constructor

        # Invoke dget on a automatically constructed URL
        def dget
            %x[dget -xu #{url_constructor("dsc")}]
        end

        # Invoke wget on a automatically constructed URL for the orig.tar.gz tarball
        # (replaces the file if it is already present in the working directory)
        def get_tar
            wget(url_constructor("orig.tar.gz"))
        end

        # Invoke wget on a automatically constructed URL for the dsc file
        # (replaces the file if it is already present in the working directory)
        def get_dsc
            wget(url_constructor("dsc"))
        end

        # Invoke wget on a automatically constructed URL for the diff.gz file
        # (replaces the file if it is already present in the working directory)
        def get_diff
            wget(url_constructor("diff.gz"))
        end
    end

    class Bzr
        def initialize(branch,path,cd=false)
            @branch    = branch
            @path      = path
            @branch_cd = cd
        end

        def cder(path=@path)
            if path and File.exist?(path) and @branch_cd and not File.basename(Dir.pwd) == path
                @pwd = Dir.pwd
                Dir.chdir(path)
            end
        end
        private :cder

        def branch
            %x[bzr branch #{@branch} #{@path}]
        end

        def commit
            cder
            %x[bzr commit]
            cder(@pwd)
        end

        def merge
            cder
            %x[bzr merge #{@branch}]
            cder(@pwd)
        end

        def push
            cder
            %x[bzr push #{@branch}]
            cder(@pwd)
        end
    end

    class OrigBuilder
        attr_reader :bz, :tar, :gz, :orig, :name, :version

        def initialize(bz,name=nil,version=nil)
            basename = File.basename(bz,".tar.bz2")
            @bz  = bz
            @tar = basename + ".tar"
            @gz  = @tar + ".gz"

            # when trying to autodetect name and version assume only the portion
            # after last hyphen is the version (always true for core KDE)
            unless name
                @name = basename.split("-")[0..-2].join("-").to_s
            else
                @name = name
            end

            unless version
                @version = basename.split("-")[-1].to_s
            else
                @version = version
            end

            @orig = @name + "_" + @version + ".orig.tar.gz"
        end

        def bunzip
            pt "bunzipping #{bz}"
            %x[bunzip2 #{bz}]
        end

        def gzip
            pt "gzipping #{tar}"
            %x[gzip -9fn #{tar}]
        end

        def rename
            pt "renaming to #{orig}"
            mv(gz,orig)
        end

        def convert
            bunzip
            gzip
            rename
        end
    end
end

# import namespace
include Bat
