#--
# Copyright (C) 2008-2009 Harald Sitter <apachelogger@ubuntu.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of
# the License or (at your option) version 3 or any later version
# accepted by the membership of KDE e.V. (or its successor approved
# by the membership of KDE e.V.), which shall act as a proxy
# defined in Section 14 of version 3 of the license.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#++
# Wrapper library for KDE script utilities kwriteconfig and kreadconfig.
#--

class KConfig
    # Ensures the given file exists, is a file and is writable. If the file is
    # in fact a relative path it will be hold relative to KDE's local config
    # directory (usually $HOME/.kde/share/config).
    #
    #   KConfig.new("../apps/amarok/dbrc") => "/home/user/.kde/share/apps/amarok/dbrc"
    #
    # If the initialization gets invoked with the relative variable set to true
    # the path will be hold relative to the current working directory instead of
    # KDE's local config.
    #
    #   KConfig.new("../apps/amarok/dbrc",true) => "/home/user/apps/amarok/dbrc"
    def initialize(file,relative=false)
        if relative?(file)
            unless relative
                cpath = %x{kde4-config --path config}.split(":")[0]
                file = File.expand_path(file,cpath)
            else
                file = File.expand_path(file)
            end
        end
        if not File.file?(file)
            raise "#{file} is not a file."
        end
        @rfile = file
        @file  = "--file #{file} "
    end

    # Read config entries using kreadconfig.
    # * group is the configuration group (e.g. [KDE])
    # * key is the configuration key (e.g. Color)
    #
    # To read the following config:
    #   [KDE]
    #   Color=true
    #
    #   KConfig.new("kderc").read("KDE","Color","bool") => true
    def read(group,key)
        if not File.exist?(@rfile)
            raise "#{@rfile} does not exist."
        end
        @cmd = "kreadconfig "
        worker(group,key)
    end

    # Write config entries using kwriteconfig.
    # * group is the configuration group (e.g. [KDE])
    # * key is the configuration key (e.g. Color)
    # * value is the key value (e.g. Blue)
    #
    # To write the following config:
    #   [KDE]
    #   Color=Blue
    #
    #   KConfig.new("kderc").write("KDE","Color","Blue")
    def write(group,key,value)
        if not File.exist?(@rfile)
            File.new(@rfile, "w+")
        end
        if not File.writable?(@rfile)
            raise "#{@rfile} not writable."
        end
        @cmd = "kwriteconfig "
        worker(group,key,value)
    end

    def worker(group,key,value=nil)
        cmd = @cmd
        cmd += @file
        cmd += "--group \"#{group}\" "
        cmd += "--key \"#{key}\" "
        cmd += "\"#{value}\"" if value

        output = %x{#{cmd}}.chomp!()
        if $? != 0
            puts("The command failed to exit properly (#{cmd}).")
            exit 1
        elsif not output
            return
        elsif not output.empty?
            return output
        else
            puts("No entry in #{group}, #{key}.")
            return false
        end
    end
    private :worker

    def chop_basename(path)
        base = File.basename(path)
        if /\A#{File::SEPARATOR}?\z/ =~ base
            return nil
        else
            return path[0, path.rindex(base)], base
        end
    end
    private :chop_basename

    def relative?(path)
        while r = chop_basename(path)
            path, basename = r
        end
        path == ''
    end
    private :relative?
end
