using System;
using System.Collections;
using System.Xml;
using System.Xml.Serialization;
using System.Xml.Schema;

namespace Data
{

public enum AccountType
{
    Asset,
    Liability,
    Equity,
    Income,
    Expense
}

public delegate void AccountEvent (Account account);

public class AccountComparer : IComparer
{
    public int Compare (object x, object y)
    {
        Account accountX = x as Account;
        Account accountY = y as Account;

        return accountX.No.CompareTo (accountY.No);
    }
}

public class AccountSortComparer : IComparer
{
    public int Compare (object x, object y)
    {
        Account accountX = x as Account;
        Account accountY = y as Account;

        return accountX.SortOrder.CompareTo (accountY.SortOrder);
    }
}


[XmlRoot]
public class Account : ICloneable
{
    [XmlAttribute] public AccountType Type;
    [XmlAttribute] public int SortOrder = -1;
    [XmlAttribute] public int No = -1;
    [XmlAttribute] public int ParentRef = -1;
    [XmlAttribute] public string Name;
    [XmlAttribute] public string Description = "";

    [XmlArrayItem(Type=typeof(Entry))]
    public ArrayList Entries = new ArrayList ();

    public object Clone ()
    {
        Account obj = new Account ();
        obj.Type = this.Type;
        obj.SortOrder = SortOrder;
        obj.No = No;
        obj.ParentRef = ParentRef;
        obj.Name = (string) Name.Clone ();
        obj.Description = (string) Description.Clone ();

        foreach (Entry entry in Entries)
        {
            obj.Entries.Add (entry.Clone ());
        }

        return obj;
    }

    [XmlIgnore] public decimal Debit
    {
        get
        {
            if (Entries.Count > 0)
                return (Entries[Entries.Count - 1] as Entry).DebitBalance;
            else
                return 0;
        }
    }

    [XmlIgnore] public decimal Credit
    {
        get
        {
            if (Entries.Count > 0)
                return (Entries[Entries.Count - 1] as Entry).CreditBalance;
            else
                return 0;
        }
    }

    [XmlIgnore] public decimal GroupDebit
    {
        get
        {
            decimal debit = 0;

            foreach (Account account in Children)
            {
                debit += account.GroupDebit;
            }

            return debit + Debit;
        }
    }

    [XmlIgnore] public decimal GroupCredit
    {
        get
        {
            decimal credit = 0;

            foreach (Account account in Children)
            {
                credit += account.GroupCredit;
            }

            return credit + Credit;
        }
    }

    [XmlIgnore] public Entry FirstEntry
    {
        get
        {
            if (Entries.Count > 0)
                return Entries[0] as Entry;
            else
                return null;
        }
    }

    [XmlIgnore] public Entry LastEntry
    {
        get
        {
            if (Entries.Count > 0)
                return Entries[Entries.Count - 1] as Entry;
            else
                return null;
        }
    }

    [XmlIgnore] public ArrayList Children = new ArrayList ();

    [XmlIgnore] public Account parentAccount;
    [XmlIgnore] public Account ParentAccount
    {
        get
        {
            return parentAccount;
        }
        set
        {
            parentAccount = value;
            if (value != null)
                ParentRef = value.No;
            else
                ParentRef = -1;
        }
    }

    [XmlIgnore] public object Node;
    [XmlIgnore] public object ParentNode;

    [XmlIgnore] public decimal DatedDebit;
    [XmlIgnore] public decimal DatedCredit;

    [XmlIgnore] public decimal Balance
    {
        get
        {
            if ( Type == AccountType.Asset || Type == AccountType.Expense )
                return Debit - Credit;
            else
                return Credit - Debit;
        }
    }

    [XmlIgnore] public decimal GroupBalance
    {
        get
        {
            if ( Type == AccountType.Asset || Type == AccountType.Expense )
                return GroupDebit - GroupCredit;
            else
                return GroupCredit - GroupDebit;
        }
    }

    [XmlIgnore] public decimal DatedBalance
    {
        get
        {
            if ( Type == AccountType.Asset || Type == AccountType.Expense )
                return DatedDebit - DatedCredit;
            else
                return DatedCredit - DatedDebit;
        }
    }



    [XmlIgnore] public decimal DatedDebitBalance
    {
        get
        {
            if (DatedDebit > DatedCredit)
                return DatedDebit - DatedCredit;
            else
                return 0;
        }
    }

    [XmlIgnore] public decimal DatedCreditBalance
    {
        get
        {
            if (DatedCredit > DatedDebit)
                return DatedCredit - DatedDebit;
            else
                return 0;
        }
    }

    [XmlIgnore] public decimal DatedGroupDebit
    {
        get
        {
            decimal debit = 0;

            foreach (Account account in Children)
            {
                debit += account.DatedGroupDebit;
            }

            return debit + DatedDebit;

        }
    }

    [XmlIgnore] public decimal DatedGroupCredit
    {
        get
        {
            decimal credit = 0;

            foreach (Account account in Children)
            {
                credit += account.DatedGroupCredit;
            }

            return credit + DatedCredit;
        }
    }

    [XmlIgnore] public decimal DatedGroupBalance
    {
        get
        {
            if ( Type == AccountType.Asset || Type == AccountType.Expense )
                return DatedGroupDebit - DatedGroupCredit;
            else
                return DatedGroupCredit - DatedGroupDebit;
        }
    }

    public event EntryEvent EntryAdded;
    public event EntryEvent EntryRemoved;

    public void DoEntryAdded (Entry entry)
    {
        if (EntryAdded != null)
            EntryAdded (entry);
    }

    public void DoEntryRemoved (Entry entry)
    {
        if (EntryRemoved != null)
            EntryRemoved (entry);
    }

    public Account ()
    {
    }

    public Account (string name, AccountType type)
    {
        this.Name = name;
        this.Type = type;
    }

    public void Recalculate (int startNo)
    {
        int c = Entries.Count;

        if (startNo < 0 || startNo >= c)
            return;


        System.Diagnostics.Debug.WriteLine ("Account " + Name +
                                            " recalculated, starting from entry " +
                                            (startNo + 1).ToString () + " of " + c.ToString ());

        if (c < 1)
            return;

        Entry prevEntry = null;
        Entry entry = null;

        if (startNo == 0)
        {
            entry = Entries [0] as Entry;
            entry.DebitBalance = entry.Debit;
            entry.CreditBalance = entry.Credit;
            startNo++;
        }

        for (int i = startNo; i < c; i++)
        {
            entry = Entries [i] as Entry;
            prevEntry = Entries [i - 1] as Entry;

            entry.DebitBalance = entry.Debit + prevEntry.DebitBalance;
            entry.CreditBalance = entry.Credit + prevEntry.CreditBalance;
        }
    }
}



}
