using System;
using Gtk;
using Glade;
using GtkExtra;

namespace Kurush
{

public class SummaryFrame : Frame
{
    public override void UpdateActions (Actions actions)
    {
        actions.EditAdd.Enabled = true;
        actions.FilePrint.Enabled = true;
        actions.ViewExpandCollapse.Enabled = true;

        networthLabel.Text = Format.Value (Owner.DataBase.Networth);
        ViewSelectionChanged (null, null);
    }

    [Glade.Widget] private ScrolledWindow summaryViewContainer;
    [Glade.Widget] private Label networthLabel;

    private SummaryView view;
    private NodeStore store;
    private SummaryNode[] groupNodes;
    private Data.Account selectedAccount = null;
    private Data.Group selectedGroup = null;

    public bool ActionAdd = false;
    public bool ActionRemove = false;
    public bool ActionDetails = false;
    public bool ActionProperties = false;

    private Menu editContextMenu;

    private void AddAccount (Data.Account account)
    {
        SummaryNode node = new SummaryNode (account);
        account.Node = node;

        if (account.ParentRef < 0)
        {
            account.ParentNode = groupNodes[(int) account.Type];
            groupNodes[(int) account.Type].AddChild (node);
        }
        else
        {
            account.ParentNode = account.ParentAccount.Node;
            (account.ParentNode as TreeNode).AddChild (node);
        }

        foreach (Data.Account childAccount in account.Children)
        {
            AddAccount (childAccount);
        }
    }

    public SummaryFrame (FrameManager owner) : base (owner, "summaryFrame")
    {
        MenuVisible = false;

        groupNodes = new SummaryNode [5];

        store = new NodeStore (typeof (SummaryNode));

        Owner.AccountsView = new SummaryView (store, false);
        Owner.AccountsView.Show ();
        ScrolledWindow sw = new ScrolledWindow ();
        sw.Add (Owner.AccountsView);
        sw.ShadowType = ShadowType.In;
        sw.Show ();
        Owner.AccountsPopup = new Window (WindowType.Popup);
        Owner.AccountsPopup.Add (sw);

        for (int i = 0; i < 5; i++)
        {
            groupNodes[i] = new SummaryNode (Owner.DataBase.Groups[i]);
            store.AddNode (groupNodes[i]);
        }

        foreach (Data.Group group in Owner.DataBase.Groups)
        {
            foreach (Data.Account account in group.Children)
            {
                AddAccount (account);
            }
        }

        view = new SummaryView (store, true);
        view.Show ();
        summaryViewContainer.Add (view);
        view.Selection.Changed += new EventHandler (ViewSelectionChanged);
        view.DoubleClicked += new EventHandler (ViewDoubleClicked);
        Owner.DataBase.AccountAdded += new Data.AccountEvent (DataAccountAdded);
        Owner.DataBase.AccountRemoved += new Data.AccountEvent (DataAccountRemoved);

    }

    private void DataAccountAdded (Data.Account account)
    {
        AddAccount (account);
    }

    private void DataAccountRemoved (Data.Account account)
    {
        (account.ParentNode as TreeNode).RemoveChild (account.Node as TreeNode);
    }

    private void ViewSelectionChanged (object obj, EventArgs args)
    {
        if (!Content.Visible)
            return;

        SummaryNode node = view.GetSelected () as SummaryNode;

        if (node != null)
        {
            selectedGroup = node.Group;
            selectedAccount = node.Account;
        }
        else
        {
            selectedGroup = null;
            selectedAccount = null;
        }


        if (selectedAccount != null)
        {
            Owner.Actions.EditDetails.Enabled = true;
            Owner.Actions.EditProperties.Enabled = true;
            Owner.Actions.EditRemove.Enabled = true;
            Owner.Actions.EditMove.Enabled = true;
        }
        else
        {
            Owner.Actions.EditDetails.Enabled = false;
            Owner.Actions.EditProperties.Enabled = false;
            Owner.Actions.EditRemove.Enabled = false;
            Owner.Actions.EditMove.Enabled = false;
        }
    }

    public override void Add ()
    {
        AccountPropertiesFrame accountProperties =
            new AccountPropertiesFrame (Owner, null);

        if (selectedAccount != null)
        {
            accountProperties.Type = selectedAccount.Type;
            accountProperties.ParentAccount = selectedAccount;
        }
        else if (selectedGroup != null)
            accountProperties.Type = selectedGroup.Type;


        Owner.Go (accountProperties);
    }

    public override void Remove ()
    {
        if (selectedAccount != null)
            Owner.DataBase.RemoveAccount (selectedAccount);
    }

    private void ViewDoubleClicked  (object obj, EventArgs args)
    {
        Details ();
    }

    public override void Details ()
    {
        if (selectedAccount != null)
        {
            if (!Owner.Activate ( typeof(DetailsFrame),
                                  selectedAccount) )
            {
                DetailsFrame detailsFrame =
                    new DetailsFrame (Owner, selectedAccount);

                Owner.Go (detailsFrame);
            }
        }
    }

    public override void Properties ()
    {
        if (selectedAccount != null)
        {
            if (!Owner.Activate ( typeof(AccountPropertiesFrame),
                                  selectedAccount) )
            {
                AccountPropertiesFrame accountProperties =
                    new AccountPropertiesFrame (Owner, selectedAccount);

                Owner.Go (accountProperties);
            }
        }
    }

    public override void PrintReport ()
    {
        new SummaryPrintDialog (Owner);
    }

    public override void ExpandAll ()
    {
        view.ExpandAll ();
    }

    public override void CollapseAll ()
    {
        view.CollapseAll ();
    }

    public override void MoveUp ()
    {

    }

    public override void MoveDown ()
    {

    }

}

}
