#! /bin/sh

###############################################################################
#
#	Run this to generate the Makefile templates, etc.
#
###############################################################################

set -e

function RUN_TOOL()
{
	if [ -z "$*" ]; then return; fi
	echo "... Running $*"
	$*
}

###############################################################################
#
#
PKG_NAME="kvirc"
DIE=0
GEN_FILES="configure.ac"
ACLOCAL_FLAGS="-I m4"

echo "... Generating build files for $PKG_NAME"

(autoconf --version) < /dev/null > /dev/null 2>&1 || {
	echo
	echo "**Error**: You must have GNU \`autoconf' installed."
	DIE=1
}

(grep "^AM_PROG_LIBTOOL" $GEN_FILES > /dev/null) && {
	(libtool --version) < /dev/null > /dev/null 2>&1 || {
		echo
		echo "**Error**: You must have GNU \`libtool' installed."
		DIE=1
	}
}

(automake --version) < /dev/null > /dev/null 2>&1 || {
	echo
	echo "**Error**: You must have GNU \`automake' installed."
	DIE=1
	NO_AUTOMAKE=yes
}

# if no automake, do not bother testing for aclocal
test -n "$NO_AUTOMAKE" || (aclocal --version) < /dev/null > /dev/null 2>&1 || {
	echo
	echo "**Error**: Missing \`aclocal'. The version of \`automake'"
	echo "installed does not appear recent enough."
	DIE=1
}

if test "$DIE" -eq 1; then
	exit 1
fi

case $CC in
xlc)
	am_opt=--include-deps;;
esac

#
# We have already patched the necessary files to work with gettext
# but we run it here to generate the m4/ and po/ directories.
# We'll run a patched gettextize to avoid the annoying (and unnecessary)
# prompt at the end.
#
cp -f `which gettextize` .
patch -p0 < admin/patches/gettextize.patch || \
	(echo Ignoring this error... you may want update admin/patches/gettextize.patch; true)
RUN_TOOL ./gettextize --copy --force
#
# Clean up Makefile.am and configure.ac after gettextize is done
#
patch -p0 < admin/patches/gettext.patch
rm -f gettextize Makefile.am~ ChangeLog~ configure.ac~

for coin in $GEN_FILES
do
	dr=`dirname $coin`
	if test -f $dr/NO-AUTO-GEN; then
		echo "... skipping $dr -- flagged as no auto-gen"
	else
		echo "... processing \"$dr\""
		macrodirs=`sed -n -e 's,AM_ACLOCAL_INCLUDE(\(.*\)),\1,gp' < $coin`
		(
			cd $dr
			aclocalinclude="$ACLOCAL_FLAGS"
			for k in $macrodirs; do
				if test -d $k; then
					aclocalinclude="$aclocalinclude -I $k"
				fi
			done

			MACRO_FILES="\
				admin/macros/acinclude.m4.in admin/macros/find-kde.m4.in  \
				admin/macros/find-dl.m4.in   admin/macros/find-file.m4.in \
				admin/macros/find-qt.m4.in   admin/macros/find-x.m4.in    \
				admin/macros/find-jpeg.m4.in admin/macros/check-localize.m4.in"

			cat $MACRO_FILES > acinclude.m4

			RUN_TOOL libtoolize --copy --force
			RUN_TOOL aclocal $aclocalinclude
			if grep "^AM_CONFIG_HEADER" $coin > /dev/null; then
				RUN_TOOL autoheader
			fi
			RUN_TOOL automake --add-missing --copy $am_opt
			RUN_TOOL autoconf
			touch stamp-h.in
		)
	fi
done
#
###############################################################################
