// =============================================================================
//
//      --- kvi_chanlabel.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviChanLabel"

#include <qdrawutil.h>
#include <qpainter.h>
#include <qtoolbutton.h>

#include "kvi_malloc.h"
#include "kvi_mirccntrl.h"
#include "kvi_chanlabel.h"

#define KVI_CHAN_LABEL_BORDER 2

/**
 * A non themeable (from outside) label.<br>
 * Used for the channel labels.
 */
KviChanLabel::KviChanLabel(const char *text, QWidget *parent, const char *name)
	: KviLabel(parent, name)
{
	m_bCanEdit          = false;
	m_szText            = text ? text : "";
	m_iFirstVisibleChar = 0;
	m_textColor         = colorGroup().text();
	m_backColor         = colorGroup().button();
	m_pPixPointer       = 0;
	setBackgroundMode(NoBackground);

	m_bLeft  = new QToolButton(LeftArrow,  this);
	m_bRight = new QToolButton(RightArrow, this);

	m_bLeft->setFocusPolicy(NoFocus);
	m_bLeft->setAutoRepeat(true);

	m_bRight->setFocusPolicy(NoFocus);
	m_bRight->setAutoRepeat(true);

	m_bLeft->hide();
	m_bRight->hide();

	connect(m_bLeft,  SIGNAL(pressed()), this, SLOT(scrollLeft()));
	connect(m_bRight, SIGNAL(pressed()), this, SLOT(scrollRight()));
}

KviChanLabel::~KviChanLabel()
{
	if( m_bLeft  ) {
		delete m_bLeft;
		m_bLeft  = 0;
	}
	if( m_bRight ) {
		delete m_bRight;
		m_bRight = 0;
	}
}

const char *KviChanLabel::strippedText()
{
	KviStr text;
	char *p = m_szText.ptr();
	while( *p ) {
		switch( *p ) {
			case KVI_TEXT_BOLD:
			case KVI_TEXT_REVERSE:
			case KVI_TEXT_RESET:
			case KVI_TEXT_UNDERLINE: {
				p++;
			} break;
			case KVI_TEXT_COLOR: {
				p++;
				if( (*p) && (*p >= '0') && (*p <= '9') ) {
					p++;
					if( (*p) && (*p >= '0') && (*p <= '9') )
						p++;
					if( (*p) && (*p == ',') ) {
						p++;
						if( (*p) && (*p >= '0') && (*p <= '9') )
							p++;
						if( (*p) && (*p >= '0') && (*p <= '9') )
							p++;
					}
				}
			} break;
			default: {
				text.append(*p);
				p++;
			}
		}
	}
	p = (char *) kvi_malloc(text.len() + 1);
	qstrcpy(p, text.ptr());
	return p;
}

void KviChanLabel::setBackgroundPixmapPointer(QPixmap *pix)
{
	m_pPixPointer = pix;
	update();
}

void KviChanLabel::setTextColor(QColor &clr)
{
	m_textColor = clr;
	update();
}

void KviChanLabel::setBackColor(QColor &clr)
{
	m_backColor = clr;
	update();
}

void KviChanLabel::paintEvent(QPaintEvent *e)
{
	QRect r = rect();
	int w = width();
	int h = height();
	QPainter pa(this);
	if( m_pPixPointer )
		pa.drawTiledPixmap(r, *m_pPixPointer);
	else
		pa.fillRect(r, m_backColor);
	qDrawShadeRect(&pa, 0, 0, w+1, h+1, colorGroup(), true, 1, 0);

	const char *sT = strippedText();
	KviStr pT(sT);
	kvi_free(sT);
	if( m_iFirstVisibleChar < 0 )
		m_iFirstVisibleChar = 0;
	KviStr t(pT.right(pT.len() - m_iFirstVisibleChar));
	if( fontMetrics().width(pT.ptr()) < w ) {
		m_bLeft->hide();
		m_bRight->hide();
	} else {
		QSize bs;
		bs.setHeight(h);
		bs.setWidth(h * 5 / 8); // Reciprocal golden mean; ~0.618
		m_bLeft->resize(bs);
		m_bRight->resize(bs);
		int buttonWidth = m_bRight->width();
		int x = w - buttonWidth;
		m_bLeft->move(x-buttonWidth, 0);
		m_bLeft->setEnabled((m_iFirstVisibleChar != 0));
		m_bRight->move(x, 0);
		m_bRight->setEnabled(fontMetrics().width(t.ptr()) >= w - m_bLeft->width() - m_bRight->width());
		m_bLeft->show();
		m_bRight->show();
	}

	if( t.len() ) {
		r.setLeft(r.left() + 2); // Some margin
		pa.setPen(m_textColor);
		pa.drawText(r, Qt::AlignLeft | Qt::AlignVCenter, t.ptr(), t.len());
	}
}

void KviChanLabel::setText(const char *text)
{
	m_szText = text ? text : "";
	m_iFirstVisibleChar = 0;
	update();
}

void KviChanLabel::scrollLeft()
{
	if( m_iFirstVisibleChar ) m_iFirstVisibleChar--;
	update();
}

void KviChanLabel::scrollRight()
{
	KviStr pT(strippedText());
	KviStr t(pT.right(pT.len() - m_iFirstVisibleChar));

	if( fontMetrics().width(t.ptr()) >= width() - m_bLeft->width() - m_bRight->width() )
		m_iFirstVisibleChar++;
	update();
}

void KviChanLabel::mousePressEvent(QMouseEvent *e)
{
	if( e->button() & RightButton )
		emit rightClicked();
	else
		QWidget::mousePressEvent(e);
}

void KviChanLabel::mouseDoubleClickEvent(QMouseEvent *)
{
	m_bCanEdit = true;
}

void KviChanLabel::mouseReleaseEvent(QMouseEvent *)
{
	if( m_bCanEdit )
	{
		emit doubleClicked();
		m_bCanEdit = false;
	}
}

QSize KviChanLabel::sizeHint()
{
	QFontMetrics fm = fontMetrics();
	return QSize(100, fm.height() + (KVI_CHAN_LABEL_BORDER << 1));
}

#include "m_kvi_chanlabel.moc"
