// =============================================================================
//
//      --- kvi_helpwindow.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviHelpWindow"

#include <qcursor.h>

#include "kvi_app.h"
#include "kvi_cursor.h"
#include "kvi_debug.h"
#include "kvi_frame.h"
#include "kvi_helpwindow.h"
#include "kvi_iconloader.h"
#include "kvi_locale.h"
#include "kvi_maintoolbar.h"
#include "kvi_options.h"
#include "kvi_process.h"
#include "kvi_textbrowser.h"

static const QString tb_back("back.png");
static const QString tb_close("fileclose.png");
static const QString tb_copy("editcopy.png");
static const QString tb_forward("forward.png");
static const QString tb_index("info.png");
static const QString tb_search("goto.png");

/**
 * ============ KviHelpWindow ============
 */
KviHelpWindow::KviHelpWindow(KviFrame *pFrm, QWidget *parent, const char *name)
	: QWidget(parent, name ? name : "help_window", WStyle_StaysOnTop)
{
	m_backButton    = 0;
	m_closeButton   = 1;
	m_copyButton    = 2;
	m_forwardButton = 3;
	m_homeButton    = 4;
	m_searchButton  = 5;
	m_inputField    = 6;

	m_pFrame = pFrm;
	setCaption(KVI_HELPWINDOW_WINDOW_NAME);
	setIcon(KviIconLoader::loadIcon(tb_helpBrowser));

	createBrowser();

	m_pToolBar = new KviToolBar(_i18n_("Help toolbar"), pFrm, this, Qt::Unmanaged, false, "help_toolbar");

	m_pToolBar->insertButton(
		tb_close, m_closeButton, SIGNAL(clicked()), m_pFrame, SLOT(closeHelpBrowser()), true, 	 
		_i18n_("Close"), _i18n_("Closes the help window") 	 
	);

	m_pToolBar->insertButton(
		tb_index, m_homeButton, SIGNAL(clicked()), this, SLOT(showIndex()), true,
		_i18n_("Index"), _i18n_("Shows the help index")
	);

	m_pToolBar->insertButton(
		tb_back, m_backButton, SIGNAL(clicked()), m_pBrowser, SLOT(backward()), false,
		_i18n_("Back"), _i18n_("Shows the previous document")
	);

	// Icon needs to be a QPixmap here, or setEnabled(...) will not work
	m_pToolBar->insertButton(
		tb_forward, m_forwardButton, SIGNAL(clicked()), m_pBrowser, SLOT(forward()), false,
		_i18n_("Forward") , _i18n_("Shows the next document")
	);

	m_pToolBar->insertButton(
		tb_copy, m_copyButton, SIGNAL(clicked()), this, SLOT(copyToClipboard()), true,
		 _i18n_("Copy"), _i18n_("Copy to clipboard")
	);

	m_pToolBar->insertLined(
		"", m_inputField, SIGNAL(returnPressed()), this, SLOT(doExactSearch()), true, _i18n_("Search string")
	);
	m_pToolBar->setItemAutoSized(m_inputField);

	m_pToolBar->insertButton(
		tb_search, m_searchButton, SIGNAL(clicked()), this, SLOT(doExactSearch()), true,
		_i18n_("Search"), _i18n_("Search the documentation")
	);

	m_pSearchJob = 0;
	setDocument(g_pApp->m_szLastHelpDocument.ptr());

	m_pToolBar->setBackgroundMode(PaletteBackground);
	m_pToolBar->toggleBigIcons(g_pOptions->m_bUseBigToolbarPixmaps);
	m_pToolBar->toggleIconText(g_pOptions->m_bUseTextToolbarComment);
}

/**
 * ============ ~KviHelpWindow ============
 */
KviHelpWindow::~KviHelpWindow()
{
	if( m_pSearchJob ) {
		delete m_pSearchJob;
		m_pSearchJob = 0;
	}
}

void KviHelpWindow::reloadDocument()
{
	KviStr tmp = g_pApp->m_szLastHelpDocument.ptr();
	showIndex(); // Hack to make it reload the doc
	setDocument(tmp.ptr());
}

void KviHelpWindow::createBrowser()
{
	m_pBrowser = new KviTextBrowser(this);
	m_pBrowser->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	m_pBrowser->setFocusPolicy(NoFocus);
	connect(m_pBrowser, SIGNAL(backwardAvailable(bool)), this, SLOT(backwardAvailable(bool)));
	connect(m_pBrowser, SIGNAL(forwardAvailable(bool)),  this, SLOT(forwardAvailable(bool)));
	m_pBrowser->show();
}

void KviHelpWindow::resetStyleSheet()
{
	KviStr tmp = g_pApp->m_szLastHelpDocument.ptr();
	// Hack hack hack hack hack hack hacK
	delete m_pBrowser;
	m_pBrowser = 0;
	createBrowser();
	resizeEvent(0);
	setDocument(tmp.ptr());
	// Hack hack hack hack hack hack hack
}

void KviHelpWindow::copyToClipboard()
{
	m_pBrowser->copy();
}

void KviHelpWindow::setDocument(const char *topic)
{
	m_pBrowser->setSource(_CHAR_2_QSTRING(topic));
}

void KviHelpWindow::showIndex()
{
	setDocument(KVI_HELP_QML_MAIN_TOPIC_FILE);
}

void KviHelpWindow::backwardAvailable(bool bAvailable)
{
	m_pToolBar->setItemEnabled(m_backButton, bAvailable);
}

void KviHelpWindow::forwardAvailable(bool bAvailable)
{
	m_pToolBar->setItemEnabled(m_forwardButton, bAvailable);
}

void KviHelpWindow::doExactSearch()
{
	setCursor(KviCursor::waitCursor());
	m_pBrowser->viewport()->setCursor(KviCursor::waitCursor());
	QString txt = m_pToolBar->getLinedText(m_inputField);
	if( !txt.isEmpty() ) {
		m_pToolBar->setItemEnabled(m_inputField, false);
		doExactSearchForString(txt.ascii());
	}
}

void KviHelpWindow::doExactSearchForString(const char *str)
{
	if( m_pSearchJob ) delete m_pSearchJob;
	m_pSearchJob = new KviProcess();
	connect(m_pSearchJob, SIGNAL(processExited(KviProcess *)), this, SLOT(searchJobTerminated(KviProcess *)));
	KviStr filesToSearch;
	g_pApp->getGlobalKVIrcDirectory(filesToSearch, KviApp::Help, "*.kvihelp");
	KviStr fileToProduce;
	g_pApp->getLocalKVIrcDirectory(fileToProduce, KviApp::Help, "qh_last_search_result.kvihelp");

	KviStr tmp;
	tmp.sprintf("kvi_search_help.sh %s %s %s", filesToSearch.ptr(), fileToProduce.ptr(), str);
	if( !m_pSearchJob->run(tmp.ptr(), false, true) ) {
		delete m_pSearchJob;
		m_pSearchJob = 0;
		g_pApp->warningBox(_i18n_("Could not execute command\n%s"), tmp.ptr());
	} else m_pToolBar->setItemEnabled(m_searchButton, false);
}

void KviHelpWindow::searchJobTerminated(KviProcess *proc)
{
	__range_valid(proc == m_pSearchJob);
	delete m_pSearchJob;
	m_pSearchJob = 0;
	m_pToolBar->setItemEnabled(m_inputField, true);
	showIndex(); // Hack hack hack
	KviStr fileToProduce;
	g_pApp->getLocalKVIrcDirectory(fileToProduce, KviApp::Help, "qh_last_search_result.kvihelp");
	setDocument(fileToProduce.ptr());
	m_pToolBar->setItemEnabled(m_searchButton, true);
	setCursor(KviCursor::arrowCursor());
	m_pBrowser->viewport()->setCursor(KviCursor::arrowCursor());
}

/**
 * ================ resizeEvent ===============
 */
void KviHelpWindow::resizeEvent(QResizeEvent *)
{
	int hght = m_pToolBar->sizeHint().height();
	m_pToolBar->setGeometry(0, 0, width(), hght);
	m_pBrowser->setGeometry(0, hght, width(), height()-hght);
}

#include "m_kvi_helpwindow.moc"
