// =============================================================================
//
//      --- kvi_dirbrowser_widget.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviDirBrowserWidget"

#include <qdir.h>
#include <qfileinfo.h>

#include "kvi_app.h"
#include "kvi_debug.h"
#include "kvi_dirbrowser_widget.h"
#include "kvi_dirview.h"
#include "kvi_dirview_item.h"
#include "kvi_frame.h"
#include "kvi_lineedit.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_popupmenu.h"
#include "kvi_statusbar.h"
#include "kvi_toolbar.h"

static const QString tb_bookmark("bookmark.png");
static const QString tb_close("fileclose.png");
static const QString tb_home("gohome.png");
static const QString tb_home_kvirc("gohome_kvirc.png");
static const QString tb_reload("reload.png");
static const QString tb_up("up.png");

KviDirectoryBrowserWidget::KviDirectoryBrowserWidget(QWidget *parent, KviFrame *lpFrm, const char *dirname)
	: QWidget(parent)
{
	m_pFrm = lpFrm;
	setBackgroundMode(NoBackground);
	m_pToolBar = new KviToolBar("Browser toolbar", m_pFrm, this, Qt::Unmanaged);
	m_pToolBar->setBackgroundMode(PaletteBackground);

	m_bookmarkButton  = 0;
	m_closeButton     = 1;
	m_homeButton      = 2;
	m_kvircHomeButton = 3;
	m_reloadButton    = 4;
	m_upButton        = 5;

	m_pToolBar->insertButton(
		tb_close, m_closeButton, SIGNAL(clicked()), m_pFrm, SLOT(closeDirBrowser()), true,
		__tr("Close"), __tr("Closes the directory browser")
	);

	m_pToolBar->insertButton(
		tb_up, m_upButton, SIGNAL(clicked()), this, SLOT(cdUp()), true,
		__tr("Go Up"), __tr("Go one directory up")
	);

	m_pToolBar->insertButton(
		tb_reload, m_reloadButton, SIGNAL(clicked()), this, SLOT(cdReload()), true,
		__tr("Reload"), __tr("Reload current directory")
	);

	m_pToolBar->insertButton(
		tb_home, m_homeButton, SIGNAL(clicked()), this, SLOT(cdHome()), true,
		__tr("Home"), __tr("Enter your home directory")
	);

	m_pToolBar->insertButton(
		tb_home_kvirc, m_kvircHomeButton, SIGNAL(clicked()), this, SLOT(cdKviHome()), true,
		__tr("KVIrc Home"), __tr("Enter local KVIrc directory")
	);

	m_pToolBar->insertButton(
		tb_bookmark, m_bookmarkButton, SIGNAL(clicked()), this, SLOT(showBookmarksPopup()), true,
		__tr("Bookmarks"), __tr("Show bookmarks")
	);

	QFrame *f = new QFrame(m_pToolBar);
	m_pToolBar->setStretchableWidget(f);

	m_pBookmarksPopup = new KviPopupMenu();

	m_pDirView = new KviDirectoryView(this, m_pFrm);
	m_pDirView->setFocusPolicy(ClickFocus);
	m_pDirView->viewport()->setFocusPolicy(ClickFocus);
	connect(m_pDirView, SIGNAL(executed(QIconViewItem *)), this, SLOT(itemExecuted(QIconViewItem *)));
	connect(m_pDirView, SIGNAL(selectionChanged()), this, SLOT(selectionChanged()));
	connect(m_pDirView, SIGNAL(needUpdate()), this, SLOT(cdReload()));
	m_pDirPathEdit  = new KviLineEdit(this);
	m_pDirPathEdit->setText("/");
	m_pDirPathEdit->setFocusPolicy(ClickFocus);
	connect(m_pDirPathEdit, SIGNAL(returnPressed()), this, SLOT(returnPressed()));
	m_szDirPath    = "/";
	m_pJob         = 0;
	m_pChildThread = 0;

	g_pApp->refreshMimeTypeIcons();

	if( !loadDirectory(dirname) ) {
		// Failed to read the first directory
		if( dirname )
			loadDirectory(0); // Try the last browsed directory, or home
		else
			debug("Cannot find a suitable start directory!");
	}

	m_pToolBar->toggleBigIcons(g_pOptions->m_bUseBigToolbarPixmaps);
	m_pToolBar->toggleIconText(g_pOptions->m_bUseTextToolbarComment);
}

/**
 * ============ ~KviDirectoryBrowserWidget ============
 */
KviDirectoryBrowserWidget::~KviDirectoryBrowserWidget()
{
	stop();
	g_pOptions->m_szLastBrowsedDirectory = m_szDirPath;
	delete m_pBookmarksPopup;
	m_pBookmarksPopup = 0;
}

bool KviDirectoryBrowserWidget::loadDirectory(const char *dirname)
{
	stop();
	if( setDirPathAndEnsureReadable(dirname) ) {
		m_pDirPathEdit->setText(m_szDirPath.ptr());
		if( beginReadCurrentDirectory() ) {
			g_pOptions->m_szLastBrowsedDirectory = m_szDirPath;
			m_pDirView->m_szDirPath              = m_szDirPath;
			return true;
		}
	}
	return false;
}

bool KviDirectoryBrowserWidget::setDirPathAndEnsureReadable(const char *dirname)
{
	if( dirname ) {
		if( isDirReadable(dirname) ) {
			m_szDirPath = dirname;
			return true;
		} else return false; // Tried, but had no success
	}
	// Dirname is 0. Try the defaults
	if( isDirReadable(g_pOptions->m_szLastBrowsedDirectory.ptr()) ) {
		m_szDirPath = g_pOptions->m_szLastBrowsedDirectory;
		return true;
	}
	// Last browsed dir disappeared?
	KviStr tmp = QDir::homeDirPath();
	if( isDirReadable(tmp.ptr()) ) {
		m_szDirPath = tmp;
		return true;
	}
	if( isDirReadable("/") ) {
		m_szDirPath = "/";
		return true;
	}
	return false;
}

bool KviDirectoryBrowserWidget::isDirReadable(const char *dirname)
{
	QFileInfo fi(dirname);
	return (fi.exists() && fi.isDir() && fi.isReadable());
}

bool KviDirectoryBrowserWidget::beginReadCurrentDirectory()
{
	__range_invalid(m_pJob);
	m_pJob                    = new KviDirBrowserJob();
	m_pJob->szDirPath         = m_szDirPath;
	m_pJob->parent            = this;
	m_pJob->entries           = 0;
	m_pJob->bShowHidden       = g_pOptions->m_bDirBrowserShowHiddenFiles;
	m_pJob->bGroupByExtension = g_pOptions->m_bDirBrowserGroupFilesByExtension;
	m_pJob->bSortBySize       = g_pOptions->m_bDirBrowserSortBySize;
	m_pJob->bCaseInsensitive  = g_pOptions->m_bDirBrowserSortCaseInsensitive;
	// m_pJob is readonly from now on.
	// No mutex should be needed
	m_pChildThread = new KviDirectoryThread(m_pJob);
	if( !m_pChildThread ) {
		delete m_pJob;
		m_pJob = 0;
		debug("Failed to start the I/O child thread");
		return false;
	}
	m_pChildThread->start();
	// m_pJob is owned by the child thread now, until the user event is received
	return true;
}

void KviDirectoryBrowserWidget::stop()
{
	if( !m_pJob )
		return; // No need to stop; readonly access if the child thread is running: no mutex needed
	if( m_pChildThread ) {
		// Cancel the running thread
		m_pChildThread->stop();
		m_pChildThread->wait();
		delete m_pChildThread;
		m_pChildThread = 0;
	}
	m_pJob = 0; // This is dead for us
}

bool KviDirectoryBrowserWidget::event(QEvent *e)
{
	if( e->type() == QEvent::User ) {
		// Ensure that we are in sync
		if( m_pJob != ((KviDirBrowserJob *) ((QCustomEvent *) e)->data()) ) {
			// Cancelled!...
			// Do not leak the job.
			KviDirBrowserJob *j = (KviDirBrowserJob *) ((QCustomEvent *) e)->data();
			if( j->entries )
				delete j->entries;
			delete j;
			return true;
		}
		if( m_pJob->entries ) {
			m_pDirView->clear();
			KviDirectoryViewItem *it = 0;
			m_pJob->entries->setAutoDelete(false);
			for( KviDirBrowserEntry *e = m_pJob->entries->first(); e; e = m_pJob->entries->next() ) {
				it = new KviDirectoryViewItem(m_pDirView, e);
				it->setKey(e->szKey.ptr());
			}
		}
		m_pChildThread->stop();
		m_pChildThread->wait();
		delete m_pChildThread;
		m_pChildThread = 0;
		return true;
	} else return QWidget::event(e);
}

/**
 * ================ resizeEvent ===============
 */
void KviDirectoryBrowserWidget::resizeEvent(QResizeEvent *)
{
	int hght  = m_pToolBar->sizeHint().height();
	int hght2 = m_pDirPathEdit->sizeHint().height();
	m_pToolBar->setGeometry(0, 0, width(), hght);
	m_pDirPathEdit->setGeometry(0, hght, width(), hght2);
	m_pDirView->setGeometry(0, hght + hght2, width(), height() - (hght + hght2));
}

void KviDirectoryBrowserWidget::cdUp()
{
	QDir d(m_szDirPath.ptr());
	d.cdUp();
	m_szDirPath = d.absPath();
	loadDirectory(m_szDirPath.ptr());
}

void KviDirectoryBrowserWidget::itemExecuted(QIconViewItem *it)
{
	KviDirectoryViewItem *i = (KviDirectoryViewItem *) it;
	if( !(i->m_data->bIsReadable) )
		return;
	if( i->m_data->bIsDirectory ) {
		QDir d(m_szDirPath.ptr());
		d.cd(i->m_data->szFileName.ptr());
		m_szDirPath = d.absPath();
		loadDirectory(m_szDirPath.ptr());
	} else if( i->m_data->bIsFile ) {
		g_pApp->executeFileWithCommand(i->m_data->szCommandline.ptr(), i->m_data->szAbsFilePath.ptr(), m_pFrm);
	}
}

void KviDirectoryBrowserWidget::selectionChanged()
{
	QPtrList<KviDirectoryViewItem> l;
	l.setAutoDelete(false);
	KviDirectoryViewItem *it;
	for( it = (KviDirectoryViewItem *) m_pDirView->firstItem(); it; it = (KviDirectoryViewItem *) it->nextItem() ) {
		if( it->isSelected() )
			l.append(it);
	}

	if( l.count() <= 0 ) return;
	if( l.count() == 1 ) {
		KviStr tmp;
		KviDirectoryViewItem *it = l.first();
		if( it->m_data->bIsReadable ) {
			if( it->m_data->bIsSymLink ) {
				if( it->m_data->bIsDirectory ) {
					tmp.sprintf(
						__tr("%s [Symbolic link to %s (Directory)][Double-click to open]"),
						it->m_data->szFileName.ptr(), it->m_data->szMimeName.ptr()
					);
				} else {
					tmp.sprintf(
						__tr("%s [Symbolic link to %s]"),
						it->m_data->szFileName.ptr(), it->m_data->szMimeName.ptr()
					);
				}
			} else if( it->m_data->bIsDirectory ) {
				tmp.sprintf(__tr("%s [Directory][Double-click to open]"), it->m_data->szFileName.ptr());
			} else if( it->m_data->bIsFile ) {
				// Regular file
				tmp.sprintf(
					__tr("%s [%s][%u bytes][Double-click: %s]"),
					it->m_data->szFileName.ptr(), it->m_data->szMimeName.ptr(), it->m_data->uSize,
					it->m_data->szCommandline.hasData() ? it->m_data->szCommandline.ptr() : __tr("No command specified")
				);
			} else if( it->m_data->bIsSocket ) {
				tmp.sprintf(__tr("%s [Socket or FIFO]"), it->m_data->szFileName.ptr());
			} else if( it->m_data->bIsDevice ) {
				tmp.sprintf(__tr("%s [Block or character device]"), it->m_data->szFileName.ptr());
			} else {
				tmp.sprintf(__tr("%s [Unknown][%u bytes]"), it->m_data->szFileName.ptr(), it->m_data->uSize);
			}
			m_pFrm->m_pStatusBar->tempText(tmp.ptr());
		} else {
			// Not readable
			if( it->m_data->bIsDirectory ) {
				tmp.sprintf(__tr("%s [Unreadable directory][%u bytes]"), it->m_data->szFileName.ptr(), it->m_data->uSize);
			} else {
				tmp.sprintf(__tr("%s [Unreadable file][%u bytes]"), it->m_data->szFileName.ptr(), it->m_data->uSize);
			}
		}
	} else {
		unsigned int mb    = 0;
		unsigned int kb    = 0;
		unsigned int bytes = 0;
		unsigned int by;
		for( KviDirectoryViewItem *it = l.first(); it; it = l.next() ) {
			if( !it->m_data->bIsDirectory ) {
				by     = it->m_data->uSize;
				mb    += by / (1024 * 1024);
				by     = by % (1024 * 1024);
				kb    += by / 1024;
				bytes += by % 1024;
				if( bytes > 1024 ) {
					kb   += bytes / 1024;
					bytes = bytes % 1024;
				}
				if( kb > 1024 ) {
					mb += kb / 1024;
					kb  = kb % 1024;
				}
			}
		}
		KviStr tmp(KviStr::Format, __tr("Multiple selection [%u files][%u mb %u kb %u bytes]"), l.count(), mb, kb, bytes);
		m_pFrm->m_pStatusBar->tempText(tmp.ptr());
	}
}

void KviDirectoryBrowserWidget::cdReload()
{
	loadDirectory(m_szDirPath.ptr());
}

void KviDirectoryBrowserWidget::cdHome()
{
	KviStr tmp = QDir::homeDirPath();
	loadDirectory(tmp.ptr());
}

void KviDirectoryBrowserWidget::cdKviHome()
{
	KviStr tmp;
	g_pApp->getLocalKVIrcDirectory(tmp);
	loadDirectory(tmp.ptr());
}

void KviDirectoryBrowserWidget::returnPressed()
{
	KviStr tmp = m_pDirPathEdit->text();
	if( kvi_strEqualCIN("file:/", tmp.ptr(), 6) )
		tmp.cutLeft(6);
	loadDirectory(tmp.ptr());
}

void KviDirectoryBrowserWidget::showBookmarksPopup()
{
	m_pBookmarksPopup->clear();
	m_pBookmarksPopup->insertItem(__tr("&Add Bookmark"));
	m_pBookmarksPopup->insertSeparator();
	for( KviStr *s = g_pOptions->m_pDirBookmarks->first(); s; s = g_pOptions->m_pDirBookmarks->next() ) {
		m_pBookmarksPopup->insertItem(_CHAR_2_QSTRING(s->ptr()));
	}
	QWidget *w = (QWidget *) m_pToolBar->getButton(m_bookmarkButton);
	__range_valid(w);
	m_pBookmarksPopup->popup(w->mapToGlobal(QPoint(0, w->height())));
	connect(m_pBookmarksPopup, SIGNAL(activated(int)), this, SLOT(bookmarkSelected(int)));
}

void KviDirectoryBrowserWidget::applyOptions()
{
	m_pDirView->applyOptions();
}

void KviDirectoryBrowserWidget::bookmarkSelected(int id)
{
	KviStr tmp = m_pBookmarksPopup->text(id);
	if( kvi_strEqualCSN("/", tmp.ptr(), 1) ) {
		// A path item
		loadDirectory(tmp.ptr());
	} else {
		// Non path item, "Add to bookmarks"
		// Add the current path to bookmarks.
		// Avoid double entries.
		KviStr tmp = m_szDirPath;
		if( (tmp.len() > 1) && (tmp.lastCharIs('/')) )
			tmp.cutRight(1);
		tmp.stripWhiteSpace();
		if( tmp.isEmpty() )
			return;

		for( KviStr *s = g_pOptions->m_pDirBookmarks->first(); s; s = g_pOptions->m_pDirBookmarks->next() ) {
			if( kvi_strEqualCS(s->ptr(), m_szDirPath.ptr()) )
				return;
		}
		if( g_pOptions->m_pDirBookmarks->count() == KVI_DIRBROWSER_MAX_BOOKMARKS ) {
			g_pOptions->m_pDirBookmarks->removeFirst();
		}
		g_pOptions->m_pDirBookmarks->append(new KviStr(m_szDirPath));
	}
}

#include "m_kvi_dirbrowser_widget.moc"
