// =============================================================================
//
//      --- kvi_modeeditor.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviModeEditor"

#include <qcheckbox.h>
#include <qlayout.h>

#include "kvi_label.h"
#include "kvi_lineedit.h"
#include "kvi_locale.h"
#include "kvi_modeeditor.h"
#include "kvi_pushbutton.h"

KviModeDescriptionEntry KviModeEditor::modeTable[6] =
{
	{ 'i', _i19n_("i - Invite-only")      },
	{ 'm', _i19n_("m - Moderated")        },
	{ 'n', _i19n_("n - No external messages") },
	{ 'p', _i19n_("p - Private")          },
	{ 's', _i19n_("s - Secret")           },
	{ 't', _i19n_("t - Topic change limited") }
};

KviModeEditor::KviModeEditor(QWidget *parent)
	: QFrame(parent)
{
	setFrameStyle(QFrame::Panel | QFrame::Sunken);

	QGridLayout *g = new QGridLayout(this, 13, 2, 2, 2);

	KviLabel *l = new KviLabel(__tr("<b>Channel Modes</b>"), this);
	g->addMultiCellWidget(l, 0, 0, 0, 1);

	QFrame *f = new QFrame(this);
	f->setFrameStyle(QFrame::Sunken | QFrame::HLine);
	g->addMultiCellWidget(f, 1, 1, 0, 1);

	for( int i = 0; i < 6; i++ ) {
		m_pCheck[i] = new QCheckBox(_i18n_translateNoText(modeTable[i].description), this);
		g->addMultiCellWidget(m_pCheck[i], i + 2, i + 2, 0, 1);
		m_pCheck[i]->setFocusPolicy(ClickFocus);
	}

	m_pLimitCheck = new QCheckBox(__tr("l - User limit:"), this);
	g->addWidget(m_pLimitCheck, 8, 0);

	m_pLimitEdit = new KviLineEdit(this);
	g->addWidget(m_pLimitEdit, 8, 1);

	m_pKeyCheck = new QCheckBox(__tr("k - Key:"), this);
	g->addWidget(m_pKeyCheck, 9, 0);

	m_pKeyEdit = new KviLineEdit(this);
	g->addWidget(m_pKeyEdit, 9, 1);

	l = new KviLabel(this);
	g->addMultiCellWidget(l, 10, 10, 0, 1);

	m_pCommit = new KviPushButton(__tr("&OK"), this);
	m_pCommit->setDefault(true);
	g->addMultiCellWidget(m_pCommit, 11, 11, 0, 1);
	connect(m_pCommit, SIGNAL(clicked()), this, SLOT(commitChanges()));

	KviPushButton *b = new KviPushButton(__tr("&Cancel"), this);
	g->addMultiCellWidget(b, 12, 12, 0, 1);
	connect(b, SIGNAL(clicked()), this, SLOT(cancel()));

	g->setRowStretch(10, 1);
}

KviModeEditor::~KviModeEditor()
{
	// Nothing here
}

void KviModeEditor::setCanCommit(bool bCanCommit)
{
	m_pCommit->setEnabled(bCanCommit);
}

void KviModeEditor::commitChanges()
{
	hide();
	KviStr minus = "-";
	KviStr plus  = "+";
	KviStr args;
	KviStr additionalModeChange;

	for( int i = 0; i < 6; i++ ) {
		if( m_bInitState[i] ) {
			// It was set at the beginning
			if( !m_pCheck[i]->isChecked() ) {
				// And was unset
				minus += modeTable[i].flag;
			}
		} else {
			// It was NOT set at the beginning
			if( m_pCheck[i]->isChecked() ) {
				// And is now set
				plus += modeTable[i].flag;
			}
		}
	}

	KviStr curLimit = m_pLimitEdit->text();
	curLimit.stripWhiteSpace();
	bool bOk = false;
	unsigned int uLimit = curLimit.toUInt(&bOk);

	if( m_szInitLimit.hasData() ) {
		// There was a limit before
		if( m_pLimitCheck->isChecked() ) {
			// We still want a limit
			if( bOk && (uLimit > 0) ) {
				unsigned int oldLimit = m_szInitLimit.toUInt(&bOk);
				if( oldLimit != uLimit ) {
					plus.append('l');
					args = curLimit;
				}
			}
		} else minus.append('l'); // Remove the l mode
	} else {
		if( m_pLimitCheck->isChecked() ) {
			// We want a limit now
			if( bOk && (uLimit > 0) ) {
				plus.append('l');
				args = curLimit;
			}
		}
	}

	KviStr curKey = m_pKeyEdit->text();
	curKey.stripWhiteSpace();

	if( m_szInitKey.hasData() ) {
		// There was a key before
		if( m_pKeyCheck->isChecked() ) {
			// We still want a key
			if( !kvi_strEqualCS(m_szInitKey.ptr(), curKey.ptr()) ) {
				// The key has been changed!
				minus.append('k');
				if( args.hasData() ) args.append(' ');
				args.append(m_szInitKey);
				additionalModeChange.sprintf("+k %s", curKey.ptr());
			}
		} else {
			// We do not want it anymore
			minus.append('k');
			if( args.hasData() ) args.append(' ');
			args.append(m_szInitKey);
		}
	} else {
		// There was no key before
		if( m_pKeyCheck->isChecked() && curKey.hasData() ) {
			// And we want it now
			plus.append('k');
			if( args.hasData() ) args.append(' ');
			args.append(curKey);
		}
	}

	KviStr tmp;
	if( minus.len() > 1 ) tmp += minus;
	if( plus.len() > 1  ) tmp += plus;
	if( args.hasData() ) {
		tmp.append(' ');
		tmp.append(args);
	}
	emit modeChanged(tmp.ptr(), additionalModeChange.ptr());
}

void KviModeEditor::cancel()
{
	emit modeChanged("", "");
}

void KviModeEditor::doEdit(KviStr &flags, KviStr &limit, KviStr &key, QObject *receiver, const char *slot)
{
	disconnect(this, SIGNAL(modeChanged(const char *, const char *)), 0, 0);
	connect(this, SIGNAL(modeChanged(const char *, const char *)), receiver, slot);

	for( int i = 0; i < 6; i++ ) {
		m_bInitState[i] = flags.findFirstIdx(modeTable[i].flag) != -1;
		m_pCheck[i]->setChecked(m_bInitState[i]);
	}

	if( limit.hasData() ) {
		m_pLimitCheck->setChecked(true);
		m_pLimitEdit->setText(limit.ptr());
		m_szInitLimit = limit;
	}

	if( key.hasData() ) {
		m_pKeyCheck->setChecked(true);
		m_pKeyEdit->setText(key.ptr());
		m_szInitKey = key;
	}

	show();
	setFocus();
}

#include "m_kvi_modeeditor.moc"
