// =============================================================================
//
//      --- kvi_serverparser.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviServerParser"

#include "kvi_antispam.h"
#include "kvi_app.h"
#include "kvi_asyncwhois.h"
#include "kvi_chanlabel.h"
#include "kvi_channel.h"
#include "kvi_console.h"
#include "kvi_debug.h"
#include "kvi_event.h"
#include "kvi_frame.h"
#include "kvi_irc_socket.h"
#include "kvi_irc_userlist.h"
#include "kvi_links.h"
#include "kvi_listwindow.h"
#include "kvi_locale.h"
#include "kvi_mirccntrl.h"
#include "kvi_numeric.h"
#include "kvi_options.h"
#include "kvi_query.h"
#include "kvi_rawevent.h"
#include "kvi_regusersdb.h"
#include "kvi_serverparser.h"
#include "kvi_statusbar.h"
#include "kvi_userlistbox.h"
#include "kvi_userparser.h"

// TODO: KILL and PONG

// Declared in kvi_app.cpp
extern KviEventManager    *g_pEventManager;
extern KviRawEventManager *g_pRawEventManager;

KviServerParser::KviServerParser(KviFrame *pFrm)
	: QObject()
{
	m_iCtcpTimerId     = 0;
	m_iCtcpCount       = 0;
	m_iQueryTimerId    = 0;
	m_iQueryCount      = 0;
	m_pFrm             = pFrm;
	m_pSocket          = 0;
	m_pConsole         = 0;
	m_pStatus          = pFrm->m_pStatusBar;
	m_pUserParser      = 0;
	m_iNetsplitTimerId = 0;
}

KviServerParser::~KviServerParser()
{
	// Nothing here
}

void KviServerParser::setup(KviIrcSocket *pSock, KviConsole *pConsole, KviUserParser *pUParser)
{
	m_pSocket     = pSock;
	m_pConsole    = pConsole;
	m_pUserParser = pUParser;
}

bool KviServerParser::isMe(const KviIrcUser &user)
{
	return kvi_strEqualCI(user.nick(), m_pFrm->m_global.szCurrentNick.ptr());
}

bool KviServerParser::isMe(const char *nick)
{
	return kvi_strEqualCI(nick, m_pFrm->m_global.szCurrentNick.ptr());
}

const char *KviServerParser::skipToTrailing(const char *str)
{
	__range_valid(str);
	while( (*str) && (*str != ':') )
		str++;
	if( *str == ':' )
		str++;
	return str;
}

const char *KviServerParser::skipWord(const char *str)
{
	__range_valid(str);
	// Run to the word
	while( (*str) && (*str == ' ') )
		str++;
	// Run to the space
	while( (*str) && (*str != ' ') )
		str++;
	// Run to next word
	while( (*str) && (*str == ' ') )
		str++;
	return str;
}

const char *KviServerParser::skipToUsername(const char *str)
{
	__range_valid(str);
	// nickname!username@host
	// ^  --->  ^
	const char *backup = str;
	while( (*str) && (*str != '!') )
		str++;
	if( *str )
		str++;
	else
		str = backup; // No username? Just a nick/server_name
	return str;
}

/**
 *
 * Oikarinen & Reed                                                [Page 8]
 *
 * RFC 1459              Internet Relay Chat Protocol              May 1993
 *
 *    The protocol messages must be extracted from the contiguous stream of
 *    octets.  The current solution is to designate two characters, CR and
 *    LF, as message separators.   Empty  messages  are  silently  ignored,
 *    which permits  use  of  the  sequence  CR-LF  between  messages
 *    without extra problems.
 *
 *    The extracted message is parsed into the components <prefix>,
 *    <command> and list of parameters matched either by <middle> or
 *    <trailing> components.
 *
 *    The BNF representation for this is:
 *
 * <message>  ::= [':' <prefix> <SPACE> ] <command> <params> <crlf>
 * <prefix>   ::= <servername> | <nick> [ '!' <user> ] [ '@' <host> ]
 * <command>  ::= <letter> { <letter> } | <number> <number> <number>
 * <SPACE>    ::= ' ' { ' ' }
 * <params>   ::= <SPACE> [ ':' <trailing> | <middle> <params> ]
 * <middle>   ::= <Any *non-empty* sequence of octets not including SPACE
 *                or NUL or CR or LF, the first of which may not be ':'>
 * <trailing> ::= <Any, possibly *empty*, sequence of octets not including
 *                  NUL or CR or LF>
 * <crlf>     ::= CR LF
 *
 * NOTES:
 *
 *   1)    <SPACE> is consists only of SPACE character(s) (0x20).
 *         Specially notice that TABULATION, and all other control
 *         characters are considered NON-WHITE-SPACE.
 *
 *   2)    After extracting the parameter list, all parameters are equal,
 *         whether matched by <middle> or <trailing>. <Trailing> is just
 *         a syntactic trick to allow SPACE within parameter.
 *
 *   3)    The fact that CR and LF cannot appear in parameter strings is
 *         just artifact of the message framing. This might change later.
 *
 *   4)    The NUL character is not special in message framing, and
 *         basically could end up inside a parameter, but as it would
 *         cause extra complexities in normal C string handling. Therefore
 *         NUL is not allowed within messages.
 *
 *   5)    The last parameter may be an empty string.
 *
 *   6)    Use of the extended prefix (['!' <user> ] ['@' <host> ]) must
 *         not be used in server to server communications and is only
 *         intended for server to client messages in order to provide
 *         clients with more useful information about who a message is
 *         from without the need for additional queries.
 *
 *    Most protocol messages specify additional semantics and syntax for
 *    the extracted parameter strings dictated by their position in the
 *    list.  For example, many server commands will assume that the first
 *    parameter after the command is the list of targets, which can be
 *    described with:
 *
 *    <target>     ::= <to> [ ", " <target> ]
 *    <to>         ::= <channel> | <user> '@' <servername> | <nick> | <mask>
 *    <channel>    ::= ('#' | '&') <chstring>
 *    <servername> ::= <host>
 *    <host>       ::= see RFC 952 [DNS:4] for details on allowed hostnames
 *    <nick>       ::= <letter> { <letter> | <number> | <special> }
 *    <mask>       ::= ('#' | '$') <chstring>
 *    <chstring>   ::= <any 8bit code except SPACE, BELL, NUL, CR, LF and
 *                      comma (',')>
 *
 *    Other parameter syntaxes are:
 *
 *    <user>       ::= <nonwhite> { <nonwhite> }
 *    <letter>     ::= 'a' ... 'z' | 'A' ... 'Z'
 *    <number>     ::= '0' ... '9'
 *    <special>    ::= '-' | '[' | ']' | '\' | '`' | '^' | '{' | '}'
 *    <nonwhite>   ::= <any 8bit code except SPACE (0x20), NUL (0x0), CR
 *                      (0xd), and LF (0xa)>
 *
 *    Numeric replies
 *
 *    Most of the messages sent to the server generate a reply of some
 *    sort.  The most common reply is the numeric reply, used for both
 *    errors and normal replies.  The numeric reply must be sent as one
 *    message consisting of the sender prefix, the three digit numeric, and
 *    the target of the reply.  A numeric reply is not allowed to originate
 *    from a client; any such messages received by a server are silently
 *    dropped. In all other respects, a numeric reply is just like a normal
 *    message, except that the keyword is made up of 3 numeric digits
 *    rather than a string of letters.
 *
 */

/**
 * === parseMessage ==========================================================
 *
 * Main entry routine.
 * Called directly from KviIrcSocket
 *
 */
void KviServerParser::parseMessage(char *msg_buffer)
{
	KviIrcMessage msg;
	msg.bHasBeenHandled = false;

	char *aux_ptr;
	// Skip any initial spaces
	while( *msg_buffer == ' ' )
		msg_buffer++;
	// Check for <prefix>
	if( *msg_buffer == ':' ) {
		aux_ptr = ++msg_buffer;
		while( *msg_buffer && (*msg_buffer != ' ') )
			msg_buffer++;
		msg.szPrefix.extractFromString(aux_ptr, msg_buffer);
		while( *msg_buffer == ' ' )
			msg_buffer++;
	}
	// Now pointing to <command>
	aux_ptr = msg_buffer;
	while( *msg_buffer && (*msg_buffer != ' ') )
		msg_buffer++;
	msg.szCommand.extractFromString(aux_ptr, msg_buffer);
	while( *msg_buffer == ' ')
		msg_buffer++;
	// Now pointing to <params>
	msg.szParams = msg_buffer;
	// OK... message split
	// Now check if the command is numeric
	bool bOk;
	msg.uCommand = msg.szCommand.toUInt(&bOk);
	if( bOk ) {
		// Trigger a raw event if necessary
		QPtrList<KviRawEvent> *lp = g_pRawEventManager->numericHandlerFor(msg.uCommand, msg.szCommand.ptr());
		if( lp ) {
			if( !triggerRawEvent(&msg, lp) )
				return; // No further processing at all (DANGEROUS)
		}
		parseNumericMessage(&msg);
		return;
	}
	// Trigger a raw event if necessary
	msg.szCommand.toUpper();
	QPtrList<KviRawEvent> *lp = g_pRawEventManager->literalHandlerFor(msg.szCommand.ptr());
	if( lp ) {
		if( !triggerRawEvent(&msg, lp) )
			return; // No further processing at all (DANGEROUS)
	}
	parseLiteralMessage(&msg);
}

bool KviServerParser::triggerRawEvent(KviIrcMessage *msg, QPtrList<KviRawEvent> *lp)
{
	for( KviRawEvent *e = lp->first(); e; e = lp->next() ) {
		KviStr retVal;
		if( ((e->szSource.len() == 1) && (*(e->szSource.ptr()) == '*')) ||
		    kvi_matchWildExpr(e->szSource.ptr(), msg->szPrefix.ptr())
		) {
			retVal = "";
			if( m_pUserParser->callRawEvent(
			    msg->szCommand, msg->szPrefix, e->szBuffer, msg->szParams, e->szSource, retVal)
			) {
				// Halted
				msg->bHasBeenHandled = true;
			}
			if( retVal.hasData() ) {
				if( kvi_strEqualCI(retVal.ptr(), "stop") ) {
					// Has been handled... no further processing at all!
					return false;
				}
			}
		}
	}
	return true;
}

/**
 * === NUMERIC MESSAGES ======================================================
 * FORMAT OF ALL NUMERICS:
 * :server_name XXX my_nick <params>
 * <szPrefix>   <c> <szParams>...
 */
void KviServerParser::parseNumericMessage(KviIrcMessage *msg)
{
	if( msg->uCommand < 200 ) { // TODO: perhaps use a hashlist instead of split tables?
		// This range of messages is used only at the
		// beginning of the connection.
		if( m_pFrm->state() != KviFrame::Connected ) {
			__range_valid(m_pFrm->state() == KviFrame::LoggingIn);
			// Make sure that the server accepted exactly our nickname.
			// Some servers will cut long nicks, or remove invalid chars.
			KviStr szTmp;
			kvi_extractToken(szTmp, msg->szParams.ptr(), ' ');
			m_pFrm->ircLoginSuccessful(szTmp);
		}
		parseNumeric000_199(msg);
		return;
	}
	if( msg->uCommand < 400 ) {
		parseNumeric200_399(msg);
		return;
	}
	parseNumeric400_999(msg);
}

/**
 * Numerics 000-199: Information.
 * When we receive one of these messages, we are
 * sure that we are successfully logged in.
 */
void KviServerParser::parseNumeric000_199(KviIrcMessage *msg)
{
	// All messages contain our nickname at the beginning.
	// Skip the nickname
	register char *ptr = msg->szParams.ptr();
	while( *ptr && *ptr != ' ' )
		ptr++;
	while( *ptr && *ptr == ' ' )
		ptr++;
	// Skip the ':' if it is there
	if( *ptr == ':' )
		ptr++;

	// OK... now ptr points to the first param after the target nickname
	switch( msg->uCommand ) {
		case RPL_WELCOME:    // Welcome to the Internet Relay Network %s
		case RPL_YOURHOST:   // Your host is <serverhostname>, running version %s
		case RPL_CREATED:    // This server was created on <date_time>
		case RPL_BOUNCE:     // Try server %s, port %s (or %s%s or %s :are available on this server)
		                     // (never encountered till now)
		case RPL_MAPMORE:    // %s%s --> *more*
		case RPL_MAPEND:     // End of /MAP
		case RPL_SNOMASK:    // %s :: Server notice mask (%#x)
		case RPL_STATMEMTOT: // %u %u :Bytes Blocks
		case RPL_STATMEM:    // %u %u %s
		case RPL_YOURCOOKIE: // <cookie> :is your reconnection cookie
			if( !msg->bHasBeenHandled ) {
				if( g_pOptions->m_bServerInfoToConsole )
					m_pConsole->outputNoFmt(KVI_OUT_INFO, ptr);
				else
					m_pFrm->activeWindow()->outputNoFmt(KVI_OUT_INFO, ptr);
			break;
		}
		case RPL_MYINFO:     // <server_name> <server_version> <u_modes> <ch_modes>
			parseNumericMyInfo(msg);
			break;
		default:
			parseUnhandledNumeric(msg);
			break;
	}
}

void KviServerParser::parseNumeric200_399(KviIrcMessage *msg)
{
	// All messages contain our nickname at the beginning.
	// Skip the nickname
	register char *ptr = msg->szParams.ptr();
	while( *ptr && *ptr != ' ' )
		ptr++;
	while( *ptr && *ptr == ' ' )
		ptr++;
	// Skip the ':' if it is there
	if( *ptr == ':' )
		ptr++;

	switch( msg->uCommand ) {
		// Server info
		case RPL_LUSERCLIENT:     // 251 ":There are %d users and %d invisible on %d servers"
		case RPL_LUSEROP:         // 252 "%d :IRC Operators online"
		case RPL_LUSERUNKNOWN:    // 253 "%d :unknown connection(s)"
		case RPL_LUSERCHANNELS:   // 254 "%d :channels formed"
		case RPL_LUSERME:         // 255 ":I have %d clients and %d servers"
		case RPL_ADMINME:         // 256 ":Administrative info about %s"
		case RPL_ADMINLOC1:       // 257 ":%s"
		case RPL_ADMINLOC2:       // 258 ":%s"
		case RPL_ADMINEMAIL:      // 259 ":%s"
		case RPL_LOCALUSERS:      // 265 ":Current local users :num Max:num"
		case RPL_GLOBALUSERS:     // 266 ":Current global users :num Max:num"
		case RPL_INFOSTART:       // 373 ":Server INFO"
		case RPL_INFO:            // 371 ":%s"
		case RPL_ENDOFINFO:       // 374 ":End of /INFO list"
			if( !msg->bHasBeenHandled ) {
				if( g_pOptions->m_bServerInfoToConsole )
					m_pConsole->outputNoFmt(KVI_OUT_INFO, ptr);
				else
					m_pFrm->activeWindow()->outputNoFmt(KVI_OUT_INFO, ptr);
			break;
		}
		case RPL_VERSION:         // 351 "%s.%s %s :%s"
			parseNumericVersion(msg);
			break;
		// MOTD
		case RPL_MOTDSTART:       // 375 ":- %s Message of the Day - "
		case RPL_MOTD:            // 372 ":- %s"
		case RPL_MOTD2:           // 377 ":- %s"
		case RPL_ENDOFMOTD:       // 376 ":End of /MOTD command"
			if( !(g_pOptions->m_bSkipMotd || msg->bHasBeenHandled) ) {
				if( g_pOptions->m_bServerInfoToConsole )
					m_pConsole->outputNoFmt(KVI_OUT_MOTD, ptr);
				else
					m_pFrm->activeWindow()->outputNoFmt(KVI_OUT_MOTD, ptr);
			break;
		}
		case RPL_USERHOST:        // 302 "..."
			parseNumericUserhost(msg);
			break;
		// Channel topic
		case RPL_NOTOPIC:         // 331 "<channel> :No topic is set"
		case RPL_TOPIC:           // 332 "<channel> :<topic>"
		case RPL_TOPICWHOTIME:    // 333 "<channel> <topicwho> [topictime]"
			parseNumericTopic(msg);
			break;
		case RPL_CREATIONTIME:    // 329 "<channel> %lu"
			parseNumericCreationTime(msg);
			break;
		case RPL_NAMREPLY:        // 353 "= <channel> :<space_separated_list_of_nicks>"
			parseNumericNamesReply(msg);
			break;
		case RPL_ENDOFNAMES:      // 366 "<channel> :End of /NAMES list."
			parseNumericEndOfNames(msg);
			break;
		case RPL_CHANNELMODEIS:   // 324 "<channel> +<mode>"
			parseNumericChannelMode(msg);
			break;
		case RPL_BANLIST:         // 367 "<channel> <banmask>"
		case RPL_ENDOFBANLIST:    // 368 "<channel> :End of channel Ban List"
			parseNumericBanList(msg);
			break;
		case RPL_EXCEPTLIST:      // 348 "<channel> <exceptionmask>"
		case RPL_ENDOFEXCEPTLIST: // 349 "<channel> :End of Channel Exception List"
			parseNumericExceptionList(msg);
			break;
		case RPL_INVITELIST:      // 346 "<channel> <exceptionmask>"
		case RPL_ENDOFINVITELIST: // 347 "<channel> :End of Channel Exception List"
			parseNumericInviteList(msg);
			break;
		case RPL_WHOREPLY:        // 352 "<chn> <usr> <hst> <srv> <nck> <stat> :<hops> <real>"
		case RPL_ENDOFWHO:        // 315 "<channel> :End of /WHO list."
			parseNumericWhoReply(msg);
			break;
		case RPL_ISON:            // 303 ":<list of nicks>"
			parseNumericIsonReply(msg);
			break;
		case RPL_LISTSTART:       // 321 Channel :Users Name
		case RPL_LIST:            // 322 <channel> <users> :<topic>
		case RPL_LISTEND:         // 323 :End of /LIST
			parseNumericList(msg);
			break;
		case RPL_WHOISUSER:       // 311 "%s %s %s * :%s"
		case RPL_WHOISSERVER:     // 312 "%s %s :%s"
		case RPL_WHOISOPERATOR:   // 313 "%s :is an IRC Operator"
		case RPL_WHOWASUSER:      // 314 "%s %s %s * :%s"
		case RPL_WHOISCHANOP:     // 316 "???"
		case RPL_WHOISIDLE:       // 317 "%s %ld %ld :seconds idle, signon time"
		case RPL_ENDOFWHOIS:      // 318 "%s :End of /WHOIS list."
		case RPL_WHOISCHANNELS:   // 319 "%s :%s"
		case RPL_ENDOFWHOWAS:     // 369 "%s :END of /WHOWAS"
		case RPL_WHOISREGNICK:    // 307 "???"
		case RPL_WHOISADMIN:      // 308 "???"
		case RPL_WHOISSADMIN:     // 309 "???"
		case RPL_WHOISHELPOP:     // 310 "???"
		case RPL_AWAY:            // 301 "%s :%s"
			parseNumericWhoisReply(msg);
			break;
		case RPL_INVITING: {      // 341 [I, E, U, D]
			KviStr target;
			const char *aux = kvi_extractToken(target, ptr, ' ');
			while( *aux && (*aux != '&') && (*aux != '#') && (*aux != '!') )
				aux++;
			KviWindow *pOut = g_pOptions->m_bInviteMessagesToConsole ? (KviWindow *) m_pConsole : m_pFrm->findChannel(aux);
			if( !pOut )
				pOut = m_pFrm->activeWindow();
			if( !msg->bHasBeenHandled ) {
				pOut->output(KVI_OUT_INVITE, __tr("%s has been invited to %s"), target.ptr(), aux);
			}
			break;
		}
		case RPL_LINKS:           // 364 [I, E, U, D]
		case RPL_ENDOFLINKS:      // 365 [I, E, U, D]
			parseNumericLinks(msg);
			break;
		default:
			parseUnhandledNumeric(msg);
			break;
	}
}

void KviServerParser::parseNumeric400_999(KviIrcMessage *msg)
{
	// All errors contain our nickname at the beginning.
	// Skip the nickname.
	register char *ptr = msg->szParams.ptr();
	while( *ptr && *ptr != ' ' )
		ptr++;
	while( *ptr && *ptr == ' ' )
		ptr++;
	// Skip the ':' if it is there
	if( *ptr == ':' )
		ptr++;

	switch( msg->uCommand ) {
		case ERR_NOSUCHNICK:       // 401 "%s :No such nick/channel"
		case ERR_NOSUCHSERVER:     // 402 "%s :No such server"
		case ERR_WASNOSUCHNICK: {  // 406 "%s :There was no such nickname"
			KviStr target;
			const char *aux = kvi_extractToken(target, ptr, ' ');
			if( *aux == ':' )
				aux = skipToTrailing(aux);
			KviAsyncWhoisEntry *e = m_pUserParser->m_pAsyncWhoisController->findPendingWhois(target.ptr());
			if( e )
				m_pUserParser->m_pAsyncWhoisController->end(e, false);
			else {
				if( !msg->bHasBeenHandled ) {
					KviWindow *pOut = m_pFrm->findQuery(target.ptr());
					if( pOut )
						pOut->output(KVI_OUT_NICKNAME_ERROR, _i18n_("%s: %s"), target.ptr(), aux);
					else {
						pOut = g_pOptions->m_bNicknameErrorsToConsole ? m_pConsole : m_pFrm->activeWindow();
						pOut->output(KVI_OUT_NICKNAME_ERROR,
							_i18n_("[%s][%s] %s: %s"), msg->szPrefix.ptr(), msg->szCommand.ptr(), target.ptr(), aux
						);
					}
				}
			}
			break;
		}
		case ERR_CANNOTSENDTOCHAN:  // 404 "%s :Cannot send to channel" [I, E, U, D]
			if( !msg->bHasBeenHandled ) {	KviStr target;
				const char *aux = kvi_extractToken(target, ptr, ' ');
				if( *aux == ':' )
					aux = skipToTrailing(aux);
				KviWindow *pOut = m_pFrm->findChannel(target.ptr());
				if( pOut )
					pOut->output(KVI_OUT_UNHANDLED_ERROR, _i18n_("%s: %s"), target.ptr(), aux);
				else {
					pOut = g_pOptions->m_bUnhandledServerErrorsToConsole ? m_pConsole : m_pFrm->activeWindow();
					pOut->output(KVI_OUT_UNHANDLED_ERROR,
						_i18n_("[%s][%s] \r!join $1\r%s\r: %s"),
						msg->szPrefix.ptr(), msg->szCommand.ptr(), target.ptr(), aux
					);
				}
			}
			break;
		case ERR_NONICKNAMEGIVEN:  // 431 ":No nickname given"
		case ERR_ERRONEUSNICKNAME: // 432 "%s :Erroneus Nickname"
		case ERR_NICKNAMEINUSE:    // 433 "%s :Nickname is already in use."
		case ERR_NICKCOLLISION:    // 436 "%s :Nickname collision KILL" [I, E, U, D]
		case ERR_UNAVAILRESOURCE:  // 437 ???
		case ERR_NICKTOOFAST:      // 438 ???
			if( m_pFrm->isLoggingIn() )
				m_pFrm->loginNickRefused(); // 431, 432 or 433
			else {
				// Already logged in. Simple nickname error
				if( !msg->bHasBeenHandled ) {
					KviWindow *pOut = g_pOptions->m_bNicknameErrorsToConsole ? m_pConsole : m_pFrm->activeWindow();
					pOut->output(KVI_OUT_NICKNAME_ERROR,
						_i18n_("[%s][%s] %s"), msg->szPrefix.ptr(), msg->szCommand.ptr(), msg->szParams.ptr()
					);
				}
			}
			break;
		case ERR_UNKNOWNMODE:      // 472 "%c :is unknown mode char to me"
			parseNumericErrorUnknownModeChar(msg);
			break;
		case ERR_CHANNELISFULL:    // 471 "%s :Cannot join channel (+l)"
		case ERR_INVITEONLYCHAN:   // 473 "%s :Cannot join channel (+i)"
		case ERR_BANNEDFROMCHAN:   // 474 "%s :Cannot join channel (+b)"
		case ERR_BADCHANNELKEY: {  // 475 "%s :Cannot join channel (+k)"
			KviStr target;
			const char *aux = kvi_extractToken(target, ptr, ' ');
			if( *aux == ':' )
				aux = skipToTrailing(aux);
			KviChannel *chan = m_pFrm->findChannel(target.ptr());
			if( chan )
				m_pFrm->closeWindow(chan);
			if( !msg->bHasBeenHandled ) {
				KviWindow *pOut = g_pOptions->m_bUnhandledServerErrorsToConsole ? m_pConsole : m_pFrm->activeWindow();
				pOut->output(KVI_OUT_UNHANDLED_ERROR,
					_i18n_("[%s][%s] \r!join $1\r%s\r: %s"), msg->szPrefix.ptr(), msg->szCommand.ptr(), target.ptr(), aux
				);
			}
			break;
		}
		case ERR_UNKNOWNCOMMAND:   // 421 "%s :Unknown command"
			parseNumericErrorUnknownCommand(msg);
			break;
		case RPL_LOGON:            // 600 "<nick> <user> <host> <logintime> :logged online
		case RPL_LOGOFF:           // 601 "<nick> <user> <host> <logintime> :logged offline
		case RPL_WATCHOFF:         // 602 "<nick> <user> <host> <logintime> :stopped watching
		case RPL_WATCHSTAT:        // 603 ":You have <n> and are on <n> WATCH entries
		case RPL_NOWON:            // 604 "<nick> <user> <host> <logintime> :is online
		case RPL_NOWOFF:           // 605 "<nick> <user> <host> 0 :is offline
		case RPL_WATCHLIST:        // 606 ?
		case RPL_ENDOFWATCH:       // 607 ":End of WATCH <char l=list s=stats>
			parseNumericWatchList(msg);
			break;
		default:
			parseUnhandledNumericError(msg);
			break;
	}
}

/**
 *
 * [04:10:41] Raw to server: watch +Praggy +[Pragma] +Diabl0
 * [04:10:43] [viking.no.eu.dal.net][605] [Pragma] Praggy * * 0 :is offline
 * [04:10:43] [viking.no.eu.dal.net][604] [Pragma] [Pragma] ~nobody a-si6-20.tin.it 927778264 :is online
 * [04:10:43] [viking.no.eu.dal.net][605] [Pragma] Diabl0 * * 0 :is offline
 * Other server conneciton : login as Diabl0
 * [04:12:28] [viking.no.eu.dal.net][600] [Pragma] Diabl0 ~nobody a-si6-20.tin.it 927778397 :logged online
 * And changed nick to Praggy
 * [04:12:34] [viking.no.eu.dal.net][601] [Pragma] Diabl0 ~nobody a-si6-20.tin.it 927778403 :logged offline
 * [04:12:34] [viking.no.eu.dal.net][600] [Pragma] Praggy ~nobody a-si6-20.tin.it 927778403 :logged online
 * Praggy logoff
 * [04:13:02] [viking.no.eu.dal.net][601] [Pragma] Praggy ~nobody a-si6-20.tin.it 927778431 :logged offline
 * [04:14:09] Raw to server: watch
 * [04:14:09] [viking.no.eu.dal.net][604] [Pragma] [Pragma] ~nobody a-si6-20.tin.it 927778264 :is online
 * [04:14:09] [viking.no.eu.dal.net][607] [Pragma] :End of WATCH l
 * [04:14:33] Raw to server: watch -Diabl0
 * [04:14:33] [viking.no.eu.dal.net][602] [Pragma] Diabl0 * * 0 :stopped watching
 * [04:14:40] Raw to server: watch -Praggy
 * [04:14:40] [viking.no.eu.dal.net][602] [Pragma] Praggy * * 0 :stopped watching
 * [04:14:45] Raw to server: watch -[Pragma]
 * [04:14:45] [viking.no.eu.dal.net][602] [Pragma] [Pragma] ~nobody a-si6-20.tin.it 927778264 :stopped watching
 * [04:23:12] Raw to server: watch stats,
 * [04:23:13] [viking.no.eu.dal.net][603] [Pragma] :You have 0 and are on 0 WATCH entries
 * [04:23:13] [viking.no.eu.dal.net][607] [Pragma] :End of WATCH s
 *
 */
void KviServerParser::parseNumericWatchList(KviIrcMessage *msg)
{
	const char *aux = skipWord(msg->szParams.ptr());
	KviWindow *pOut = g_pOptions->m_bNotifyListChangesToConsole ? m_pConsole : m_pFrm->activeWindow();

	if( (msg->uCommand == RPL_LOGON) || (msg->uCommand == RPL_LOGOFF) ||
	    (msg->uCommand == RPL_NOWON) || (msg->uCommand == RPL_NOWOFF) ||
	    (msg->uCommand == RPL_WATCHOFF)
	) {
		KviStr nick, user, host, logtime_t, logtime;
		aux = kvi_extractToken(nick, aux, ' ');
		aux = kvi_extractToken(user, aux, ' ');
		aux = kvi_extractToken(host, aux, ' ');
		aux = kvi_extractToken(logtime_t, aux, ' ');
		while( *aux == ' ' )
			aux++;
		while( *aux == ':' )
			aux++;
		getDateTimeStringFromCharTimeT(logtime, logtime_t.ptr());
		pOut->output(KVI_OUT_WATCH, __tr("%s!%s@%s (%s) %s"), nick.ptr(), user.ptr(), host.ptr(), logtime.ptr(), aux);
		if( (msg->uCommand == RPL_LOGON) || (msg->uCommand == RPL_NOWON) ) {
			if( !(m_pConsole->m_pListBox->findUser(nick.ptr())) ) {
				KviIrcUser u;
				u.setNick(nick.ptr());
				u.setUsername(user.ptr());
				u.setHost(host.ptr());
				m_pConsole->m_pListBox->join(u, 0, 0, 0, 0, 0, true);
			}
		} else {
			if( m_pConsole->m_pListBox->findUser(nick.ptr()) ) {
				m_pConsole->m_pListBox->part(nick.ptr());
			}
		}
	} else {
		while( *aux == ':' )
			aux++;
		pOut->output(KVI_OUT_WATCH, "WATCH LIST: %s", aux);
	}
}

/**
 * [04:10:43] [viking.no.eu.dal.net][605] [Pragma] :Bobo Praggy Diabl0
 */
void KviServerParser::parseNumericIsonReply(KviIrcMessage *msg)
{
	const char *aux  = skipToTrailing(msg->szParams.ptr());
	bool bDoRepaint  = false;
	KviWindow *pOut  = g_pOptions->m_bNotifyListChangesToConsole ? m_pConsole : m_pFrm->activeWindow();
	QPtrList<KviStr> *l = m_pConsole->m_pListBox->createNickStringList();
	KviStr token;

	while( *aux ) {
		aux = kvi_extractToken(token, aux, ' ');
		if( token.hasData() ) {
			KviStr *nk = 0;
			for( nk = l->first(); nk; nk = l->next() ) {
				if( kvi_strEqualCI(nk->ptr(), token.ptr()) )
					break;
			}
			if( !nk ) {
				// Was not there
				KviIrcUser u;
				u.setNick(token.ptr());
				m_pConsole->m_pListBox->join(u, 0, 0, 0, 0, 0, false);
				bDoRepaint   = true;
				bool bOutput = true;
				if( g_pEventManager->eventEnabled(KviEvent_OnNotify) ) {
					KviStr eventparms(KviStr::Format, "%s 1", token.ptr());
					if( m_pUserParser->callEvent(KviEvent_OnNotify, m_pConsole, eventparms) )
						bOutput = false;
				}
				if( bOutput )
					pOut->output(KVI_OUT_WATCH, __tr("%s is on IRC"), token.ptr());
			} // else: no changes
			if( nk )
				l->removeRef(nk);
		}
	}
	if( bDoRepaint )
		m_pConsole->m_pListBox->doRepaint();
	bool bOutput = true;
	for( KviStr *n = l->first(); n; n = l->next() ) {
		// The rest parts
		if( g_pEventManager->eventEnabled(KviEvent_OnNotify) ) {
			KviStr eventparms(KviStr::Format, "%s 0", n->ptr());
			if( m_pUserParser->callEvent(KviEvent_OnNotify, m_pConsole, eventparms) )
				bOutput = false;
		}
		m_pConsole->m_pListBox->part(n->ptr());
		if( bOutput )
			pOut->output(KVI_OUT_WATCH, __tr("%s has left IRC"), n->ptr());
	}
	delete l;
}

void KviServerParser::parseNumericMyInfo(KviIrcMessage *msg)
{
	const char *aux = skipWord(msg->szParams.ptr());
	if( !msg->bHasBeenHandled ) {
		if( g_pOptions->m_bServerInfoToConsole )
			m_pConsole->output(KVI_OUT_INFO, __tr("Server info: %s"), aux);
		else
			m_pFrm->activeWindow()->output(KVI_OUT_INFO, __tr("Server info: %s"), aux);
	}

	// Skip version
	aux = skipWord(aux);
	// Skip server name
	aux = skipWord(aux);
	// Skip user modes

	const char *aux2 = skipWord(aux);
	KviStr umodes(aux, aux2);
	KviStr chanmodes = aux2;
	umodes.stripWhiteSpace();
	chanmodes.stripWhiteSpace();
	// We have channel modes in chanmodes
	m_pFrm->m_global.bServerSupportsEMode = ((chanmodes.findFirstIdx('e') != -1) && (chanmodes.findFirstIdx('I') != -1));

	if( g_pOptions->m_bBeVerbose ) {
		if( m_pFrm->m_global.bServerSupportsEMode ) {
			m_pConsole->output(KVI_OUT_INTERNAL,
				__tr("The server supports ban and invite exception lists (channel modes e and I)")
			);
		} else {
			m_pConsole->output(KVI_OUT_INTERNAL,
				__tr("The server has no support for invite and ban exception lists (channel modes e and I)")
			);
		}
	}

	if( g_pOptions->m_bShowExtendedServerInfo && (!msg->bHasBeenHandled) ) {
		if( g_pOptions->m_bServerInfoToConsole )
			m_pConsole->outputNoFmt(KVI_OUT_INFO, __tr("Available user modes:"));
		else
			m_pFrm->activeWindow()->outputNoFmt(KVI_OUT_INFO, __tr("Available user modes:"));

		aux = umodes.ptr();
		KviStr tmp;
		while( *aux ) {
			switch( *aux ) {
				case 'O': tmp = __tr("O: local IRC operator (LOCOP)");                break;
				case 'o': tmp = __tr("o: global IRC operator (OPER)");                break;
				case 'a': tmp = __tr("a: services administrator (SVSADMIN)");         break;
				case 'A': tmp = __tr("A: server administrator (ADMIN)");              break;
				case 'N': tmp = __tr("N: network administrator (NETADMIN)");          break;
				case 'i': tmp = __tr("i: invisible");                                 break;
				case 'q': tmp = __tr("q: quiet");                                     break;
				case 'm': tmp = __tr("m: blocking messages");                         break;
				case 'p': tmp = __tr("p: blocking CTCP");                             break;
				case 'e': tmp = __tr("e: blocking DCC");                              break;
				case 'L': tmp = __tr("L: language filtering");                        break;
				case 'h': tmp = __tr("h: recipient for helpops messages");            break;
				case 'w': tmp = __tr("w: recipient for wallops messages");            break;
				case 'r': tmp = __tr("r: registered nick/restricted connection");     break;
				case 's': tmp = __tr("s: recipient for server notices");              break;
				case 'z': tmp = __tr("z: recipient for oper wallop messages");        break;
				case 'c': tmp = __tr("c: recipient for cconn messages");              break;
				case 'k': tmp = __tr("k: recipient for server kill messages");        break;
				case 'f': tmp = __tr("f: recipient for full server notices");         break;
				case 'y': tmp = __tr("y: spy");                                       break;
				case 'd': tmp = __tr("d: obscure 'DEBUG' flag");                      break;
				case 'n': tmp = __tr("n: recipient for nick changes or newsflashes"); break;
				case 'x': tmp = __tr("x: masked hostname");                           break;
				default:  tmp = __tr(": unknown user mode"); tmp.prepend(*aux);       break;
			}
			if( tmp.hasData() ) {
				if( g_pOptions->m_bServerInfoToConsole )
					m_pConsole->outputNoFmt(KVI_OUT_INFO, tmp.ptr());
				else
					m_pFrm->activeWindow()->outputNoFmt(KVI_OUT_INFO, tmp.ptr());
			}
			tmp = "";
			aux++;
		}
		if( g_pOptions->m_bServerInfoToConsole )
			m_pConsole->outputNoFmt(KVI_OUT_INFO, __tr("Available channel modes:"));
		else
			m_pFrm->activeWindow()->outputNoFmt(KVI_OUT_INFO, __tr("Available channel modes:"));
		aux = chanmodes.ptr();
		tmp = "";
		while( *aux ) {
			switch( *aux ) {
				case 'o': tmp = __tr("o: channel operator");                       break;
				case 'h': tmp = __tr("h: halfop");                                 break;
				case 'v': tmp = __tr("v: voiced user");                            break;
				case 'u': tmp = __tr("u: userop");                                 break;
				case 'b': tmp = __tr("b: ban masks");                              break;
				case 'e': tmp = __tr("e: ban exception masks/no CTCPs");           break;
				case 'I': tmp = __tr("I: invite exception masks");                 break;
				case 's': tmp = __tr("s: secret");                                 break;
				case 'p': tmp = __tr("p: private");                                break;
				case 'a': tmp = __tr("a: anonymous");                              break;
				case 'r': tmp = __tr("r: registered");                             break;
				case 'n': tmp = __tr("n: no external messages");                   break;
				case 'q': tmp = __tr("q: quiet/channel owner");                    break;
				case 'm': tmp = __tr("m: moderated");                              break;
				case 'c': tmp = __tr("c: no colors");                              break;
				case 'L': tmp = __tr("L: language filtering");                     break;
				case 't': tmp = __tr("t: topic change restricted");                break;
				case 'd': tmp = __tr("d: nick change restricted");                 break;
				case 'i': tmp = __tr("i: invite-only");                            break;
				case 'j': tmp = __tr("j: Java clients only");                      break;
				case 'l': tmp = __tr("l: limited number of users");                break;
				case 'O': tmp = __tr("O: redirect if full");                       break;
				case 'k': tmp = __tr("k: key");                                    break;
				default:  tmp = __tr(": unknown channel mode"); tmp.prepend(*aux); break;
			}
			if( tmp.hasData() ) {
				if( g_pOptions->m_bServerInfoToConsole )
					m_pConsole->outputNoFmt(KVI_OUT_INFO, tmp.ptr());
				else
					m_pFrm->activeWindow()->outputNoFmt(KVI_OUT_INFO, tmp.ptr());
			}
			tmp = "";
			aux++;
		}
	}
}

/**
 * [irc.pragma.net][472] newbie %c :is unknown mode char to me
 */
void KviServerParser::parseNumericErrorUnknownModeChar(KviIrcMessage *msg)
{
	KviStr modes;
	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());
	aux = kvi_extractToken(modes, aux, ' ');
	__range_valid(*aux == ':');

	if( m_pFrm->m_global.bServerSupportsEMode ) {
		// Do not do the same error again
		if( modes.contains('e') || modes.contains('I') ) {
			m_pFrm->m_global.bServerSupportsEMode = false;
			if( g_pOptions->m_bBeVerbose ) {
				m_pConsole->output(KVI_OUT_INTERNAL,
					__tr("This server does not support the 'e/I' umode flag: disabling ban/invite exception handling.")
				);
			}
			return;
		}
	}
	parseUnhandledNumericError(msg);
}

/**
 * Handle the WATCH command error
 * [irc.pragma.net][421] newbie %s :Unknown command
 */
void KviServerParser::parseNumericErrorUnknownCommand(KviIrcMessage *msg)
{
	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());
	KviStr cmd;
	aux = kvi_extractToken(cmd, aux, ' ');
	__range_valid(*aux == ':');
	if( kvi_strEqualCI(cmd.ptr(), "watch") ) {
		if( m_pFrm->m_global.bServerSupportsWatchList ) {
			if( g_pOptions->m_bBeVerbose ) {
				m_pConsole->output(KVI_OUT_INTERNAL,
					__tr("This server does not support the 'watch' notify method: using standard 'ison ping' method.")
				);
			}
			m_pFrm->m_global.bServerSupportsWatchList = false;
			m_pFrm->updateNotifyOrWatchList();
			return;
		}
	}
	parseUnhandledNumericError(msg);
}

/**
 * [irc.pragma.net][331] newbie #kvirc :No topic is set
 * [irc.pragma.net][332] newbie #kvirc :Topiko time
 * [irc.pragma.net][333] newbie #kvirc pragma 923579937 <--- Some networks send the nick only and not the time
 */
void KviServerParser::parseNumericTopic(KviIrcMessage *msg)
{
	KviStr channel;
	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());
	aux = kvi_extractToken(channel, aux, ' ');
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		// Oops, desync with the server.
		KviWindow *pOut = g_pOptions->m_bDesyncMsgsToConsole ? m_pConsole : m_pFrm->activeWindow();
		pOut->output(KVI_OUT_DESYNC,
			__tr("Local desync: received topic info: %u %s"), msg->uCommand, msg->szParams.ptr()
		);
		return;
	} else {
		// OK
		if( msg->uCommand == RPL_TOPIC ) {
			chan->setTopic(skipToTrailing(aux));
			KviWindow *pOut = g_pOptions->m_bTopicMsgsToConsole ? m_pConsole : ((KviWindow *) chan);
			pOut->output(KVI_OUT_TOPIC, __tr("Topic for %s is: %s"), channel.ptr(), skipToTrailing(aux));
			chan->gotTopicReply();
		} else if( msg->uCommand == RPL_NOTOPIC ) {
			chan->setTopic(__tr("[ No topic is set ]"));
			chan->output(KVI_OUT_TOPIC, __tr("No topic has been set for %s"), channel.ptr());
			chan->gotTopicReply();
		} else {
			KviStr topicWho, topicTime;
			aux = kvi_extractToken(topicWho, aux, ' ');
			aux = kvi_extractToken(topicTime, aux, ' ');
			if( !msg->bHasBeenHandled ) {
				topicWho.stripLeft(':');  // Ensure that there are no leading ':'
				topicTime.stripLeft(':'); // (there should not be, anyway)
				bool bOk = false;
				uint uTopicTime = topicTime.toUInt(&bOk);
				if( bOk ) {
					QDateTime dt;
					dt.setTime_t(uTopicTime);
					topicTime = dt.toString();
					chan->output(KVI_OUT_TOPIC, __tr("Topic was set by %s on %s"), topicWho.ptr(), topicTime.ptr());
				} else chan->output(KVI_OUT_TOPIC, __tr("Topic was set by %s"), topicWho.ptr());
			}
		}
	}
}

/**
 * [15:31:26] [irc.pragma.net][321] pragma Channel :Users  Name
 * [15:31:26] [irc.pragma.net][322] pragma #linux-it 1 :
 * [15:31:26] [irc.pragma.net][322] pragma #kvirc 1 :[ Channel topic unk
 * [15:31:26] [irc.pragma.net][323] pragma :End of /LIST
 */
void KviServerParser::parseNumericList(KviIrcMessage *msg)
{
	KviListWindow *w = m_pFrm->getListWindow();
	switch( msg->uCommand ) {
		case RPL_LIST:
			w->processListEntry(skipWord(msg->szParams.ptr()));
			break;
		case RPL_LISTEND:
			w->endOfList();
			break;
		case RPL_LISTSTART:
			w->startOfList();
			break;
	}
}

/**
 * [18:11:42] [irc.tin.it][364] [newbie] irc.tin.it irc.tin.it :0 - Telecom Italia Net - Italy
 * [18:11:42] [irc.tin.it][364] [newbie] ircd.tin.it irc.tin.it :1 - Telecom Italia Net - Italy
 * [18:11:42] [irc.tin.it][364] [newbie] irctest.tin.it ircd.tin.it :2 [194.243.155.212] - Telecom Italia Net - Italy
 * [18:11:42] [irc.tin.it][364] [newbie] irc.flashnet.it ircd.tin.it :2 Flashnet Telecomunicazioni - Cybernet Group
 * [18:11:42] [irc.tin.it][364] [newbie] irc.fun.uni.net irc.flashnet.it :3 - UniNet Rome, Italy
 * [18:11:42] [irc.tin.it][364] [newbie] irc.ccii.unipi.it irc.flashnet.it :3 University of Pisa, Italy
 * [18:11:44] [irc.tin.it][365] [newbie] * :End of /LINKS list.
 */
void KviServerParser::parseNumericLinks(KviIrcMessage *msg)
{
	KviLinksWindow *w = m_pFrm->getLinksWindow();

	if( msg->uCommand == RPL_ENDOFLINKS ) {
		w->endOfLinks();
		return;
	}
	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());
	w->processLink(aux);
}

/**
 * [15:49:32] [irc.pragma.net][329] newbie #kvirc 923579125
 */
void KviServerParser::parseNumericCreationTime(KviIrcMessage *msg)
{
	KviStr channel, creationTime;
	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());
	aux = kvi_extractToken(channel, aux, ' ');
	aux = kvi_extractToken(creationTime, aux, ' ');
	// Get the creation time
	creationTime.stripLeft(':');
	bool bOk = false;
	uint uCreationTime = creationTime.toUInt(&bOk);
	if( !bOk )
		creationTime.append(__tr(" [Unrecognized time format]"));
	else {
		QDateTime dt;
		dt.setTime_t(uCreationTime);
		creationTime = dt.toString();
	}

	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		// Oops, desync with the server.
		KviWindow *pOut = g_pOptions->m_bChannelModeChangesToConsole ? m_pConsole : m_pFrm->activeWindow();
		pOut->output(KVI_OUT_TIME, __tr("Channel %s creation time: %s"), channel.ptr(), creationTime.ptr());
		return;
	} else {
		if( !msg->bHasBeenHandled )
			chan->output(KVI_OUT_TIME, __tr("Channel was created on %s"), creationTime.ptr());
	}
}

/**
 * [irc.pragma.net][366] newbie #kvirc :End of /NAMES list.
 */
void KviServerParser::parseNumericEndOfNames(KviIrcMessage *msg)
{
	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());
	// Extract the channel
	KviStr channel;
	aux = kvi_extractToken(channel, aux, ' ');
	__range_valid(*aux == ':');
	// Find the channel
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( chan && !chan->hasAllNames() ) { // [ more comments in parseNumericNamesReply below ]
		// OK. notify the channel that the user list is finished
		chan->endOfNamesList();
		if( g_pOptions->m_bRequestWhoAfterEndOfNames ) {
			m_pSocket->sendFmtData("WHO %s", channel.ptr()); // Request userlist
			if( g_pOptions->m_bBeVerbose ) {
				if( (*channel.ptr() == '#') || (*channel.ptr() == '!') || (*channel.ptr() == '&') ) {
					channel.prepend("\r!join $1\r");
					channel.append("\r");
				}
				m_pConsole->output(KVI_OUT_INTERNAL, __tr("Requesting WHO list for %s"), channel.ptr());
			}
		} else m_pFrm->startUpdatingUserList(); // Nope. Need to update the userlist
		chan->setModeChangeEnabled(chan->isMeOwner() || chan->isMeOp() || chan->isMeHalfOp());
		chan->setTopicChangeEnabled(chan->isMeOwner() || chan->isMeOp() || chan->isMeHalfOp() || !(chan->isModeActive('t')));
		return;
	} // else: it is unhandled; result of a /NAMES or a desync
	if( !msg->bHasBeenHandled ) {
		KviWindow *pOut = g_pOptions->m_bNamesReplyToConsole ? m_pConsole : m_pFrm->activeWindow();
		if( (*channel.ptr() == '#') || (*channel.ptr() == '!') || (*channel.ptr() == '&') ) {
			channel.prepend("\r!join $1\r");
			channel.append("\r");
		}
		pOut->output(KVI_OUT_NAMES, __tr("End of names for %s"), channel.ptr());
	}
}

/**
 * [irc.pragma.net][353] newbie [=|*|@] #kvirc :newbie pragma
 */
void KviServerParser::parseNumericNamesReply(KviIrcMessage *msg)
{
	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());
	// We are pointing to the type of the channel:
	//     = --> public
	//     * --> private
	//     @ --> secret
	// ... but we ignore it
	aux = skipWord(aux);
	__range_valid((*aux == '#') || (*aux == '&') || (*aux == '!'));
	// Now the channel name
	KviStr channel;
	aux = kvi_extractToken(channel, aux, ' ');
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	// And run to the first nickname
	aux = skipToTrailing(aux);
	while( (*aux) && (*aux == ' ') )
		aux++;
	// Now check if we have that channel
	if( chan ) {
		// If the names list of the channel is not complete yet...
		if( !chan->hasAllNames() ) {
			// OK. Time to parse a lot of data
			KviStr aNick;
			char aMode;

			while( *aux ) {
				if( (*aux == '@') || (*aux == '+') || (*aux == '%') || (*aux == '*') ) {
					// Leading umode flag
					aMode = *aux;
					aux++;
				} else aMode = 0;
				// Extract the nickname
				aux = kvi_extractToken(aNick, aux, ' ');
				// And make it join
				if( aNick.hasData() )
					chan->join(aNick.ptr(), (aMode == '@'), (aMode == '+'), (aMode == '%'), 0, (aMode == '*'), false);
				// Run to the next nick (or the end)
				while( (*aux) && (*aux == ' ') )
					aux++;
			}
			// Finished a block, repaint the listbox
			chan->repaintListBox();
			return;
		}
	}
	// Otherwise it is unhandled for us:
	// channel not found, or already has all names.
	// So it is a result of a /NAMES command or a local desync
	// We handle it in a cool way.
	if( !msg->bHasBeenHandled ) {
		KviWindow *pOut = g_pOptions->m_bNamesReplyToConsole ? m_pConsole : m_pFrm->activeWindow();
		if( (*channel.ptr() == '#') || (*channel.ptr() == '!') || (*channel.ptr() == '&') ) {
			channel.prepend("\r!join $1\r");
			channel.append("\r");
		}
		pOut->output(KVI_OUT_NAMES, __tr("Names for %s: %s"), channel.ptr(), aux);
	}
}

/**
 * [04:16:39] [irc.tin.it][324] newbie #linux-it +tn
 * [02:57:49] [services1.undernet.org][324] [newbie] #azz +
 */
void KviServerParser::parseNumericChannelMode(KviIrcMessage *msg)
{
	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());
	KviStr channel;
	aux = kvi_extractToken(channel, aux, ' ');
	KviWindow *pOut;
	KviChannel *chan;
	if( (*channel.ptr() == '#') || (*channel.ptr() == '&') || (*channel.ptr() == '!') ) {
		chan = m_pFrm->findChannel(channel.ptr());
		while( *aux && *aux != '+' )
			aux++;
		if( *aux )
			aux++;
		if( !chan ) {
			pOut = g_pOptions->m_bChannelModeChangesToConsole ? m_pConsole : m_pFrm->activeWindow();
			if( (*aux) && (*aux != ' ') )
				pOut->output(KVI_OUT_CHANMODE, __tr("Channel mode for \r!join $1\r%s\r is +%s"), channel.ptr(), aux);
			else
				pOut->output(KVI_OUT_CHANMODE, __tr("Channel \r!join $1\r%s\r is modeless"), channel.ptr());
			return;
		}
	} else {
		while( *aux && *aux == ' ')
			aux++;
		pOut = g_pOptions->m_bUserModeChangesToConsole ? m_pConsole : m_pFrm->activeWindow();
		if( *aux )
			pOut->output(KVI_OUT_UMODE, __tr("Mode for %s is +%s"), channel.ptr(), aux);
		else
			pOut->output(KVI_OUT_UMODE, __tr("%s is modeless"), channel.ptr());
		return;
	}

	pOut = g_pOptions->m_bChannelModeChangesToConsole ? m_pConsole : ((KviWindow *) chan);

	bool bNoOutput = false;
	if( (*aux) && (*aux != ' ') ) {
		if( g_pEventManager->eventEnabled(KviEvent_OnChannelMode) ) {
			KviStr eventparms(KviStr::Format,"%s %s %s %s", chan->caption(), msg->szPrefix.ptr(), msg->szPrefix.ptr(), aux);
			bNoOutput = m_pUserParser->callEvent(KviEvent_OnChannelMode, chan, eventparms);
		}
		if( !(bNoOutput || msg->bHasBeenHandled) )
			pOut->output(KVI_OUT_CHANMODE, __tr("Channel mode for %s is +%s"), channel.ptr(), aux);
		const char *parms = aux;
		while( *parms && (*parms != ' ') )
			parms++;
		while( *parms == ' ' )
			parms++;
		while( *aux && (*aux != ' ') ) {
			switch( *aux ) {
				case 'k': // Fall-through
				case 'l': // Fall-through
				case 'O':
					chan->setModeFlag(*aux, true, parms);
					while( *parms && (*parms != ' ') )
						parms++;
					while( *parms == ' ' )
						parms++;
					break;
				default:
					chan->setModeFlag(*aux, true);
					break;
			}
			aux++;
		}
	} else {
		if( g_pEventManager->eventEnabled(KviEvent_OnChannelMode) ) {
			KviStr eventparms(KviStr::Format, "%s %s %s", chan->caption(), msg->szPrefix.ptr(), msg->szPrefix.ptr());
			bNoOutput = m_pUserParser->callEvent(KviEvent_OnChannelMode, chan, eventparms);
		}
		if( !(bNoOutput || msg->bHasBeenHandled) )
			pOut->output(KVI_OUT_CHANMODE, __tr("Channel %s is modeless"), channel.ptr());
		// Clear mode label
		chan->m_pModeLabel->setText("");
	}
	chan->gotModeReply();
}

void KviServerParser::getDateTimeStringFromCharTimeT(KviStr &buffer, const char *time_t_string)
{
	KviStr tmp = time_t_string;
	bool bOk = false;
	unsigned int uTime = tmp.toUInt(&bOk);
	if( bOk ) {
		QDateTime dt;
		dt.setTime_t(uTime);
		buffer = dt.toString();
	} else buffer = __tr("[Unknown]");
}

/**
 * [irc.tin.it][367] newbie #linux-it *!*@*jkkkjk.jkjkjk.ru
 * [irc.tin.it][367] newbie #linux-it *!*@*a-ars*.r.org
 * [irc.tin.it][367] newbie #linux-it *!*@*cuc.xoomics.net
 * [irc.tin.it][367] newbie #linux-it *!*@ray.webweb.ooo.ca
 * [irc.tin.it][367] newbie #linux-it *!fino*@*
 * [irc.tin.it][368] newbie #linux-it :End of Channel Ban List
 */
void KviServerParser::parseNumericBanList(KviIrcMessage *msg)
{
	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());

	KviStr channel;
	aux = kvi_extractToken(channel, aux, ' ');
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());

	if( *aux == ':' )
		aux++;
	KviStr mask;
	aux = kvi_extractToken(mask, aux, ' ');
	while( (*aux == ' ') || (*aux == ':') )
		aux++;
	if( !(*aux) )
		aux = __tr("[Unknown]");

	KviStr banSetBy;
	aux = kvi_extractToken(banSetBy, aux, ' ');
	while( (*aux == ' ') || (*aux == ':') )
		aux++;

	KviStr banSetAt;
	getDateTimeStringFromCharTimeT(banSetAt, aux);

	if( !chan ) {
		// Oops, desync with the server.
		KviWindow *pOut = g_pOptions->m_bBanAndExceptionListToConsole ? m_pConsole : m_pFrm->activeWindow();
		if( msg->uCommand == RPL_BANLIST ) {
			pOut->output(KVI_OUT_BAN,
				__tr("Ban entry %s (%s:%s) for channel \r!join $1\r%s\r"),
				mask.ptr(), banSetBy.ptr(), banSetAt.ptr(), channel.ptr()
			);
		} else pOut->output(KVI_OUT_BAN, __tr("End of ban list for channel \r!join $1\r%s\r"), channel.ptr());
		return;
	}

	if( msg->uCommand == RPL_BANLIST )
		chan->banMask(mask.ptr(), banSetBy.ptr(), banSetAt.ptr(), true);
	else
		chan->gotBanlistReply();

	if( g_pOptions->m_bShowBanAndExceptionList && (!msg->bHasBeenHandled) ) {
		KviWindow *pOut = (g_pOptions->m_bBanAndExceptionListToConsole ? m_pConsole : ((KviWindow *) chan));
		if( msg->uCommand == RPL_BANLIST ) {
			pOut->output(KVI_OUT_BAN,
				__tr("Ban entry for %s: %s (%s:%s)"), channel.ptr(), mask.ptr(), banSetBy.ptr(), banSetAt.ptr()
			);
		} else pOut->output(KVI_OUT_BAN, __tr("End of channel ban list for %s"), channel.ptr());
	}

	if( (msg->uCommand == RPL_BANLIST) &&
	    (g_pEventManager->eventEnabled(KviEvent_OnBan) || g_pEventManager->eventEnabled(KviEvent_OnMeBan))
	) {
		KviStr eventparms(KviStr::Format,
			"%s %s %s %s %s %s",
			chan->caption(), msg->szPrefix.ptr(), msg->szPrefix.ptr(), mask.ptr(), banSetBy.ptr(), banSetAt.ptr()
		);
		KviIrcUser u(m_pFrm->m_global.szCurrentMaskFromServer.ptr());
		if( u.matches(mask.ptr()) ) { // It is me
			if( g_pEventManager->eventEnabled(KviEvent_OnMeBan) ) {
				if( m_pUserParser->callEvent(
					KviEvent_OnMeBan,
					(g_pOptions->m_bBanAndExceptionListToConsole ? m_pConsole : ((KviWindow *) chan)),
					eventparms)
				) {
					return;
				}
			}
		} else if( g_pEventManager->eventEnabled(KviEvent_OnBan) ) {
			if( m_pUserParser->callEvent(
				KviEvent_OnBan,
				(g_pOptions->m_bBanAndExceptionListToConsole ? m_pConsole : ((KviWindow *) chan)),
				eventparms)
			) {
				return;
			}
		}
	}
}

/**
 * [irc.tin.it][348] newbie #linux-it *!*@0233321.po0.f.it
 * [irc.tin.it][349] newbie #linux-it :End of Channel Exception List
 */
void KviServerParser::parseNumericExceptionList(KviIrcMessage *msg)
{
	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());

	KviStr channel;
	aux = kvi_extractToken(channel, aux, ' ');
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());

	if( *aux == ':' )
		aux++;
	KviStr mask;
	aux = kvi_extractToken(mask, aux, ' ');
	while( (*aux == ' ') || (*aux == ':') )
		aux++;
	if( !(*aux) )
		aux = __tr("[Unknown]");

	KviStr excSetBy;
	aux = kvi_extractToken(excSetBy, aux, ' ');
	while( (*aux == ' ') || (*aux == ':') )
		aux++;

	KviStr excSetAt;
	getDateTimeStringFromCharTimeT(excSetAt, aux);

	if( !chan ) {
		// Oops, desync with the server.
		KviWindow *pOut = g_pOptions->m_bBanAndExceptionListToConsole ? m_pConsole : m_pFrm->activeWindow();
		if( msg->uCommand == RPL_EXCEPTLIST ) {
			pOut->output(KVI_OUT_BAN,
				__tr("Ban exception entry %s (%s:%s) for channel \r!join $1\r%s\r"),
				mask.ptr(), excSetBy.ptr(), excSetAt.ptr(), channel.ptr()
			);
		} else pOut->output(KVI_OUT_BAN, __tr("End of ban exception list for channel \r!join $1\r%s\r"), channel.ptr());
		return;
	}
	if( msg->uCommand == RPL_EXCEPTLIST )
		chan->banExceptionMask(mask.ptr(), excSetBy.ptr(), excSetAt.ptr(), true);
	// else: RPL_ENDOFEXCEPTLIST; simply ignore it

	if( g_pOptions->m_bShowBanAndExceptionList && (!msg->bHasBeenHandled) ) {
		KviWindow *pOut = g_pOptions->m_bBanAndExceptionListToConsole ? m_pConsole : ((KviWindow *) chan);
		if( msg->uCommand == RPL_EXCEPTLIST ) {
			pOut->output(KVI_OUT_BAN,
				__tr("Ban exception entry for %s: %s (%s:%s)"), channel.ptr(), mask.ptr(), excSetBy.ptr(), excSetAt.ptr()
			);
		} else pOut->output(KVI_OUT_BAN, __tr("End of channel ban exception list for %s"), channel.ptr());
	}

	if( (msg->uCommand == RPL_EXCEPTLIST) &&
		(g_pEventManager->eventEnabled(KviEvent_OnBanException) || g_pEventManager->eventEnabled(KviEvent_OnMeBanException))
	) {
		KviStr eventparms(KviStr::Format,
			"%s %s %s %s %s %s",
			chan->caption(), msg->szPrefix.ptr(), msg->szPrefix.ptr(), mask.ptr(), excSetBy.ptr(), excSetAt.ptr()
		);
		KviIrcUser u(m_pFrm->m_global.szCurrentMaskFromServer.ptr());
		if( u.matches(mask.ptr()) ) { // It is me
			if( g_pEventManager->eventEnabled(KviEvent_OnMeBanException) ) {
				if( m_pUserParser->callEvent(
					KviEvent_OnMeBanException,
					(g_pOptions->m_bBanAndExceptionListToConsole ? m_pConsole : ((KviWindow *) chan)),
					eventparms)
				) {
					return;
				}
			}
		} else if( g_pEventManager->eventEnabled(KviEvent_OnBanException) ) {
			if( m_pUserParser->callEvent(
				KviEvent_OnBanException,
				(g_pOptions->m_bBanAndExceptionListToConsole ? m_pConsole : ((KviWindow *) chan)),
				eventparms)
			) {
				return;
			}
		}
	}
}

/**
 * [irc.tin.it][346] newbie #linux-it *!*@0233321.po0.f.it
 * [irc.tin.it][347] newbie #linux-it :End of Channel Invite List
 */
void KviServerParser::parseNumericInviteList(KviIrcMessage *msg)
{
	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());

	KviStr channel;
	aux = kvi_extractToken(channel, aux, ' ');
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());

	if( *aux == ':' )
		aux++;
	KviStr mask;
	aux = kvi_extractToken(mask, aux, ' ');
	while( (*aux == ' ') || (*aux == ':') )
		aux++;
	if( !(*aux) )
		aux = __tr("[Unknown]");

	KviStr excSetBy;
	aux = kvi_extractToken(excSetBy, aux, ' ');
	while( (*aux == ' ') || (*aux == ':') )
		aux++;

	KviStr excSetAt;
	getDateTimeStringFromCharTimeT(excSetAt, aux);

	if( !chan ) {
		// Oops, desync with the server.
		KviWindow *pOut = g_pOptions->m_bBanAndExceptionListToConsole ? m_pConsole : m_pFrm->activeWindow();
		if( msg->uCommand == RPL_INVITELIST ) {
			pOut->output(KVI_OUT_BAN,
				__tr("Invite exception entry %s (%s:%s) for channel \r!join $1\r%s\r"),
				mask.ptr(), excSetBy.ptr(), excSetAt.ptr(), channel.ptr()
			);
		} else pOut->output(KVI_OUT_BAN, __tr("End of invite exception list for channel \r!join $1\r%s\r"), channel.ptr());
		return;
	}
	if( msg->uCommand == RPL_INVITELIST )
		chan->inviteExceptionMask(mask.ptr(), excSetBy.ptr(), excSetAt.ptr(), true);
	// else: RPL_ENDOFINVITELIST: simply ignore it

	if( g_pOptions->m_bShowBanAndExceptionList && (!msg->bHasBeenHandled) ) {
		KviWindow *pOut = g_pOptions->m_bBanAndExceptionListToConsole ? m_pConsole : ((KviWindow *) chan);
		if( msg->uCommand == RPL_INVITELIST ) {
			pOut->output(KVI_OUT_BAN,
				__tr("Invite exception entry for %s: %s (%s:%s)"),
				channel.ptr(), mask.ptr(), excSetBy.ptr(), excSetAt.ptr()
			);
		} else pOut->output(KVI_OUT_BAN, __tr("End of channel invite exception list for %s"), channel.ptr());
	}

	if( (msg->uCommand == RPL_INVITELIST) &&
		(g_pEventManager->eventEnabled(KviEvent_OnInviteException) ||
		 g_pEventManager->eventEnabled(KviEvent_OnMeInviteException))
	) {
		KviStr eventparms(KviStr::Format,
			"%s %s %s %s %s %s",
			chan->caption(), msg->szPrefix.ptr(), msg->szPrefix.ptr(), mask.ptr(), excSetBy.ptr(), excSetAt.ptr()
		);
		KviIrcUser u(m_pFrm->m_global.szCurrentMaskFromServer.ptr());
		if( u.matches(mask.ptr()) ) { // It is me
			if( g_pEventManager->eventEnabled(KviEvent_OnMeInviteException)) {
				if( m_pUserParser->callEvent(
					KviEvent_OnMeInviteException,
					(g_pOptions->m_bBanAndExceptionListToConsole ? m_pConsole : ((KviWindow *) chan)),
					eventparms)
				) {
					return;
				}
			}
		} else if( g_pEventManager->eventEnabled(KviEvent_OnInviteException) ) {
			if( m_pUserParser->callEvent(
				KviEvent_OnInviteException,
				(g_pOptions->m_bBanAndExceptionListToConsole ? m_pConsole : ((KviWindow *) chan)),
				eventparms)
			) {
				return;
			}
		}
	}
}

/**
 * [irc.tin.it][352] newbie #linux-it ^nobody a-si6-39.tin.it irc.tin.it newbie H :0 Using KVIrc 1.0.0 'Millenium' by Szymon Stefanek
 * [irc.tin.it][352] newbie #linux-it ~Mik ip014.pool-300.flashnet.it irc.flashnet.it |Wallace| H :2 Gustavo La Mazza
 * [irc.tin.it][352] newbie #linux-it dira server.speedcom.it *.webbernet.net fhex G@ :7 dira@speedcom.it
 * [irc.tin.it][352] newbie #linux-it buzzz 195.223.77.66 irc.flashnet.it buzzzo G@ :2 LiCe | http://www.mistik.net/lice
 * [irc.tin.it][352] newbie #linux-it slash irc.flashnet.it irc.flashnet.it sLASh H@ :2 sLASh
 * [irc.tin.it][352] newbie #linux-it rusty 194.243.114.217 irc.fun.uni.net rustino G@ :4 In her eyes my life...
 * [irc.tin.it][315] newbie #linux-it :End of /WHO list.
 */
void KviServerParser::parseNumericWhoReply(KviIrcMessage *msg)
{
	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());

	KviStr channel;
	aux = kvi_extractToken(channel, aux, ' ');
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());

	if( !chan ) {
		// Local desync, or result of a /WHO call.
		// Show it normally
		if( !msg->bHasBeenHandled ) {
			KviStr user;
			aux = kvi_extractToken(user, aux, ' ');   // Username
			KviStr host;
			aux = kvi_extractToken(host, aux, ' ');   // Hostname
			KviStr server;
			aux = kvi_extractToken(server, aux, ' '); // Using server
			KviStr nick;
			aux = kvi_extractToken(nick, aux, ' ');   // Nickname

			KviWindow *pOut = g_pOptions->m_bWhoMsgsToConsole ? m_pConsole : m_pFrm->activeWindow();
			if( (*channel.ptr() == '#') || (*channel.ptr() == '!') || (*channel.ptr() == '&') ) {
				channel.prepend("\r!join $1\r");
				channel.append("\r");
			}
			if( msg->uCommand == RPL_WHOREPLY ) {
				pOut->output(KVI_OUT_WHO,
					__tr("WHO entry for %s: \x02\r!query $1\r%s\r\x02 %s@\r!host $1\r%s\r on \r!motd $s\r%s\r %s"),
					channel.ptr(), nick.ptr(), user.ptr(), host.ptr(), server.ptr(), aux
				);
			} else pOut->output(KVI_OUT_WHO, __tr("End of WHO list for %s"), channel.ptr());
		}
		return;
	}

	// Who request for channel?
	if( chan->whoRequestDone() || g_pOptions->m_bShowInternalWhoMessages ) {
		// Result of a /WHO command
		if( !msg->bHasBeenHandled ) {
			KviStr user;
			aux = kvi_extractToken(user, aux, ' ');  // Username
			KviStr host;
			aux = kvi_extractToken(host, aux, ' ');  // Hostname
			KviStr dummy;
			aux = kvi_extractToken(dummy, aux, ' '); // Using server
			KviStr nick;
			aux = kvi_extractToken(nick, aux, ' ');  // Nickname

			KviWindow *pOut = g_pOptions->m_bWhoMsgsToConsole ? m_pConsole : ((KviWindow *) chan);
			if( msg->uCommand == RPL_WHOREPLY ) {
				pOut->output(KVI_OUT_WHO,
					__tr("WHO entry for %s: \x02\r!query $1\r%s\r\x02 %s@\r!host $1\r%s\r on %s %s"),
					channel.ptr(), nick.ptr(), user.ptr(), host.ptr(), dummy.ptr(), aux
				);
			} else pOut->output(KVI_OUT_WHO, __tr("End of WHO list for %s"), channel.ptr());
		}
		if( chan->whoRequestDone() )
			return; // If already updated, return
		// else: the show for internal messages was requested; still need to update the list
	}

	if( msg->uCommand == RPL_ENDOFWHO )
		chan->endOfWho();
	else {
		// Update the userlist
		KviStr user;
		aux = kvi_extractToken(user, aux, ' ');  // Username
		KviStr host;
		aux = kvi_extractToken(host, aux, ' ');  // Hostname
		KviStr dummy;
		aux = kvi_extractToken(dummy, aux, ' '); // Using server
		KviStr nick;
		aux = kvi_extractToken(nick, aux, ' ');  // Nickname

		KviIrcUser *u = chan->m_pListBox->findUser(nick.ptr());
		if( u ) {
			u->setUsername(user.ptr());
			u->setHost(host.ptr());
		} else {
			// Desync?
			KviWindow *pOut = g_pOptions->m_bDesyncMsgsToConsole ? m_pConsole : m_pFrm->activeWindow();
			pOut->output(KVI_OUT_DESYNC,
				__tr("Local desync: received internal WHO reply for channel %s: %s %s %s %s %s"),
				channel.ptr(), user.ptr(), host.ptr(), dummy.ptr(), nick.ptr(), aux
			);
			return;
		}
	}
}

void KviServerParser::parseNumericUserhost(KviIrcMessage *msg)
{
	const char *aux = skipToTrailing(msg->szParams.ptr());
	KviIrcUser user; // Empty user
	while( *aux ) {
		aux = user.setUserhostMask(aux);
		if( user.hasNick() )
			m_pFrm->m_pUserList->updateUser(user);
	}
	if( m_pFrm->m_global.bUserHostInProgress )
		m_pFrm->m_global.bUserHostInProgress = false;
	else {
		KviWindow *pOut = g_pOptions->m_bWhoMsgsToConsole ? m_pConsole : m_pFrm->activeWindow();
		pOut->output(KVI_OUT_WHO,
			__tr("USERHOST reply: \x02\r!query $1\r%s\r\x02 %s@\r!host $1\r%s\r"),
			user.nick(), user.username(), user.host()
		);
	}
}

/**
 * [05:03:36] [irc.tin.it][311] newbie Diabl0 ~Diablin0 ulisse.iuav.unive.it * :Please Warm Up Me
 * [05:03:36] [irc.tin.it][319] newbie Diabl0 :@#skatafascio
 * [05:03:36] [irc.tin.it][312] newbie Diabl0 *.webbernet.net :[ircd.webbernet.net] Web Net Worldwide
 * [05:03:36] [irc.tin.it][318] newbie Diabl0 :End of /WHOIS list.
 * 		case RPL_WHOISUSER:     // 311 "%s %s %s * :%s"
 * 		case RPL_WHOISSERVER:   // 312 "%s %s :%s"
 * 		case RPL_WHOISOPERATOR: // 313 "%s :is an IRC Operator"
 * 		case RPL_WHOWASUSER:    // 314 "%s %s %s * :%s"
 * 		case RPL_WHOISCHANOP:   // 316 "???"
 * 		case RPL_WHOISIDLE:     // 317 "%s %ld %ld :seconds idle, signon time"
 * 		case RPL_ENDOFWHOIS:    // 318 "%s :End of /WHOIS list."
 * 		case RPL_WHOISCHANNELS: // 319 "%s :%s"
 * 		case RPL_ENDOFWHOWAS:   // 369 "%s :END of /WHOWAS"
 * 		case RPL_WHOISREGNICK:  // 307 "???"
 * 		case RPL_WHOISADMIN:    // 308 "???"
 * 		case RPL_WHOISSADMIN:   // 309 "???"
 * 		case RPL_WHOISHELPOP:   // 310 "???"
 * 		case RPL_AWAY:          // 301 "%s :%s"
 */
void KviServerParser::parseNumericWhoisReply(KviIrcMessage *msg)
{
	// TODO: Do updateUserlist() on WHOIS!!!

	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());

	KviStr target;
	aux = kvi_extractToken(target, aux, ' ');
	KviWindow *pOut = g_pOptions->m_bWhoisRepliesToConsole ? m_pConsole : m_pFrm->activeWindow();
	if( *aux == ':' )
		aux = skipToTrailing(aux);

	KviStr tmp;
	switch( msg->uCommand ) {
		case RPL_WHOISUSER: {
			KviIrcUser u;
			bool bQuiet = false;
			aux = kvi_extractToken(tmp, aux, ' ');
			u.setNick(target.ptr());
			u.setUsername(tmp.ptr());
			aux = kvi_extractToken(tmp, aux, ' ');
			u.setHost(tmp.ptr());
			aux = skipToTrailing(aux);
			m_pFrm->m_pUserList->updateUser(u);
			if( !m_pFrm->m_global.pDnsWhoisPending->isEmpty() ) {
				for( KviStr *s = m_pFrm->m_global.pDnsWhoisPending->first()
				    ; s
				    ; s = m_pFrm->m_global.pDnsWhoisPending->next()
				) {
					if( kvi_strEqualCI(s->ptr(), target.ptr()) ) {
						m_pFrm->m_global.pDnsWhoisPending->removeRef(s);
						if( !m_pUserParser->doAsyncDnsCallForHost(u.host(), u.nick()) ) {
							pOut->output(KVI_OUT_ERROR,
								__tr("Cannot look up %s's hostname: cannot start the DNS thread"), u.nick()
							);
						}
						bQuiet = true;
					}
				}
			}

			KviAsyncWhoisEntry *e = m_pUserParser->m_pAsyncWhoisController->findPendingWhois(target.ptr());
			if( e ) {
				bQuiet      = true;
				e->user     = u.username();
				e->host     = u.host();
				e->realName = aux;
			}
			if( !bQuiet ) {
				if( g_pEventManager->eventEnabled(KviEvent_OnWhoisReply) ) {
					KviStr eventparms(KviStr::Format, "USER %s %s %s %s", u.nick(), u.username(), u.host(), aux);
					if( m_pUserParser->callEvent(KviEvent_OnWhoisReply, m_pConsole, eventparms) )
						return;
				}

				if( !msg->bHasBeenHandled ) {
					pOut->output(KVI_OUT_WHO,
						__tr("%c%s%c is:        \r!query $1\r%s\r!%s@\r!host $1\r%s\r"),
						KVI_TEXT_BOLD, target.ptr(), KVI_TEXT_BOLD, u.nick(), u.username(), u.host()
					);
					pOut->output(KVI_OUT_WHO, __tr("%c%s%c real name: %s"), KVI_TEXT_BOLD, target.ptr(), KVI_TEXT_BOLD, aux);
				}
			}
			break;
		}
		case RPL_WHOISSERVER: {
			KviAsyncWhoisEntry *e = m_pUserParser->m_pAsyncWhoisController->findPendingWhois(target.ptr());
			if( e )
				e->server = aux;
			else {
				if( g_pEventManager->eventEnabled(KviEvent_OnWhoisReply) ) {
					KviStr eventparms(KviStr::Format, "SERVER %s %s", target.ptr(), aux);
					if( m_pUserParser->callEvent(KviEvent_OnWhoisReply, m_pConsole, eventparms) )
						return;
				}

				KviStr server;
				aux = kvi_extractToken(server, aux, ' ');
				if( *aux == ':' )
					aux = skipToTrailing(aux);
				if( !msg->bHasBeenHandled ) {
					pOut->output(KVI_OUT_WHO,
						__tr("%c%s%c server:    \r!motd $1\r%s\r: %s"),
						KVI_TEXT_BOLD, target.ptr(), KVI_TEXT_BOLD, server.ptr(), aux
					);
				}
			}
			break;
		}
		case RPL_WHOISOPERATOR: {
			KviAsyncWhoisEntry *e = m_pUserParser->m_pAsyncWhoisController->findPendingWhois(target.ptr());
			if( e )
				e->bIrcOp = true;
			else {
				if( g_pEventManager->eventEnabled(KviEvent_OnWhoisReply) ) {
					KviStr eventparms(KviStr::Format, "OPER %s", target.ptr());
					if( m_pUserParser->callEvent(KviEvent_OnWhoisReply, m_pConsole, eventparms) )
						return;
				}
				if( !msg->bHasBeenHandled ) {
					pOut->output(KVI_OUT_WHO,
						__tr("%c%s%c status:    IRC Operator"), KVI_TEXT_BOLD, target.ptr(), KVI_TEXT_BOLD
					);
				}
			}
			break;
		}
		case RPL_WHOWASUSER:
			if( g_pEventManager->eventEnabled(KviEvent_OnWhoisReply) ) {
				KviStr eventparms(KviStr::Format, "WAS %s %s", target.ptr(), aux);
				if( m_pUserParser->callEvent(KviEvent_OnWhoisReply, m_pConsole, eventparms) )
					return;
			}
			if( !msg->bHasBeenHandled ) {
				pOut->output(KVI_OUT_WHO, __tr("%c%s%c was:       %s"), KVI_TEXT_BOLD, target.ptr(), KVI_TEXT_BOLD, aux);
			}
			break;
		case RPL_WHOISREGNICK:  // 307 "???"
		case RPL_WHOISADMIN:    // 308 "???"
		case RPL_WHOISSADMIN:   // 309 "???"
		case RPL_WHOISHELPOP:   // 310 "???"
		case RPL_WHOISCHANOP:
		case RPL_AWAY: {
			KviAsyncWhoisEntry *e = m_pUserParser->m_pAsyncWhoisController->findPendingWhois(target.ptr());
			if( e )
				e->status = aux;
			else {
				if( g_pEventManager->eventEnabled(KviEvent_OnWhoisReply) ) {
					KviStr eventparms(KviStr::Format, "STATUS %s %s", target.ptr(), aux);
					if( m_pUserParser->callEvent(KviEvent_OnWhoisReply, m_pConsole, eventparms) )
						return;
				}
				if( !msg->bHasBeenHandled ) {
					pOut->output(KVI_OUT_WHO,
						__tr("%c%s%c %s: %s"), KVI_TEXT_BOLD, target.ptr(), KVI_TEXT_BOLD,
						(msg->uCommand == RPL_AWAY) ? "away     " : "status   ", aux
					);
				}
			}
			break;
		}
		case RPL_WHOISIDLE: {
			aux = kvi_extractToken(tmp, aux, ' ');
			KviAsyncWhoisEntry *e = m_pUserParser->m_pAsyncWhoisController->findPendingWhois(target.ptr());
			if( e )
				e->idle = tmp.ptr();
			else {
				bool bOk    = false;
				bool bQuiet = false;
				unsigned int secs = tmp.toUInt(&bOk);
				if( bOk ) {
					unsigned int hours = secs / 3600;
					secs -= hours * 3600;
					unsigned int mins  = secs /   60;
					secs -= mins * 60;

					if( g_pEventManager->eventEnabled(KviEvent_OnWhoisReply) ) {
						KviStr eventparms(KviStr::Format, "IDLE %s %s %u %u %u", target.ptr(), tmp.ptr(), hours, mins, secs);
						bQuiet = m_pUserParser->callEvent(KviEvent_OnWhoisReply, m_pConsole, eventparms);
					}
					if( !(msg->bHasBeenHandled || bQuiet) ) {
						pOut->output(KVI_OUT_WHO,
							__tr("%c%s%c idle time: %s seconds (%u hrs %u mins %u secs)"),
							KVI_TEXT_BOLD, target.ptr(), KVI_TEXT_BOLD, tmp.ptr(), hours, mins, secs
						);
					}
				} else {
					if( g_pEventManager->eventEnabled(KviEvent_OnWhoisReply) ) {
						KviStr eventparms(KviStr::Format, "UIDLE %s %s", target.ptr(), tmp.ptr());
						bQuiet = m_pUserParser->callEvent(KviEvent_OnWhoisReply, m_pConsole, eventparms);
					}
					if( !(msg->bHasBeenHandled || bQuiet) ) {
						pOut->output(KVI_OUT_WHO,
							__tr("%c%s%c idle time: %s seconds"), KVI_TEXT_BOLD, target.ptr(), KVI_TEXT_BOLD, tmp.ptr()
						);
					}
				}
				if( *aux && isdigit(*aux) ) {
					aux = kvi_extractToken(tmp, aux, ' ');
					bOk = false;
					unsigned int uI = tmp.toUInt(&bOk);
					if( bOk ) {
						QDateTime dt;
						dt.setTime_t(uI);
						tmp = dt.toString();

						if( g_pEventManager->eventEnabled(KviEvent_OnWhoisReply) ) {
							KviStr eventparms(KviStr::Format, "SIGNON %s %s", target.ptr(), tmp.ptr());
							if( m_pUserParser->callEvent(KviEvent_OnWhoisReply, m_pConsole, eventparms) )
								return;
						}
						if( !msg->bHasBeenHandled ) {
							pOut->output(KVI_OUT_WHO,
								__tr("%c%s%c signon at: %s"), KVI_TEXT_BOLD, target.ptr(), KVI_TEXT_BOLD, tmp.ptr()
							);
						}
					}
				}
			}
			break;
		}
		case RPL_ENDOFWHOIS: {
			bool bQuiet = false;
			if( !m_pFrm->m_global.pDnsWhoisPending->isEmpty() ) {
				for( KviStr *s = m_pFrm->m_global.pDnsWhoisPending->first()
				    ; s
				    ; s = m_pFrm->m_global.pDnsWhoisPending->next()
				) {
					if( kvi_strEqualCI(s->ptr(), target.ptr()) ) {
						m_pFrm->m_global.pDnsWhoisPending->removeRef(s);
						bQuiet = true;
					}
				}
			}

			KviAsyncWhoisEntry *e = m_pUserParser->m_pAsyncWhoisController->findPendingWhois(target.ptr());
			if( e ) {
				bQuiet = true;
				m_pUserParser->m_pAsyncWhoisController->end(e);
			}
			if( !bQuiet ) {
				if( g_pEventManager->eventEnabled(KviEvent_OnWhoisReply) ) {
					KviStr eventparms(KviStr::Format, "ENDWHOIS %s %s", target.ptr(), msg->szPrefix.ptr());
					if( m_pUserParser->callEvent(KviEvent_OnWhoisReply, m_pConsole, eventparms) )
						return;
				}
				if( !(msg->bHasBeenHandled) ) {
					pOut->output(KVI_OUT_WHO,
						__tr("%c%s%c WHOIS from \r!motd $1\r%s\r"),
						KVI_TEXT_BOLD, target.ptr(), KVI_TEXT_BOLD, msg->szPrefix.ptr()
					);
				}
			}
			break;
		}
		case RPL_ENDOFWHOWAS:
			if( g_pEventManager->eventEnabled(KviEvent_OnWhoisReply) ) {
				KviStr eventparms(KviStr::Format, "ENDWHOWAS %s %s", target.ptr(), msg->szPrefix.ptr());
				if( m_pUserParser->callEvent(KviEvent_OnWhoisReply, m_pConsole, eventparms) )
					return;
			}
			if( !msg->bHasBeenHandled ) {
				pOut->output(KVI_OUT_WHO,
					__tr("%c%s%c WHOWAS from \r!motd $1\r%s\r"),
					KVI_TEXT_BOLD, target.ptr(), KVI_TEXT_BOLD, msg->szPrefix.ptr()
				);
			}
			break;
		case RPL_WHOISCHANNELS: {
			KviAsyncWhoisEntry *e = m_pUserParser->m_pAsyncWhoisController->findPendingWhois(target.ptr());
			if( e )
				e->chans = aux;
			else {
				if( g_pEventManager->eventEnabled(KviEvent_OnWhoisReply) ) {
					KviStr eventparms(KviStr::Format, "CHANS %s %s", target.ptr(), aux);
					if( m_pUserParser->callEvent(KviEvent_OnWhoisReply, m_pConsole, eventparms) )
						return;
				}
				if( !msg->bHasBeenHandled ) {
					KviStr channels;
					const char *aux2;
					while( *aux ) {
						aux2 = aux;
						while( *aux && (*aux != '#') && (*aux != '!') && (*aux != '&') )
							aux++;
						if( *aux ) {
							channels.append(aux2, aux - aux2);
							channels.append("\r!join $1\r");
							aux2 = aux;
							while( *aux && (*aux != ' ') && (*aux != ',') )
								aux++;
							channels.append(aux2, aux - aux2);
							channels.append('\r');
						}
					}
					pOut->output(KVI_OUT_WHO,
						__tr("%c%s%c channels:  %s"), KVI_TEXT_BOLD, target.ptr(), KVI_TEXT_BOLD, channels.ptr()
					);
				}
			}
			break;
		}
	}
}

/**
 * "%s.%s %s :%s"
 */
void KviServerParser::parseNumericVersion(KviIrcMessage *msg)
{
	if( msg->bHasBeenHandled ) return;

	// Skip the leading nickname
	const char *aux = skipWord(msg->szParams.ptr());

	if( g_pOptions->m_bServerInfoToConsole )
		m_pConsole->output(KVI_OUT_INFO, __tr("Server %s: %s"), msg->szPrefix.ptr(), aux);
	else
		m_pFrm->activeWindow()->output(KVI_OUT_INFO, __tr("Server %s: %s"), msg->szPrefix.ptr(), aux);

	aux = skipToTrailing(aux);
	KviStr tmp;
	tmp.stripWhiteSpace();

	while( *aux ) {
		switch( *aux ) {
			case 'A': tmp.append("A:SENDQ_ALWAYS ");                      break;
			case 'a': tmp.append("a:NO_IDENT ");                          break;
			case 'c': tmp.append("c:CHROOTDIR ");                         break;
			case 'C': tmp.append("C:CMDLINE_CONFIG ");                    break;
			case 'd': tmp.append("d:DO_ID (or RANDOM_NDELAY) ");          break;
			case 'D': tmp.append("D:DEBUGMODE ");                         break;
			case 'e': tmp.append("e:LOCOP_REHASH ");                      break;
			case 'E': tmp.append("E:OPER_REHASH ");                       break;
			case 'f': tmp.append("f:FOLLOW_IDENT_RFC (or SLOW_ACCEPT) "); break;
			case 'F': tmp.append("F:HIDE_FAKES (or CLONE_CHECK) " );      break;
			case 'g': tmp.append("g:SUN_GSO_BUG ");                       break;
			case 'G': tmp.append("G:SHOW_GHOSTS ");                       break;
			case 'H': tmp.append("H:HUB ");                               break;
			case 'h': tmp.append("h:BETTER_CDELAY ");                     break;
			case 'i': tmp.append("i:SHOW_INVISIBLE_LUSERS ");             break;
			case 'I': tmp.append("I:NO_DEFAULT_INVISIBLE ");              break;
			case 'K': tmp.append("K:OPER_KILL ");                         break;
			case 'k': tmp.append("k:LOCAL_KILL_ONLY ");                   break;
			case 'L': tmp.append("L:LEAST_IDLE ");                        break;
			case 'm': tmp.append("m:M4_PREPROC ");                        break;
			case 'M': tmp.append("M:IDLE_FROM_MSG ");                     break;
			case 'N': tmp.append("N:DELAY_NICKS (or NPATH) ");            break;
			case 'n': tmp.append("n:BETTER_NDELAY ");                     break;
			case 'p': tmp.append("p:CRYPT_OPER_PASSWORD ");               break;
			case 'P': tmp.append("P:CRYPT_LINK_PASSWORD ");               break;
			case 'r': tmp.append("r:LOCOP_RESTART ");                     break;
			case 'R': tmp.append("R:OPER_RESTART ");                      break;
			case 's': tmp.append("s:SECUNREG (or USE_SERVICES) ");        break;
			case 'S': tmp.append("S:ENABLE_SUMMON ");                     break;
			case 't': tmp.append("t:OPER_REMOTE ");                       break;
			case 'T': tmp.append("T:TRACE_STATS ");                       break;
			case 'u': tmp.append("u:IRCII_KLUDGE (or NO_PREFIX) ");       break;
			case 'U': tmp.append("U:ENABLE_USERS ");                      break;
			case 'V': tmp.append("V:VALLOC ");                            break;
			case 'w': tmp.append("w:NOWRITEALARM ");                      break;
			case 'X': tmp.append("X:UNIXPORT ");                          break;
			case 'Y': tmp.append("Y:USE_SYSLOG ");                        break;
			case 'Z': tmp.append("Z:ZIP_LINKS ");                         break;
			case '8': tmp.append("8:V28PlusOnly ");                       break;
			case '$': tmp.append("$:MIRC_KLUDGE ");                       break;
			default:
				tmp.append(*aux);
				tmp.append(":UNKNOWN_FLAG ");
				break;
		}
		aux++;
	}

	if( g_pOptions->m_bServerInfoToConsole )
		m_pConsole->output(KVI_OUT_INFO, __tr("Server %s: %s"), msg->szPrefix.ptr(), tmp.ptr());
	else
		m_pFrm->activeWindow()->output(KVI_OUT_INFO, __tr("Server %s: %s"), msg->szPrefix.ptr(), tmp.ptr());
}

void KviServerParser::parseUnhandledNumeric(KviIrcMessage *msg)
{
	if( msg->bHasBeenHandled ) return;

	if( g_pEventManager->eventEnabled(KviEvent_OnUnhandledNumeric) ) {
		KviStr tmp(KviStr::Format, "%s %s %s", msg->szPrefix.ptr(), msg->szCommand.ptr(), msg->szParams.ptr());
		if( m_pUserParser->callEvent(KviEvent_OnUnhandledNumeric, m_pConsole, tmp) )
			return;
	}

	KviWindow *pOut = g_pOptions->m_bUnhandledNumericsToConsole ? m_pConsole : m_pFrm->activeWindow();
	pOut->output(KVI_OUT_UNHANDLED, _i18n_("[%s][%s] %s"), msg->szPrefix.ptr(), msg->szCommand.ptr(), msg->szParams.ptr());
}

void KviServerParser::parseUnhandledNumericError(KviIrcMessage *msg)
{
	if( msg->bHasBeenHandled ) return;

	if( g_pEventManager->eventEnabled(KviEvent_OnUnhandledNumeric) ) {
		KviStr tmp(KviStr::Format, "%s %s %s", msg->szPrefix.ptr(), msg->szCommand.ptr(), msg->szParams.ptr());
		if( m_pUserParser->callEvent(KviEvent_OnUnhandledNumeric, m_pConsole, tmp) )
			return;
	}

	KviWindow *pOut = g_pOptions->m_bUnhandledServerErrorsToConsole ? m_pConsole : m_pFrm->activeWindow();
	pOut->output(KVI_OUT_UNHANDLED_ERROR,
		_i18n_("[%s][%s] %s"), msg->szPrefix.ptr(), msg->szCommand.ptr(), msg->szParams.ptr()
	);
}

/**
 * === LITERAL MESSAGES ======================================================
 */
KviIrcCommandEntry KviServerParser::cmdTable[] =
{
	{ "JOIN"   , 4, &KviServerParser::parseJoin    },
	{ "PING"   , 4, &KviServerParser::parsePing    },
	{ "PART"   , 4, &KviServerParser::parsePart    },
	{ "KICK"   , 4, &KviServerParser::parseKick    },
	{ "QUIT"   , 4, &KviServerParser::parseQuit    },
	{ "MODE"   , 4, &KviServerParser::parseMode    },
	{ "NICK"   , 4, &KviServerParser::parseNick    },
	{ "PRIVMSG", 7, &KviServerParser::parsePrivmsg },
	{ "NOTICE" , 6, &KviServerParser::parseNotice  },
	{ "TOPIC"  , 5, &KviServerParser::parseTopic   },
	{ "INVITE" , 6, &KviServerParser::parseInvite  },
	{ "LOCOPS" , 6, &KviServerParser::parseLocops  },
	{ "GLOBOPS", 7, &KviServerParser::parseGlobops },
	{ "WALLOPS", 7, &KviServerParser::parseWallops },
	{ "ERROR"  , 5, &KviServerParser::parseError   },
	{ 0        , 0, 0                              }
};

void KviServerParser::parseLiteralMessage(KviIrcMessage *msg)
{
	for( int i = 0; cmdTable[i].cmdName; i++ ) {
		// msg->szCommand has already been transformed toUpper()
		if( kvi_strEqualCSN(msg->szCommand.ptr(), cmdTable[i].cmdName, cmdTable[i].cmdLen) ) {
			(this->*(cmdTable[i].cmdProc))(msg);
			return;
		}
	}
	parseUnhandledLiteral(msg);
}

void KviServerParser::parseLocops(KviIrcMessage *msg)
{
	KviWindow *outW = g_pOptions->m_bWallopsToConsole ? m_pConsole : m_pFrm->activeWindow();
	if( !msg->bHasBeenHandled )
		outW->output(KVI_OUT_WALLOPS, __tr("LOCOPS from %s: %s"), msg->szPrefix.ptr(), skipToTrailing(msg->szParams.ptr()));
}

void KviServerParser::parseGlobops(KviIrcMessage *msg)
{
	KviWindow *outW = g_pOptions->m_bWallopsToConsole ? m_pConsole : m_pFrm->activeWindow();
	if( !msg->bHasBeenHandled )
		outW->output(KVI_OUT_WALLOPS, __tr("GLOBOPS from %s: %s"), msg->szPrefix.ptr(), skipToTrailing(msg->szParams.ptr()));
}

void KviServerParser::parseWallops(KviIrcMessage *msg)
{
	KviWindow *outW = g_pOptions->m_bWallopsToConsole ? m_pConsole : m_pFrm->activeWindow();
	if( g_pEventManager->eventEnabled(KviEvent_OnWallops) ) {
		KviStr tmp(KviStr::Format, "%s %s", msg->szPrefix.ptr(), msg->szParams.ptr());
		if( m_pUserParser->callEvent(KviEvent_OnWallops, m_pConsole, tmp) )
			return; // Stop the output
	}
	if( !msg->bHasBeenHandled )
		outW->output(KVI_OUT_WALLOPS, __tr("WALLOPS from %s: %s"), msg->szPrefix.ptr(), skipToTrailing(msg->szParams.ptr()));
}

/**
 * [04:16:49] [][ERROR] :Closing Link: newbie[^nobody@a-si6-32.tin.it] (Cya!)
 * [22:26:42] [][ERROR] :Closing Link: newbie[^nobody@a-si4-57.tin.it] ircd.tin.it (Killed (ircd.nl.uu.net ((^nobody@a-si4-57.tin.it)irc2.sci.kun.nl <- (test@in206.pem.cam.ac.uk)ircnet.demon.co.uk)))
 */
void KviServerParser::parseError(KviIrcMessage *msg)
{
	if( msg->szPrefix.isEmpty() )
		msg->szPrefix = m_pFrm->m_global.szCurrentServerHost;
	if( g_pEventManager->eventEnabled(KviEvent_OnError) ) {
		KviStr tmp(KviStr::Format, "%s %s", msg->szPrefix.ptr(), msg->szParams.ptr());
		if( m_pUserParser->callEvent(KviEvent_OnError, m_pConsole, tmp) )
			return; // Stop the output
	}
	if( !msg->bHasBeenHandled )
		m_pConsole->output(KVI_OUT_SERVERROR, __tr("[%s ERROR] %s"), msg->szPrefix.ptr(), skipToTrailing(msg->szParams.ptr()));
}

void KviServerParser::parsePing(KviIrcMessage *msg)
{
	if( msg->szPrefix.isEmpty() )
		msg->szPrefix = m_pFrm->m_global.szCurrentServerHost;
	m_pSocket->sendFmtData("PONG %s", msg->szParams.ptr());
	// Call the event
	if( g_pEventManager->eventEnabled(KviEvent_OnServerPing) ) {
		KviStr tmp(KviStr::Format, "%s %s", msg->szPrefix.ptr(), msg->szParams.ptr());
		if( m_pUserParser->callEvent(KviEvent_OnServerPing, m_pConsole, tmp) )
			return; // Stop the output
	}
	// If we are still here, the event not stopped us; show the ping pong (if enabled)
	if( g_pOptions->m_bShowPingPong && (!msg->bHasBeenHandled) ) {
		m_pConsole->output(KVI_OUT_PING,
			__tr("Ping request from %s [PING %s]... PONG!"), msg->szPrefix.ptr(), msg->szParams.ptr()
		);
	}
	KviStr tmp(KviStr::Format, __tr("Received PING from %s"), msg->szPrefix.ptr());
	m_pStatus->tempText(tmp.ptr(), KVI_TEMP_TEXT_TIMEOUT);
}

/**
 * Check for extended join syntax... used in splits only (AFAIK)
 *
 * [newbie!nobody@irc.pragma.net][JOIN] :#kvirc
 *  :joiner_mask                  JOIN  :<channel>
 *  szPrefix                   szCommand szParams
 *  Nick!User@host JOIN :#channel\x07[o|v]
 */
void KviServerParser::parseJoin(KviIrcMessage *msg)
{
	KviIrcUser joiner(msg->szPrefix.ptr());
	const char *pExt = msg->szParams.ptr();
	char cExtMode = 0;
	while( (*pExt) && (*pExt != 0x07) )
		pExt++;
	if( *pExt ) {
		// Extended join char
		pExt++;
		if( *pExt ) {
			// Get the o or v flag (if any)
			cExtMode = (*pExt);
			msg->szParams.cutRight(2); // Well, here I am assuming that I am at the end of the string!
		} else {
			// \x07\0
			msg->szParams.cutRight(1); // Incorrect syntax from server (AFAIK, 0x07 is not allowed in channel names; or yes?)
		}
	}

	const char *chName = skipToTrailing(msg->szParams.ptr());
	if( !*chName )
		chName = msg->szParams.ptr();
	KviChannel *chan = m_pFrm->findChannel(chName);

	// It must be me, or we are desynced with the server.
	if( isMe(joiner) ) {
		// First, update my mask
		m_pFrm->m_global.szCurrentMaskFromServer = msg->szPrefix;
		m_pFrm->m_global.szLocalHostName         = joiner.host();
		// OK, a new chan window is going to be created
		if( !chan )
			chan = m_pFrm->createChannel(chName);
		else {
			chan->m_bOnChannel    = true;
			chan->m_joinTime      = QTime::currentTime();
			chan->m_iChanSyncMask = KVI_CHANSYNC_NOTINSYNC;
			chan->m_pListBox->partAll();
		}

		KviStr tmp(KviStr::Format, __tr("Joined %s"), chName);
		m_pStatus->tempText(tmp.ptr(), KVI_TEMP_TEXT_TIMEOUT);

		m_pFrm->addRecentChannel(chName);

		// Request the channel mode
		m_pSocket->sendFmtData("MODE %s", chName);
		m_pSocket->sendFmtData("MODE %s b", chName);
		if( g_pOptions->m_bBeVerbose )
			m_pConsole->output(KVI_OUT_INTERNAL, __tr("Requesting channel mode and ban list for %s"), chName);
		if( g_pOptions->m_bRequestBanExceptionListOnJoin && m_pFrm->m_global.bServerSupportsEMode ) {
			m_pSocket->sendFmtData("MODE %s e", chName);
			m_pSocket->sendFmtData("MODE %s I", chName);
			if( g_pOptions->m_bBeVerbose )
				m_pConsole->output(KVI_OUT_INTERNAL, __tr("Requesting ban and invite exception lists for %s"), chName);
		}
		// OK, now join
		chan->join(joiner,
			((cExtMode == 'o') ? 1 : 0), ((cExtMode == 'v') ? 1 : 0),
			((cExtMode == 'h') ? 1 : 0), ((cExtMode == 'u') ? 1 : 0), ((cExtMode == 'q') ? 1 : 0)
		);
		// Now call the event
		if( g_pEventManager->eventEnabled(KviEvent_OnMeJoin) ) {
			KviStr tmp(chName);
			if( m_pUserParser->callEvent(KviEvent_OnMeJoin, chan, tmp) )
				return;
		}
	} else if( chan ) {
		// Another user joined the channel
		chan->join(joiner,
			((cExtMode == 'o') ? 1 : 0), ((cExtMode == 'v') ? 1 : 0),
			((cExtMode == 'h') ? 1 : 0), ((cExtMode == 'u') ? 1 : 0), ((cExtMode == 'q') ? 1 : 0)
		);
		// Call the event
		if( g_pEventManager->eventEnabled(KviEvent_OnJoin) ) {
			KviStr tmp(KviStr::Format, "%s %s %s %s", chName, joiner.nick(), joiner.username(), joiner.host());
			if( m_pUserParser->callEvent(KviEvent_OnJoin, chan, tmp) )
				return;
		}
	} else {
		// Oops, desync with the server.
		KviWindow *pOut = g_pOptions->m_bDesyncMsgsToConsole ? m_pConsole : m_pFrm->activeWindow();
		pOut->output(KVI_OUT_DESYNC,
			__tr("Local desync: received JOIN for channel %s from %s"), chName, msg->szPrefix.ptr());
		return;
	}
	// And say it to the right window (if we are still here)
	if( !msg->bHasBeenHandled ) {
		KviWindow *pOut = g_pOptions->m_bJoinMsgsToConsole ? m_pConsole : ((KviWindow *) chan);
		if( cExtMode ) {
			// Have an implicit mode change (return from split?)
			pOut->output(KVI_OUT_JOIN,
				__tr("%s [%s] has joined %s [implicit +%c umode change]"),
				joiner.nick(), skipToUsername(msg->szPrefix.ptr()), chName, cExtMode
			);
		} else {
			pOut->output(KVI_OUT_JOIN,
				__tr("%s [%s] has joined %s"), joiner.nick(), skipToUsername(msg->szPrefix.ptr()), chName
			);
		}
	}
}

/**
 * [newbie!nobody@irc.pragma.net][PART] #kvirc :part_message
 */
void KviServerParser::parsePart(KviIrcMessage *msg)
{
	KviIrcUser parter(msg->szPrefix.ptr());

	// Why not? He may be on other channels
	m_pFrm->m_pUserList->updateUser(parter);

	KviStr channel;
	const char *aux  = kvi_extractToken(channel, msg->szParams.ptr(), ' ');
	KviChannel *chan = m_pFrm->findChannel(channel.ptr());

	aux = skipToTrailing(aux);
	if( !chan ) {
		// Oops, desync with the server.
		KviWindow *pOut = g_pOptions->m_bDesyncMsgsToConsole ? m_pConsole : m_pFrm->activeWindow();
		if( isMe(parter) ) {
			pOut->output(KVI_OUT_DESYNC,
				__tr("Local desync: seems that you have left the channel %s: [%s%c]"), channel.ptr(), aux, KVI_TEXT_RESET
			);
		} else {
			pOut->output(KVI_OUT_DESYNC,
				__tr("Local desync: received PART for channel %s from %s: [%s%c]"),
				channel.ptr(), msg->szPrefix.ptr(), aux, KVI_TEXT_RESET
			);
		}
		return;
	}

	bool bStop = false;
	if( isMe(parter) ) {
		// Call the event
		if( g_pEventManager->eventEnabled(KviEvent_OnMePart) ) {
			KviStr tmp(KviStr::Format, "%s %s", channel.ptr(), aux);
			bStop = m_pUserParser->callEvent(KviEvent_OnMePart, chan, tmp);
		}
		// I have left a channel, destroy the window
		m_pFrm->closeWindow(chan);
		KviStr tmp(KviStr::Format, __tr("Parted from %s"), channel.ptr());
		m_pStatus->tempText(tmp.ptr(), KVI_TEMP_TEXT_TIMEOUT);
		if( bStop )
			return;
		// And output if needed
		if( g_pOptions->m_bShowOwnParts && (!msg->bHasBeenHandled) ) {
			if( *aux ) {
				m_pConsole->output(KVI_OUT_PART,
					__tr("You have left channel %s: [%s%c]"), channel.ptr(), aux, KVI_TEXT_RESET
				);
			} else m_pConsole->output(KVI_OUT_PART, __tr("You have left channel %s"), channel.ptr());
		}
	} else {
		// A user left the channel, just part.
		// Call the event
		if( g_pEventManager->eventEnabled(KviEvent_OnPart) ) {
			KviStr tmp(KviStr::Format,
				"%s %s %s %s %s", channel.ptr(), parter.nick(), parter.username(), parter.host(), aux
			);
			bStop = m_pUserParser->callEvent(KviEvent_OnPart, chan, tmp);
		}
		chan->part(parter);
		if( bStop )
			return;
		// And output if needed
		if( !msg->bHasBeenHandled ) {
			KviWindow *pOut = g_pOptions->m_bPartMsgsToConsole ? m_pConsole : ((KviWindow *) chan);
			if( *aux ) {
				pOut->output(KVI_OUT_PART,
					__tr("%s [%s] has left %s: [%s%c]"),
					parter.nick(), skipToUsername(msg->szPrefix.ptr()), channel.ptr(), aux, KVI_TEXT_RESET
				);
			} else {
				pOut->output(KVI_OUT_PART,
					__tr("%s [%s] has left %s"), parter.nick(), skipToUsername(msg->szPrefix.ptr()), channel.ptr()
				);
			}
		}
	}
}

/**
 * [xxx!xxxx@x.unix.x][KICK] #italia yyyyyy :Bitch-X BaBy!
 */
void KviServerParser::parseKick(KviIrcMessage *msg)
{
	KviIrcUser kicker(msg->szPrefix.ptr());

	m_pFrm->m_pUserList->updateUser(kicker);

	KviStr channel, kicked;
	const char *aux = kvi_extractToken(channel, msg->szParams.ptr(), ' ');
	aux = kvi_extractToken(kicked, aux, ' ');
	aux = skipToTrailing(aux);

	KviChannel *chan = m_pFrm->findChannel(channel.ptr());
	if( !chan ) {
		// Oops, desync with the server.
		KviWindow *pOut = g_pOptions->m_bDesyncMsgsToConsole ? m_pConsole : m_pFrm->activeWindow();
		if( isMe(kicked.ptr()) ) {
			pOut->output(KVI_OUT_DESYNC,
				__tr("Local desync: seems that you have been kicked from %s by %s [%s]: [%s%c]"),
				channel.ptr(), kicker.nick(), skipToUsername(msg->szPrefix.ptr()), aux, KVI_TEXT_RESET
			);
		} else {
			pOut->output(KVI_OUT_DESYNC,
				__tr("Local desync: received KICK %s on channel %s from %s [%s]: [%s%c]"),
				kicked.ptr(), channel.ptr(), kicker.nick(), skipToUsername(msg->szPrefix.ptr()), aux, KVI_TEXT_RESET
			);
		}
		return;
	}

	if( isMe(kicked.ptr()) ) { // I have left a channel, destroy the window
		// TODO: When kicked from a channel do not destroy the userlist! (So do not request it again from the server)
		bool bDumped = false;
		KviStr fName;
		if( g_pOptions->m_bDumpChannelOnKick ) {
			// Dump the log to a file
			g_pApp->getChannelDumpLogFileName(fName);
			bDumped = chan->kickDumpChanStatus(
				fName.ptr(), kicked.ptr(), kicker.nick(), skipToUsername(msg->szPrefix.ptr()), aux
			);
		}
		// Close the window
		m_pFrm->closeWindow(chan);
		// Call the event
		bool bHalt = false;
		if( g_pEventManager->eventEnabled(KviEvent_OnMeKick) ) {
			KviStr tmp(KviStr::Format,
				"%s %s %s %s %s", channel.ptr(), kicker.nick(), kicker.username(), kicker.host(), aux
			);
			bHalt = m_pUserParser->callEvent(KviEvent_OnMeKick, m_pConsole, tmp);
		}
		// And say that we have been kicked
		if( !(bHalt || msg->bHasBeenHandled) ) {
			m_pConsole->output(KVI_OUT_KICK,
				__tr("You have been kicked from %s by %s [%s]: [%s%c]"),
				channel.ptr(), kicker.nick(), skipToUsername(msg->szPrefix.ptr()), aux, KVI_TEXT_RESET
			);
		}
		if( g_pOptions->m_bDumpChannelOnKick ) {
			if( bDumped )
				m_pConsole->output(KVI_OUT_INTERNAL, __tr("Channel kick log dumped to file %s"), fName.ptr());
			else
				m_pConsole->output(KVI_OUT_ERROR, __tr("Could not write channel kick log to file %s"), fName.ptr());
		}
		// Now set the status text
		KviStr tmp(KviStr::Format, __tr("Kicked from %s by %s: %s"), channel.ptr(), kicker.nick(), aux);
		m_pStatus->tempText(tmp.ptr(), KVI_TEMP_TEXT_TIMEOUT);
		// And auto-rejoin if requested
		if( g_pOptions->m_bAutoRejoinOnKick )
			m_pSocket->sendFmtData("JOIN %s", channel.ptr());
	} else {
		// A user was kicked
		chan->part(kicked.ptr());
		if( g_pEventManager->eventEnabled(KviEvent_OnKick) ) {
			KviStr tmp(KviStr::Format,
				"%s %s %s %s %s %s", channel.ptr(), kicked.ptr(), kicker.nick(), kicker.username(), kicker.host(), aux
			);
			if( m_pUserParser->callEvent(KviEvent_OnKick, chan, tmp) )
				return;
		}
		if( !msg->bHasBeenHandled ) {
			KviWindow *pOut = g_pOptions->m_bKickMsgsToConsole ? m_pConsole : ((KviWindow *) chan);
			pOut->output(KVI_OUT_KICK,
				__tr("%s has been kicked from %s by %s [%s]: [%s%c]"),
				kicked.ptr(), channel.ptr(), kicker.nick(), skipToUsername(msg->szPrefix.ptr()), aux, KVI_TEXT_RESET
			);
		}
	}
}

/**
 * [pragma!kvirc@irc.pragma.net][QUIT] :Read error: 0 (Success)
 * [Solitario!xnzct@x.x.x][QUIT] :Ping timeout
 * [Giulio!~tor@000.000.000.000][QUIT] :7th Sphere v3.0  1997 7th Sphere...
 * [pljdnh!bubu@foo.it][QUIT] :Excess Flood
 * []-[ermeS!Corrupt0r@foo.it][QUIT] :Connection timed out
 * [Evai!ouh@foobar.tin.it][QUIT] :Connection reset by peer
 * SPLITS
 * [StAr^DuSt!stardash@foo.foo.foo.com][QUIT] :*.webbernet.net irc.anet.com
 * [NANDO^!~Odissea@foo.dowco.com][QUIT] :*.webbernet.net irc.anet.com
 * [[Astral]!kaos@dontknow.com][QUIT] :*.webbernet.net irc.anet.com
 * [BiriKin0!~Alex@foo.bigbadbug.com][QUIT] :*.webbernet.net irc.anet.com
 */
void KviServerParser::parseQuit(KviIrcMessage *msg)
{
	KviIrcUser quitter(msg->szPrefix.ptr());
	// Now determine if it was a split
	const char *aux = skipToTrailing(msg->szParams.ptr());
	bool bWasSplit = false;
	// Determine if signoff string matches "%.% %.%", and only one space (from eggdrop code)
	char *p = strchr(aux, ' ');
	if( p && (p == strrchr(aux, ' ')) ) {
		// One space detected: go ahead
		char *z1, *z2;
		*p = 0;
		z1 = strchr(p + 1, '.');
		z2 = strchr(aux, '.');
		if( z1 && z2 && (*(z1 + 1) != 0) && (z1 - 1 != p) && (z2 + 1 != p) && (z2 != aux) ) {
			// Server split, or else it looked like it anyway
			*p = ' ';
			bWasSplit = true;
			KviStr tmp(KviStr::Format, __tr("Netsplit detected: %s"), aux);
			m_pStatus->tempText(tmp.ptr(), KVI_TEMP_TEXT_TIMEOUT);
			if( !kvi_strEqualCI(m_szLastNetsplit.ptr(), aux) ) {
				m_szLastNetsplit = aux;
				if( m_iNetsplitTimerId != 0 )
					killTimer(m_iNetsplitTimerId);
				m_iNetsplitTimerId = startTimer(5000);
				if( !msg->bHasBeenHandled ) {
					KviWindow *pOut = g_pOptions->m_bSplitWarningsToConsole ? m_pConsole : m_pFrm->activeWindow();
					pOut->outputNoFmt(KVI_OUT_SPLIT, tmp.ptr());
				}
			}
		} else *p = ' ';
	}

	if( g_pEventManager->eventEnabled(KviEvent_OnQuit) ) {
		KviStr tmp(KviStr::Format,
			"%s %s %s %s %s", quitter.nick(), quitter.username(), quitter.host(), bWasSplit ? "1" : "0", aux
		);
		msg->bHasBeenHandled = m_pUserParser->callEvent(KviEvent_OnQuit, m_pConsole, tmp);
	}

	// Time to handle it
	for( KviWindow *w = m_pFrm->m_pWinList->first(); w; w = m_pFrm->m_pWinList->next() ) {
		if( w->type() == KVI_WND_TYPE_CHANNEL ) {
			if( ((KviChannel *) w)->isOnChannel(quitter) ) {
				((KviChannel *) w)->part(quitter);
				// TODO: Netsplit messages to console?
				if( !(g_pOptions->m_bQuitMsgsToConsole || msg->bHasBeenHandled) ) {
					if( bWasSplit ) {
						w->output(KVI_OUT_QUITSPLIT,
							__tr("%s [%s] has quit IRC: NETSPLIT: [%s%c]"),
							quitter.nick(), skipToUsername(msg->szPrefix.ptr()), aux, KVI_TEXT_RESET
						);
					} else {
						w->output(KVI_OUT_QUIT,
							__tr("%s [%s] has quit IRC: [%s%c]"),
							quitter.nick(), skipToUsername(msg->szPrefix.ptr()), aux, KVI_TEXT_RESET
						);
					}
				}
			}
		} else if( w->type() == KVI_WND_TYPE_QUERY ) {
			if( kvi_strEqualCI(w->caption(), quitter.nick()) ) {
				if( !msg->bHasBeenHandled ) {
					if( bWasSplit ) {
						w->output(KVI_OUT_QUITSPLIT,
							__tr("%s [%s] has quit IRC: NETSPLIT: [%s%c]"),
							quitter.nick(), skipToUsername(msg->szPrefix.ptr()), aux, KVI_TEXT_RESET
						);
					} else {
						w->output(KVI_OUT_QUIT,
							__tr("%s [%s] has quit IRC: [%s%c]"),
							quitter.nick(), skipToUsername(msg->szPrefix.ptr()), aux, KVI_TEXT_RESET
						);
					}
					w->output(KVI_OUT_INTERNAL, __tr("Warning: remote end of this query is no longer online"));
				}
			}
		}
	}

	// If we have to say it to the console, do it only once.
	if( g_pOptions->m_bQuitMsgsToConsole && (!msg->bHasBeenHandled) ) {
		if( bWasSplit ) {
			m_pConsole->output(KVI_OUT_QUITSPLIT,
				__tr("%s [%s] has quit IRC: NETSPLIT: [%s%c]"),
				quitter.nick(), skipToUsername(msg->szPrefix.ptr()), aux, KVI_TEXT_RESET
			);
		} else {
			m_pConsole->output(KVI_OUT_QUIT,
				__tr("%s [%s] has quit IRC: [%s%c]"),
				quitter.nick(), skipToUsername(msg->szPrefix.ptr()), aux, KVI_TEXT_RESET
			);
		}
	}
}

/**
 * [Pragtest!^nobody@a-si6-7.tin.it][NICK] :newbie
 * [kaos`away!entropy@199.217.176.44][NICK] :kaos`work
 */
void KviServerParser::parseNick(KviIrcMessage *msg)
{
	KviIrcUser nicker(msg->szPrefix.ptr());
	const char *newNick = skipToTrailing(msg->szParams.ptr());

	int event_ix = KviEvent_OnNick;
	KviStr channels;
	KviStr eventStr(KviStr::Format, "%s %s@%s %s", nicker.nick(), nicker.username(), nicker.host(), newNick);

	if( isMe(nicker) ) {
		event_ix = KviEvent_OnMeNick;
		m_pFrm->m_global.szCurrentMaskFromServer = newNick;
		m_pFrm->m_global.szCurrentNick           = newNick;
		m_pFrm->m_global.szCurrentMaskFromServer.append('!');
		m_pFrm->m_global.szCurrentMaskFromServer.append(skipToUsername(msg->szPrefix.ptr()));
		m_pFrm->updateCaption();
		KviStr tmp(KviStr::Format, __tr("Changed nick to %s"), newNick);
		m_pFrm->addRecentNickname(m_pFrm->m_global.szCurrentNick);
		m_pStatus->tempText(tmp.ptr(), KVI_TEMP_TEXT_TIMEOUT);
		if( g_pOptions->m_bShowOwnNickChanges && (!msg->bHasBeenHandled) ) {
			m_pConsole->output(KVI_OUT_NICK, __tr("You have changed your nickname to %s"), newNick);
		}
	}

	QPtrList<KviWindow> tlist;
	for( KviWindow *w = m_pFrm->m_pWinList->first(); w; w = m_pFrm->m_pWinList->next() ) {
		if( w->type() == KVI_WND_TYPE_CHANNEL ) {
			if( ((KviChannel *) w)->nickChange(nicker, newNick) ) {
				if( channels.hasData() )
					channels += ',';
				channels += w->caption();
				tlist.append(w);
			}
		} else if( (w->type() == KVI_WND_TYPE_QUERY) ) {
			if( kvi_strEqualCI(w->caption(), nicker.nick()) ) {
				if( channels.hasData() )
					channels += ',';
				channels += w->caption();
				w->setWindowCaption(newNick);
				tlist.append(w);
			}
		}
	}

	if( g_pEventManager->eventEnabled(event_ix) ) {
		eventStr += ' ' + channels;
		if( m_pUserParser->callEvent(event_ix, m_pConsole, eventStr) )
			return;
	}

	// If we have to say it to the console, do it only once.
	if( g_pOptions->m_bNickMsgsToConsole && (!msg->bHasBeenHandled) ) {
		m_pConsole->output(KVI_OUT_NICK,
			__tr("%s [%s] is now known as %s"), nicker.nick(), skipToUsername(msg->szPrefix.ptr()), newNick
		);
		return;
	}

	if( !msg->bHasBeenHandled ) {
		for( KviWindow *w = tlist.first(); w; w = tlist.next() ) {
			if( !g_pOptions->m_bNickMsgsToConsole ) {
				w->output(KVI_OUT_NICK,
					__tr("%s [%s] is now known as %s"), nicker.nick(), skipToUsername(msg->szPrefix.ptr()), newNick
				);
			}
			if( w->type() == KVI_WND_TYPE_QUERY ) {
				w->output(KVI_OUT_INTERNAL, __tr("Warning: this query has a new remote end"));
			}
		}
	}
}

/**
 * [x!~x@zzz.zz.zzz][MODE] #italia +o yyy
 * [goo!^goo@goo.goo.goo.ret][MODE] #italia -bb *!*@*.chem.uu.nl *!e@a-lt*.tin.it
 * [zzzz!p@o.it][MODE] #italia +b *!*@*.chem.uu.nl
 * [cccc!^r@222222.net][MODE] #italia +o StAr^DuSt
 * [rodata!jmp@mov.eax.eax][MODE] #italia -b *!*@*.chem.uu.nl
 */
void KviServerParser::parseMode(KviIrcMessage *msg)
{
	KviIrcUser source(msg->szPrefix.ptr());
	m_pFrm->m_pUserList->updateUser(source);

	KviStr target;
	const char *aux = kvi_extractToken(target, msg->szParams.ptr(), ' ');
	__range_valid(*aux != ' ');

	if( *aux == ':' ) {
		aux++;
		while( *aux == ' ' )
			aux++;
	}

	if( !(*aux) ) {
		debug("WARNING: malformed MODE message");
		return;
	}
	__range_valid((*aux == '+') || (*aux == '-'));

	if( isMe(target.ptr()) ) {
		// My UMode
		m_pFrm->userModeChange(aux);
		if( !msg->bHasBeenHandled ) {
			KviWindow *pOut = g_pOptions->m_bUserModeChangesToConsole ? m_pConsole : m_pFrm->activeWindow();
			pOut->output(KVI_OUT_UMODE, __tr("%s sets mode %s %s"), source.nick(), target.ptr(), aux);
		}
		return;
	}

	KviChannel *chan = m_pFrm->findChannel(target.ptr());
	if( !chan ) {
		// Oops, desync with the server.
		KviWindow *pOut = g_pOptions->m_bDesyncMsgsToConsole ? m_pConsole : m_pFrm->activeWindow();
		if( isMe(source) ) {
			pOut->output(KVI_OUT_DESYNC,
				__tr("Local desync: seems that you have set the mode %s %s"), target.ptr(), aux
			);
		} else {
			pOut->output(KVI_OUT_DESYNC,
				__tr("Local desync: %s [%s] sets mode %s %s"),
				source.nick(), skipToUsername(msg->szPrefix.ptr()), target.ptr(), aux
			);
		}
		return;
	}

	KviStr modeString(aux);
	KviStr modes;
	aux = kvi_extractToken(modes, aux, ' ');
	__range_valid(*aux != ' ');
	if( modes.isEmpty() ) {
		debug("WARNING: malformed MODE message");
		return;
	}

	bool bAdd = true;
	const char *mPtr = modes.ptr();

	KviStr tmp;
	while( *mPtr ) {
		switch( *mPtr ) {
			case '+': bAdd = true;  break;
			case '-': bAdd = false; break;
			case 'q': // Fall-through
			case 'o': // Fall-through
			case 'h': // Fall-through
			case 'v': // Fall-through
			case 'u': // Fall-through
			case 'b': // Fall-through
			case 'e': // Fall-through
			case 'I': // Fall-through
			case 'a':
				aux = kvi_extractToken(tmp, aux, ' ');
				if( !tmp.isEmpty() )
					handleChanUserMode(msg, chan, bAdd, *mPtr, source.nick(), msg->szPrefix.ptr(), tmp.ptr());
				else
					handleChanMode(msg, chan, bAdd, *mPtr, source.nick(), skipToUsername(msg->szPrefix.ptr()), 0);
				break;
			default:
				aux = kvi_extractToken(tmp, aux, ' ');
				if( !tmp.isEmpty() )
					handleChanMode(msg, chan, bAdd, *mPtr, source.nick(), skipToUsername(msg->szPrefix.ptr()), tmp.ptr());
				else
					handleChanMode(msg, chan, bAdd, *mPtr, source.nick(), skipToUsername(msg->szPrefix.ptr()), 0);
			break;
		}
		mPtr++;
	}

	if( !msg->bHasBeenHandled && g_pOptions->m_bSingleMessageForChannelModes ) {
		if( g_pOptions->m_bChannelModeChangesToConsole ) {
			m_pConsole->output(KVI_OUT_CHANMODE,
				__tr("%s [%s] sets mode %s on %s"),
				source.nick(), skipToUsername(msg->szPrefix.ptr()), modeString.ptr(), chan->caption()
			);
		} else {
			 chan->output(KVI_OUT_CHANMODE,
			 	__tr("%s [%s] sets mode %s"), source.nick(), skipToUsername(msg->szPrefix.ptr()), modeString.ptr()
			);
		}
	}
	__range_valid(*aux == '\0');
}

void KviServerParser::handleChanUserMode(
	KviIrcMessage *msg, KviChannel *chan, bool bAdd, char mode,
	const char *src_nick, const char *src_mask, const char *dst_nick)
{
	__range_valid(chan);
	__range_valid(src_nick);
	__range_valid(src_mask);
	int type = KVI_OUT_CHANMODE;
	KviStr tmp;

	switch( mode ) {
		case 'q':
			chan->owner(dst_nick, bAdd);
			type = bAdd ? KVI_OUT_OWNER : KVI_OUT_DEOWNER;
			if( isMe(dst_nick) ) {
				chan->setModeChangeEnabled(bAdd);
				chan->setTopicChangeEnabled(bAdd || (!(chan->isModeActive('t'))));
				if( !callChanUserModeEvent(
					chan, (bAdd ? KviEvent_OnMeOwner : KviEvent_OnMeDeowner), src_nick, src_mask, dst_nick, true)
				) {
					return; // Stop output and statusbar
				}
				tmp.sprintf(__tr("%s owner status on %s"), bAdd ? __tr("Gained") : __tr("Lost"), chan->caption());
			} else if( !callChanUserModeEvent(
				chan, (bAdd ? KviEvent_OnOwner : KviEvent_OnDeowner), src_nick, src_mask, dst_nick)
			) {
				return; // Stop output and statusbar
			}
			break;
		case 'o':
			chan->op(dst_nick, bAdd);
			type = bAdd ? KVI_OUT_OP : KVI_OUT_DEOP;
			if( isMe(dst_nick) ) {
				chan->setModeChangeEnabled(bAdd);
				chan->setTopicChangeEnabled(bAdd || (!(chan->isModeActive('t'))));
				if( !callChanUserModeEvent(
					chan, (bAdd ? KviEvent_OnMeOp : KviEvent_OnMeDeop), src_nick, src_mask, dst_nick, true)
				) {
					return; // Stop output and statusbar
				}
				tmp.sprintf(__tr("%s operator status on %s"), bAdd ? __tr("Gained") : __tr("Lost"), chan->caption());
			} else if( !callChanUserModeEvent(
				chan, (bAdd ? KviEvent_OnOp : KviEvent_OnDeop), src_nick, src_mask, dst_nick)
			) {
				return; // Stop output and statusbar
			}
			break;
		case 'h':
			chan->halfop(dst_nick, bAdd);
			type = bAdd ? KVI_OUT_HALFOP : KVI_OUT_DEHALFOP;
			if( isMe(dst_nick) ) {
				if( !callChanUserModeEvent(
					chan, (bAdd ? KviEvent_OnMeHalfop : KviEvent_OnMeDehalfop), src_nick, src_mask, dst_nick, true)
				) {
					return; // Stop output and statusbar
				}
				tmp.sprintf(__tr("%s halfop status on %s"), bAdd ? __tr("Gained") : __tr("Lost"), chan->caption());
			} else if( !callChanUserModeEvent(
				chan, (bAdd ? KviEvent_OnHalfop : KviEvent_OnDehalfop), src_nick, src_mask, dst_nick)
			) {
				return; // Stop output and statusbar
			}
			break;
		case 'v':
			chan->voice(dst_nick, bAdd);
			type = bAdd ? KVI_OUT_VOICE : KVI_OUT_DEVOICE;
			if( isMe(dst_nick) ) {
				if( !callChanUserModeEvent(
					chan, (bAdd ? KviEvent_OnMeVoice : KviEvent_OnMeDevoice), src_nick, src_mask, dst_nick, true)
				) {
					return; // Stop output and statusbar
				}
				tmp.sprintf(__tr("%s voice status on %s"), bAdd ? __tr("Gained") : __tr("Lost"), chan->caption());
			} else if( !callChanUserModeEvent(
				chan, (bAdd ? KviEvent_OnVoice : KviEvent_OnDevoice), src_nick, src_mask, dst_nick)
			) {
				return; // Stop output and statusbar
			}
			break;
		case 'u':
			chan->userop(dst_nick, bAdd);
			type = bAdd ? KVI_OUT_USEROP : KVI_OUT_DEUSEROP;
			if( isMe(dst_nick) ) {
				if( !callChanUserModeEvent(
					chan, (bAdd ? KviEvent_OnMeUserop : KviEvent_OnMeDeuserop), src_nick, src_mask, dst_nick, true)
				) {
					return; // Stop output and statusbar
				}
				tmp.sprintf(__tr("%s userop status on %s"), bAdd ? __tr("Gained") : __tr("Lost"), chan->caption());
			} else if( !callChanUserModeEvent(
				chan, (bAdd ? KviEvent_OnUserop : KviEvent_OnDeuserop), src_nick, src_mask, dst_nick)
			) {
				return; // Stop output and statusbar
			}
			break;
		case 'b': {
			KviIrcUser u(m_pFrm->m_global.szCurrentMaskFromServer.ptr());
			chan->banMask(dst_nick, src_mask, 0, bAdd);
			type = bAdd ? KVI_OUT_BAN : KVI_OUT_UNBAN;
			if( u.matches(dst_nick) ) { // It is me
				if( !callChanUserModeEvent(
					chan, (bAdd ? KviEvent_OnMeBan : KviEvent_OnMeUnban), src_nick, src_mask, dst_nick)
				) {
					return; // Stop output and statusbar
				}
			} else if( !callChanUserModeEvent(
				chan, (bAdd ? KviEvent_OnBan : KviEvent_OnUnban), src_nick, src_mask, dst_nick)
			) {
				return; // Stop output and statusbar
			}
			break;
		}
		case 'e': {
			KviIrcUser u(m_pFrm->m_global.szCurrentMaskFromServer.ptr());
			chan->banExceptionMask(dst_nick, src_mask, 0, bAdd);
			type = bAdd ? KVI_OUT_EXCEPT : KVI_OUT_UNEXCEPT;
			if( u.matches(dst_nick) ) { // It is me
				if( !callChanUserModeEvent(
					chan, (bAdd ? KviEvent_OnMeBanException : KviEvent_OnMeBanExceptionRemove),
					src_nick, src_mask, dst_nick)
				) {
					return; // Stop output and statusbar
				}
			} else if(!callChanUserModeEvent(
				chan, (bAdd ? KviEvent_OnBanException : KviEvent_OnBanExceptionRemove),
				src_nick, src_mask, dst_nick)
			) {
				return; // Stop output and statusbar
			}
		break;
		}
		case 'I': {
			KviIrcUser u(m_pFrm->m_global.szCurrentMaskFromServer.ptr());
			chan->inviteExceptionMask(dst_nick, src_mask, 0, bAdd);
			type = bAdd ? KVI_OUT_EXCEPT : KVI_OUT_UNEXCEPT;
			if( u.matches(dst_nick) ) { // It is me
				if( !callChanUserModeEvent(
					chan, (bAdd ? KviEvent_OnMeInviteException : KviEvent_OnMeInviteExceptionRemove),
					src_nick, src_mask, dst_nick)
				) {
					return; // Stop output and statusbar
				}
			} else if( !callChanUserModeEvent(
				chan, (bAdd ? KviEvent_OnInviteException : KviEvent_OnInviteExceptionRemove),
				src_nick, src_mask, dst_nick)
			) {
				return; // Stop output and statusbar
			}
			break;
		}
		default:
			// Never here
			break;
	}

	if( tmp.hasData() )
		m_pStatus->tempText(tmp.ptr(), KVI_TEMP_TEXT_TIMEOUT);

	if( !msg->bHasBeenHandled && !g_pOptions->m_bSingleMessageForChannelModes ) {
		if( g_pOptions->m_bChannelModeChangesToConsole ) {
			m_pConsole->output(type,
				__tr("%s [%s] sets mode %c%c %s on %s"),
				src_nick, skipToUsername(src_mask), bAdd ? '+' : '-', mode, dst_nick, chan->caption()
			);
		} else {
			chan->output(type,
				__tr("%s [%s] sets mode %c%c %s"), src_nick, skipToUsername(src_mask), bAdd ? '+':'-', mode, dst_nick
			);
		}
	}
}

bool KviServerParser::callChanUserModeEvent(
	KviChannel *chan, int event, const char *src_nick, const char *src_mask,
	const char *dst_nick, bool isme, char mode, bool bAdd, const char *param)
{
	if( g_pEventManager->eventEnabled(event) ) {
		if( !dst_nick && mode ) {
			KviStr eventparms(KviStr::Format,
				"%s %s %s %c%c %s", chan->caption(), src_nick, src_mask, bAdd ? '+' : '-', mode, param
			);
			if( m_pUserParser->callEvent(
				event, (g_pOptions->m_bChannelModeChangesToConsole ? m_pConsole : ((KviWindow *) chan)), eventparms)
			) {
				return false;
			}
		} else if( !isme ) {
			KviStr eventparms(KviStr::Format,
				"%s %s %s %s", chan->caption(), src_nick, src_mask, dst_nick
			);
			if( m_pUserParser->callEvent(
				event, (g_pOptions->m_bChannelModeChangesToConsole ? m_pConsole : ((KviWindow *) chan)), eventparms)
			) {
				return false;
			}
		} else {
			KviStr eventparms(KviStr::Format, "%s %s %s", chan->caption(), src_nick, src_mask);
			if( m_pUserParser->callEvent(
				event, (g_pOptions->m_bChannelModeChangesToConsole ? m_pConsole : ((KviWindow *) chan)), eventparms)
			) {
				return false;
			}
		}
	}
	return true;
}

void KviServerParser::handleChanMode(
	KviIrcMessage *msg, KviChannel *chan, bool bAdd, char mode,
	const char *src_nick, const char *src_mask, const char *param)
{
	__range_valid(chan);
	__range_valid(src_nick);
	__range_valid(src_mask);

	chan->setModeFlag(mode, bAdd, param);

	if( g_pEventManager->eventEnabled(KviEvent_OnChannelMode) ) {
		// Call the event and eventually stop the output
		KviStr eventparms(KviStr::Format,
			"%s %s %s %c%c", chan->caption(), src_nick, src_mask, bAdd ? '+' : '-', mode
		);
		if( param ) {
			eventparms.append(' ');
			eventparms.append(param);
		}
		if( m_pUserParser->callEvent(KviEvent_OnChannelMode, chan, eventparms) )
			return;
	}

	if( !msg->bHasBeenHandled && !g_pOptions->m_bSingleMessageForChannelModes ) {
		KviWindow *pOut = g_pOptions->m_bChannelModeChangesToConsole ? m_pConsole : ((KviWindow *) chan);
		switch( mode ) {
			case 'k': // Fall-through
			case 'l':
				pOut->output(
					(mode == 'k') ? KVI_OUT_KEY : KVI_OUT_LIMIT,
					__tr("%s [%s] sets mode %c%c for %s"), src_nick, src_mask, bAdd ? '+' : '-', mode, chan->caption()
				);
				break;
			default:
				if( param ) {
					pOut->output(KVI_OUT_CHANMODE,
						__tr("%s [%s] sets mode %c%c %s for %s"), src_nick, src_mask,
						bAdd ? '+' : '-', mode, param, chan->caption()
					);
				} else {
					pOut->output(KVI_OUT_CHANMODE,
						__tr("%s [%s] sets mode %c%c for %s"), src_nick, src_mask, bAdd ? '+' : '-', mode, chan->caption()
					);
				}
			break;
		}
	}
}

/**
 * ================================================================================================
 * PRIVMSG
 * ================================================================================================
 *
 * [lains!Socmel@foo][PRIVMSG] #italia : ACTION e' Away.[Auto away: idle  dalle 05:07am ]
 *
 */
void KviServerParser::parsePrivmsg(KviIrcMessage *msg)
{
	if( msg->szPrefix.isEmpty() )
		msg->szPrefix = m_pFrm->m_global.szCurrentServerHost;

	KviIrcUser talker(msg->szPrefix.ptr());
	m_pFrm->m_pUserList->updateUser(talker);

	if( g_pOptions->m_bEnableIgnoreOnPrivmsg ) {
		KviRegisteredUser *u = g_pOptions->m_pRegUsersDb->findIgnoredUserByMask(msg->szPrefix.ptr());
		if( u ) {
			if( g_pOptions->m_bVerboseIgnore ) {
				m_pConsole->output(KVI_OUT_INTERNAL,
					__tr("Ignoring PRIVMSG from %s: %s"), msg->szPrefix.ptr(), msg->szParams.ptr()
				);
			}
			return;
		}
	}

	KviStr target;
	const char *aux = kvi_extractToken(target, msg->szParams.ptr(), ' ');
	aux = skipToTrailing(aux);

	if( *aux == 0x01 ) {
		const char *aux2 = aux;
		while( *aux2 )
			aux2++;
		aux2--;
		if( (aux != aux2) && (*aux2 == 0x01) ) {
			aux++;
			msg->szParams.cutRight(1);
			parseCTCPPrivmsg(talker, target.ptr(), aux);
			return;
		}
	}

	// Normal PRIVMSG
	if( isMe(target.ptr()) ) {
		if( g_pOptions->m_bListenToMultimediaFileRequests ) {
			if( *aux == '!' ) {
				const char *tmp = aux;
				tmp++;
				if( kvi_strEqualCI(tmp, m_pFrm->m_global.szCurrentNick.ptr()) ) {
					// A multimedia file request?
					tmp += m_pFrm->m_global.szCurrentNick.len();
					if( ((*tmp) == ' ') || ((*tmp) == '\t') ) {
						while( ((*tmp) == ' ') || ((*tmp) == '\t') )
							tmp++;
						if( *tmp ) {
							KviStr file = tmp;
							KviStr filePath;
							m_pFrm->findMultimediaFileOffer(filePath, file);
							if( filePath.hasData() ) {
								m_pFrm->activeWindow()->output(KVI_OUT_INTERNAL,
									__tr("%s requests previously offered file %s: sending (%s)"),
									talker.nick(), file.ptr(), filePath.ptr()
								);
								QString cmd;
								cmd.sprintf("DCC SEND %s %s", talker.nick(), filePath.ptr());
								m_pFrm->m_pUserParser->parseUserCommand(cmd, m_pConsole);
								return;
							} else {
								m_pFrm->activeWindow()->output(KVI_OUT_INTERNAL,
									__tr("%s requests file %s: no such file was offered, ignoring"),
									talker.nick(), file.ptr()
								);
								return;
							}
						}
					}
				}
			}
		}

		KviQuery *query = m_pFrm->findQuery(talker.nick());
		if( !query ) {
			// New query requested
			if( g_pOptions->m_bUseAntiQueryFlood ) {
				if( g_pOptions->m_iMaxTotalQueries > -1 ) {
					int total = 0;
					for( KviWindow *w = m_pFrm->m_pWinList->first(); w; w = m_pFrm->m_pWinList->next() ) {
						if( w->type() == KVI_WND_TYPE_QUERY )
							total++;
					}
					if( g_pOptions->m_iMaxTotalQueries <= total ) {
						// Flood!
						KviWindow *pOut = g_pOptions->m_bQueryFloodWarningsToConsole ? m_pConsole : m_pFrm->activeWindow();
						if( !msg->bHasBeenHandled ) {
							pOut->output(KVI_OUT_FLOOD,
								__tr("Clone flood detected: maximum number of query windows exceeded by %s [%s]: %s"),
								talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux
							);
						}
						return;
					}
				}
				if( g_pOptions->m_iMaxQueryCount > 0 ) {
					// At least one query allowed
					m_iQueryCount++;
					// Start the timer that will reset the query count
					if( m_iQueryTimerId == 0 ) {
						if( g_pOptions->m_iMaxQueryTime < 1 )
							g_pOptions->m_iMaxQueryTime = 1; // Need at least one second
						m_iQueryTimerId = startTimer(g_pOptions->m_iMaxQueryTime * 1000);
					} // else: the timer is already running
				}
				if( m_iQueryCount > g_pOptions->m_iMaxQueryCount ) {
					// Flood!
					KviWindow *pOut = g_pOptions->m_bQueryFloodWarningsToConsole ? m_pConsole : m_pFrm->activeWindow();
					if( !msg->bHasBeenHandled ) {
						pOut->output(KVI_OUT_FLOOD,
							__tr("Clone flood detected: not creating query window for %s [%s]: %s"),
							talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux
						);
					}
					return;
				}
			}
			if( g_pOptions->m_bUseSmallAntispamOnPrivmsg ) {
				if( kvi_mayBeSpamMsg(aux) ) {
					if( !msg->bHasBeenHandled ) {
						m_pConsole->output(KVI_OUT_ANTISPAM,
							__tr("Probable spam message from %s [%s]: %s"),
							talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux
						);
					}
					return;
				}
			}
			if( g_pEventManager->eventEnabled(KviEvent_OnMePrivateMessage) ) {
				KviStr eventparms(KviStr::Format, "%s %s %s", talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux);
				if( m_pUserParser->callEvent(KviEvent_OnMePrivateMessage, m_pConsole, eventparms) )
					return; // Halted
			}
			// Not halted by the event
			if( g_pOptions->m_bCreateQueryOnPrivmsg ) {
				query = m_pFrm->createQuery(talker.nick(), true);
				if( !msg->bHasBeenHandled ) {
					query->output(KVI_OUT_INTERNAL,
						__tr("Private conversation requested by %s [%s]"), talker.nick(), skipToUsername(msg->szPrefix.ptr())
					);
				}
				m_pFrm->outputPrivmsg(query, KVI_OUT_NONE, talker.nick(), msg->szPrefix.ptr(), aux);
			} else {
				m_pFrm->activeWindow()->output(KVI_OUT_PRIVMSG,
					__tr("PRIVMSG from %s [%s]: %s"), talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux
				);
			}
		} else {
			// A query exists
			if( g_pEventManager->eventEnabled(KviEvent_OnMePrivateMessage) ) {
				KviStr eventparms(KviStr::Format, "%s %s %s", talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux);
				if( m_pUserParser->callEvent(KviEvent_OnMePrivateMessage, query, eventparms) )
					return;
			}
			m_pFrm->outputPrivmsg(query, KVI_OUT_NONE, talker.nick(), msg->szPrefix.ptr(), aux);
		}
		return;
	}

	// Channel PRIVMSG
	KviChannel *chan = m_pFrm->findChannel(target.ptr());
	if( !chan ) {
		// Oops, desync with the server? (or broadcast message)
		m_pFrm->activeWindow()->output(KVI_OUT_PRIVMSG,
			__tr("PRIVMSG from %s [%s] (to %s): %s"), talker.nick(), skipToUsername(msg->szPrefix.ptr()), target.ptr(), aux
		);
		return;
	}
	if( g_pEventManager->eventEnabled(KviEvent_OnChannelMessage) ) {
		KviStr eventparms(KviStr::Format, "%s %s %s", talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux);
		if( m_pUserParser->callEvent(KviEvent_OnChannelMessage, chan, eventparms) )
			return;
	}
	m_pFrm->outputPrivmsg(chan, KVI_OUT_NONE, talker.nick(), msg->szPrefix.ptr(), aux);
}

/**
 * ================================================================================================
 * NOTICE
 * ================================================================================================
 *
 * [lains!Socmel@foo][NOTICE] #italia : ACTION e' Away.[Auto away: idle  dalle 05:07am ]
 *
 */
void KviServerParser::parseNotice(KviIrcMessage *msg)
{
	if( msg->szPrefix.isEmpty() )
		msg->szPrefix = m_pFrm->m_global.szCurrentServerHost;
	KviIrcUser talker(msg->szPrefix.ptr());
	m_pFrm->m_pUserList->updateUser(talker);

	// Check ignore
	if( g_pOptions->m_bEnableIgnoreOnNotice ) {
		KviRegisteredUser *u = g_pOptions->m_pRegUsersDb->findIgnoredUserByMask(msg->szPrefix.ptr());
		if( u ) {
			// Ignored user
			if( g_pOptions->m_bVerboseIgnore ) {
				m_pConsole->output(KVI_OUT_INTERNAL,
					__tr("Ignoring NOTICE from %s: %s"), msg->szPrefix.ptr(), msg->szParams.ptr()
				);
			}
			return;
		}
	}

	KviStr target;
	const char *aux = kvi_extractToken(target, msg->szParams.ptr(), ' ');
	aux = skipToTrailing(aux);

	if( *aux == 0x01 ) {
		const char *aux2 = aux;
		while( *aux2 )
			aux2++;
		aux2--;
		if( (aux != aux2) && (*aux2 == 0x01) ) {
			aux++;
			msg->szParams.cutRight(1);
			parseCTCPNotice(talker, target.ptr(), aux);
			return;
		}
	}

	// First check if talker is a server.
	if( msg->szPrefix.findFirstIdx('@') == -1 ) {
		if( msg->szPrefix.findFirstIdx('!') == -1 ) {
			// Sure, it is a server (or something as CHANSERV or NICKSERV or AUTH)
			// Services use a strange format here.
			// [empty prefix] NOTICE <Service> :<text>
			if( g_pEventManager->eventEnabled(KviEvent_OnServerNotice) ) {
				KviStr eventparms(KviStr::Format, "%s %s %s", msg->szPrefix.ptr(), target.ptr(), aux);
				if( m_pUserParser->callEvent(KviEvent_OnServerNotice, m_pConsole, eventparms) )
					return;
			}
			if( isMe(target.ptr()) ) {
				KviWindow *pOut = g_pOptions->m_bServerNoticesToConsole ? m_pConsole : m_pFrm->activeWindow();
				pOut->output(KVI_OUT_NOTICE, "[%s]: %s", msg->szPrefix.ptr(), aux);
			} else {
				// The target may be a channel
				KviWindow *pOut = g_pOptions->m_bServerNoticesToConsole
					? m_pConsole
					: ((KviWindow *) m_pFrm->findChannel(target.ptr())
				);
				if( !pOut )
					pOut = m_pFrm->activeWindow();
				pOut->output(KVI_OUT_NOTICE, "[%s:%s]: %s", msg->szPrefix.ptr(), target.ptr(), aux);
			}
			return;
		}
	}
	// Normal NOTICE
	if( isMe(target.ptr()) ) {
		KviQuery *query = m_pFrm->findQuery(talker.nick());
		if( !query ) {
			// New query requested
			if( g_pOptions->m_bUseSmallAntispamOnPrivmsg ) {
				if( kvi_mayBeSpamMsg(aux) ) {
					if( !msg->bHasBeenHandled ) {
						m_pConsole->output(KVI_OUT_ANTISPAM,
							__tr("Probable spam message from %s [%s]: %s"),
							talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux
						);
					}
					return;
				}
			}
			if( g_pEventManager->eventEnabled(KviEvent_OnMeNotice) ) {
				KviStr eventparms(KviStr::Format, "%s %s %s", talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux);
				if( m_pUserParser->callEvent(KviEvent_OnMeNotice, m_pConsole, eventparms) )
					return;
			}
			// Not halted by the event
			if( g_pOptions->m_bCreateQueryOnNotice ) {
				query = m_pFrm->createQuery(talker.nick(), true);
				if( !msg->bHasBeenHandled ) {
					query->output(KVI_OUT_INTERNAL,
						__tr("Private conversation requested by %s [%s]"), talker.nick(), skipToUsername(msg->szPrefix.ptr())
					);
				}
				m_pFrm->outputPrivmsg(query, KVI_OUT_NOTICE, talker.nick(), msg->szPrefix.ptr(), aux);
			} else {
				KviWindow *pOut;
				if( g_pOptions->m_bOffChannelNoticesToConsole ) {
					pOut = m_pConsole;
					for( KviWindow *wnd = m_pFrm->m_pWinList->first(); wnd; wnd = m_pFrm->m_pWinList->next() ) {
						if( (wnd->type() == KVI_WND_TYPE_CHANNEL) &&
						    (((KviChannel *) wnd)->isOnChannel(talker.nick()))
						) {
							pOut = m_pFrm->activeWindow();
							break;
						}
					}
				} else pOut = m_pFrm->activeWindow();
				pOut->output(KVI_OUT_NOTICE,
					__tr("NOTICE from %s [%s]: %s"), talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux
				);
			}
		} else {
			// A query exists
			if( g_pEventManager->eventEnabled(KviEvent_OnMeNotice) ) {
				KviStr eventparms(KviStr::Format, "%s %s %s", talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux);
				if( m_pUserParser->callEvent(KviEvent_OnMeNotice, query, eventparms) )
					return;
			}
			m_pFrm->outputPrivmsg(query, KVI_OUT_NOTICE, talker.nick(), msg->szPrefix.ptr(), aux, false);
		}
		return;
	}

	// Channel NOTICE.
	// Check if it is a WALLCHOPS message
	bool oNotice = false;
	if( *(target.ptr()) == '@' ) {
		// It is :)
		target.cutLeft(1);
		oNotice = true;
	}

	KviChannel *chan = m_pFrm->findChannel(target.ptr());
	if( !chan ) {
		// Oops, desync with the server?
		m_pFrm->activeWindow()->output(KVI_OUT_PRIVMSG,
			__tr("NOTICE from %s [%s] (to %s): %s"), talker.nick(), skipToUsername(msg->szPrefix.ptr()), target.ptr(), aux
		);
		return;
	}

	if( oNotice ) {
		if( g_pEventManager->eventEnabled(KviEvent_OnChannelOpNotice) ) {
			KviStr eventparms(KviStr::Format, "%s %s %s", talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux);
			if( m_pUserParser->callEvent(KviEvent_OnChannelOpNotice, chan, eventparms) )
				return;
		}
		chan->output(KVI_OUT_ONOTICE,
			__tr("OP-NOTICE from %s [%s] to %s: %s"), talker.nick(), skipToUsername(msg->szPrefix.ptr()), target.ptr(), aux
		);
	} else {
		if( g_pEventManager->eventEnabled(KviEvent_OnChannelNotice) ) {
			KviStr eventparms(KviStr::Format, "%s %s %s", talker.nick(), skipToUsername(msg->szPrefix.ptr()), aux);
			if( m_pUserParser->callEvent(KviEvent_OnChannelNotice, chan, eventparms) )
				return;
		}
		m_pFrm->outputPrivmsg(chan, KVI_OUT_NOTICE, talker.nick(), msg->szPrefix.ptr(), aux);
	}
}

/**
 * ================================================================================================
 * TOPIC
 * ================================================================================================
 *
 * [000!aaa@bbb.earth][TOPIC] #channel :topik_text
 *
 */
void KviServerParser::parseTopic(KviIrcMessage *msg)
{
	KviIrcUser topiker(msg->szPrefix.ptr());
	m_pFrm->m_pUserList->updateUser(topiker);

	KviStr target;
	const char *aux = kvi_extractToken(target, msg->szParams.ptr(), ' ');
	aux = skipToTrailing(aux);

	KviChannel *chan = m_pFrm->findChannel(target.ptr());
	if( !chan ) {
		// Oops, desync with the server.
		KviWindow *pOut = g_pOptions->m_bDesyncMsgsToConsole ? m_pConsole : m_pFrm->activeWindow();
		pOut->output(KVI_OUT_DESYNC,
			__tr("Local desync: TOPIC from %s [%s] for channel %s: %s"),
			topiker.nick(), skipToUsername(msg->szParams.ptr()), target.ptr(), aux
		);
		return;
	}

	chan->setTopic(aux);

	if( g_pEventManager->eventEnabled(KviEvent_OnTopic) ) {
		KviStr tmp(KviStr::Format, "%s %s %s %s", target.ptr(), topiker.nick(), msg->szPrefix.ptr(), aux);
		if( m_pUserParser->callEvent(KviEvent_OnTopic, chan, tmp) )
			return;
	}

	if( !msg->bHasBeenHandled ) {
		KviWindow *pOut = g_pOptions->m_bTopicMsgsToConsole ? m_pConsole : ((KviWindow *) chan);
		pOut->output(KVI_OUT_TOPIC,
			__tr("%s [%s] sets topic for %s to: %s"), topiker.nick(), skipToUsername(msg->szPrefix.ptr()), target.ptr(), aux
		);
	}
}

void KviServerParser::parseInvite(KviIrcMessage *msg)
{
	KviIrcUser inviter(msg->szPrefix.ptr());
	m_pFrm->m_pUserList->updateUser(inviter);

	const char *aux = msg->szParams.ptr();
	while( *aux && (*aux != '&') && (*aux != '#') && (*aux != '!') )
		aux++;

	KviWindow *pOut = g_pOptions->m_bInviteMessagesToConsole ? m_pConsole : m_pFrm->activeWindow();
	if( *aux ) {
		KviStr chan;
		kvi_extractToken(chan, aux, ' ');

		if( g_pEventManager->eventEnabled(KviEvent_OnInvite) ) {
			KviStr tmp(KviStr::Format,
				"%s %s %s %c",
				inviter.nick(), msg->szPrefix.ptr(), chan.ptr(), (g_pOptions->m_bAutoJoinChanOnInvite ? '1' : '0')
			);
			if( m_pUserParser->callEvent(KviEvent_OnInvite, m_pConsole, tmp) )
				return;
		}

		if( g_pOptions->m_bAutoJoinChanOnInvite ) {
			if( !msg->bHasBeenHandled ) {
				pOut->output(KVI_OUT_INVITE,
					__tr("%s [%s] invites you to join channel \r!join $1\r%s\r: auto-joining"),
					inviter.nick(), skipToUsername(msg->szPrefix.ptr()), chan.ptr()
				);
			}
			m_pSocket->sendFmtData("JOIN %s", chan.ptr());
		} else {
			if( !msg->bHasBeenHandled ) {
				pOut->output(KVI_OUT_INVITE,
					__tr("%s [%s] invites you to join channel \r!join $1\r%s\r"),
					inviter.nick(), skipToUsername(msg->szPrefix.ptr()), chan.ptr()
				);
			}
		}
	} else {
		// Broken invite?
		if( !msg->bHasBeenHandled ) {
			m_pConsole->output(KVI_OUT_INTERNAL,
				__tr("Broken invite message from %s [%s]: %s"),
				inviter.nick(), skipToUsername(msg->szPrefix.ptr()), msg->szParams.ptr()
			);
		}
	}
}

/**
 * ======================================================================================================
 * Unhandled Literal
 * ======================================================================================================
 */
void KviServerParser::parseUnhandledLiteral(KviIrcMessage *msg)
{
	if(msg->bHasBeenHandled ) return;

	if( g_pEventManager->eventEnabled(KviEvent_OnUnhandledLiteral) ) {
		KviStr tmp(KviStr::Format, "%s %s %s", msg->szPrefix.ptr(), msg->szCommand.ptr(), msg->szParams.ptr());
		if( m_pUserParser->callEvent(KviEvent_OnUnhandledLiteral, m_pConsole, tmp) )
			return;
	}

	KviWindow *pOut = g_pOptions->m_bUnhandledLiteralsToConsole ? m_pConsole : m_pFrm->activeWindow();
	pOut->output(KVI_OUT_UNHANDLED,
		_i18n_("[%s][%s] %s"), msg->szPrefix.ptr(), msg->szCommand.ptr(), msg->szParams.ptr()
	);
}

/**
 * ======================================================================================================
 * Timer Event for CTCP, Query, Netsplit
 * ======================================================================================================
 */
void KviServerParser::timerEvent(QTimerEvent *e)
{
	if( e->timerId() == m_iQueryTimerId ) {
		m_iQueryCount -= g_pOptions->m_iMaxQueryCount;
		if( m_iQueryCount <= 0 ) {
			m_iQueryCount = 0;
			killTimer(m_iQueryTimerId);
			m_iQueryTimerId = 0;
		} else if( m_iQueryCount > g_pOptions->m_iMaxQueryCount ) {
			// If m_iQueryCount grows too much,
			// queries can be disabled for too long after the flood has ceased.
			// Keep it quite small
			m_iQueryCount = g_pOptions->m_iMaxQueryCount;
		}
	} else if( e->timerId() == m_iCtcpTimerId ) {
		m_iCtcpCount -= g_pOptions->m_iMaxCtcpCount;
		if( m_iCtcpCount <= 0 ) {
			m_iCtcpCount = 0;
			killTimer(m_iCtcpTimerId);
			m_iCtcpTimerId = 0;
		} else if( m_iCtcpCount > g_pOptions->m_iMaxCtcpCount ) {
			// If m_iCtcpCount grows too much,
			// CTCP's can be disabled for too long after the flood has ceased.
			// Keep it quite small
			m_iCtcpCount = g_pOptions->m_iMaxCtcpCount;
		}
	} else if( e->timerId() == m_iNetsplitTimerId ) {
		m_szLastNetsplit   = "";
		killTimer(m_iNetsplitTimerId);
		m_iNetsplitTimerId = 0;
	}
}

/**
 * [04:16:49] [][ERROR] :Closing Link: newbie[^nobody@a-si6-32.tin.it] (Cya!)
 * [22:26:42] [ircd.nl.uu.net][436] newbie newbie :Nickname collision KILL from test@in206.pem.cam.ac.uk
 * [22:26:42] [ircd.nl.uu.net][KILL] newbie :ircd.tin.it!irc.flashnet.it!irc.ccii.unipi.it!*.fi[unknown@ircd.funet.fi]!irc2.sci.kun.nl!*.nl.uu.net[unknown@ircd.nl.uu.net]!ircd.nl.uu.net ((^nobody@a-si4-57.tin.it)irc2.sci.kun.nl <- (test@in206.pem.cam.ac.uk)ircnet.demon.co.uk)
 * [22:26:42] [][ERROR] :Closing Link: newbie[^nobody@a-si4-57.tin.it] ircd.tin.it (Killed (ircd.nl.uu.net ((^nobody@a-si4-57.tin.it)irc2.sci.kun.nl <- (test@in206.pem.cam.ac.uk)ircnet.demon.co.uk)))
 * [22:26:42] Disconnected : Remote end closed connection
 * [22:30:16] Unrecognized CTCP request from Jim20 [Jim20@ip192.t19.ds.pwr.wroc.pl] to [newbie] : AWAY...
 *
 * /LIST >100 (dal.net)
 * [04:22:30] [viking.no.eu.dal.net][322] [Pragma] #filipino 153 :@@Congratulations   To  AJ  LYN   &  PINTOY    son  his   on   A  HONOR    ROLL  students...We   Love   u  AJ@@
 * [04:22:30] [viking.no.eu.dal.net][322] [Pragma] #teenzone 104 :( join #mp3s )
 * [04:22:31] [viking.no.eu.dal.net][322] [Pragma] #?5 400 :*??????????? PaxCapital R2:86.43/R1:84.97/Piv:82.53/S1:81.07/S2:78.63????????????*
 * [04:22:31] [viking.no.eu.dal.net][322] [Pragma] #CyberChat 180 :*`* *`*  Welcome to #Cyberchat  State of Origin 99... QUEENSLAND beats NSW 9-8!!!!   *`**`* ** Visit our Homepage at: www.cyber-chat.org
 * [04:22:31] [viking.no.eu.dal.net][323] [Pragma] :End of /LIST
 *
 */

#include "m_kvi_serverparser.moc"
