// =============================================================================
//
//      --- kvi_userwindow.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//   Copyright (C) 1999-2000 Till Busch (buti@geocities.com)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviUserWindow"

#include <qsplitter.h>

#include "kvi_event.h"
#include "kvi_frame.h"
#include "kvi_input.h"
#include "kvi_irc_view.h"
#include "kvi_listbox.h"
#include "kvi_locale.h"
#include "kvi_mdi_child.h"
#include "kvi_options.h"
#include "kvi_statusbar.h"
#include "kvi_userparser.h"
#include "kvi_userwindow.h"
#include "kvi_winproperties.h"

// Declared in kvi_app.cpp and managed by KviApp class
extern QPixmap         *g_pixViewOut[KVI_OUT_NUM_IMAGES];
extern KviEventManager *g_pEventManager;

KviUserWindow::KviUserWindow(KviFrame *lpFrm, const char *name)
	: KviWindow(name, KVI_WND_TYPE_UWINDOW, lpFrm)
{
	m_pSplitter  = new QSplitter(QSplitter::Horizontal, this);
	m_pSplitter->setOpaqueResize();
	m_pView      = new KviIrcView(m_pSplitter, lpFrm, this);
	m_pView->enableDnD(true);
	connect(m_pView, SIGNAL(fileDropped(const char *)), this, SLOT(fileDroppedOnView(const char *)));
	connect(m_pView, SIGNAL(dndEntered()), this, SLOT(dndEnteredView()));

	m_pListBox   = new KviListBox(m_pSplitter);
	m_pListBox->setHScrollBarMode(QScrollView::AlwaysOff);
	connect(m_pListBox, SIGNAL(selected(int)), this, SLOT(listBoxDoubleClicked(int)));
	connect(m_pListBox, SIGNAL(highlighted(int)), this, SLOT(listBoxHighlighted(int)));

	m_pInput     = new KviInput(this, lpFrm->m_pUserParser);
	m_bHasInput  = true;
	connect(m_pView, SIGNAL(contextPopupRequested(KviIrcView *)), this, SLOT(viewRightClicked(KviIrcView *)));

	// Reasonable default for the splitter
	QValueList<int> l;
	l.append(80);
	l.append(20);
	m_pSplitter->setSizes(l);

	m_bCloseEnabled = true;
}

/**
 * ============ ~KviUserWindow ============
 */
KviUserWindow::~KviUserWindow()
{
	// Nothing here
}

void KviUserWindow::saveProperties()
{
	KviWindowProperty p;
	p.rect           = externalGeometry();
	p.isDocked       = isAttached();
	KviWindow *w     = m_pFrm->activeWindow();
	p.isMaximized    = isAttached() && w ? w->isMaximized() : isMaximized();
	p.topSplitWidth1 = 0;
	p.topSplitWidth2 = 0;
	p.topSplitWidth3 = 0;
	g_pOptions->m_pWinPropertiesList->setProperty(caption(), &p);
}

void KviUserWindow::dndEnteredView()
{
	m_pFrm->m_pStatusBar->tempText(__tr("Drop the file to start the DCC Send"), 5000);
}

void KviUserWindow::fileDroppedOnView(const char *fileName)
{
	if( g_pEventManager->eventEnabled(KviEvent_OnUserWindowFileDrop) ) {
		KviStr eventparms(KviStr::Format, "%s %s", m_szCaption.ptr(), fileName);
		m_pFrm->m_pUserParser->callEvent(KviEvent_OnUserWindowFileDrop, this, eventparms);
	}
}

void KviUserWindow::viewRightClicked(KviIrcView *)
{
	// Nothing here
}

void KviUserWindow::viewPopupClicked(const KviStr &dataBuffer)
{
	// Nothing here
}

/**
 * ================= listAdd ============
 */
void KviUserWindow::listAdd(const char *name)
{
	m_pListBox->insertItem(name);
}

/**
 * ================= listRemove ============
 */
void KviUserWindow::listRemove(const int item)
{
	m_pListBox->removeItem(item);
}

/**
 * ================= listClear ============
 */
void KviUserWindow::listClear()
{
	m_pListBox->clear();
}

/**
 * ================= itemCount ==============
 */
int KviUserWindow::itemCount()
{
	return m_pListBox->count();
}

/**
 * =========== listBoxDoubleClicked ============
 */
void KviUserWindow::listBoxDoubleClicked(int index)
{
	if( g_pEventManager->eventEnabled(KviEvent_OnUserWindowSelected) ) {
		KviStr tmp = m_pListBox->text(index);
		KviStr eventparms(KviStr::Format, "%s %c %d %s", m_szCaption.ptr(), '1', index, tmp.ptr());
		m_pFrm->m_pUserParser->callEvent(KviEvent_OnUserWindowSelected, this, eventparms);
	}
}

/**
 * ============ listBoxHighlighted ============
 */
void KviUserWindow::listBoxHighlighted(int index)
{
	if( g_pEventManager->eventEnabled(KviEvent_OnUserWindowSelected) ) {
		KviStr tmp = m_pListBox->text(index);
		KviStr eventparms(KviStr::Format, "%s %c %d %s", m_szCaption.ptr(), '0', index, tmp.ptr());
		m_pFrm->m_pUserParser->callEvent(KviEvent_OnUserWindowSelected, this, eventparms);
	}
}

/**
 * ================ myIconPtr =================
 */
QPixmap *KviUserWindow::myIconPtr()
{
	return g_pixViewOut[KVI_OUT_WND_UWINDOW];
}

/**
 * =============== applyOptions ================
 */
void KviUserWindow::applyOptions()
{
	m_pView->setFont(g_pOptions->m_fntView);
	m_pView->setShowImages(g_pOptions->m_bShowImages, false);
	m_pView->setTimestamp(g_pOptions->m_bTimestamp);
	m_pView->setMaxBufferSize(g_pOptions->m_iViewMaxBufferSize);
}

/**
 * =============== hasInput ================
 */
void KviUserWindow::hasInput(bool i)
{
	if( i ) {
		if( !m_bHasInput ) {
			int inputSize = m_pInput->heightHint();
			m_pSplitter->setGeometry(0, 0, width(), height() - inputSize);
			m_pInput->setGeometry(0, height() - inputSize, width(), inputSize);
		}
	} else {
		if( m_bHasInput ) {
			m_pSplitter->setGeometry(0, 0, width(), height());
			m_pInput->setGeometry(0, 0, width(), 0);
		}
	}
	m_bHasInput = i;
}

/**
 * =============== hasList ================
 */
void KviUserWindow::hasList(bool i)
{
	if( i ) {
		if( !m_pListBox->isVisible() ) {
			m_pListBox->show();
			m_pListBox->setEnabled(true);
		}
	} else {
		if( m_pView->isVisible() && m_pListBox->isVisible() ) {
			m_pListBox->hide();
			m_pListBox->setEnabled(false);
		}
	}
}

/**
 * =============== hasView ================
 */
void KviUserWindow::hasView(bool i)
{
	if( i ) {
		if( !m_pView->isVisible() ) {
			m_pView->show();
			m_pView->setEnabled(true);
		}
	} else {
		if( m_pListBox->isVisible() && m_pView->isVisible() ) {
			m_pView->hide();
			m_pView->setEnabled(false);
		}
	}
}

/**
 * ================ resizeEvent ===============
 */
void KviUserWindow::resizeEvent(QResizeEvent *)
{
	int inputSize = (m_bHasInput ? m_pInput->heightHint() : 0);
	m_pSplitter->setGeometry(0, 0, width(), height() - inputSize);
	if( m_bHasInput )
		m_pInput->setGeometry(0, height() - inputSize, width(), inputSize);
}

void KviUserWindow::youAreAttached(KviMdiChild *lpC)
{
	KviWindow::youAreAttached(lpC);
	lpC->enableClose(m_bCloseEnabled);
}

void KviUserWindow::enableClose(bool bEnable)
{
	m_bCloseEnabled = bEnable;
	if( mdiParent() )
		mdiParent()->enableClose(m_bCloseEnabled);
}

#include "m_kvi_userwindow.moc"
