// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptWidget"

#include <qmetaobject.h>
#include <qvariant.h>
#include <qwidget.h>

#include "kvi_defines.h"
#include "kvi_error.h"
#include "kvi_frame.h"
#include "kvi_locale.h"
#include "kvi_malloc.h"
#include "kvi_mirccntrl.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_script_objectcontroller.h"
#include "kvi_script_widget.h"
#include "kvi_window.h"

// TODO: Fonts and colors, more events, icons!
// TODO: OnDblCLick?
// TODO: Set tooltip?
// TODO: Font properties
// TODO: QFont & QPixmap?
// TODO: OnGenericEvent
// TODO: Settable properties of the KVIrc windows

/*
	@class: widget
	@short:
		Base class for all widgets
	@inherits:
		<a href="class_object.kvihelp">object</a>
	@functions:

		!fn: $show(&lt;bShowChildren&gt;)
		Shows the widget.<br>
		The argument bShowChildren is optional, if it is present and it is non-zero (true)
		the children of the widget are also shown.<br>

		!fn: $hide()
		Hides the widget

		!fn: $x()
		Returns the x coordinate of the widget in pixels, relative to the parent widget
		or to the desktop of the widget is toplevel.<br>

		!fn: $y()
		Returns the y coordinate of the widget in pixels, relative to the parent widget
		or to the desktop of the widget is toplevel.<br>

		!fn: $width()
		Returns the width of the widget in pixels.<br>

		!fn: $height()
		Returns the height of the widget in pixels.<br>

		!fn: $setMinimumWidth()
		Sets the minimum width of the widget.<br>

		!fn: $setMinimumHeight()
		Sets the minimum height of the widget.<br>

		!fn: $isToplevel()
		returns 1 (true) if this widget is a toplevel one (has no parent widget).<br>
		Please note that this does NOT mean that the object itself has no parent!.<br>

		!fn: $isVisible()
		returns 1 (true) if this widget is visible (mapped), 0 otherwise.<br>

		!fn: $setGeometry(&lt;x&gt;, &lt;y&gt;, &lt;width&gt;, &lt;height&gt;)
		Sets the geometry of the widget.<br>
		All the coordinates are expressed in screen pixels.<br>
		The &lt;x&gt; and &lt;y&gt; coordinates are relative to the parent widget,
		or to the desktop (if this widget is a toplevel one).<br>
		Returns 0 if one of the parameters passed was invalid (not a number?),
		1 otherwise.<br>

		!fn: $isEnabled()
		Returns 1 if the widget is enabled, 0 otherwise.<br>

		!fn: $setEnabled(&lt;state&gt;)
		Sets the enable state to &lt;state&gt; that is a boolean value:<br>
		If &lt;state&gt; is 0 the widget is disabled, for any other value the widget
		is enabled.<br>

		!fn: $parentWidget()
		If this widget is a toplevel one, this functon is equivalent to parent()
		(inherited from the object class), otherwise it returns the STRING "0".<br>

		!fn: $setCaption()
		Sets the caption text for this widget.
		This function is meaningful only for the toplevel widgets.<br>

		!fn: $caption()
		Returns the caption text of this widget.<br>

		!fn: $raise()
		Brings this widget to the top of the stack of widgets of its parent widget.<br>

		!fn: $lower()
		Brings this widget to the bottom of the stack of widget of its parent widget.<br>

		!fn: $setFocus()
		Sets the keyboard focus to this widget.<br>

		!fn: $hasFocus()
		Returns 1 if this widget currently has the keyboard focus.<br>

		!fn: $setBackgroundColor(&lt;rgb&gt;)
		Sets the background color of the widget to the color specified
		by the rgb string.<br>
		The rgb string is a hexadecimal string of 6 characters.<br>
		The first two hex-digits rappresent the red component of the color,
		the third and fourth rappresent the green comonent and the remaining part
		represents the blue component.<br>
		So for example: 000000 represents black, FFFFFF represents white,
		FF0000 represents red, FFA000 is a mix of red and green that gives an orange colour.<br>
		Returns 0 if the rgb parameter was invalid, 1 otherwise.<br>

		!fn: $backgroundColor()
		Returns the background color as hexadecimal string as accepted by $setBackgroundColor().<br>

		!fn: $qtProperty(&lt;property_name&gt;)
		This is for really advanced scripting.<br>
		All KVIrc widgets are based on the Qt library ones.<br>
		The Qt library widgets allow to set and read special properties.<br>
		You will have to take a look at the Qt documentation for each widget type
		to see the available property names.<br>
		The supported property types are: Rect, Size, Point, Color, String, CString,
		Int, UInt, Bool and enumeration types.<br>
		For example, the widget's x coordinate can be retrieved by using the [classfnc:widget]$x[/classfnc]()
		function or by calling $property(x).<br>
		There are many properties that are available ony through the "$property()" call:<br>
		For example, you can find out if the widget accepts drops by calling $property(acceptDrops).<br>
		This function will be mainly useful in the [class]qtwrapper[/class] class.<br>

		!fn: $setQtProperty(&lt;property_name&gt;, &lt;property_value&gt;)
		Sets a qt property for this widget.<br>
		This is for advanced scripting, and can control really many features of the Qt widgets.<br>
		For example, the [class]multilineedit[/class] widgets can be set to
		the "password" echo mode only by using this function call:<br>
		<example>
			%X = [fnc]$new[/fnc]([class]multilineedit[/class], [fnc]$root[/fnc], a_name)
			%X->[classfnc:widget]$show[/classfnc]()
			%X->[classfnc:widget]$setQtProperty[/classfnc](echoMode, Password)
		</example>
		The available properties to be set are listed by [classfnc:widget]$listQtProperties[/classfnc]()
		and must appear in the list as writeable.<br>
		This function will be mainly useful in the [class]qtwrapper[/class] class.<br>

		!fn: $listQtProperties()
		Lists the qt properties supported by this object.<br>
		This function will be mainly useful in the [class]qtwrapper[/class] class.<br>

	@events:
		!ev: OnMouseButtonPress($1 = Button; $2 = x $3 = y)
		Triggered when the user press a mouse button while the pointer is on this widget.<br>
		The first parameter is "left", "right" or "mid" depenging on the mouse button
		that has been pressed.<br>
		The x and y parameters are the coordinates in pixels of the press-point relative to this widget.<br>
		Please note that if the user drags the mouse out of the widget while the button is
		being pressed, the OnMouseButtonRelease event may also be triggered in a different widget.<br>

		!ev: OnMouseButtonRelease($1 = Button; $2 = x $3 = y)
		Triggered when the user releases a mouse button while the pointer is on this widget.<br>
		The first parameter is "left", "right" or "mid" depenging on the mouse button
		that has been released.<br>
		The x and y parameters are the coordinates in pixels of the release-point relative to this widget.<br>
		Please note that this event may also NOT follow a OnMouseButtonPress event.<br>

		!ev: OnMouseButtonDblClick($1 = Button; $2 = x $3 = y)
		Triggered when the user "double-clicks" a mouse button while the pointer is on this widget.<br>
		The first parameter is "left", "right" or "mid" depenging on the mouse button
		that has been double-pressed.<br>
		The x and y parameters are the coordinates in pixels of the click-point relative to this widget.<br>

		!ev: OnClose()
		Triggered when the user closes the window with the close caption button.<br>
		This widget object will be destroyed in a while.<br>
		This event is obviously triggered only for toplevel widgets.<br>

		!ev: OnResize($1 = newWidth, $2 = newHeight)
		Triggered when the widget is being resized either by the user or
		by using $setGeometry.<br>
		Please note that the new widget size is available through the parameters of this
		event, and may not yet correspond to the size returned by $width() and $height().<br>
		It depends on the order in that the events are processed.<br>

		!ev: OnMove($1 = newX, $2 = newY)
		Triggered when the widget is being moved either by the user or
		by a $setGeometry() function call.<br>
		Please nothe the the new widget position is available through the parameters
		of this event, and may not correspond (yet) to the position returned by $x() and $y().<br>

		!ev: OnFocusIn()
		Triggered when the widget gains keyboard focus.<br>

		!ev: OnFocusOut()
		Triggered when the widget loses keyboard focus.<br>

	@description:
		This class is the base for all the widget classes.<br>
		It encapsulates the basic functionality of a "physical" widget.<br>
		You create a widget by using the <a href="s_new.kvihelp">$new()</a> function.<br>
		If the object passed as parent inherits the "widget" class, its "physical"
		widget is taken as parent, otherwise this widget becomes a toplevel one (parentless,
		or if you want, with the desktop as parent).<br>
		The $isToplevel() member function will tell you whether the widget
		was created as toplevel or not.<br>
		The $parentWidget() function will return the parent object id, if it inherits the
		widget class, the string "0" otherwise (and so this widget is toplevel).<br>
		A toplevel widget usually has a caption bar provided by the window manager.<br>
		It is possible to set the text of the caption by using the $setCaption() function.<br>
		The position of the widget relative to the parent widget (or to the desktop) can be
		set by using $setGeometry() and retrieved by using $x(), $y(), $width() and $height().<br>
		You can react to mouse clicks by implementing the handlers for OnMouseButtonPress,
		OnMouseButtonRelease and OnMouseButtonDblClick.<br>
		When the widget is resized or moved by the user (or by the $setGeometry() function)
		the events OnMove and OnResize are triggered.<br>
		If the widget is being closed by the user by clicking on the caption "close" button
		(if provided by the window manager), the OnClose event is triggered and after that
		the object is automatically destroyed.<br>

	@examples:
		First simple example: create a toplevel widget and respond to the button press events.<br>
		<example>
		%tmp = <a href="s_new.kvihelp">$new</a>(widget, <a href="s_root.kvihelp">$root</a>, aName)
		<a href="obj_setevent.kvihelp">obj_setevent</a>(%tmp, OnMouseButtonPress){ <a href="s_this.kvihelp">$this</a>-&gt;$setCaption(Pressed); }
		<a href="obj_setevent.kvihelp">obj_setevent</a>(%tmp, OnMouseButtonRelease){ <a href="s_this.kvihelp">$this</a>-&gt;$setCaption(Released); }
		<a href="obj_setevent.kvihelp">obj_setevent</a>(%tmp, OnClose){ <a href="echo.kvihelp">echo</a> Widget closed!; }
		%tmp-&gt;$show();
		</example>
		Now just click on the widget to see the caption changing.<br>
		When you close the widget by clicking the close button in the caption
		the object will be automatically destroyed.<br>
		An alternative implementation using class definitions:<br>
		<example>
		<a href="class.kvihelp">class</a>(mywidget, widget)
		{
		&nbsp;&nbsp;event(OnMouseButtonPress)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;$setCaption(Pressed)
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnMouseButtonRelease)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;$setCaption(Released)
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnClose)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> Widget destroyed!
		&nbsp;&nbsp;}
		}
		%tmp = <a href="s_new.kvihelp">$new</a>(mywidget, <a href="s_root.kvihelp">$root</a>, aName)
		%tmp-&gt;$show();
		</example>
		If you test this script twice, and you get errors about "Classes already defined"
		try to use the <a href="clearobjects.kvihelp">clearobjects</a> command.<br>
		Another example:<br>
		Create a toplevel widget and a child one with a special background color.<br>
		React to clicks on the child widget, and to the parent resize event to adjust
		the child size.<br>
		<example>
		<a href="class.kvihelp">class</a>(myw1, widget)
		{
		&nbsp;&nbsp;event(OnDestroy)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> Object myw1 destroyed
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnMouseButtonPress)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="if.kvihelp">if</a>("<a href="s_this.kvihelp">$this</a>-&gt;$parentWidget()" != "0")<a href="s_this.kvihelp">$this</a>-&gt;$parent()-&gt;$setCaption(Parent click)
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;$setBackgroundColor(000000)
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnMouseButtonRelease)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="if.kvihelp">if</a>("<a href="s_this.kvihelp">$this</a>-&gt;$parentWidget()" != "0")<a href="s_this.kvihelp">$this</a>-&gt;$parent()-&gt;$setCaption(Parent release)
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;$setBackgroundColor(80, 80, 80)
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnResize)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> Child resized : <a href="s_this.kvihelp">$this</a>-&gt;$x(), <a href="s_this.kvihelp">$this</a>-&gt;$y() , $1, $2
		&nbsp;&nbsp;}
		}

		<a href="class.kvihelp">class</a>(myw2, widget)
		{
		&nbsp;&nbsp;function(constructor)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;%child1 = <a href="s_new.kvihelp">$new</a>(myw1, <a href="s_this.kvihelp">$this</a>, myw)
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;$setBackgroundColor(255, 255, 255)
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="setreturn.kvihelp">setreturn</a> 1
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnResize)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;%child1-&gt;$setGeometry(4, 4, <a href="s_calc.kvihelp">$calc</a>($1 - 8), <a href="s_calc.kvihelp">$calc</a>($2 - 8));
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnDestroy)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> Object myw2 destroyed
		&nbsp;&nbsp;}
		}

		%O = <a href="s_new.kvihelp">$new</a>(myw2, <a href="s_root.kvihelp">$root</a>, 0)
		%O-&gt;$show()
		</example>
	@seealso:
		<a href="class_object.kvihelp">Object class</a>,
		<a href="class_layout.kvihelp">Layout class</a>,
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/
void KviScriptWidget::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("show",               (scriptObjectFunction) &KviScriptWidget::builtinFunction_SHOW);
	d->addBuiltinFunction("hide",               (scriptObjectFunction) &KviScriptWidget::builtinFunction_HIDE);
	d->addBuiltinFunction("x",                  (scriptObjectFunction) &KviScriptWidget::builtinFunction_X);
	d->addBuiltinFunction("y",                  (scriptObjectFunction) &KviScriptWidget::builtinFunction_Y);
	d->addBuiltinFunction("width",              (scriptObjectFunction) &KviScriptWidget::builtinFunction_WIDTH);
	d->addBuiltinFunction("height",             (scriptObjectFunction) &KviScriptWidget::builtinFunction_HEIGHT);
	d->addBuiltinFunction("setMinimumWidth",    (scriptObjectFunction) &KviScriptWidget::builtinFunction_SETMINIMUMWIDTH);
	d->addBuiltinFunction("setMinimumHeight",   (scriptObjectFunction) &KviScriptWidget::builtinFunction_SETMINIMUMHEIGHT);
	d->addBuiltinFunction("isToplevel",         (scriptObjectFunction) &KviScriptWidget::builtinFunction_ISTOPLEVEL);
	d->addBuiltinFunction("isVisible",          (scriptObjectFunction) &KviScriptWidget::builtinFunction_ISVISIBLE);
	d->addBuiltinFunction("setGeometry",        (scriptObjectFunction) &KviScriptWidget::builtinFunction_SETGEOMETRY);
	d->addBuiltinFunction("isEnabled",          (scriptObjectFunction) &KviScriptWidget::builtinFunction_ISENABLED);
	d->addBuiltinFunction("setEnabled",         (scriptObjectFunction) &KviScriptWidget::builtinFunction_SETENABLED);
	d->addBuiltinFunction("parentWidget",       (scriptObjectFunction) &KviScriptWidget::builtinFunction_PARENTWIDGET);
	d->addBuiltinFunction("setCaption",         (scriptObjectFunction) &KviScriptWidget::builtinFunction_SETCAPTION);
	d->addBuiltinFunction("caption",            (scriptObjectFunction) &KviScriptWidget::builtinFunction_CAPTION);
	d->addBuiltinFunction("raise",              (scriptObjectFunction) &KviScriptWidget::builtinFunction_RAISE);
	d->addBuiltinFunction("lower",              (scriptObjectFunction) &KviScriptWidget::builtinFunction_LOWER);
	d->addBuiltinFunction("setFocus",           (scriptObjectFunction) &KviScriptWidget::builtinFunction_SETFOCUS);
	d->addBuiltinFunction("hasFocus",           (scriptObjectFunction) &KviScriptWidget::builtinFunction_HASFOCUS);
	d->addBuiltinFunction("setBackgroundColor", (scriptObjectFunction) &KviScriptWidget::builtinFunction_SETBACKGROUNDCOLOR);
	d->addBuiltinFunction("backgroundColor",    (scriptObjectFunction) &KviScriptWidget::builtinFunction_BACKGROUNDCOLOR);
	d->addBuiltinFunction("setQtProperty",      (scriptObjectFunction) &KviScriptWidget::builtinFunction_SETQTPROPERTY);
	d->addBuiltinFunction("qtProperty",         (scriptObjectFunction) &KviScriptWidget::builtinFunction_QTPROPERTY);
	d->addBuiltinFunction("listQtProperties",   (scriptObjectFunction) &KviScriptWidget::builtinFunction_LISTQTPROPERTIES);
}

KviScriptWidget::KviScriptWidget(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptObject(cntrl, p, name, pDef)
{
	m_pWidget = 0;
	m_bAutoDestroyControlledWidget = false;
}

KviScriptWidget::~KviScriptWidget()
{
	if( m_pWidget ) {
		m_pWidget->removeEventFilter(this);
		disconnect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
		if( m_bAutoDestroyControlledWidget ) {
			delete m_pWidget;
			m_pWidget = 0;
		}
	}
}

bool KviScriptWidget::init(QPtrList<KviStr> *)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new QWidget(((KviScriptWidget *) parent())->m_pWidget, name());
		}
	}
	if( !m_pWidget )
		m_pWidget = new QWidget(0, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	return true;
}

void KviScriptWidget::widgetDestroyed()
{
	m_pWidget = 0;
	dieOutOfThisEventStep();
}

bool KviScriptWidget::eventFilter(QObject *o, QEvent *e)
{
	if( o == m_pWidget ) {
		KviStr parms;
		KviStr tmp;
		switch( e->type() ) {
			case QEvent::MouseButtonPress:
				if( ((QMouseEvent *) e)->button() & LeftButton )       tmp = "left";
				else if( ((QMouseEvent *) e)->button() & RightButton ) tmp = "right";
				else if( ((QMouseEvent *) e)->button() & MidButton )   tmp = "mid";
				parms.sprintf("%s %d %d", tmp.ptr(), ((QMouseEvent *) e)->pos().x(), ((QMouseEvent *) e)->pos().x());
				triggerEvent("OnMouseButtonPress", parms);
				break;
			case QEvent::MouseButtonRelease:
				if( ((QMouseEvent *) e)->button() & LeftButton )       tmp = "left";
				else if( ((QMouseEvent *) e)->button() & RightButton ) tmp = "right";
				else if( ((QMouseEvent *) e)->button() & MidButton )   tmp = "mid";
				parms.sprintf("%s %d %d", tmp.ptr(), ((QMouseEvent *) e)->pos().x(), ((QMouseEvent *) e)->pos().x());
				triggerEvent("OnMouseButtonRelease", parms);
				break;
			case QEvent::MouseButtonDblClick:
				if( ((QMouseEvent *) e)->button() & LeftButton )       tmp = "left";
				else if( ((QMouseEvent *) e)->button() & RightButton ) tmp = "right";
				else if( ((QMouseEvent *) e)->button() & MidButton )   tmp = "mid";
				parms.sprintf("%s %d %d", tmp.ptr(), ((QMouseEvent *) e)->pos().x(), ((QMouseEvent *) e)->pos().x());
				triggerEvent("OnMouseButtonDblClick", parms);
				break;
			case QEvent::Resize:
				parms.sprintf("%d %d", ((QResizeEvent *) e)->size().width(), ((QResizeEvent *) e)->size().height());
				triggerEvent("OnResize", parms);
				break;
			case QEvent::Move:
				parms.sprintf("%d %d", ((QMoveEvent *) e)->pos().x(), ((QMoveEvent *) e)->pos().y());
				triggerEvent("OnMove", parms);
				break;
			case QEvent::Close:
				triggerEvent("OnClose", parms);
				dieOutOfThisEventStep();
				break;
			case QEvent::FocusIn:
				triggerEvent("OnFocusIn", parms);
				break;
			case QEvent::FocusOut:
				triggerEvent("OnFocusOut", parms);
				break;
			default:
				break;
		}
	}
	return KviScriptObject::eventFilter(o, e);
}

int KviScriptWidget::builtinFunction_SHOW(QPtrList<KviStr> *params, KviStr &buffer)
{
	bool bShowChildren = false;
	if( params ) {
		KviStr *pS = params->first();
		if( pS )
			bShowChildren = pS->toInt();
	}
	if( bShowChildren )
		recursiveShow();
	else
		m_pWidget->show();

	return KVI_ERROR_Success;
}

void KviScriptWidget::recursiveShow()
{
	m_pWidget->show();
	QPtrList<KviScriptObject> *l = childrenList();
	if( l ) {
		for( KviScriptObject *o = l->first(); o; o = l->next() ) {
			if( o->inherits("KviScriptWidget") )
				((KviScriptWidget *) o)->recursiveShow();
		}
	}
}

int KviScriptWidget::builtinFunction_LISTQTPROPERTIES(QPtrList<KviStr> *params, KviStr &buffer)
{
	KviWindow *w = controller()->mainFrame()->activeWindow();
	w->output(KVI_OUT_INTERNAL,
		__tr("%cListing Qt Properties for widget object %s (%s)"), KVI_TEXT_BOLD, getName(), id()
	);

	QMetaObject *o = m_pWidget->metaObject();
	do {
		w->output(KVI_OUT_INTERNAL, __tr("Properties class: %c%s%c"), KVI_TEXT_BOLD, o->className(), KVI_TEXT_BOLD);
		QStrList l = o->propertyNames(false);
		int idx = 0;
		for( char *c = l.first(); c; c = l.next() ) {
			const QMetaProperty *p = o->property(idx);
			if( p ) {
				KviStr tmp(KviStr::Format,
					__tr("Property: %c%s%c, type: %s"), KVI_TEXT_BOLD, p->name(), KVI_TEXT_BOLD, p->type()
				);
				if( p->isEnumType() ) {
					tmp.append(__tr(", enum ("));
					QStrList le = p->enumKeys();
					int i = 0;
					for( char *c2 = le.first(); c2; c2 = le.next() ) {
						if( i == 0 )
							i++;
						else
							tmp.append(", ");
						tmp.append(c2);
					}
					tmp.append(')');
				}
				if( p->isSetType() )
					tmp.append(__tr(", set"));
				if( p->writable() )
					tmp.append(__tr(", writeable"));
				w->output(KVI_OUT_INTERNAL, tmp.ptr());
			} else debug("Oops... no such property: %s", c);
			idx++;
		}
		o = o->superClass();
	} while( o );
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_QTPROPERTY(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			const QMetaProperty *p = m_pWidget->metaObject()->property(
				m_pWidget->metaObject()->findProperty(pS->ptr(), true), true
			);
			if( !p )
				return KVI_ERROR_NoSuchQtProperty;
			QVariant v = m_pWidget->property(pS->ptr());
			if( !v.isValid() )
				return KVI_ERROR_NoSuchQtProperty;
			if( p->isEnumType() ) {
				const char *c = p->valueToKey(v.toInt());
				buffer.append(c ? c : "");
			} else switch( v.type() ) {
				case QVariant::Int: {
					KviStr tmp(KviStr::Format, "%d", v.toInt());
					buffer.append(tmp.ptr());
					break;
				}
				case QVariant::Point: {
					QPoint p = v.toPoint();
					KviStr tmp(KviStr::Format, "%d,%d", p.x(), p.y());
					buffer.append(tmp.ptr());
					break;
				}
				case QVariant::Size: {
					QSize s = v.toSize();
					KviStr tmp(KviStr::Format, "%d,%d", s.width(), s.height());
					buffer.append(tmp.ptr());
					break;
				}
				case QVariant::Rect: {
					QRect r = v.toRect();
					KviStr tmp(KviStr::Format, "%d,%d,%d,%d", r.x(), r.y(), r.width(), r.height());
					buffer.append(tmp.ptr());
					break;
				}
				case QVariant::Color: {
					QColor clr = v.toColor();
					char buf[3];
					buf[0] = clr.red();
					buf[1] = clr.green();
					buf[2] = clr.blue();
					KviStr tmp;
					tmp.bufferToHex(buf, 3);
					buffer.append(tmp);
					break;
				}
				case QVariant::UInt: {
					KviStr tmp(KviStr::Format, "%u", v.toUInt());
					buffer.append(tmp.ptr());
					break;
				}
				case QVariant::Bool: {
					KviStr tmp(KviStr::Format, "%d", v.toInt());
					buffer.append(tmp.ptr());
					break;
				}
				case QVariant::String:
					buffer.append(v.toString());
					break;
				case QVariant::CString: {
					const char *c = v.toCString().data();
					if( c )
						buffer.append(c);
					break;
				}
				default:
					return KVI_ERROR_UnsupportedQtProperty;
					break;
			}
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptWidget::builtinFunction_SETQTPROPERTY(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		KviStr *pV = params->next();
		if( pS ) {
			const QMetaProperty *p = m_pWidget->metaObject()->property(
				m_pWidget->metaObject()->findProperty(pS->ptr(), true), true
			);
			if( !p )
				return KVI_ERROR_NoSuchQtProperty;
			KviStr type = p->type();
			if( p->isEnumType() ) {
				if( !pV )
					return KVI_ERROR_MissingParameter;
				int val = p->keyToValue(pV->ptr());
				QVariant v(val);
				m_pWidget->setProperty(pS->ptr(), v);
			} else if( kvi_strEqualCI("QString", type.ptr()) ) {
				QVariant v(_CHAR_2_QSTRING(pV ? pV->ptr() : ""));
				m_pWidget->setProperty(pS->ptr(), v);
			} else if( kvi_strEqualCI("QCString", type.ptr()) ) {
				QVariant v(QCString(pV ? pV->ptr() : ""));
				m_pWidget->setProperty(pS->ptr(), v);
			} else if( kvi_strEqualCI("int", type.ptr()) ) {
				if( !pV )
					return KVI_ERROR_MissingParameter;
				QVariant v(pV ? pV->toInt() : 0);
				m_pWidget->setProperty(pS->ptr(), v);
			} else if( kvi_strEqualCI("uint", type.ptr()) ) {
				if( !pV )
					return KVI_ERROR_MissingParameter;
				QVariant v(pV ? pV->toUInt() : 0);
				m_pWidget->setProperty(pS->ptr(), v);
			} else if( kvi_strEqualCI("bool", type.ptr()) ) {
				if( !pV )
					return KVI_ERROR_MissingParameter;
				QVariant v(pV ? pV->toInt() : 0, 0);
				m_pWidget->setProperty(pS->ptr(), v);
			} else if( kvi_strEqualCI("QPoint", type.ptr()) ) {
				if( !pV )
					return KVI_ERROR_MissingParameter;
				KviStr *pV2 = params->next();
				if( !pV2 )
					return KVI_ERROR_MissingParameter;
				QVariant v(QPoint(pV->toInt(), pV2->toInt()));
				m_pWidget->setProperty(pS->ptr(), v);
			} else if( kvi_strEqualCI("QSize", type.ptr()) ) {
				if( !pV )
					return KVI_ERROR_MissingParameter;
				KviStr *pV2 = params->next();
				if( !pV2 )
					return KVI_ERROR_MissingParameter;
				QVariant v(QSize(pV->toInt(), pV2->toInt()));
				m_pWidget->setProperty(pS->ptr(), v);
			} else if( kvi_strEqualCI("QRect", type.ptr()) ) {
				if( !pV )
					return KVI_ERROR_MissingParameter;
				KviStr *pV2 = params->next();
				if( !pV2 )
					return KVI_ERROR_MissingParameter;
				KviStr *pV3 = params->next();
				if( !pV3 )
					return KVI_ERROR_MissingParameter;
				KviStr *pV4 = params->next();
				if( !pV4 )
					return KVI_ERROR_MissingParameter;
				QVariant v(QRect(pV->toInt(), pV2->toInt(), pV3->toInt(), pV4->toInt()));
				m_pWidget->setProperty(pS->ptr(), v);
			} else if( kvi_strEqualCI("QColor", type.ptr()) ) {
				if( !pV )
					return KVI_ERROR_MissingParameter;
				char *buf = 0;
				int len = pS->hexToBuffer(&buf, false);
				if( len == 3 ) {
					QVariant v(QColor(((unsigned char) buf[0]), ((unsigned char) buf[1]), ((unsigned char) buf[2])));
					m_pWidget->setProperty(pS->ptr(), v);
					kvi_free(buf);
				} else return KVI_ERROR_InvalidParameter;
				if( len > 0 )
					kvi_free(buf);
			} else return KVI_ERROR_UnsupportedQtProperty;
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptWidget::builtinFunction_HIDE(QPtrList<KviStr> *, KviStr &buffer)
{
	m_pWidget->hide();
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_X(QPtrList<KviStr> *, KviStr &buffer)
{
	KviStr tmp(KviStr::Format, "%d", m_pWidget->x());
	buffer.append(tmp);
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_Y(QPtrList<KviStr> *, KviStr &buffer)
{
	KviStr tmp(KviStr::Format, "%d", m_pWidget->y());
	buffer.append(tmp);
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_WIDTH(QPtrList<KviStr> *, KviStr &buffer)
{
	KviStr tmp(KviStr::Format, "%d", m_pWidget->width());
	buffer.append(tmp);
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_HEIGHT(QPtrList<KviStr> *, KviStr &buffer)
{
	KviStr tmp(KviStr::Format, "%d", m_pWidget->height());
	buffer.append(tmp);
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_ISTOPLEVEL(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(m_pWidget->isTopLevel() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_ISVISIBLE(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(m_pWidget->isVisible() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_ISENABLED(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(m_pWidget->isEnabled() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_SETGEOMETRY(QPtrList<KviStr> *params, KviStr &buffer)
{
	int x = m_pWidget->x();
	int y = m_pWidget->y();
	int w = m_pWidget->width();
	int h = m_pWidget->height();
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			x  = pS->toInt();
			pS = params->next();
			if( pS ) {
				y  = pS->toInt();
				pS = params->next();
				if( pS ) {
					w  = pS->toInt();
					pS = params->next();
					if( pS )
						h = pS->toInt();
				}
			}
		}
		m_pWidget->setGeometry(x, y, w, h);
		buffer.append('1');
	} else buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_SETENABLED(QPtrList<KviStr> *params, KviStr &buffer)
{
	bool bEn = true;
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			int tmp = pS->toInt();
			if( tmp == 0 )
				bEn = false;
		}
		m_pWidget->setEnabled(bEn);
	}
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_SETMINIMUMHEIGHT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			int tmp = pS->toInt();
			m_pWidget->setMinimumHeight(tmp);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptWidget::builtinFunction_SETMINIMUMWIDTH(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			int tmp = pS->toInt();
			m_pWidget->setMinimumWidth(tmp);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptWidget::builtinFunction_PARENTWIDGET(QPtrList<KviStr> *, KviStr &buffer)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			buffer.append(((KviScriptObject *) parent())->id());
			return KVI_ERROR_Success;
		}
	}
	buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_SETCAPTION(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS )
			m_pWidget->setCaption(pS->ptr());
	}
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_CAPTION(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(m_pWidget->caption());
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_RAISE(QPtrList<KviStr> *, KviStr &buffer)
{
	m_pWidget->raise();
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_LOWER(QPtrList<KviStr> *, KviStr &buffer)
{
	m_pWidget->lower();
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_SETFOCUS(QPtrList<KviStr> *, KviStr &buffer)
{
	m_pWidget->setFocus();
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_HASFOCUS(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(m_pWidget->hasFocus() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_SETBACKGROUNDCOLOR(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			char *buf = 0;
			int len   = pS->hexToBuffer(&buf, false);
			if( len == 3 ) {
				m_pWidget->setBackgroundColor(
					QColor(((unsigned char) buf[0]), ((unsigned char) buf[1]), ((unsigned char) buf[2]))
				);
				buffer.append('1');
				kvi_free(buf);
				return KVI_ERROR_Success;
			}
			if( len > 0 )
				kvi_free(buf);
		}
	}
	buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_BACKGROUNDCOLOR(QPtrList<KviStr> *params, KviStr &buffer)
{
	QColor clr = m_pWidget->backgroundColor();
	char buf[3];
	buf[0] = clr.red();
	buf[1] = clr.green();
	buf[2] = clr.blue();
	KviStr tmp;
	tmp.bufferToHex(buf, 3);
	buffer.append(tmp);
	return KVI_ERROR_Success;
}

#include "m_kvi_script_widget.moc"
