// =============================================================================
//
//      --- kvi_biff_socket.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Authors:
//       Krzysztof Godlewski    <kristoff@poczta.wprost.pl>
//       Szymon Stefanek        <stefanek@tin.it>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviBiffSocket"

#include <sys/socket.h>
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>

#include <qsocketnotifier.h>

#define __KVIRC_PLUGIN__

#include "kvi_biff.h"
#include "kvi_biff_mailbox.h"
#include "kvi_biff_message.h"
#include "kvi_biff_socket.h"
#include "kvi_locale.h"
#include "kvi_netutils.h"
#include "kvirc_plugin.h"

extern KviBiff *g_pBiff;

KviBiffSocket::KviBiffSocket()
	: QObject(0)
{
	m_fd          = -1;
	m_pNotifier   = 0;
	m_szHostIp    = "127.0.0.1";
	m_uPort       = 110;
	m_pDNS        = 0;
	m_lastCommand = BIFF_NULL_STRING;
}

KviBiffSocket::~KviBiffSocket()
{
	if( m_pDNS )      { delete m_pDNS;      m_pDNS      = 0; }
	if( m_pNotifier ) { delete m_pNotifier; m_pNotifier = 0; }
	if( m_fd != -1 ) close(m_fd);
}

const char *KviBiffSocket::hostname() const
{
	return m_szHostname.ptr();
}

const char *KviBiffSocket::hostIp() const
{
	return m_szHostIp.ptr();
}

unsigned short int KviBiffSocket::port() const
{
	return m_uPort;
}

bool KviBiffSocket::run(KviBiffMailbox *box)
{
	if( m_pDNS || (m_fd != -1) )
	    return false; // Already running

	m_pMailbox   = box; // DO NOT USE THIS POINTER!... it may be deleted from outside
	m_uPort      = box->port();
	m_szHostname = box->hostname();
	m_szUsername = box->username();
	m_szPassword = box->password();

	m_pDNS = new KviDns();
	connect(m_pDNS, SIGNAL(finished(KviDnsData *)), this, SLOT(dnsFinished(KviDnsData *)));
	emit resolving();
	m_pDNS->resolve(box->hostname());

	return true;
}

int KviBiffSocket::connectToHost(const char *addr)
{
	m_szHostIp = addr;
	KviStr s;
	s.sprintf("Host resolved to %s", addr);
	g_pBiff->systrayMsg(s.ptr());

	struct sockaddr_in serv_sin;
	serv_sin.sin_family = AF_INET;
	serv_sin.sin_port   = htons(m_uPort);

	if( !kvi_stringIpToBinaryIp(m_szHostIp.ptr(), &(serv_sin.sin_addr)) ) {
	    emit error(__tr("Internal error"));
	    return -1;
	}

	if( (m_fd = socket(PF_INET, SOCK_STREAM,IPPROTO_TCP)) < 0 ) {
	    emit error(__tr("Socket creation failure"));
	    return -1;
	}

	if( fcntl(m_fd, F_SETFL, O_NONBLOCK) < 0 ) {
	    close(m_fd);
	    m_fd = -1;
	    emit error(__tr("Internal error: fcntl()"));
	    return -1;
	}

	if( ::connect(m_fd, (struct sockaddr *) &serv_sin, sizeof(serv_sin)) < 0 ) {
	    if( errno != EINPROGRESS ) {
	        close(m_fd);
	        m_fd = -1;
			// TODO: Maybe a more verbose error code?
			emit error(__tr("Connect failed"));
	        return -1;
	    }
	}
	return 0;
}

void KviBiffSocket::finished(KviDnsData *dns)
{
	if( dns->iError != KVI_ERROR_Success ) {
	    KviStr tmp(KviStr::Format, __tr("DNS failure: %s"), kvi_getErrorString(dns->iError));
	    delete m_pDNS;
	    m_pDNS = 0;
	    emit error(tmp.ptr());
	    return;
	}
	// Hostname resolved
	int result = -1;
	KviIpAddresses::Iterator it = dns->addresses.begin();
	for( ; it != dns->addresses.end(); ++it ) {
		QHostAddress addr = dns->addresses.first();
		if( !addr.isNull() )
			result = connectToHost(addr.toString().ascii());
		if( result == 0 )
			break;
	}
	delete m_pDNS;
	m_pDNS = 0;
	if( result != 0 )
		return;

	m_pNotifier = new QSocketNotifier(m_fd, QSocketNotifier::Write);
	QObject::connect(m_pNotifier, SIGNAL(activated(int)), this, SLOT(writeNotifierFired(int)));
	m_pNotifier->setEnabled(true);

	KviStr m;
	m.sprintf("Connecting to %s", m_pMailbox->hostname());
	g_pBiff->systrayMsg(m.ptr());
	// TODO: Add a timer with a timeout here?
	// TODO: Emit a message saying "connecting..."
}

void KviBiffSocket::writeNotifierFired(int)
{
	// Step 3: connected... remove the write notifier
	//         setup the read one and go on
	//         with the pop3 protocol.
	//         read data in readNotifierFired() and write
	//         to the socket accordingly.
	delete m_pNotifier;
	m_pNotifier = 0;
	int sockError;
	socklen_t iSize = sizeof(int);
	if( getsockopt(m_fd, SOL_SOCKET, SO_ERROR, (void *) &sockError, &iSize) < 0 )
	    sockError = -1;
	if( sockError != 0 ) {
	    close(m_fd);
	    m_fd = -1;
	    emit error(strerror(sockError));
	    return;
	}
	m_pNotifier = new QSocketNotifier(m_fd, QSocketNotifier::Read);
#if 0 // TODO: fix the readNotifier() call
	QObject::connect(m_pNotifier, SIGNAL(activated(int)), this, SLOT(readNotifierFired(int)));
	m_pNotifier->setEnabled(true);
#endif

	emit connected();
	// TODO: Probably will need to keep track of the state here

	char buffer[1024];
	int readLength = 0;

	// TODO: should just move this into a function.
	//
	// I am lazy :) A macro to make my life easier.
	// Do some reading / writing and some checks.
	// Its argument MUST be KviStr, the second one is the output buffer.
	// There is some delay between writing to the socket and reading from it. It
	// is just right for connections to the local server, so in the Net it can be
	// smaller (lagzz). On the other hand, when the communication is too fast the
	// plugin will segfault! I do not know how to prevent this... I can only hope
	// that when readNotifierFired() starts to work fine everything will be ok.
	//
	// The "for" loop is here to make this whole thing not block the GUI.
#define BIFF_SEND_CMD(x, y) \
write(m_fd, x.ptr(), x.len()); \
for( int i = 0; i < 20; i ++ ) {\
	usleep(15000); \
	qApp->processOneEvent(); \
} \
memset((y), '\0', sizeof((y))); \
readLength = read(m_fd, (y), sizeof((y))); \
if( readLength <= 0 ) {\
	close(m_fd); \
	m_fd = -1; \
	emit error(__tr("Disconnected")); \
	return; \
} \
if( y[0] == '-' ) {\
	KviStr s; \
	KviStr _s = KviStr((y)).cutToFirst(' '); \
	int _idx = _s.findFirstIdx('\r'); \
	if( _idx != -1 ) _s.cut(_idx, 1); \
	s.sprintf(__tr("Error in command %s, server replied: %s" ), m_lastCommand.ptr(), _s.ptr()); \
	write(m_fd, "QUIT\r\n", 6); \
	emit error(s.ptr()); \
	return; \
}

	// USER
	KviStr tmp(KviStr::Format, "USER %s\r\n", m_szUsername.ptr());
	m_lastCommand = "USER";
	BIFF_SEND_CMD(tmp, buffer)

	// PASS
	tmp.sprintf("PASS %s\r\n", m_szPassword.ptr());
	m_lastCommand = "PASS";
	BIFF_SEND_CMD(tmp, buffer)

	emit loggedIn();

	// STAT
	tmp.sprintf("STAT\r\n");
	m_lastCommand = "STAT";
	BIFF_SEND_CMD(tmp, buffer)
	KviStr numMsg(buffer);
	tmp = numMsg.middle(strlen("+OK "), 3);
	uint num_msg = tmp.cutFromFirst(' ').toUInt();

	// UIDL
	tmp.sprintf("UIDL\r\n");
	BIFF_SEND_CMD(tmp, buffer);
	tmp = buffer;
	KviStr uids(tmp.cutToFirst('1', false));

	// TODO: Find a decent way to know new messages number. It cannot be done simply
	// by something like newMsgCount = num_msg - m_pMailbox->messageCount(), because
	// usually after you check your account with biff you download your mail, and it
	// is deleted from server, so m_pMailbox->messageList()->clear() would have to
	// be called at every mail check. But this does not solve the problem because not
	// all mail has to be retrieved from the server, and for example there can be:
	// * on first check 10 msgs
	// * on second one 11 msgs - one new? This might be not true, because user
	// could have gotten 11 mails between two checks, retrieving the 10 mentioned
	// earlier!
	// I thought about keeping 2 QPtrList<KviBiffMessage> objects. One would be
	// currentMail list, and the other one lastMailList. Then we could compare the
	// messages uid's and I think this would work. But maybe there is another, less
	// complicated way? Maybe something with the pop3 protocol (message status?). Any
	// ideas?

	for( uint i = 0; i < num_msg; i++ ) {
		// Find the message's unique id
	    uids.getLine(tmp);                   // Get the first line of the uids
	    KviStr uid = tmp.cutToFirst(' ');    // Remove the message number
	    int idx = uid.findFirstIdx('\r');
	    if( idx != -1 ) uid.cut(idx, 1);

	    tmp.sprintf("TOP %u 0\r\n", i + 1);
	    BIFF_SEND_CMD(tmp, buffer)
	    tmp = buffer;
	    KviStr from(tmp.middle(tmp.findFirstIdx("From: "), 100));
	    from.getLine(tmp);
	    from = tmp.middle(strlen("From: "), 100);
	    from.findFirstIdx('\r');
	    if( idx != -1 ) from.cut(idx, 1);
	    tmp = buffer;
	    KviStr subject(tmp.middle(tmp.findFirstIdx("Subject: "), 100));
	    subject.getLine(tmp);
	    subject = tmp.middle(strlen("Subject: "), 100);
	    idx = subject.findFirstIdx('\r');
	    if( idx != -1 ) subject.cut(idx, 1);

	    if( !m_pMailbox->findMessageByUid(uid.ptr()) ) {
			// The message was deleted from the account
	        KviBiffMessage *m = new KviBiffMessage(from.ptr(), subject.ptr(), uid.ptr());
	        m_pMailbox->messageList()->append(m);
	    }
	}

	tmp.sprintf("QUIT\r\n");
	m_lastCommand = "QUIT";
	BIFF_SEND_CMD(tmp, buffer)

	emit jobDone();
}

void KviBiffSocket::readNotifierFired(int fd)
{
	// TODO: Still cannot get the readNotifier to work!
	//
	// Sometimes all I get is "Server ready message", sometimes the output reaches
	// LIST command... strange.. I have only tested in on my machine (popd), do not know
	// how it will act in the Net...
	// Maybe the commands are being sent to fast and the server does not reply
	// because of the flood? Can it be possible?
	//
	// Ok, so I have added usleep() and it helped. The problem is that read notifier
	// is getting fired not after every command sent to server, but after some time
	// This is strange, but I have no idea what the problem is, since I do not know
	// too much about sockets and stuff :)) Guess I will just have to keep on
	// experimenting with this.
	//
	// Right. For now I will do it without the read notifier. Maybe Pragma will
	// be able to fix this. I will do it the simplest way it can be done :)))
	//

	// All the communication work is done here
	char buffer[1025];
	int readLength = read(m_fd,buffer, sizeof(buffer));
	if( readLength <= 0 ) {
	    close(m_fd);
	    m_fd = -1;
	    emit error(__tr("Disconnected"));
	    return;
	}

	buffer[readLength - 1] = '\0';    // readLength - 1 is a '\n' char - remove it

	// TODO: Need to keep track of the state and chat with the server consequently
	if( buffer[0] == '-' ) { // "-ERROR" from server
	    KviStr s;
	    s.sprintf(__tr("Error in command %s (server replied: %s"), m_lastCommand.ptr(), buffer);
	    emit error(s.ptr());
	    return;
	}

	// TODO: Just for testing
	// DO NOT REFERENCE this AFTER THIS SIGNAL HAS BEEN EMITTED!
	emit jobDone();
}

#include "m_kvi_biff_socket.moc"
