// =============================================================================
//
//      --- kvi_stat_window.cpp ---
//
//   This is a plugin for KVIrc 2. It allows you to see
//   statistics of your IRC sessions - total words said etc.
//   More features are still to come.
//
//   (c) Krzysztof Godlewski <kristoff@poczta.wprost.pl>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviStatWindow"

#include <qvbox.h>

#include "kvi_label.h"
#include "kvi_listview.h"
#include "kvi_locale.h"
#include "kvi_messagebox.h"
#include "kvi_pushbutton.h"
#include "kvi_stat_chan.h"
#include "kvi_stat_controller.h"
#include "kvi_stat_window.h"
#include "kvi_string.h"

extern KviStatController *g_pStatPluginController;

KviStatWindow::KviStatWindow()
	: KviTabDialog(0, "KviStatWindow", false)
{
	QString cap;
	cap.sprintf(__tr("Stats since %s"), g_pStatPluginController->statData().startDate.ptr());
	setCaption(cap);
	labels.setAutoDelete(true);

	QVBox *vb = addVBoxPage(_CHAR_2_QSTRING(__tr("Words/Letters")));
	vb->setMargin(5);
	vb->resize(290, 270);

	setFixedSize(vb->width() + 20, vb->height() + 20);

	int lab_count = 0;

	// Words / letters tab
	for( ; lab_count < 8; lab_count++ ) {
		labels.append(new KviLabel(vb));
		labels.at(lab_count)->setTextFormat(PlainText);
	}

	// Chanlist
	vb = addVBoxPage(_CHAR_2_QSTRING(__tr("Channels")));
	vb->setMargin(5);
	m_chansView = new KviListView(vb);

	const char *columns[] =
	{
		__tr("Name"),
		__tr("Joins"),
		__tr("Words"),
		__tr("Kicks"),
		__tr("Bans"),
		__tr("Topics")
	};

#define Col_Name       0
#define Col_Joins      1
#define Col_Words      2
#define Col_Kicks      3
#define Col_Bans       4
#define Col_Topics     5

	for( uint i = 0; i < sizeof(columns) / sizeof(columns[0]); i++ )
		m_chansView->addColumn(_CHAR_2_QSTRING(columns[i]));

	m_chansView->setColumnWidth(Col_Name, 62);

	QHBox *hb = new QHBox(vb);
	hb->setMargin(5);
	hb->setSpacing(10);

	KviPushButton *b = new KviPushButton(_CHAR_2_QSTRING(__tr("Remove channel")), hb);
	connect(b, SIGNAL(clicked()), SLOT(slotRemoveChan()));

	b = new KviPushButton(_CHAR_2_QSTRING(__tr("Remove all")), hb);
	connect(b, SIGNAL(clicked()), g_pStatPluginController, SLOT(slotRemoveAllChans()));

	// Misc
	vb = addVBoxPage(_CHAR_2_QSTRING(__tr("Misc")));
	vb->setMargin(5);

	for( ; lab_count < 14; lab_count++ ) {
		labels.append(new KviLabel(vb));
		labels.at(lab_count)->setTextFormat(PlainText);
	}

	setButtonDefaultText(_CHAR_2_QSTRING(__tr("Reset")));
	connect(this, SIGNAL(defaultButtonPressed()), g_pStatPluginController, SLOT(slotReset()));

	updateStats();
}

KviStatWindow::~KviStatWindow()
{
	// Nothing here
}

void KviStatWindow::updateStats()
{
	const KviStatData sd = g_pStatPluginController->statData();

	// TODO: rework this?
	//    Ugly... but only this way can I make Qt display iso8859-2 (and others) correctly I think.
	//    s.local8Bit() does not seem to work...

	// Words/Letters
	QString s;
	QString n;
	s = _CHAR_2_QSTRING(__tr("Total Words:                 "));
	n.sprintf("%i.", sd.totalWords);
	labels.at(0)->setText(s + n);
	s = _CHAR_2_QSTRING(__tr("Total Letters:               "));
	n.sprintf("%i.", sd.totalLetters);
	labels.at(1)->setText(s + n);
	s = _CHAR_2_QSTRING(__tr("Channels Words:              "));
	n.sprintf("%i.", sd.chansWords);
	labels.at(2)->setText(s + n);
	s = _CHAR_2_QSTRING(__tr("Channels Letters:            "));
	n.sprintf("%i.", sd.chansLetters);
	labels.at(3)->setText(s + n);
	s = _CHAR_2_QSTRING(__tr("Queries Words:               "));
	n.sprintf("%i.", sd.queriesWords);
	labels.at(4)->setText(s + n);
	s = _CHAR_2_QSTRING(__tr("Queries Letters:             "));
	n.sprintf("%i.", sd.queriesLetters);
	labels.at(5)->setText(s + n);
	s = _CHAR_2_QSTRING(__tr("DCCs Words:                  "));
	n.sprintf("%i.", sd.dccsWords);
	labels.at(6)->setText(s + n);
	s = _CHAR_2_QSTRING(__tr("DCCs Letters:                "));
	n.sprintf("%i.", sd.dccsLetters);
	labels.at(7)->setText(s + n);

	// Misc
	s = _CHAR_2_QSTRING(__tr("Number of IRC sessions:      "));
	n.sprintf("%i.", sd.onIrc);
	labels.at(8)->setText(s + n);
	s = _CHAR_2_QSTRING(__tr("Words Recorded:              "));
	n.sprintf("%i.", sd.wordsRecord);
	labels.at(9)->setText(s + n);
	s = _CHAR_2_QSTRING(__tr("Number of kicks given:       "));
	n.sprintf("%i.", sd.kicks);
	labels.at(10)->setText(s + n);
	s = _CHAR_2_QSTRING(__tr("Number of bans set:          "));
	n.sprintf("%i.", sd.bans);
	labels.at(11)->setText(s + n);
	s = _CHAR_2_QSTRING(__tr("Number of joins to channels: "));
	n.sprintf("%i.", sd.joins);
	labels.at(12)->setText(s + n);
	s = _CHAR_2_QSTRING(__tr("Number of topic changes:     "));
	n.sprintf("%i.", sd.topics);
	labels.at(13)->setText(s + n);

	// Channels
	m_chansView->clear();
	for( const KviStatChan *c = g_pStatPluginController->chanList()->first();
		 c;
		 c = g_pStatPluginController->chanList()->next() )
	{
		KviListViewItem *vi = new KviListViewItem(m_chansView);
		vi->setText(Col_Name,   c->name());
		vi->setText(Col_Words,  s.setNum(c->words()));
		vi->setText(Col_Joins,  s.setNum(c->joins()));
		vi->setText(Col_Kicks,  s.setNum(c->kicks()));
		vi->setText(Col_Bans,   s.setNum(c->bans()));
		vi->setText(Col_Topics, s.setNum(c->topics()));
		m_chansView->insertItem(vi);
	}

	m_chansView->setSorting(Col_Words);
}

void KviStatWindow::slotRemoveChan()
{
	if( !m_chansView->selectedItem() ) return;

	KviStr cn = m_chansView->selectedItem()->text(Col_Name).latin1();
	if( KviMessageBox::questionYesNo(
		_CHAR_2_QSTRING(__tr("Are you sure you want to remove this channel from your statistics?")),
		_CHAR_2_QSTRING(__tr("Remove Channel"))) == KviMessageBox::Yes
	) {
		KviStatChan *chan = g_pStatPluginController->findStatChan(cn.ptr());
		m_chansView->removeItem(m_chansView->selectedItem());
		g_pStatPluginController->removeChan(chan);
	}
}

#include "m_kvi_stat_window.moc"
