//=============================================================================
//
//   File : kvi_kvs_coresimplecommands_mr.cpp
//   Created on Fri 31 Oct 2003 00:04:25 by Szymon Stefanek
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Szymon Stefanek <pragma at kvirc dot net>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_kvs_coresimplecommands.h"

#include "kvi_window.h"
#include "kvi_out.h"
#include "kvi_locale.h"
#include "kvi_app.h"
#include "kvi_options.h"
#include "kvi_fileutils.h"
#include "kvi_filedialog.h"

#include "kvi_kvs_variantlist.h"
#include "kvi_kvs_script.h"

#include "kvi_kvs_popupmanager.h"

#include <qcursor.h>

namespace KviKvsCoreSimpleCommands
{
	/*
		@doc: parse
		@type:
			command
		@title:
			parse
		@syntax:
			parse [-q] [-e] [-f] [-r] <filename> [parameters]
		@short:
			Executes commands from a file
		@switches:
			!sw: -e | --fail-on-load
			Causes the command to terminate with an error when the file
			can't be loaded: the error termination stops the parsing
			of the current script too. If the -e switch is not used then
			file load failures are eventually reported only as a warning
			or not reported at all (if -q is used).
			!sw: -f | --fail-on-error
			Causes the command to terminate with an error when the execution
			of the script loaded from disk generates an error.
			This error termination stops the parsing of the current script too.
			If the -f switch is not used then the loaded script errors
			do not stop the current script.
			!sw: -q | --quiet
			Causes the command to run quietly without printing warnings about
			empty <filename> and file loading errors.
			!sw: -r | --propagate-return
			Causes the return value of the script to be propagated to the
			calling context. This allows the usage of ${ } trick to extract
			this return value. See the examples section for a sample usage.
			
		@description:
			Executes commands from the external file <filename>.[br]
			<filename> can be an absolute or relative path.[br]
			[parameters] is a space separated string of parameters to be
			passed to the script. The parameter $0 will contain the
			name of the file being parsed, the other parameters will
			be available starting from $1.[br]
			If <filename> is an existing directory name a file dialog 
			will pop up, allowing you to choose a file to parse.
			The file dialog will start in the directory specified by <filename>.
			If you don't know the name of an existing directory, just pass "."
			or "/". The command just prints a warning if the file cannot be loaded for
			some reason. If you want the command to stop with an error in
			this case then use the -e switch.[br]
		@examples:
			[example]
				parse /home/pragma/myscript.kvs
			[/example]
			[example]
				[comment]# Here we are assuming that fetchdata.kvs returns a string[/comment]
				[comment]# We evaluate the return value thru ${} and echo it[/comment]
				echo ${ parse -r /home/pragma/fetchdata.kvs; };
			[/example]
	*/
	
	_KVS_CORESIMPLECOMMAND(parse)
	{
		QString szFileName;
		pParams->firstAsString(szFileName);

		KviFileUtils::adjustFilePath(szFileName);

		if(szFileName.isEmpty())
		{
			if(!pSwitches->find('q',"quiet"))c->warning(__tr2qs("Missing file name to parse"));
			return true;
		}

		if(KviFileUtils::directoryExists(szFileName))
		{
			// FIXME: Throw away KviStr!!!
			KviStr szTmp;
			
			c->enterBlockingSection();
			
			bool bResult = KviFileDialog::askForOpenFileName(szTmp,__tr2qs("Choose a file to parse"),szFileName.latin1(),"*.kvs");

			if(!c->leaveBlockingSection())return false; // need to stop immediately
			if(!bResult)return true;

			szFileName = szTmp;

			if(szFileName.isEmpty())return true; // done
			KviFileUtils::adjustFilePath(szFileName);
		}

		QString * pszBuffer = new QString;
		if(!KviFileUtils::loadFile(szFileName,*pszBuffer,true))
		{
			delete pszBuffer;
            pszBuffer = 0;
			if(pSwitches->find('e',"fail-on-load"))
			{
				c->error(__tr2qs("Failed to load the file '%Q' for parsing"),&szFileName);
				return false;
			} else {
				if(!pSwitches->find('q',"quiet"))c->warning(__tr2qs("Failed to load the file '%Q' for parsing"),&szFileName);
				return true;
			}
		}

		KviKvsScript s(szFileName,pszBuffer);

		KviKvsVariant vFileName(szFileName);
		KviKvsVariantList parms(&vFileName);
		for(KviKvsVariant * v = pParams->next();v;v = pParams->next())parms.append(v);
		parms.setAutoDelete(false);

		KviKvsVariant * pRetVal = pSwitches->find('r',"propagate-return") ? c->returnValue() : 0;

		if(!s.run(c->window(),&parms,pRetVal,KviKvsScript::PreserveParams))
		{
			if(pSwitches->find('f',"fail-on-error"))return false;
		}

		return true;
	}
	
	//////////////////////////////////////////////////////////////////////////////////////////////////////////////
	
	/*
		@doc: popup
		@type:
			command
		@title:
			popup
		@syntax:
			popup [-p=<screen_coordinates>] <popup_name> [parameters]
		@short:
			Shows a popup menu
		@switches:
			!sw: -p=<screen_coordinates> | --point=<screen_coordinates>
		@description:
			Shows the popup menu <popup_name> at the current cursor position,
			eventually passing the [parameters]. Please note that you can't
			use this command inside one of the [cmd]defpopup[/cmd] command
			handlers for <popup_name>. In other words, you can't "popup
			a popup" while it is being popped up. :) (This would
			be an endless recursive behaviour).[br]
			If the -p switch is used , then <screen_coordinates> must be
			in the form <x>,<y> and will be used as screen coordinates
			for the placement of the popup (instead of the current cursor position).[br]
		@seealso:
			[cmd]defpopup[/cmd]
	*/
	
	_KVS_CORESIMPLECOMMAND(popup)
	{
		QString szPopupName;
		pParams->firstAsString(szPopupName);
		
		if(szPopupName.isEmpty())
		{
			c->error(__tr2qs("A popup name is expected as parameter"));
			return false;
		}

		// copy parameters
		KviKvsVariantList * pPopupParams = new KviKvsVariantList();
		while(KviKvsVariant * v = pParams->next())pPopupParams->append(new KviKvsVariant(*v));

		KviKvsPopupMenu * pMenu = KviKvsPopupManager::instance()->lookup(szPopupName);
		
		if(!pMenu)
		{
			delete pPopupParams;
            pPopupParams = 0;
			c->error(__tr2qs("Popup %Q is not defined"),&szPopupName);
			return false;
		}

		if(pMenu->isLocked())
		{
			delete pPopupParams;
            pPopupParams = 0;
			c->error(__tr2qs("A popup menu cannot be popped up twice"));
			return false;
		}

		QPoint pnt = QCursor::pos();

		KviKvsVariant * pCoords = pSwitches->find('p',"point");
		if(pCoords)
		{
			QString szCoords;
			pCoords->asString(szCoords);

			int idx = szCoords.find(',');
			bool bCoordsOk = true;
			if(idx == -1)bCoordsOk = false;
			else {
				QString szX = szCoords.left(idx);
				szCoords.remove(0,idx + 1);
				bool bOk1,bOk2;
				int iX = szX.toInt(&bOk1);
				int iY = szCoords.toInt(&bOk2);
				if(bOk1 && bOk2)pnt = QPoint(iX,iY);
				else bCoordsOk = false;
			}
			
			if(!bCoordsOk)c->warning(__tr2qs("Invalid syntax for screen coordinates, using cursor position"));
		}

		pMenu->doPopup(pnt,c->window(),pPopupParams);
		return true;
	}

	//////////////////////////////////////////////////////////////////////////////////////////////////////////////
	
	/*
		@doc: rebind
		@type:
			command
		@title:
			rebind:
		@syntax:
			rebind [-q] <window_id>
		@short:
			Rebinds a command sequence to a specified window
		@switches:
			!sw: -q | --quiet
			Causes the command to run quietly, print no warnings
			and don't fail when the window identifier passed is empty.
		@description:
			Rebinds the current command sequence to the window specified by <window_id>.[br]
			The new window will be inherited by all the subcommands and aliases called.[br]
			-q causes the command to run quietly.[br]
			[b]Handle with care.[/b]
		@examples:
			[comment]# Try this example in a channel or query window[/comment]
			[comment]# Remember the current window id[/comment]
			%winid = $window
			[comment]# Rebind to the console of the current irc context[/comment]
			rebind $console
			echo "Hello from the console :)"
			echo "Hello again.. still in the console"
			[comment]# Rebind back[/comment]
			rebind %winid
			echo "Again in this window :)"
	*/
	
	_KVS_CORESIMPLECOMMAND(rebind)
	{
		QString szWinId;
		pParams->firstAsString(szWinId);
		
		if(szWinId.isEmpty())
		{
			if(!pSwitches->find('q',"quiet"))
			{
				c->error(__tr2qs("Can't rebind the command sequence: missing window identifier"));
				return false;
			}
		}

		KviWindow * pAux = g_pApp->findWindow(szWinId.latin1());
		if(pAux)c->setWindow(pAux);
		else {
			if(!pSwitches->find('q',"quiet"))
				c->warning(__tr2qs("Window with id %Q not found: no rebinding performed"),&szWinId);
		}

		return true;
	}


	//////////////////////////////////////////////////////////////////////////////////////////////////////////////
	
	/*
		@doc: return
		@type:
			command
		@title:
			return
		@syntax:
			return <string>
		@short:
			Returns from the current command sequence and sets it return value
		@description:
			Sets the return value of the current command sequence to <string>
			and stops the execution.[br]
			This is more or less equivalent to calling [cmd]setreturn[/cmd] <string>
			and then [cmd]halt[/cmd], but has no additional semantics in events.[br]
			Starting from version 3.0.0 of kvirc you can also return
			arrays and hashes just like any other variable types.
		@examples:
			return $array(item1,item2,3213,itemX);
		@seealso:
			[cmd]setreturn[/cmd], [cmd]break[/cmd], [cmd]halt[/cmd]
	*/
	
	_KVS_CORESIMPLECOMMAND(return_CKEYWORDWORKAROUND)
	{
		if(pParams->count() == 0)
		{
			c->returnValue()->setNothing();
			return false;
		}
		if(pParams->count() == 1)
		{
			c->returnValue()->copyFrom(*(pParams->first()));
			return false;
		}
	
		QString all;
		pParams->allAsString(all);
		c->returnValue()->setString(all);
		return false;
	}


};

