#ifndef _KVI_KVS_MODULEINTERFACE_H_
#define _KVI_KVS_MODULEINTERFACE_H_
//=============================================================================
//
//   File : kvi_kvs_moduleinterface.h
//   Created on Tue 16 Dec 2003 00:27:54 by Szymon Stefanek
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 2003 Szymon Stefanek <pragma at kvirc dot net>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================


#include "kvi_settings.h"

#include <qdict.h>

#include "kvi_kvs_runtimecontext.h"

class KviModule;


class KviKvsModuleRunTimeCall
{
protected:
	KviModule             * m_pModule;
	KviKvsRunTimeContext  * m_pContext;
	KviKvsVariantList     * m_pParams;
public:
	KviKvsModuleRunTimeCall(KviModule * pModule,
					KviKvsRunTimeContext * pContext,
					KviKvsVariantList * pParams)
		: m_pModule(pModule),m_pContext(pContext),m_pParams(pParams){};
	~KviKvsModuleRunTimeCall(){};				
public:
	// the script runtime context
	// this is zero only in event calls
	KviKvsRunTimeContext * context(){ return m_pContext; };
	
	KviWindow * window(){ return m_pContext->window(); };

	KviModule * module(){ return m_pModule; };
	
	// parameter list access & helpers
	KviKvsVariantList * parameterList(){ return m_pParams; };
	KviKvsVariantList * params(){ return m_pParams; }; // the short version
	unsigned int paramCount(){ return m_pParams->count(); };
	unsigned int parameterCount(){ return m_pParams->count(); };
	KviKvsVariant * firstParam(){ return m_pParams->first(); };
	KviKvsVariant * nextParam(){ return m_pParams->next(); };
};

class KviKvsModuleEventCall : public KviKvsModuleRunTimeCall
{
public:
	KviKvsModuleEventCall(KviModule * pModule,
					KviKvsVariantList * pParams)
		: KviKvsModuleRunTimeCall(pModule,0,pParams){};
	~KviKvsModuleEventCall(){};
};

class KviKvsModuleCommandCall : public KviKvsModuleRunTimeCall
{
protected:
	KviKvsSwitchList * m_pSwitchList;
public:
	KviKvsModuleCommandCall(KviModule * pModule,
					KviKvsRunTimeContext * pContext,
					KviKvsVariantList * pParams,
					KviKvsSwitchList * pSwitches)
		: KviKvsModuleRunTimeCall(pModule,pContext,pParams), m_pSwitchList(pSwitches){};
	~KviKvsModuleCommandCall(){};
public:
	KviKvsSwitchList * switches(){ return m_pSwitchList; };
	KviKvsSwitchList * switchList(){ return m_pSwitchList; };
};

class KviKvsModuleCallbackCommandCall : public KviKvsModuleCommandCall
{
protected:
	const KviKvsScript * m_pCallback;
public:
	KviKvsModuleCallbackCommandCall(KviModule * pModule,
					KviKvsRunTimeContext * pContext,
					KviKvsVariantList * pParams,
					KviKvsSwitchList * pSwitches,
					const KviKvsScript * pCallback)
		: KviKvsModuleCommandCall(pModule,pContext,pParams,pSwitches), m_pCallback(pCallback){};
	~KviKvsModuleCallbackCommandCall(){};
public:
	const KviKvsScript * callback(){ return m_pCallback; };
};


class KviKvsModuleFunctionCall : public KviKvsModuleRunTimeCall
{
	friend class KviKvsTreeNodeModuleFunctionCall;
protected:
	KviKvsVariant * m_pResult;
protected:
	KviKvsModuleFunctionCall(KviModule * pModule,
					KviKvsRunTimeContext * pContext,
					KviKvsVariantList * pParams,
					KviKvsVariant * pResult)
		: KviKvsModuleRunTimeCall(pModule,pContext,pParams), m_pResult(pResult)
		{};
	~KviKvsModuleFunctionCall(){};
public:
	void setResult(const QString &szValue){ m_pResult->setString(szValue); };
	void setResult(int iValue){ m_pResult->setInteger(iValue); };
	void setResult(double dValue){ m_pResult->setReal(dValue); };
	void setNullResult(){ m_pResult->setNothing(); };
	KviKvsVariant * result(){ return m_pResult; };
};



typedef bool (*KviKvsModuleSimpleCommandExecRoutine)(KviKvsModuleCommandCall * c);
typedef bool (*KviKvsModuleFunctionExecRoutine)(KviKvsModuleFunctionCall * c);
typedef bool (*KviKvsModuleCallbackCommandExecRoutine)(KviKvsModuleCallbackCommandCall * c);
typedef bool (*KviKvsModuleEventHandlerRoutine)(KviKvsModuleEventCall * c);



class KVIRC_API KviKvsModuleInterface
{
public:
	KviKvsModuleInterface();
	~KviKvsModuleInterface();
protected:
	QDict<KviKvsModuleSimpleCommandExecRoutine>         * m_pModuleSimpleCommandExecRoutineDict;
	QDict<KviKvsModuleFunctionExecRoutine>              * m_pModuleFunctionExecRoutineDict;
	QDict<KviKvsModuleCallbackCommandExecRoutine>       * m_pModuleCallbackCommandExecRoutineDict;
public:
	void kvsRegisterSimpleCommand(const QString &szCommand,KviKvsModuleSimpleCommandExecRoutine r);
	void kvsRegisterCallbackCommand(const QString &szCommand,KviKvsModuleCallbackCommandExecRoutine r);
	void kvsRegisterFunction(const QString &szFunction,KviKvsModuleFunctionExecRoutine r);
	bool kvsRegisterAppEventHandler(unsigned int iEventIdx,KviKvsModuleEventHandlerRoutine r);
	bool kvsRegisterRawEventHandler(unsigned int iRawIdx,KviKvsModuleEventHandlerRoutine r);

	void kvsUnregisterSimpleCommand(const QString &szCommand)
		{ m_pModuleSimpleCommandExecRoutineDict->remove(szCommand); };
	void kvsUnregisterCallbackCommand(const QString &szCommand)
		{ m_pModuleCallbackCommandExecRoutineDict->remove(szCommand); };
	void kvsUnregisterFunction(const QString &szFunction)
		{ m_pModuleFunctionExecRoutineDict->remove(szFunction); };
	void kvsUnregisterAppEventHandler(unsigned int iEventIdx);
	void kvsUnregisterRawEventHandler(unsigned int iRawIdx);

	void kvsUnregisterAllSimpleCommands()
		{ m_pModuleSimpleCommandExecRoutineDict->clear(); };
	void kvsUnregisterAllCallbackCommands()
		{ m_pModuleCallbackCommandExecRoutineDict->clear(); };
	void kvsUnregisterAllFunctions()
		{ m_pModuleFunctionExecRoutineDict->clear(); };
	void kvsUnregisterAllAppEventHandlers();
	void kvsUnregisterAllRawEventHandlers();
	void kvsUnregisterAllEventHandlers();

	KviKvsModuleSimpleCommandExecRoutine * kvsFindSimpleCommand(const QString &szCommand)
		{ return m_pModuleSimpleCommandExecRoutineDict->find(szCommand); };
	KviKvsModuleCallbackCommandExecRoutine * kvsFindCallbackCommand(const QString &szCommand)
		{ return m_pModuleCallbackCommandExecRoutineDict->find(szCommand); };
	KviKvsModuleFunctionExecRoutine * kvsFindFunction(const QString &szFunction)
		{ return m_pModuleFunctionExecRoutineDict->find(szFunction); };
};



#define KVSM_REGISTER_SIMPLE_COMMAND(_pModule,_szCmd,_procname) \
	_pModule->kvsRegisterSimpleCommand(_szCmd,_procname);

#define KVSM_UNREGISTER_SIMPLE_COMMAND(_pModule,_szCmd) \
	_pModule->kvsUnregisterSimpleCommand(_szCmd);

#define KVSM_REGISTER_CALLBACK_COMMAND(_pModule,_szCmd,_procname) \
	_pModule->kvsRegisterCallbackCommand(_szCmd,_procname);

#define KVSM_UNREGISTER_CALLBACK_COMMAND(_pModule,_szCmd) \
	_pModule->kvsUnregisterCallbackCommand(_szCmd);

#define KVSM_REGISTER_FUNCTION(_pModule,_szFnc,_procname) \
	_pModule->kvsRegisterFunction(_szCmd,_procname); \

#define KVSM_UNREGISTER_FUNCTION(_pModule,_szFnc) \
	_pModule->kvsUnregisterFunction(_szFnc);

#define KVSM_UNREGISTER_ALL_SIMPLE_COMMANDS(_pModule) \
	_pModule->kvsUnregisterAllSimpleCommands();

#define KVSM_UNREGISTER_ALL_CALLBACK_COMMANDS(_pModule) \
	_pModule->kvsUnregisterAllCallbackCommands();

#define KVSM_UNREGISTER_ALL_FUNCTIONS(_pModule) \
	_pModule->kvsUnregisterAllFunctions();




#endif //!_KVI_KVS_MODULEINTERFACE_H_
