//=============================================================================
//
//   File : kvi_kvs_parser.cpp
//   Creation date : Thu 25 Sep 2003 05.12 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2003 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__


#include "kvi_kvs_parser.h"

#include "kvi_kvs_treenode.h"

#include "kvi_kvs_report.h"
#include "kvi_kvs_kernel.h"

#include "kvi_kvs_parser_macros.h"

#include "kvi_locale.h"


//FIXME: @ == $$-> == $this->

KviKvsParser::KviKvsParser()
{
	// no need to initialize m_pBuffer
	// no need to initialize m_ptr
	// no need to initialize m_bError
	m_pGlobals = 0;
}

KviKvsParser::~KviKvsParser()
{
	if(m_pGlobals)delete m_pGlobals;
}




void KviKvsParser::init()
{
	KviKvsKernel * pKern = KviKvsKernel::instance();

#define _REG_CNTRL_CMD(__cntrlCmdName,__parsingRoutine) \
	{ \
		KviKvsSpecialCommandParsingRoutine * r = new KviKvsSpecialCommandParsingRoutine; \
		r->proc = KVI_PTR2MEMBER(KviKvsParser::__parsingRoutine); \
		pKern->registerSpecialCommandParsingRoutine(QString(__cntrlCmdName),r); \
	}

	_REG_CNTRL_CMD("if",parseSpecialCommandIf);
	_REG_CNTRL_CMD("global",parseSpecialCommandGlobal);
	_REG_CNTRL_CMD("while",parseSpecialCommandWhile);
	_REG_CNTRL_CMD("break",parseSpecialCommandBreak);
	_REG_CNTRL_CMD("do",parseSpecialCommandDo);
	_REG_CNTRL_CMD("for",parseSpecialCommandFor);
	_REG_CNTRL_CMD("foreach",parseSpecialCommandForeach);
	_REG_CNTRL_CMD("switch",parseSpecialCommandSwitch);
	_REG_CNTRL_CMD("defpopup",parseSpecialCommandDefpopup);

#undef _REG_CNTRL_CMD
}


KviKvsTreeNodeInstruction * KviKvsParser::parse(const QChar * pBuffer,int iFlags)
{
	m_iFlags = iFlags;

	m_bError = false;
	if(m_pGlobals)m_pGlobals->clear(); // this shouldn't be needed since this is a one time parser

	m_pBuffer = pBuffer;
	m_ptr = pBuffer;

	if(!pBuffer)
	{
		error(0,__tr2qs("Empty script"));
		return 0;
	}

	return parseInstructionList();
}

KviKvsTreeNodeInstruction * KviKvsParser::parseAsExpression(const QChar * pBuffer,int iFlags)
{
	m_iFlags = iFlags;

	m_bError = false;
	if(m_pGlobals)m_pGlobals->clear(); // this shouldn't be needed since this is a one time parser

	m_pBuffer = pBuffer;
	m_ptr = pBuffer;

	if(!pBuffer)
	{
		error(0,__tr2qs("Empty script"));
		return 0;
	}

	KviKvsTreeNodeExpression * expr = parseExpression(0);
	if(!expr)return 0;
	return new KviKvsTreeNodeExpressionReturn(pBuffer,expr);
}

KviKvsTreeNodeInstruction * KviKvsParser::parseAsParameter(const QChar * pBuffer,int iFlags)
{
	m_iFlags = iFlags;

	m_bError = false;
	if(m_pGlobals)m_pGlobals->clear(); // this shouldn't be needed since this is a one time parser

	m_pBuffer = pBuffer;
	m_ptr = pBuffer;

	if(!pBuffer)
	{
		error(0,__tr2qs("Empty script"));
		return 0;
	}

	KviKvsTreeNodeDataList * l = parseCommandParameterList();
	if(!l)return 0;
	return new KviKvsTreeNodeParameterReturn(pBuffer,l);
}



///////////////////////////////////////////////////////////////////////////////////////////////////
// THE REAL KVS
//
// <script> ::= <instruction list> '\0'
//
// <instruction list> ::= <instruction> [ <instruction list> ]
// <instruction> ::= <instruction block> | <command> | <operation> | <comment>
// <instruction block> ::= '{' <instruction list> '}'
//
// <comment> ::= <bash style line comment> | <c++ style comment> | <c style comment>
// <bash style comment> ::='#' <any char not including newline or null> <newline or null>
// <c++ style comment> ::= '//' <any char not including newline or null> <newline or null>
// <c style comment> ::= '/*' <any char not including null or the sequence */> '*/'
//
// <command> ::= <simple command> | <callback command> | <control command>
// <simple command> ::= <core command> | <module command> | <alias command>
// <core command> ::= <command identifier> <switch list> <command parameter list> <command terminator>
// <switch list> ::= '-'<switch body> [<switch list>]
// <command parameter list> ::= <command parameter>[<space><command parameter>]
// <command parameter> ::= <command parameter part>[<command parameter>]
// <command parameter part> ::= <command literal parameter> | <string parameter> | <data evaluation>
// <command literal parameter> ::= <anything except space , null , newline , ; , " , $ or %>
// <string parameter> ::= '"'<string parameter body>'"'
// <string parameter body> ::= <anything except '"' or null>...
// <command terminator> ::= ';' | '\n' | '\0'

// <data> ::= 
// <data_reference> ::= <function_call> | <structured_data>
// <structured_data> ::= <data_structure> | <variable> | <pointer_data>
// <data_structure> ::= <array> | <hash>
// <array> ::= '%'<identifier>'[]'
// <hash> ::= '%'<identifier>'{}'
// <variable> ::= '%'<identifier> | <array_element> | <hash_element>
// <array_element> ::= '%'<identifier>'['<expression>']'
// <hash_element> ::= '%'<identifier>'{'<key>'}'
// <function_call> ::= <simple_function_call> | <pointer_function_call>
// <simple_function_call> ::= '$'<identifier>'('<function_parameter_list>')'
// <pointer_function_call> ::= <variable>'->'<function_call> | <simple_function_call>'->'<function_call>
// <pointer_data> ::= <variable>'->'<data>' | <simple_function_call>'->'<data>


//
// This must evaluate SCOPE OBJECT operators!
// thus...first evaluate the <data> or <simple_function_call>
// If it was <simple_function_call> or <data> evaluation results in returning a <variable>
// then check if there is a scope operator
// if there is then take it as the top of the tree, the <variable> or <simple_function_call>
// go at the left param of the scope operator and re-evaluate the right side <data> or <simple_function_call>
//

/*
	@doc: kvs_introduction
	@type:
		language
	@keyterms:
		kvs, compilation
	@title:
		KVIrc scripting language introduction
	@short:
		KVIrc scripting language introduction
	@body:
		[p]
		[b]KVS[/b] is the [b]KV[/b]irc [b]S[/b]cripting language.
		It was inspired by C++,sh,perl,php and mIrc scripting language implementations.
		It is a compromise between flexibility and speed, a 'workaround' for many intrinsic
		problems of an IRC-oriented scripting language.
		[/p]
		[p]
		KVS is semi-interpreted: the execution is done in two main stages.
		The first stage is the compilation where a syntactic tree is built.
		The second stage is the real execution and is performed by visiting the tree
		in the proper order. The syntactic trees are cached in memory so
		the next executions can jump directly into the second stage.
		This two-stage approach has been introduced in version 3.0.0, the previous
		versions of the language used a single-stage on-the-fly interpreter.
		[/p]
		[p]
		KVS allows you to:[br]
		[ul]
		[li]Implement automated reactions to the events generated by an IRC network[/li]
		[li]Add new complex commands[/li]
		[li]Add interface elements like popups, toolbars, buttons...[/li]
		[li]Add advanced interface elements like complete dialogs or even widgets integrated in KVIrc[/li]
		[/ul]
		[/p]
		[p]
		KVS contains all the common constructs of structured programming.
		You will find almost all the C control commands, sh/perl-like variables, arrays and and functions.
		There are also some object-oriented characteristics: you will find C++ like
		objects with constructors, destructors and class inheritance.
		There are also more exotic concepts like the signal-slots interobject-communication.
		Obviously you will also find most of the RFC1459 IRC commands and
		other tools to "play" with an IRC connection.
		[/p]
		[p]
		I'll try to explain the language by using examples
		instead of strict syntactic rules. (Actually I have even
		tried to write the rules...take a look [doc:syntactic_rules]here[/doc][br][br]
		And please...forgive me for my "fantastic" english :)
		[/p]
		Szymon Stefanek
*/

/*
	@doc: kvs_basicconcepts
	@type:
		language
	@keyterms:
		script
	@title:
		KVS basic concepts
	@short:
		KVS basic concepts
	@body:
		[big]Scripts[/big]
		[p]
		You use KVS to implement [b]scripts[/b].
		A script is basically a finite list of KVS instructions.
		When you type a command in the KVIrc input window you in fact
		execute a small one-line script. You can store
		longer scripts in KVIrc memory and execute them at later time.
		Scripts can be also read from external files by the means of the
		[cmd]parse[/cmd] command.
		[/p]

		[big]Hello world![/big]
		[p]
		This documentation contains a lot of script examples.
		They will appear like the following block of code:
		[example]
			[cmd]echo[/cmd] Hello world!
		[/example]
		The best way to experiment is to execute the scripts from an external file.
		Try to copy & paste the example above to a file and save it in
		a known place. Then in the command input window type
		[example]
			[b]/[/b][cmd]parse[/cmd] <filename>
		[/example]
		where <filename> stands for the name of the file you just have saved.
		Some simple examples (like the one above) can be also typed
		directly in the command input window.
		You must remember that the command input window needs
		a leading slash ('/') character to recognize a script.
		The command input window can also be put in multiline mode by clicking
		on the button on the right.
		Another alternative for testing scripts is the code tester window.
		You can access it by selecting "New code tester" from the Scripting menu
		at the top of the KVIrc window. You will soon have the opportunity to
		experiment with all the methods. Read on.
		[/p]

		[big]Basic syntax[/big]
		[p]
		A script contains a list of instructions separated by newlines or ';' characters.
		Placing an instruction per line does not require a terminating character,
		placing more instructions in a single line require them to be separated by ';'.
		The most common instructions in KVS are [b]commands[/b]. A command is basically
		a keyword followed by a list of space separater parameters.
		The simplest (and the most useful) command in KVS is [cmd]echo[/cmd]; it prints
		all its parameters to a KVIrc window.[br]
		The following is an example of a valid script that uses only [cmd]echo[/cmd] commands.
		[example]
			echo "This is the first line"
			ECHO This is the second line;
			echo "This is the third line"; echo This is still on the third line;
			eChO "This is the fourth line"; Echo "This is still on the fourth line"
		[/example]
		You have probably noticed that the terminating ';' character is optional
		when the command is the last in a line.
		The commands are [b]case insensitive[/b]; 'echo' is equivalent to 'Echo',
		to 'ECHO' and to 'eChO'. In fact, most of KVS is case insensitive.
		(There are obvious unavoidable exceptions for this rule; for example,
		on UNIX systems file names are case sensitive and this must be
		also reflected in KVS).
		Another interesting thing is that when you execute the script you
		don't see the enclosing quotes around the printed text: more about this
		in the following sections.[br]
		[note]
		Cryptic note (you may skip it for now):[br]
		Yes, the command terminator is a problem for those that want to use ';)' at the end
		of IRC commands like [cmd]msg[/cmd]. It is almost unavoidable (read: the cost for
		avoiding it is too high). Note that using '|' or any other character as command terminator
		will NOT solve the problem: if the terminator is too difficult to type it will annoy the
		scripters (and me), if it is too easy then there will be always someone that wants to use it 
		at the end (or in the middle) of a command with the original meaning.
		The solution is to escape the ';' character:
		[example]
			[cmd]echo[/cmd] You can do it now \;)
		[/example]
		[/note]
		[/p]

		[big]Parameter processing[/big]
		[p]
		Most of the commands accept (and sometimes require) a list of parameters.
		For example, the [cmd]join[/cmd] command (that is used to join an IRC channel)
		accepts two parameters: the first one is the channel to join and the second is
		the password. The simplified syntax for join is:
		[example]
			[cmd]join[/cmd] <channel> [password]
		[/example]
		The line above is an example of syntax specification. All the commands
		are described by such syntax lines. [cmd]join[/cmd] is the command and it stands exactly
		for the literal string "join" typed in a script. <channel> is in angular parenthesis
		and rappresents a mandatory parameter: you must substitute a real channel name in its place
		otherwise the command will fail and KVIrc will probably complain too.
		[password] is still a parameter but the square parentheses indicate that it is
		optional: if you specify it, then it will be interpreted as the channel password,
		if you don't then no password will be used.
		[note]
		The syntax is written in a simplified BNF. I say simplified because it is not
		totally strict around the KVIrc documentation. I just prefer the syntax to be
		clear and easy to read instead of being formally perfect.
		[/note]
		You can finally join a channel by writing:
		[example]
			[cmd]join[/cmd] #kvirc kvircrocks
		[/example]
		or , since #kvirc usually has no password , by writing:
		[example]
			[cmd]join[/cmd] #kvirc
		[/example]
		In the example above the optional parameter [password] is omitted.
		[note]
		In fact it is not really omitted: KVIrc interprets it as an empty string that later
		means "do not send the password to the server".
		Empty strings are equivalent to omitted ones.
		[/note]
		[/p]

		[big]Parameters, spaces and quotes[/big]
		[p]
		From the examples above is obvious that KVS command parameters are separated by spaces.
		What is not totally obvious is that multiple spaces are allowed but KVIrc
		will automatically reduce them to exactly one (just like HTML parsers or the shell
		interpreters do). This is an useful behaviour in an IRC client since spaces usually
		carry no information and in text oriented protocols make the parsing really harder (:D).
		[/p]
		[p]
		The spaces are simplified in normal processing but there are ways to force KVIrc
		to interpret the spaces just as they are.
		The first method are the quotation marks: all the spaces enclosed in quotation marks
		will be preserved.
		[example]
			[cmd]echo[/cmd] This &nbsp; &nbsp; text &nbsp; &nbsp; will &nbsp; &nbsp; have &nbsp; &nbsp; spaces &nbsp; &nbsp; simplified
			[cmd]echo[/cmd] But &nbsp; &nbsp; "this &nbsp; &nbsp; &nbsp; one &nbsp; &nbsp; not"
		[/example]
		The first example will print out with spaces simplified but the second not.
		The quotes are also a nice trick to embed spaces into a single parameter that
		would be obviously splitted in two or more.
		[example]
			[cmd]echo[/cmd] Parameter1 Parameter2 "Parameter  3 ( with spaces )" Parameter4
		[/example]
		By running the examples above you may have noticed that the spaces are preserved but the
		quotes are then stripped! Yes, this is another tricky behaviour. But don't be afraid:
		it is really easier to use than to explain.
		There is obviously a method to preserve the quotes too and it is also another
		method to preserve the spaces but that leads us to the next paragraph.
		[/p]

		[big]Escape character[/big]
		[p]
		You may have already noticed that KVS treats some characters in a special way.
		For example the double-quote characters can be used to enclose strings
		and are stripped by the parser.
		Another example of a special character is the command terminator (';'):
		it has the "special" meaning of terminating a command.
		If you want to enclose a literal quote in your text, you need to [b]escape[/b] it.
		Like in most other programming languages, the escaping character is the backslash ('\').
		[example]
			[cmd]echo[/cmd] You can smile this way too! \;)
		[/example]
		The above example will treat the ';' as a part of the parameters and print it.[br]
		In some languages the action of "escaping" a character is called "quoting".
		Altough there is some confusion in this term, the meaning is to either use quotes
		or to use the escape character to remove special meaning from some characters.
		By quoting the spaces you can include them in a parameter, by escaping the quotes
		you can include them in a command.
		[example]
			[cmd]echo[/cmd] "And he said \"Hello world!\""
		[/example]
		The example above will have the internal quotes preserved.
		You can use the escape backslash to escape a newline:
		[example]
			[cmd]echo[/cmd] This text will be \
			&nbsp; &nbsp; printed on a single line!
		[/example]
		After an escaped newline all the leading space and tab characters are skipped,
		so you must include the needed spaces [b]before[/b] the escape character.
		The previous example will be printed as:[br][br]
		[i]This text will be printed on a single line[/i][br]
		Another example:[br]
		[example]
			[cmd]echo[/cmd] "The new kvirc &nbsp; &nbsp; &nbsp \
            &nbsp; &nbsp; IS OUT!"
			[cmd]echo[/cmd] Check it out at http://www.kvi\
            &nbsp; &nbsp; rc.net!
		[/example]
		This will be printed as:[br][br]
		[i]
		The new kvirc &nbsp; &nbsp; &nbsp; IS OUT![br]
		Check it out at http://www.kvirc.org!
		[/i][br]
		Finally, you can escape an escape character to include it literally in a parameter (and
		this is really the end of these tricks :)
		Later we will discover other common usages of the backslash escape, such
		as preventing KVIrc from interpreting a literal percent character as a variable
		or separating variable names from the text.
		[/p]

		[big]Command switches[/big]
		[p]
		Many commands accept switch parameters.
		[b]A switch modifies the behaviour of a command.[/b]
		Any switch can optionally accept a parameter, that must
		be specified after an equal ('=') sign.
		[example]
			[cmd]echo[/cmd] [b]-i = 2[/b] This text uses a specific color scheme
		[/example]
		The -i switch (just for example) changes the attributes
		and the icon of the printed text.
		[b]The switch must be specified immediately after the command keyword.[/b]
		[example]
			[cmd]echo[/cmd] This -i = 2 will obviously not work...
		[/example]
		If you want to start the first parameter of a command (that is not a switch)
		with a literal '-' you must again escape it:
		[example]
			[cmd]echo[/cmd] \--- This text has three minus signs on the left
		[/example]
		or use the quotes:
		[example]
			[cmd]echo[/cmd] "--- This text has three minus signs on the left"
		[/example]
		[/p]

		[big]Command blocks[/big]
		[p]
		Commands can be 'grouped' in blocks by using the classic C++ braces.
		Here is a single line example:[br]
		[example]
			{ [cmd]echo[/cmd] First command; [cmd]echo[/cmd] Second command; } [cmd]echo[/cmd] Third command
		[/example]
		Multi line example:[br]
		[example]
			{
			    [cmd]echo[/cmd] First command
				[cmd]echo[/cmd] Second command
			}
			[cmd]echo[/cmd] Third command
		[/example]
		[note]
		Reminder : copy the example above to a text file
		and then use /[cmd]parse[/cmd] &lt;filename&gt;
		[/note]
		In this case the command block has no special meaning
		other than making the code more readable , but command blocks
		will be useful later (see [cmd]if[/cmd],[cmd]while[/cmd]...).[br]
		[note]
		Unlike in C or C++, the braces do NOT automatically define a variable scope
		(with few exceptions to this rule ... just to complicate the things a bit more).
		You will recall this last assertion later, when reading about [doc:data_structures]data structures[/doc].
		[/note]
		[/p]

		[big]Comments[/big]
		[p]
		KVirc supports comments in command sequences.[br]
		A comment startw with the character '#' and terminates with a newline.
		You can start a comment anywhere a command can start.[br]
		[example]
			# This is a comment , it occupies the whole line
			[cmd]echo[/cmd] After the comment!; # This is an end-line comment
		[/example]
		You can not escape newline characters in this case.
		(or better: escape characters have no meaning in comments...
		maybe one day I'll implement it).[br]
		Starting from version 3.0.0 kvirc supports also C++ single line and C multiline comments.[br]
		A C++ comment starts with two slashes '//' and terminates with a newline.
		A multiline C comment starts with '/*' and ends at the first '* /' encountered.
		Since KVIrc has no pre-processor, the C/C++ comments usually can't be placed in the middle of a command:
		they must start where a command would start and end before the begin of another command.[br]
		[/p]

		[big]Indentation[/big]
		[p]
		You [b]should[/b] use spaces or [b]tabs[/b] to [b]indent[/b] your code. Note that the [b]should[/b]
		word is written in bold characters: I mean that you really should indent your code.
		Indenting helps both you (the script writer) and the reader (any other user that will
		read your script). A good indenting practice is the first step to become a great programmer :)
		[note]
		Please note that the command parameters should be separated by
		space characters (ascii 32). Tabs are not granted to work as parameter separators.[br]
		[/note]
		[example]
		{
		&lt;tab&gt;[cmd]echo[/cmd] Indented command
		&lt;tab&gt;{
		&lt;tab&gt;&lt;tab&gt;# Comment
		&lt;tab&gt;&lt;tab&gt;[cmd]echo[/cmd] Really Really long indented \
		&lt;tab&gt;&lt;tab&gt;&lt;tab&gt;command
		&lt;tab&gt;}
		}
		[/example]
		Tabs behave better than spaces as indentation characters since other users can
		adjust the tab size to match their taste. I personally prefer 4 character tabs
		while most text/code editors usually come with 8 characters as default.
		[/p]
		
		[big]And now ?[/big]
		[p]
		You're now ready to really start experimenting with KVS. You can take
		a look at the [doc:commands]command index[/doc] and start trying to use them
		while keeping in mind the rules described in this document.
		The next suggested lecture is the documentation about [doc:kvs_aliasesandfunctions]the aliases
		and the functions[/doc]. Have fun :)
		[/p]
*/

/*
	@doc: kvs_aliasesandfunctions
	@type:
		language
	@keyterms:
		aliases, functions
	@title:
		KVS Functions and aliases
	@short:
		KVS Functions and aliases
	@body:
		[big]Introduction[/big]
		[p]
		Since you're here, you should already have readed about the [doc:kvs_basicconcepts]KVS basic concepts[/doc]
		and have visited the [doc:commands]command index[/doc]. If you feel ready to take the next step
		then read on.
		[/p]

		[big]Functions[/big][br]
		[p]
		KVS has many internal [doc]functions[/doc] that can be used as command parameters.[br]
		[b]All the function names start with a literal '$' character.[/b][br]
		[example]
			[cmd]echo[/cmd] This window caption is [fnc]$caption[/fnc]
		[/example]
		The [fnc]$caption[/fnc] [doc:functions]function[/doc]
		is evaluated before the command executes,
		and it is changed into the current window caption text.[br]
		The [doc]functions[/doc] can be used also as switch parameters.[br]
		[example]
			[cmd]echo[/cmd] -w = [fnc]$window[/fnc] This text will be surely \
			&nbsp; &nbsp; printed in the current window
		[/example]
		The -w switch allows to redirect the echo text to a specified window --- in this
		case the one that you are typing in.[br]
		[i](Surprise: in this case the -w switch is useless ,
		since echo prints text to the current window by default...
		but it will work correctly. :)[/i]
		[/p]
		[p]
		Normal function names can be made of "anycase" letters, digits and underscores,
		with the restriction that the first character is not a digit.[br]
		Some kind of functions can contain a dot '.' character inside the name
		and these are assumed to be module references (see [doc:modules]the modules documentation[/doc]).[br]
		[/p]
		[p]
		By now we have seen only simple functions, but there's more...[br]
		The functions can accept parameters; the general syntax for a function call is:[br]
		[b]$<function name>['('<parameter_list>')'][/b][br]
		where <parameter_list> is a list of comma separated parameters,
		eventually empty.
		[example]
			[cmd]echo[/cmd] The word 'neural' is [fnc]$str.len[/fnc](neural) characters long
		[/example]
		The function [fnc]$str.len[/fnc] accepts a single parameter and returns the
		length in characters of the parameter string. The returned value is always
		a string: in this case it can be also interpreted as a number.[br]
		When passing an empty list you can avoid the parenthesis.
		(And you have found the "simple" functions shown above).
		So the followind two calls are equal:[br]
		[example]
			[cmd]echo[/cmd] [fnc]$caption[/fnc]
			[cmd]echo[/cmd] [fnc]$caption()[/fnc]
		[/example]
		If you want to pass an "empty" string as the first parameter you have to use
		the following syntax:[br]
		[example]
			[cmd]echo[/cmd] [fnc]$str.len[/fnc]("")
		[/example]
		Obviously a function is valid as a function parameter.[br]
		[example]
			[cmd]echo[/cmd] [fnc]$str.len[/fnc]([fnc]$caption[/fnc])
		[/example]
		If you want to place a literal '(' or ')' in the function parameters
		you must escape it.
		A special case for when you want to use 'matching' parentheses:
		an opened '(' corresponds to a closed ')'.
		In this case you can omit the 'escape' character.[br]
		[example]
			[cmd]echo[/cmd] The length of '(a+b)' is : [fnc]$str.len[/fnc]( (a+b) )
		[/example]
		This is useful for algebraic and boolean expressions , like the ones
		accepted by the special function $() (see next paragraphs).[br]
		[/p]


		[big]Aliases[/big][br]
		An alias is an user defined command.  It can be used to rename the builtin kvirc commands or functions,
		to automatize complex tasks or as structured programming mean.
		Aliases can be created or destroyed by using the scriptcenter (graphic interface)
		or from the commandline (or script) by using the [cmd]alias[/cmd] command.
		Once created, an alias remains stored permanently in the KVIrc configuration files
		until it is explicitly deleted.
		A couple of examples will make the things clear.
		join is a really commonly used command. It might be a good idea to rename it to
		simply "j" .. just to type it faster.
		Nothing easier in KVirc: just try this commandline:
		[example]
			[cmd]alias[/cmd](j){ [cmd]join[/cmd] $0-; };
		[/example]

		This will create the alias "j". From this moment you can use /j as it was a normal command.
		[example]
		j #kvirc
		[/example]
		You may have notices the strange $0- function in the alias body: it stands for
		"all parameters passed to the alias". This means that when you call
		[example]
			j #kvirc testpassword
		[/example]
		then both the parameters (#kvirc and testpassword) are passed to the join command.
		The $N functions are special functions that return the positional parameters passed
		to the current script context. In an alias the script context is the script body and
		it is the alias caller that generates the parameters.
		$N (where N is a digit) returns the (N-1)-th positional parameter passed by the caller.
		It returns the parameter numbered N-1 and not N since the parameters are indexed starting
		from zero ($0 is the first parameter!).
		$N-M returns the parameters from (N-1)-th to the (M-1)-th (a parameter range) and $N- returns
		all the parameters from (N-1)-th to the last one. In the example above $0- stands for
		all the parameters starting from the first one.
		[/p]
		[p]
		To remove an alias use again the alias command with an empty body:
		[example]
			[cmd]alias[/cmd](j){}
		[/example]
		This will remove the alias "j" defined above.
		[/p]
		[p]
		A common task in channel management is the kick & ban action.
		You first ban an user from the channel and then eventually kick him
		(obviously assuming that he is actually on the channel).
		This involves using two commands: ban and then kick.
		It could be a nice idea to have a single "kb" command to perform this action.
		Well...easy:
		[example]
			[cmd]alias[/cmd](kb){ [cmd]ban[/cmd] $0; [cmd]kick[/cmd] $0-; };
		[/example]
		This adds the "kb" alias: it can be called as a normal command:
		[example]
			kb spammer You're not welcome here!
		[/example]
		This will first execute "ban spammer" and then "kick spammer You're not welcome here".
		Our kb is a really simple example... it doesn't check for the validity of the parameters:
		the server will warn us if the parameters passed to kb were empty.
		[/p]
		[p]
		The alias can be modified at any time by re-using the alias command.
		Let's make our "kb" a bit more intelligent and add a check for the parameters.
		TIP: It is a good idea to write the following examples in a text file and then use /parse <filename> to execute it.
		[example]
			[cmd]alias[/cmd](kb)
			{
				[cmd]if[/cmd]("$0" == "")
				{
					[cmd]echo[/cmd] "Usage: /kb <nickname> <kick reason>"
					[cmd]return[/cmd]
				}
				[cmd]ban[/cmd] $0
				%reason = $1-
				[cmd]if[/cmd]("%reason" == "")%reason = "You're not welcome here!"
				[cmd]kick[/cmd] $0 %reason
			}
		[/example]
		The example above will first check the validity of the <nickname> passed to kb: 
		if no nickname was passed , it will warn the user and stop.
		The next step will be the "ban <nickname>" call. Another enchancement is the "default reason":
		we first assign the remaining parameters ($1- means "from $1 to the end") to a temporary variable,
		if the variable is empty , a default kick reason is assigned.
		Finally the "kick <nickname> <reason>" will be executed.
		Get used to looking at the single command documentation pages, they will give
		you the hints necessary to fully understand the above piece of code.
		[/p]
		[p]
		Aliases can be used as a mean for structured programming.
		In large scripts you will SURELY have "common tasks" to perform (like having specially
		colored output or calculating a value from a set of other values)...
		Aliases are the way of writing the common tasks: they are equivalent to the "procedures"
		or "functions" in many high-level programming languages.
		The alias as a procedure (subroutine or sub-task) has been shown in the "kb" example above:
		it might be commonly called from complexier scripts or other aliases in case that a
		kick & ban action is needed.
		[/p]
		[p]
		The aliases can be used also as functions.
		Assume that you need really often to calculate the sum of three numbers: a function-alias is the way.
		[example]
			[cmd]alias[/cmd](sum3){ [cmd]return[/cmd] $($0 + $1 + $2); };
		[/example]
		This will add the alias "sum3" and make it available both as a command and a function.
		The "return" command sets the return value of a sequence of commands
		(an alias is a sequence of commands...remember ?) and terminates the execution (by returning
		the control to the caller).
		So return $($0 + $1 + $2); will set the return value of the alias to the value
		computed by $($0 + $1 + $2) that actually is the sum of the first three parameters passed.
		You will then use it in the following way:
		[example]
			...
			%myfirstsum = $sum3(%somevalue,%someothervalue,4)
			%anothersum = $sum3(12,%somevalue,%anothervalue)
			...
		[/example]
		Ops.. I've used some variables without actually explaining them... hehe.. please forgive me and read on.
		This example is again really simple , but you might have complexier function-aliases.
		The function-aliases are also normal aliases.... you can use it as a command:
		[example]
			/sum3 1 2 3
		[/example]
		Is a perfectly valid call.... it's just that it will have no visible results
		(just because a command call implies ignoring the return value.
		In fact there is no difference al all between function-aliases and normal-aliases: 
		the caller makes the difference: by calling an alias as a command the return value 
		just disappears in hyperspace, by calling an alias as a function , the return value
		is propagated (and in fact "used").
		(There are some "nice" exceptions to this rule...but you don't need to care about it, for now).
		If return is not called inside an alias body , the return value will be just a null value.
		[/p]
		[p]
		Aliases can accept switches just like any other command. The [fnc]$sw[/fnc] is there
		exactly for that purpose. Check it out.
		[/p]

		[big]Special functions[/big]
		[p]
		We have already seen the positional parameter functions.
		The functions of type [b]$N[-[M]][/b] (where N and M are positive
		numbers starting from 0 and N < M) evaluate to the sequence of
		[b]positional parameters[/b] from Nth to Mth."[br]
		If M is omitted , the function evaluate to the sequence of [b]positional
		parameters[/b] from Nth to the last one. If the whole -M block is omitted
		the function evaluate to the Nth positional parameter.
		We will discover more on the [b]positional parameters[/b] when talking
		of aliases and events.[br]
		[example]
			$0 evaluates to the 1st positional parameter
			$0-4 evaluates to the parameters from first to 5th
			$41- evaluates to the parameters from 41st to the last avaiable
		[/example]
		The function [b]$#[/b] evaluates to the number of positional parameters available.
		The [b]positional parameter[/b] functions do not accept parameters.[br]
		The special function [b]$(<expression>)[/b] returns the result
		of the evaluation of the <expression>. In previous versions of KVIrc this
		function was called [fnc]$calc[/fnc].[br]
		[example]
			[cmd]echo[/cmd] $(2 + (3 ^ 7) <= 1 * (3 && 2))
		[/example]
		The special function [b]${<command sequence>}[/b] evaluates to the
		return value of the <command sequence>.[br]
		The special function [b]$$[/b] evaluates to the current object id,
		but it is too early to explain it here...[br]
*/


/*
	@doc: kvs_codingtips
	@type:
		generic
	@title:
		Coding tips
	@keyterms:
		indentation,indent,readability
	@short:
		Generic coding tips for scripters (and not only)
	@body:
		Here comes a small list of "coding tips".[br]
		These apply to programming in general , not only to KVIrc scripting.[br]
		[br]
		1. [b]Comment your code[/b][br]
		A well commented code is easy to mantain, and easy to read by others.[br]
		[br]
		2. [b]Indent your code[/b][br]
		Indentation increases the code readability; this is again for you and
		other developers that will be going to read your code.[br]
		[br]
		3. [b]Use TABS to indent your code[/b][br]
		...and use ONLY TABS to indent.[br]
		Tabs are better than space since most code editors allow you
		to set the tab sice and thus to have the indentation steps smaller or bigger.[br]
		This is really important since the indentation size is really a matter of personal taste.[br]
		Mixing spaces and tabs is Evil (tm), since it makes the code look really
		ugly in editors that have the tab size different than yours; in some cases the
		code gets really unreadable.[br]
		[br]
		4. [b]Use descriptive variable names[/b][br]
		Using 'foo' as variable name implies tracing its semantic the next
		time that you're going to read the code that uses it.[br]
		This is really annoying and time-consuming, especially if the project
		is getting large.[br]
		Obviously using "thisIsACounterVariable" as name for a simple counter
		is also a suicide.[br]
		A good convention on variable names can speed up writing , debugging and mantaining code.[br]
		Encoding the type of the variable in the variable name might be also a good idea,
		but this is a matter of taste; personally I feel really well with that.[br]
		Just as example, here go my fundamental convention rules for C++:[br]
		[br]
		- The type of the variable is encoded at the beginning of the variable name:[br]
		[br]
			- b prefix for the boolean varables[br]
			- i prefix for signed integers[br]
			- u prefix for unsigned integers[br]
			- f and d prefixes for floating point stuff[br]
			- sz prefix for strings (this is rather for string classes)[br]
			- ...[br]
		[br]
		- Pointers have a "p" prefix prepended[br]
		- Global variables start with a "g_" prefix[br]
		- Member variables start with a "m_" prefix[br]
		- Exception comes for local variables with obvious semantics[br]
		[br]
			- i,j,k,l for local loop counters[br]
			- "aux" and "tmp" for local obvious short-term temporary variables[br]
		[br]
		So actually by ONLY reading "g_pszQuitMessage" I know that this is a global pointer to a string variable
		containing a quit message. :)[br]
		[/p]
*/

// FIXME: #warning "FINISH THE SYNTACTIC RULES DOC"

/*
	@doc: syntactic_rules
	@type:
		language
	@keyterms:
		productions
	@title:
		Syntactic rules
	@short:
		Syntactic rules of the KVIrc scripting language
	@body:

		In the following table you can find a good part of the
		KVIrc scripting language syntactic rules.[br]
		[br]
		<entity> indicates a ENTITY THAT CAN APPEAR EXACTLY ONE TIME.[br]
		[<entity>] indicates an OPTIONAL ENTITY.[br]
		{<entity>} indicates an ENTITY THAT CAN APPEAR ONE OR MORE TIMES.[br] 
		'entity' indicates a LITERAL ENTITY: written exactly as it is.[br]
		<entity1>|<entity2> indicates mutually exclusive choices.[br]
		The mutually exclusive choices are often separated in two or more
		rules (productions), to improve readability.[br]
		[table]
		[tr]
		[td]<command buffer>[/td]
		[td][<whitespace>][<command block>]{<command buffer>}[/td]
		[/tr]
		[tr]
		[td]<command buffer>[/td]
		[td][<whitespace>][<single command>]{<command buffer>}[/td]
		[/tr]
		[tr]
		[td]<whitespace>[/td]
	    [td]{<space>|<tab>|<newline>}['\'<newline>][<whitespace>][/td]
		[/tr]
		[tr]
		[td]<space>[/td]
		[td]' '['\'<newline>][<space>] (Ascii space character)[/td]
		[/tr]
		[tr]
		[td]<tab>[/td]
		[td]'\t' (Ascii horizontal tabulation character)[/td]
		[/tr]
		[tr]
		[td]<newline>[/td]
		[td]'\n' (Ascii line feed (LF) character)[/td]
		[/tr]
		[tr]
		[td]<command block>[/td]
		[td]'{' <command buffer>[<whitespace>] '}'[/td]
		[/tr]
		[tr]
		[td]<single command>[/td]
		[td]<comment>[/td]
		[/tr]
		[tr]
		[td]<single command>[/td]
		[td]<lvalue command> <command terminator>[/td]
		[/tr]
		[tr]
		[td]<single command>[/td]
		[td]<rvalue command> <command terminator>[/td]
		[/tr]
		[tr]
		[td]<comment>[/td]
		[td]'#' {<non comment terminator>} <comment terminator>[/td]
		[/tr]
		[tr]
		[td]<comment terminator>[/td]
		[td]<newline> | <end of string>[/td]
		[/tr]
		[tr]
		[td]<end of string>[/td]
		[td]No character (internally Ascii character 0)[/td]
		[/tr]
		[tr]
		[td]<command terminator>[/td]
		[td]<newline> | <end of string> | ';'[/td]
		[/tr]
		[tr]
		[td]<non comment-terminator>[/td]
		[td]Any Ascii character except <newline> and <end of string>[/td]
		[/tr]
		[tr]
		[td]<simple command>[/td]
		[td][<module name>'.']<command name>[<switch list>]{<space>}<command dependant part>[/td]
		[/tr]	
		[tr]
		[td]<lvalue command>[/td]
		[td]<variable>[<space>]<operation>[/td]
		[/tr]
		[tr]
		[td]<lvalue command>[/td]
		[td]<variable>'->'<object command>[/td]
		[/tr]
		[tr]
		[td]<lvalue command>[/td]
		[td]<identifier>'->'<object command>[/td]
		[/tr]
		[tr]
		[td]<operation>[/td]
		[td]<one op operator>[/td]
		[/tr]
	    [tr]
		[td]<operation>[/td]
		[td]<two op operator>[<space>]<param string>[/td]
		[/tr]
		[tr]
		[td]<switch list>[/td]
		[td]{<space>}'-'<alpha char>[{<space>}'='<single parameter>][<switch list>][/td]
		[/tr]
		[tr]
		[td]<command name>[/td]
		[td]<alphanumeric char>{<alphanumeric char>}[/td]
		[/tr]
		[tr]
		[td]<module name>[/td]
		[td]<alphanumeric char>{<alphanumeric char>}[/td]
		[/tr]
		[tr]
		[td]<alphanumeric char>[/td]
		[td]Ascii characters 'A' to 'Z' , 'a' to 'z' , '0' to '9' and '_'[/td]
		[/tr]
		[tr]
		[td]<variable>[/td]
		[td]<global variable> | <local variable>[/td]
		[/tr]
		[tr]
		[td]<global variable>[/td]
		[td]'%' <uppercase letter> [<alphanumeric char>]['['<param string>']'][/td]
		[/tr]
		[tr]
		[td]<local variable>[/td]
		[td]'%' <lowercase letter> [<alphanumeric char>]['['<param string>']'][/td]
		[/tr]
		[tr]
		[td]<param string>[/td]
		[td][<single parameter>][<space>[<param string>]][/td]
		[/tr]
		[tr]
		[td]<single parameter>[/td]
		[td]<variable> | <identifier> | <nonterminator token> | <string>[/td]
		[/tr]
		[tr]
		[td]<nonterminator token>[/td]
		[td]<nonterminator char>['\'<newline><nonterminator char>][/td]
		[/tr]
		[tr]
		[td]<nonterminator char>[/td]
		[td]Any ascii character except <space> and <command terminator>[/td]
		[/tr]
		[tr]
		[td]<command dependant part>[/td]
		[td][b]Production in each command help page[/b][/td]
		[/tr]
		[/table]
		To be continued...
*/

/*
	@doc: command_rebinding
	@type:
		language
	@keyterms:
		Not supported
	@title:
		Standard -r switch no longer supported
	@short:
		Standard -r switch no longer supported
	@body:
		Starting from version 3.0.0 the standard -r switch to commands is no longer supported.
		You should rebind your command sequences with [cmd]rebind[/cmd]
*/

/*
	@doc: variables
	@type:
		language
	@keyterms:
		variables, variable, array, hash, dictionary, global variables, local variables,variable evaluation,
		associative arrays, scalars, data types
	@title:
		Variables
	@short:
		Variables and data types
	@body:
		[big]Basic syntax[/big]
		A variable identifier is composed by a '%' (percent) sign followed
		by a sequence of letters, digits or underscores.[br]
		Examples of valid variable names are:[br]
		%i,%variable,%MyVar,%1,%thisisavar,%2ndName,%_hidden[br]
		
		[big]Local and global variables[/big]
		Variables can be local or global.[br]
		Local variables preserve their contents only inside the scope of a single script.[br]
		Global variables are shared between all the scripts and preserve their contents
		until they are explicitly unset or until KVIrc quits.[br]
		By default KVIrc assumes that the variables you're referencing are local.[br]
		If you want to force the scope of a variable to global you need to pre-declared it with
		the [cmd]global[/cmd] keyword.
		[example]
			[comment]# copy this script to a file and run /[cmd]parse[/cmd] <filename>[/comment]
			global %a
			%a = "The contents of the variable a"
			%b = "The contents of the variable b"
			[comment]# %a is a global variable now : all the other scripts can see its value[/comment]
			[comment]# %b is a local variable and no other scripts can see its value[/comment]
		[/example]
		One notable exception is the commandline. When you type the commands "manually"
		KVIrc assumes that the variables you're referencing are global. This is generally
		useful since you don't want to type 'global %x' before every command you try.[br]
		If you have executed the example above from a file (by the means of [cmd]parse[/cmd])
		then now you can type
		[example]
			[cmd]echo %a[/cmd]
		[/example]
		in the commandline to see the contents of the variable %a.[br]
		If you also try
		[example]
			[cmd]echo %b[/cmd]
		[/example]
		you will probably see nothing printed since %b was local to the parsed script.[br]
		Older KVIrc versions recognized global variables when they were starting with
		an uppercase letter and local ones when the initial letter was lowercase.[br]
		This method was incoherent since the other parts of the language are case insensitive.[br]
		The case sentive method is still supported for backward compatibility but is deprecated and
		the parser warns about it.[br]
		
		[big]Variable types[/big]
		KVS has three main data types: scalars, arrays of scalars and associative
		arrays of scalars (also known as hashes).[br]
		[b]Scalars[/b][br]
		The scalars are simple variables containing a single value (a string or an integer).[br]
		[example]
			[comment]# %a is a scalar variable[/comment]
			%a = "This is a string"
			[cmd]echo[/cmd] %a
			%a = 24.5
			[cmd]echo[/cmd] %a
		[/example]
		[b]Arrays[/b][br]
		The arrays are collections of items indexted by integers.[br]
		An easy way to create an array is to use the [fnc]$array[/fnc] function.
		[example]
			%a = $array("element1","element2","element3")
			[cmd]for[/cmd](%i=0;%i<3;%i++)
			{
				echo %a[%i];
			}
		[/example]
		
		
		Any variable is a sort of variant and can assume different type identities
		in different times.
		[example]
			# %a is a scalar
			%a = "This is a string"
			# %a becomes an array with 3 elements
			%a = $array("element1","element2","element3");
			# %a becomes a hash with two values
			%a = $hash("key1","value1","key2","value2");
		[/example]

		[note]
		Obscure KVIrc internals:[br]
		Internally KVS is an implicitly typed language: the "scalar" data type is in fact
		a set of types that KVIrc manages silently. Notable types are "integer","real" and "string".
		[/note]
		
*/

/*
	FIXME: this needs rechecking
	@doc: kvs_datatypes
	@type:
		language
	@keyterms:
		variable,variables,array,arrays,dictionary,dictionaries,percent sign,global variables,local variables,
		global variable,local variable,variable creation and destruction,variable evaluation,dictionary keys,
		associative arrays,hash,hashes,scalar,scalars,command scope,data types,extended scope,array reference,
		dictionary reference
	@title:
		Data types
	@short:
		KVIrc built-in data types
	@body:
		KVirc has three basic built-in data types: simple variables, arrays of
		simple variables and associative arrays of variables.[br]
		There is a fourth builtin data type: object
		that allows creation of complex structures and object-oriented programming;
		the object data type is described in another document.[br]

		[big]Variables[/big]
		The variables can be either global to the application or local
		to the command scope. The content of a variable is basically a string:
		any sequence of ASCII characters. In some contests the string may be interpreted
		as a number. All the variable names start with a percent '%' sign.
		
		[big]Global variables[/big][br]
		A global variable name is formed by a "percent" sign (%),
		followed by an uppercase letter from A to Z, followed
		by a sequence of characters in range ('a' to 'z','0' to '9','.','_').[br]
		"%INDEX","%My_nickname","%Foo","%Bar1" and "%Foo.BAR" are examples of valid
		global variable names.[br]
		A global variable is [b]global to the entire application[/b], not
		to the current frame or irc context; be sure to remember this.[br]
		You can type[br]
		[example]
			/%Hello = "Hello world!"
		[/example]
		in the commandline of any KVIrc window and then execute[br]
		[example]
			echo %Hello
		[/example]
		in the commandline of any other KVIrc window. You will see "Hello world!"
		printed in the second window view.

		[big]Local variables[/big][br]
		A local variable name is formed by a "percent" sign (%),
		followed by an lowercase letter from a to z, followed
		by a sequence of characters in range ('a' to 'z','0' to '9','.','_').[br]
		"%index","%my_nickname","%foo","%bAR1" and "%foo.BAR" are examples of valid
		local variable names.[br]
		A local variable exists only in the current command scope.
		The exact command scope definition is rather tricky and depends
		on the internal KVIrc implementation. Just be aware that:[br]
		- An alias body is a command scope.[br]
		- An event body is a command scope.[br]
		- Any sequence of commands executed [b]at once[/b] in a window commandline
		is a command scope.[br]
		You will notice that finding out the current command scope is rather intuitive.[br]
		When you type[br]
		[example]
			%text = "very important text";
		[/example]
		in the commandline of any KVIrc window and then try to execute[br]
		[example]
			echo %text
		[/example]
		you will see nothing printed. The variable %text was local to the "command scope"
		and disappeared immediately after the execution of the assignment.[br]
		But if you execute[br]
		[example]
			%text = "hello"; echo %text wold!
		[/example]
		you will see  "hello world!" printed in the current window.[br]

		[big]Extended scope variables[/big]
		Variables that start with a ':' character are "extended scope" variables.
		"%:index" , "%:Hello" , "%:something.else" are all valid special scope variable names.[br]
		They're actually used in popups and in timers (but later I might find other usages as well :).[br]
		"Extended scope" means that these variables are somewhere in the middle between
		global and local variables. They normally act as local , but in some cases their [b]lifetime[/b] and [b]visibility[/b]
		may be extended.[br]
		For example , in the popups , all the special scope variables
		are visible during all the "lifetime" of a popup (so from the prologue code call to
		the moment when the user selects an item and the corresponding code is executed).[br]
		This allows you to pre-calculate some data or conditions in the popup prologue
		and use this data in the popup item conditions and item handlers.[br]

		[big]Variable creation and destruction[/big]
		You don't need to declare variables.
		A variable starts to exist at the time that you assing
		something to it.[br]
		[example]
			%MyVar = some funky text
			%X = 2445833
			%SevenSpaces = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;"
		[/example]
		It terminates its existence at the time that you assingn it an empty string:
		[example]
			%MyVar = ""
			%X = %MyVar
			%SevenSpaces =
		[/example]
		The example is based on global variables, but it is valid for local ones as well.[br]
		A non existing variable is equivalent to an empty one: you will
		never get a warning about non existing variables. This convention
		allows KVIrc to automatically manage the variable creation and destruction.[br]

		[big]Variable evaluation[/big]
		A variable can appear in every place where a parameter
		is expected: so after the command name, after a switch or inside
		an identifier parameters.
		KVirc will try to extract the longest possible variable name after a literal percent '%'
		sign everywhere in the parameter string. So the command sequence[br]
		[example]
			%number = 1st; echo this is my %number variable test
		[/example]
		will first assign "1st" to the variable "%number" and then execute
		"echo this is my 1st variable test".
		The following example will NOT work as expected.[br]
		[example]
			%number = 1; echo this is my %numberst variable test
		[/example]
		KVirc will assign "1" to %number in this case but the next variable
		name extracted will be "%numberst" that is actually empty; so finally
		"echo this is my variable test" will be executed.
		To avoid this problem you can use the backslash escape character:[br]
		[example]
			%number = 1; echo this is my %number\st variable test
		[/example]

		[big]Arrays[/big]
		Arrays are collections of items indexed by numbers.[br]
		The general syntax for an array is:[br]
		[b]%<name>[<index>][/b][br]
		<name> is the name of the array and follows the rules
		valid for the simple variables: the names starting with an uppercase letter
		designate a global array, the others designate local ones.[br]
		Extended scope arrays can be created as well: normally they act as local,
		but may have extended lifetime in some scopes.[br]
		The arrays have its own namespace thus %Array[] has nothing to do with %Array
		(that is a simple variable).[br]
		<index> must be a subscript that evaluates to a positive integer and it selects an item in the array.
		The first index of the array is 0 and the last is equal to size-1.[br]
		You can obtain the size of the array by evaluating %<name>[]#.[br]
		You don't need to declare the size of the array: it is automatically
		handled. When you assign a non-empty string to an item, the array
		is automatically enlarged to contain the index that you are assigning to.
		If the array was not existing before the assignment, it is created.[br]
		If the first assignment index is greater than 0 the items below that index
		will be empty: will just behave as empty/unset variables.[br]
		[example]
			[comment]# Create an array with 21 elements[/comment]
			%Names[20]=Pragma
		[/example]
		To remove an item from the array you assign it an empty string.[br]
		When you remove the highest indexed item the array is automatically shrunk to
		the next highest non-empty item. If there are no other non-empty items the array is destroyed.[br]
		Please note that the memory occupation of the array depends on the <index>.[br]
		By assigning a value to the 10000'th item (index 9999) you allocate 10000 entries! (in fact
		at least ((10000 * 4) + value size) bytes).[br]
		An array is destroyed when it contains no more items or when it goes out of scope (a local array obviously).[br]
		[example]
			[comment]# Creating an array with 800 entries[/comment]
			%Array[799]=test
			echo %Array[]#
			%Array[299]=test
			echo %Array[]#
			%Array[799]=
			echo %Array[]#
			[comment]# Now it contains 300 elements[/comment]
		[/example]
		[br]
		[big]Array references[/big][br]
		When <index> is an empty string then the subscript %<name>[] is called "array reference" (maybe a bit unproperly :).
		You may think that notation as referencing the "whole array".[br]
		The following example shows how the array assignment.[br]
		[example]
			%A[100]=x
			%A[200]=y
			%B[]=%A[]
			echo %B[200]
		[/example]
		When you pass an array reference to a command or function, it is evaluated
		as a comma separated list of entries.[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Pluto
			%Array[2]=Paperino
			echo %Array[]
		[/example]
		By assigning a string to an array you assign it to all the array entries:[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Pluto
			%Array[2]=Paperino
			echo %Array[]
			%Array[]=undefined
			echo %Array[]
		[/example]
		This is useful when you want to unset an array: just assign an empty string to all its entries:[br]
		[example]
			%Array[200]=Test
			echo %Array[]#
			%Array[]=
			echo %Array[]#
		[/example]
		You can loop through all the array items by using the [cmd]foreach[/cmd] command:[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Pluto
			%Array[2]=Paperino
			[cmd]foreach[/cmd](%item,%Array[])[cmd]echo[/cmd] %item
		[/example]
		Obviously also the traditional [cmd]for[/cmd] and [cmd]while[/cmd] indexed-looping methods
		are available:[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Never show this
			%Array[2]=Pluto
			%Array[5]=Hidden again
			%Array[8]=Paperino
			for(%i=0;%i < %Array[]#;%i+=2)echo Entry %i: \"%Array[%i]\";
		[/example]
		[br]
		[big]Dictionaries[/big]
		Dictionaries are associative arrays of strings. They look close
		to the perl hashes. The general syntax for a dictionary name is:[br]
		[b]%<name>{<key>}[/b][br]
		<name> is the name of the dictionary and follows the same rule
		as for the variables: the names starting with an uppercase letter
		designate a global dictionary, the others designate local ones.[br]
		Again , the dictionaries have its own namespace: you can safely use
		%Names , %Names[] and %Names{} as different entities in your script.[br]
		Extended scope dictionaries can be created as well: normally they act as local,
		but may have extended lifetime in some scopes.[br]
		A dictionary associates a "data string" to each "key string" used.
		The key can be [b]any string[/b] not containing an "unescaped" '}' character
		and is [b]case insensitive[/b]: "key" is equivalent to "KEY" or "KeY".[br]
		[example]
			%Ages{Pragma} = 24
		[/example]
		This assignment associates the key "Pragma" to the number "24" in
		the global dictionary "%Ages". If the array was not existing yet,
		it is created first. If the key "Pragma" was already existing,
		the value associated is replaced with the new value.[br]
		To remove an association you simply assign the empty string to it:[br]
		[example]
			%Ages{pragma} =
		[/example]
		The dictionary is automatically destroyed when there are no more associations
		in it (eg. when you assign the empty string to the last key).[br]
		Dictionaries can be used easily to simulate arrays:[br]
		[example]
			%Links{21} = "http://www.kvirc.net"
		[/example]
		Even multidimensional ones:[br]
		[example]
			%Pixels{324.312} = 1
		[/example]
		Remember that in the example above the key "324.312" is a single string.[br]
		Obviously the key can contain variables and functions just as any other parameter.[br]
		An empty key performs operations on the whole dictionary (just like for the arrays):[br]
		You can assign dictionaries:[br]
		[example]
			%Test2{}=%Test1{}
		[/example]
		Assign a scalar to all the items
		[example]
			%Data{} = "unspecified"
		[/example]
		The example above associates the string "unspecified" to all the existing
		keys in the dictionary %Data. By using this method you can easily destroy
		a dictionary: simply assign the empty string to it.[br]
		[example]
			%Data{} =
		[/example]
		Other [doc]operators[/doc] have similar semantic when working on "empty keys".[br]

		[big]Dictionary evaluation[/big]
		A dictionary can appear in every place where a variable can appear.[br]
		[example]
			[cmd]echo[/cmd] My age is %Ages{[fnc]$mynick[/fnc]}
		[/example]
		If you pass an empty key, the dictionary evaluates to the comma separated
		list of values stored in it. The values have no defined order.
		The list is interpreted as single "string" in contexts where a "string" is required,
		and as a list of "strings" in context where lists of strings are expected. (<-- hehe :)[br]
		The special syntax %<name>{}# returns the number of keys in the
		dictionary.[br]
		The special syntax %<name>{}@ returns a comma
		separated list of keys in the dictionary; the keys have no
		defined order (well, you may be only sure that the order
		of the keys is exactly equal to the values order (%name{})).[br]
		[example]
			%Songs{Jimi Hendrix} = Voodo child
			%Songs{Shawn Lane} = Gray piano's flying
			%Songs{Mina} = Brava
			%Songs{Greg Howe} = "Full Throttle"
			# This is a "single string" evaluation context
			[cmd]echo[/cmd] %Songs{}
			# This is a "list of strings" evaluation context
			[cmd]foreach[/cmd](%var,%Songs{})[cmd]echo[/cmd] %var
		[/example]
*/



void KviKvsParser::skipSpaces()
{
	while((KVSP_curCharUnicode == ' ') || (KVSP_curCharUnicode == '\t'))
	{
		KVSP_skipChar;
	}

	if(KVSP_curCharUnicode == '\\')
	{
		KVSP_skipChar;
		if(KVSP_curCharUnicode == '\n')
		{
			KVSP_skipChar;
			skipSpaces();
		} else {
			KVSP_backChar;
		}
	}
}



void KviKvsParser::skipSpacesAndNewlines()
{
	while((KVSP_curCharUnicode == ' ') || (KVSP_curCharUnicode == '\t') || (KVSP_curCharUnicode == '\n'))
	{
		KVSP_skipChar;
	}
	
	switch(KVSP_curCharUnicode)
	{
		case '\\':
			KVSP_skipChar;
			if(KVSP_curCharUnicode == '\n')
			{
				KVSP_skipChar;
				skipSpacesAndNewlines();
			} else {
				KVSP_backChar;
			}
		break;
		case '#': 
		case '/':
			// we allow comments too!
			(void)parseComment(); // this will return 0 anyway
			skipSpacesAndNewlines();
		break;
	}
}

void KviKvsParser::skipToNextLine()
{
	while((KVSP_curCharUnicode != 0) && (KVSP_curCharUnicode != '\n'))
		KVSP_skipChar;

	if(KVSP_curCharUnicode == '\n')KVSP_skipChar;
}


KviKvsTreeNodeInstruction * KviKvsParser::parseInstructionList()
{
	KviKvsTreeNodeInstructionBlock * l = new KviKvsTreeNodeInstructionBlock(KVSP_curCharPointer);


	for(;;)
	{
		skipSpacesAndNewlines();

		if(KVSP_curCharUnicode != 0)
		{
			// instruction
			KviKvsTreeNodeInstruction * i = parseInstruction();
			if(i)l->addInstruction(i);
			else {
				if(error())
				{
					// ops...
					delete l;
					return 0;
				} // else empty instruction
			}
		} else {
			if(l->instructionCount() == 1)
			{
				// return the single instruction instead
				KviKvsTreeNodeInstruction * i = l->releaseFirst();
				delete l;
				return i;
			}
			// end of buffer
			return l;
		}
	}

	// never here
	KVSP_ASSERT(false);
	return 0;
}




KviKvsTreeNodeData * KviKvsParser::parseParameterPercentOrDollar()
{
	KVSP_ASSERT((KVSP_curCharUnicode == '%') || (KVSP_curCharUnicode == '$'));

	if(KVSP_curCharUnicode == '%')
	{
		KVSP_skipChar;
		if(!KVSP_curCharIsLetter && (KVSP_curCharUnicode != ':'))
		{
			// be flexible : allow an "alone" '%' char
			return new KviKvsTreeNodeConstantData(KVSP_curCharPointer - 1,new KviKvsVariant(QString("%")));
		}
	} else {
		KVSP_skipChar;
		if(!KVSP_curCharIsFunctionStart)
		{
			// be flexible : allow an "alone" '$' char
			return new KviKvsTreeNodeConstantData(KVSP_curCharPointer - 1,new KviKvsVariant(QString("$")));
		}
	}

	// this is surely a variable or function
	KVSP_backChar;
	return parsePercentOrDollar();
}



KviKvsTreeNodeData * KviKvsParser::parsePercentOrDollar(bool bInObjScope)
{
	KVSP_ASSERT((KVSP_curCharUnicode == '%') || (KVSP_curCharUnicode == '$'));

	KviKvsTreeNodeData * r;

	if(KVSP_curCharUnicode == '%')
	{
		r = parsePercent();
		if(!r)return 0;
	} else {
		r = parseDollar(bInObjScope);
		if(!r)return 0;
	}

	const QChar * pBegin = KVSP_curCharPointer;

	if(KVSP_curCharUnicode == '[')
	{
		// array index
		KVSP_skipChar;
		skipSpaces();
		if(KVSP_curCharUnicode == ']')
		{
			KVSP_skipChar;
			if(KVSP_curCharUnicode == '#')
			{
				// count
				KVSP_skipChar;
				return new KviKvsTreeNodeArrayCount(pBegin,r);
			}
			// a hash reference assert
			return new KviKvsTreeNodeArrayReferenceAssert(pBegin,r);
		}

		KviKvsTreeNodeExpression * e = parseExpression(']');
		if(!e)
		{
			delete r;
			return 0;
		}
		
		r = new KviKvsTreeNodeArrayElement(pBegin,r,e);
		
	} else if(KVSP_curCharUnicode == '{')
	{
		// hash key
		KVSP_skipChar;
		skipSpaces();

		if(KVSP_curCharUnicode == '}')
		{
			// entire hash ?
			KVSP_skipChar;
			if(KVSP_curCharUnicode == '#')
			{
				KVSP_skipChar;
				return new KviKvsTreeNodeHashCount(pBegin,r);
			}
			return new KviKvsTreeNodeHashReferenceAssert(pBegin,r);
		}

		KviKvsTreeNodeData * i = parseHashKey();
		if(!i)
		{
			// error
			delete r;
			return 0;
		}

		KVSP_ASSERT(KVSP_curCharUnicode == '}');

		KVSP_skipChar;

		r = new KviKvsTreeNodeHashElement(pBegin,r,i);
	}


	if(KVSP_curCharUnicode != '-')
	{
		return r;
	}


	if(!r->canEvaluateToObjectReference())return r; // FIXME: maybe print a warning ?

	// might be a scope operator

	KVSP_skipChar;
	if(KVSP_curCharUnicode != '>')
	{
		KVSP_setCurCharPointer(pBegin);
		return r;
	}

	KVSP_skipChar;
	skipSpaces();

	if((KVSP_curCharUnicode != '$') && (KVSP_curCharUnicode != '%'))
	{
		KVSP_setCurCharPointer(pBegin);
		return r;
	}

	// hmmm... there really seems to be a scope operator there...
	if(KVSP_curCharUnicode == '%')
	{
		KVSP_skipChar;
		if(!KVSP_curCharIsLetter)
		{
			// be flexible : allow an "alone" '%' char
			KVSP_setCurCharPointer(pBegin);
			return r;
		}
	} else {
		KVSP_skipChar;
		if(!KVSP_curCharIsFunctionStart)
		{
			// be flexible : allow an "alone" '$' char
			KVSP_setCurCharPointer(pBegin);
			return r;
		}
	}

	// ok : try the scope operator
	KVSP_backChar;

	pBegin = KVSP_curCharPointer;

	KviKvsTreeNodeData * r2 = parsePercentOrDollar(true);

	if(!r2)
	{
		// must be an error
		delete r;
		return 0;
	}
	
	if(!r2->canEvaluateInObjectScope())
	{
		// ops... it really wasn't
		delete r2;
		KVSP_setCurCharPointer(pBegin);
		return r;
	}

	return new KviKvsTreeNodeScopeOperator(pBegin,r,r2);
}





KviKvsTreeNodeData * KviKvsParser::parsePercent()
{
	KVSP_ASSERT(KVSP_curCharUnicode == '%');

	KVSP_skipChar;

	bool bExtScope;

	if(KVSP_curCharUnicode == ':')
	{
		bExtScope = true;
		KVSP_skipChar;
	} else bExtScope = false;

	if(!((KVSP_curCharIsLetterOrNumber) || (KVSP_curCharUnicode == '_')))
	{
		error(KVSP_curCharPointer,__tr2qs("Syntax error after '%' variable prefix. If you want to use a plain '%' in the code you need to escape it"));
		return 0;
	}

	const QChar * pIdBegin = KVSP_curCharPointer;

	while(((KVSP_curCharIsLetterOrNumber) || (KVSP_curCharUnicode == '_')))KVSP_skipChar;

	QString szIdentifier(pIdBegin,KVSP_curCharPointer - pIdBegin);

//#warning "ADD A KviKvsTreeNodeBuiltinCleanupVariablesCommand on this KviKvsParser object"
//#warning "KviKvsParser will append it to the script"

	if(bExtScope)return new KviKvsTreeNodeExtendedScopeVariable(pIdBegin,szIdentifier);
	if(m_pGlobals)
	{
		if(m_pGlobals->find(szIdentifier))return new KviKvsTreeNodeGlobalVariable(pIdBegin,szIdentifier);
	}

	if(m_iFlags & AssumeGlobals)return new KviKvsTreeNodeGlobalVariable(pIdBegin,szIdentifier);
	else {
		// old scripts compat
		if(pIdBegin->category() & QChar::Letter_Uppercase)
		{
			warning(pIdBegin,__tr2qs("Declaring global variables with an uppercase letter is deprecated. Global variables should be declared with 'global'"));
			return new KviKvsTreeNodeGlobalVariable(pIdBegin,szIdentifier);
		}
		return new KviKvsTreeNodeLocalVariable(pIdBegin,szIdentifier);
	}
}

KviKvsTreeNodeInstruction * KviKvsParser::parseInstruction()
{
	switch(KVSP_curCharUnicode)
	{
		case '#': 
		case '/':
			(void)parseComment(); // this will return 0 anyway
			return 0;
		break;
		case 0:
		case '\n':
		case ';': // empty instruction
			KVSP_skipChar;
			return 0;
		break;
		case '{': // command block
			return parseInstructionBlock();
		break;
		case '$':
		case '%':
			return parseVoidFunctionCallOrOperation();
		break;
		default:
			if(KVSP_curCharIsLetter)
			{
				// must be a command
				return parseCommand();
			} else {
				// what the heck is this ?
				error(KVSP_curCharPointer,__tr2qs("Found character '%q' (unicode %x) where an instruction was expected"),KVSP_curCharPointer,KVSP_curCharUnicode);
				return 0;
			}
		break;
	}

	// never here
	KVSP_ASSERT(false);
	return 0;
}

KviKvsTreeNodeInstruction * KviKvsParser::parseInstructionBlock()
{
	KVSP_ASSERT(KVSP_curCharUnicode == '{');

	KVSP_skipChar;

	const QChar * pBegin = KVSP_curCharPointer;

	KviKvsTreeNodeInstructionBlock * b = new KviKvsTreeNodeInstructionBlock(pBegin - 1);

	for(;;)
	{
		skipSpacesAndNewlines();

		switch(KVSP_curCharUnicode)
		{
			case 0:
				delete b;
				warning(pBegin,__tr2qs("Unterminated instruction block"));
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of script in instruction block (missing closing brace)"));
				return 0;
			break;
			case '}':
				KVSP_skipChar;
				if(b->instructionCount() <= 1)
				{
					if(b->instructionCount() < 1)
					{
						delete b;
						return 0; // just an empty block
					}
					// a single instruction
					KviKvsTreeNodeInstruction * i = b->releaseFirst();
					delete b;
					return i;
				}
				return b;
			break;
			default:
				// instruction
				KviKvsTreeNodeInstruction * i = parseInstruction();
				if(i)b->addInstruction(i);
				else {
					if(error())
					{
						// ops...
						delete b;
						return 0;
					} // else empty instruction
				}
			break;
		}
	}
}

KviKvsTreeNodeSwitchList * KviKvsParser::parseCommandSwitchList()
{
	KVSP_ASSERT(KVSP_curCharUnicode == '-');

	KviKvsTreeNodeSwitchList * sw = new KviKvsTreeNodeSwitchList(KVSP_curCharPointer);

	while(KVSP_curCharUnicode == '-')
	{
		const QChar * pBegin = KVSP_curCharPointer;
		KVSP_skipChar;
		
		bool bLong = false;
		
		if(KVSP_curCharUnicode == '-')
		{
			// long switch
			pBegin = KVSP_curCharPointer;
			KVSP_skipChar;
			bLong = true;
		}

		skipSpaces();
		if(!KVSP_curCharIsLetterOrNumber)
		{
			delete sw;
			warning(pBegin,__tr2qs("The dash after a command should be either escaped or followed by a letter or number (switch)"));

			if(KVSP_curCharUnicode == 0)
			{
				error(KVSP_curCharPointer,__tr2qs("Unexpected character '%q' (unicode %x) after a switch dash"),KVSP_curCharPointer,KVSP_curCharUnicode);
			} else {
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of script after a switch dash"));
			}
			return 0;
		}

		const QChar * pSw = KVSP_curCharPointer;

		KVSP_skipChar;
		while((KVSP_curCharIsLetterOrNumber) || (KVSP_curCharUnicode == '-'))KVSP_skipChar;

		const QChar * pSwEnd = KVSP_curCharPointer;

		skipSpaces();

		if(KVSP_curCharUnicode == '=')
		{
			KVSP_skipChar;
			skipSpaces();
			KviKvsTreeNodeData * p = parseCommandParameter();
			if(!p)
			{
				// must be an error :(
				error(pBegin,__tr2qs("The above problem might be related to the switch dash and the following equal sign"));
				delete sw;
				return 0;
			}

			if(bLong)
				sw->addLong(QString(pSw,pSwEnd - pSw),p);
			else
				sw->addShort(pSw->lower().unicode(),p);
		} else {
			if(bLong)
				sw->addLong(QString(pSw,pSwEnd - pSw),new KviKvsTreeNodeConstantData(KVSP_curCharPointer,new KviKvsVariant(1))); // empty param
			else
				sw->addShort(pSw->lower().unicode(),new KviKvsTreeNodeConstantData(KVSP_curCharPointer,new KviKvsVariant(1))); // empty param
		}
	}

	return sw;
}



KviKvsTreeNodeDataList * KviKvsParser::parseCommandParameterList()
{
	KviKvsTreeNodeDataList * l = new KviKvsTreeNodeDataList(KVSP_curCharPointer);

	for(;;)
	{
		skipSpaces();
		switch(KVSP_curCharUnicode)
		{
			case 0:
				return l;
			break;
			case '\n':
			case ';':
				KVSP_skipChar;
				return l;
			break;
			default:
				// anything else is a parameter
				KviKvsTreeNodeData * p = parseCommandParameter();
				if(!p)
				{
					// this is an error
					delete l;
					return 0;
				}
				l->addItem(p);
			break;
		}
	}

	// never here
	KVSP_ASSERT(false);
	return 0;
}


KviPtrList<QString> * KviKvsParser::parseCommaSeparatedParameterListNoTree()
{
	KviPtrList<QString> * l = new KviPtrList<QString>;
	l->setAutoDelete(true);

	KVSP_skipChar;

	for(;;)
	{
		skipSpaces();
		switch(KVSP_curCharUnicode)
		{
			case 0:
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of script in parameter list"));
				delete l;
				return 0;
			break;
			case '\n':
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of line in parameter list"));
				delete l;
				return 0;
			break;
			case ',':
				KVSP_skipChar;
			break;
			case ')':
				KVSP_skipChar;
				return l;
			break;
			default:
			{
				// anything else is a parameter
				const QChar *pBegin = KVSP_curCharPointer;
				KviKvsTreeNodeData * p = parseCommaSeparatedParameter();
				if(!p)
				{
					// this is an error
					delete l;
					return 0;
				}
				delete p;
				QString * s = new QString(pBegin,KVSP_curCharPointer - pBegin);
				s->stripWhiteSpace();
				l->append(s);
			}
			break;
		}
	}

	// never here
	KVSP_ASSERT(false);
	return 0;
}


KviKvsTreeNodeDataList * KviKvsParser::parseCommaSeparatedParameterList()
{
	KviKvsTreeNodeDataList * l = new KviKvsTreeNodeDataList(KVSP_curCharPointer);

	KVSP_skipChar;

	for(;;)
	{
		skipSpaces();
		switch(KVSP_curCharUnicode)
		{
			case 0:
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of script in parameter list"));
				delete l;
				return 0;
			break;
			case '\n':
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of line in parameter list"));
				delete l;
				return 0;
			break;
			case ',':
				KVSP_skipChar;
			break;
			case ')':
				KVSP_skipChar;
				return l;
			break;
			default:
				// anything else is a parameter
				KviKvsTreeNodeData * p = parseCommaSeparatedParameter();
				if(!p)
				{
					// this is an error
					delete l;
					return 0;
				}
				l->addItem(p);
			break;
		}
	}

	// never here
	KVSP_ASSERT(false);
	return 0;
}


#define LITERAL_PARAM_PARSING_FUNCTION_BEGIN(__funcname) \
KviKvsTreeNodeConstantData * KviKvsParser::__funcname() \
{ \
	QString szValue; \
 \
	const QChar * pBegin = KVSP_curCharPointer; \
	int iLen = 0; \
 \
	for(;;) \
	{ \
		switch(KVSP_curCharUnicode) \
		{ \
			case 0: \
			case '$': \
			case '%': \
			case '\n': \
			case '"':


#define LITERAL_PARAM_PARSING_FUNCTION_END \
				if(iLen > 0)szValue.append(QString(pBegin,iLen)); \
				{ \
					bool bOk; \
					int iVal = szValue.toInt(&bOk); \
					if(bOk)return new KviKvsTreeNodeConstantData(pBegin,new KviKvsVariant(iVal)); \
					double dVal = szValue.toDouble(&bOk); \
					if(bOk)return new KviKvsTreeNodeConstantData(pBegin,new KviKvsVariant(dVal)); \
				} \
				return new KviKvsTreeNodeConstantData(pBegin,new KviKvsVariant(szValue)); \
			break; \
			case '\\': \
				if(iLen > 0)szValue.append(QString(pBegin,iLen)); \
				KVSP_skipChar; \
				switch(KVSP_curCharUnicode) \
				{ \
					case 0: \
						warning(KVSP_curCharPointer - 1,__tr2qs("Stray backslash at the end of the script")); \
						iLen = 0; \
					break; \
					case '\n': \
						KVSP_skipChar; \
						pBegin = KVSP_curCharPointer; \
						iLen = 0; \
					break; \
					default: \
						pBegin = KVSP_curCharPointer; \
						KVSP_skipChar; \
						iLen = 1; \
					break; \
				} \
			break; \
			default: \
				KVSP_skipChar; \
				iLen++; \
			break; \
		} \
	} \
	KVSP_ASSERT(false); \
	return 0; \
}


LITERAL_PARAM_PARSING_FUNCTION_BEGIN(parseCommandLiteralParameter)
			case ';':
			case ' ':
			case '\t':
LITERAL_PARAM_PARSING_FUNCTION_END

LITERAL_PARAM_PARSING_FUNCTION_BEGIN(parseStringLiteralParameter)
LITERAL_PARAM_PARSING_FUNCTION_END

/*
LITERAL_PARAM_PARSING_FUNCTION_BEGIN(parseArrayIndexLiteralParameter)
			case '\t':
			case ' ':
			case ']':
LITERAL_PARAM_PARSING_FUNCTION_END
*/

LITERAL_PARAM_PARSING_FUNCTION_BEGIN(parseHashKeyLiteralParameter)
			case '\t':
			case ' ':
			case '}':
LITERAL_PARAM_PARSING_FUNCTION_END

LITERAL_PARAM_PARSING_FUNCTION_BEGIN(parseCommaSeparatedLiteralParameter)
			case ')':
			case ',':
			case ' ':
			case '\t':
LITERAL_PARAM_PARSING_FUNCTION_END

LITERAL_PARAM_PARSING_FUNCTION_BEGIN(parseSingleLiteralParameterInParenthesis)
			case ')':
			case ' ':
			case '\t':
LITERAL_PARAM_PARSING_FUNCTION_END


/*
KviKvsTreeNodeData * KviKvsParser::parseArrayIndex()
{
	KviPtrList<KviKvsTreeNodeData> * l = new KviPtrList<KviKvsTreeNodeData>();
	l->setAutoDelete(true);

	const QChar * pBegin = KVSP_curCharPointer;

	//KVSP_skipChar;

	for(;;)
	{
		switch(KVSP_curCharUnicode)
		{
			case 0:
				delete l;
				warning(pBegin,__tr2qs("Unterminated array index"));
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of script in array index (missing ']' character ?)"));
				return 0;
			break;
			case '\n':
				delete l;
				warning(pBegin,__tr2qs("Unterminated array index"));
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of line in array index (missing ']' character or unescaped newline)"));
				return 0;
			break;
			case ' ':
			case '\t':
				skipSpaces();
				if(KVSP_curCharUnicode != ']')
				{
					delete l;
					warning(pBegin,__tr2qs("Unterminated array index"));
					switch(KVSP_curCharUnicode)
					{
						case 0:
							error(KVSP_curCharPointer,__tr2qs("Unexpected end of script in array index (missing ']' character ?)"));
						break;
						case '\n':
							error(KVSP_curCharPointer,__tr2qs("Unexpected end of line in array index (missing ']' character or unescaped newline)"));
						break;
						default:
							error(KVSP_curCharPointer,__tr2qs("Unexpected character '%q' (unicode %x) in array index: it should be already terminated"),KVSP_curCharPointer,KVSP_curCharUnicode);
						break;
					}
					return 0;
				}
				goto end_of_the_array_index;
			break;
			case '$':
			case '%':
			{
				// this may be a data reference
				KviKvsTreeNodeData * p = parseParameterPercentOrDollar();
				if(!p)
				{
					// this is an error
					delete l;
					return 0;
				}
				l->append(p);
			}
			break;
			case ']':
			{
				// end of the array index
				goto end_of_the_array_index;
			}
			break;
			case '"':
			{
				// string (should we parse strings in array indexes anyway ?).. well "1"$count might be a valid one in the end
				KviKvsTreeNodeData * p = parseStringParameter();
				if(!p)
				{
					// error
					delete l;
					return 0;
				}
				l->append(p);
			}
			break;
			default:
			{
				// anything else is a literal
				l->append(parseArrayIndexLiteralParameter());
			}
			break;
		}
	}
end_of_the_array_index:
	if(l->count() > 1)
	{
		// complex parameter needed
		return new KviKvsTreeNodeCompositeData(l);
	} else {
		// a single parameter in the list
		l->setAutoDelete(false);
		KviKvsTreeNodeData * p = l->first();
		delete l;
		return p;
	}

}
*/




KviKvsTreeNodeData * KviKvsParser::parseHashKey()
{
	KviPtrList<KviKvsTreeNodeData> * l = new KviPtrList<KviKvsTreeNodeData>();
	l->setAutoDelete(true);

	const QChar * pBegin = KVSP_curCharPointer;

	//KVSP_skipChar;

	for(;;)
	{
		switch(KVSP_curCharUnicode)
		{
			case 0:
				delete l;
				warning(pBegin,__tr2qs("Unterminated hash key"));
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of script in hash key (missing '}' character ?)"));
				return 0;
			break;
			case '\n':
				delete l;
				warning(pBegin,__tr2qs("Unterminated hash key"));
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of line in hash key (missing '}' character or unescaped newline)"));
				return 0;
			break;
			case ' ':
			case '\t':
				skipSpaces();
				if(KVSP_curCharUnicode != '}')
				{
					// separate by single spaces
					l->append(new KviKvsTreeNodeConstantData(KVSP_curCharPointer,new KviKvsVariant(QString(" "))));
				} else {
					goto end_of_the_hash_key;
				}
			break;
			case '$':
			case '%':
			{
				// this may be a data reference
				KviKvsTreeNodeData * p = parseParameterPercentOrDollar();
				if(!p)
				{
					// this is an error
					delete l;
					return 0;
				}
				l->append(p);
			}
			break;
			case '}':
			{
				// end of the array index
				goto end_of_the_hash_key;
			}
			break;
			case '"':
			{
				// string 
				KviKvsTreeNodeData * p = parseStringParameter();
				if(!p)
				{
					// error
					delete l;
					return 0;
				}
				l->append(p);
			}
			break;
			default:
			{
				// anything else is a literal
				l->append(parseHashKeyLiteralParameter());
			}
			break;
		}
	}
end_of_the_hash_key:
	if(l->count() > 1)
	{
		// complex parameter needed
		return new KviKvsTreeNodeCompositeData(pBegin,l);
	} else {
		// a single parameter in the list
		l->setAutoDelete(false);
		KviKvsTreeNodeData * p = l->first();
		delete l;
		return p;
	}
}

/*
PARENTHESIS_PARAMETER_PARSING_FUNCTION_BEGIN(parseCommaSeparatedParameter)
	case 0:
	case ',':
	case ')':
	case '\n':
PARENTHESIS_PARAMETER_PARSING_FUNCTION_END()

#define PARENTHESIS_PARAMETER_PARSING_FUNCTION_BEGIN(_name) \
*/

KviKvsTreeNodeData * KviKvsParser::parseCommaSeparatedParameter()
{
	KviPtrList<KviKvsTreeNodeData> * l = new KviPtrList<KviKvsTreeNodeData>;
	l->setAutoDelete(true);

	const QChar * pBegin = KVSP_curCharPointer;

	for(;;)
	{
		switch(KVSP_curCharUnicode)
		{
			case 0:
			case ',':
			case ')':
			case '\n':
				// not a part of a parameter
				goto end_of_function_parameter;
			break;
			case '$':
			case '%':
			{
				// this may be a data reference
				KviKvsTreeNodeData * p = parseParameterPercentOrDollar();
				if(!p)
				{
					// this is an error
					delete l;
					return 0;
				}
				l->append(p);
			}
			break;
			case ' ':
			case '\t':
				skipSpaces();
				if((KVSP_curCharUnicode != ')') && (KVSP_curCharUnicode != ','))
				{
					// separate by single spaces
					l->append(new KviKvsTreeNodeConstantData(KVSP_curCharPointer,new KviKvsVariant(QString(" "))));
				} else {
					goto end_of_function_parameter;
				}
			break;
			case '"':
			{
				// this is a string
				KviKvsTreeNodeData * p = parseStringParameter();
				if(!p)
				{
					// this is an error
					delete l;
					return 0;
				}
				l->append(p);
			}
			break;
			default:
			{
				// anything else is a literal
				l->append(parseCommaSeparatedLiteralParameter());
			}
			break;
		}
	}
end_of_function_parameter:
	if(l->count() > 1)
	{
		// complex parameter needed
		return new KviKvsTreeNodeCompositeData(pBegin,l);
	} else {
		// a single parameter in the list
		l->setAutoDelete(false);
		KviKvsTreeNodeData * p = l->first();
		delete l;
		return p;
	}
}


KviKvsTreeNodeData * KviKvsParser::parseSingleParameterInParenthesis()
{
	KviPtrList<KviKvsTreeNodeData> * l = new KviPtrList<KviKvsTreeNodeData>;
	l->setAutoDelete(true);

	const QChar * pBegin = KVSP_curCharPointer;

	for(;;)
	{
		switch(KVSP_curCharUnicode)
		{
			case ')':
				// not a part of a parameter
				KVSP_skipChar;
				goto end_of_function_parameter;
			break;
			case 0:
			case '\n':
				// not a part of a parameter
				goto end_of_function_parameter;
			break;
			case '$':
			case '%':
			{
				// this may be a data reference
				KviKvsTreeNodeData * p = parseParameterPercentOrDollar();
				if(!p)
				{
					// this is an error
					delete l;
					return 0;
				}
				l->append(p);
			}
			break;
			case ' ':
			case '\t':
				skipSpaces();
				if((KVSP_curCharUnicode != ')') && (KVSP_curCharUnicode != ','))
				{
					// separate by single spaces
					l->append(new KviKvsTreeNodeConstantData(KVSP_curCharPointer,new KviKvsVariant(QString(" "))));
				} else {
					goto end_of_function_parameter;
				}
			break;
			case '"':
			{
				// this is a string
				KviKvsTreeNodeData * p = parseStringParameter();
				if(!p)
				{
					// this is an error
					delete l;
					return 0;
				}
				l->append(p);
			}
			break;
			default:
			{
				// anything else is a literal
				l->append(parseSingleLiteralParameterInParenthesis());
			}
			break;
		}
	}
end_of_function_parameter:
	if(l->count() > 1)
	{
		// complex parameter needed
		return new KviKvsTreeNodeCompositeData(pBegin,l);
	} else {
		// a single parameter in the list
		l->setAutoDelete(false);
		KviKvsTreeNodeData * p = l->first();
		delete l;
		return p;
	}
}


KviKvsTreeNodeData * KviKvsParser::parseStringParameter()
{
	KVSP_ASSERT(KVSP_curCharUnicode == '"');

	KviPtrList<KviKvsTreeNodeData> * l = new KviPtrList<KviKvsTreeNodeData>();
	l->setAutoDelete(true);

	const QChar * pBegin = KVSP_curCharPointer;

	KVSP_skipChar;

	for(;;)
	{
		switch(KVSP_curCharUnicode)
		{
			case 0:
				delete l;
				warning(pBegin,__tr2qs("Unterminated string constant"));
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of script in string constant (missing \" character ?)"));
				return 0;
			break;
			case '\n':
				delete l;
				warning(pBegin,__tr2qs("Unterminated string constant"));
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of line in string constant (missing \" character or unescaped newline)"));
				return 0;
			break;
			case '$':
			case '%':
			{
				// this may be a data reference
				KviKvsTreeNodeData * p = parseParameterPercentOrDollar();
				if(!p)
				{
					// this is an error
					delete l;
					return 0;
				}
				l->append(p);
			}
			break;
			case '"':
			{
				// end of the string
				KVSP_skipChar;
				goto end_of_the_string;
			}
			break;
			default:
			{
				// anything else is a literal
				l->append(parseStringLiteralParameter());
			}
			break;
		}
	}
end_of_the_string:
	if(l->count() > 1)
	{
		// complex parameter needed
		return new KviKvsTreeNodeCompositeData(pBegin,l);
	} else {
		if(l->count() > 0)
		{
			// a single parameter in the list
			l->setAutoDelete(false);
			KviKvsTreeNodeData * p = l->first();
			delete l;
			return p;
		} else {
			// no parameters at all.. empty string!!!
			delete l;
			return new KviKvsTreeNodeConstantData(pBegin,new KviKvsVariant(new QString()));
		}
	}
}

KviKvsTreeNodeData * KviKvsParser::parseCommandParameter()
{
	KviPtrList<KviKvsTreeNodeData> * l = new KviPtrList<KviKvsTreeNodeData>;
	l->setAutoDelete(true);
	
	const QChar * pBegin = KVSP_curCharPointer;

	for(;;)
	{
		switch(KVSP_curCharUnicode)
		{
			case 0:
			case ' ':
			case '\t':
			case '\n':
			case ';':
				// not a part of a parameter
				goto jumpout;
			break;
			case '$':
			case '%':
			{
				// this may be a data reference
				KviKvsTreeNodeData * p = parseParameterPercentOrDollar();
				if(!p)
				{
					// this is an error
					delete l;
					return 0;
				}
				l->append(p);
			}
			break;
			case '"':
			{
				// this is a string
				KviKvsTreeNodeData * p = parseStringParameter();
				if(!p)
				{
					// this is an error
					delete l;
					return 0;
				}
				l->append(p);
			}
			break;
			default:
			{
				// anything else is a literal
				l->append(parseCommandLiteralParameter());
			}
			break;
		}
	}
jumpout:
	if(l->count() > 1)
	{
		// complex parameter needed
		return new KviKvsTreeNodeCompositeData(pBegin,l);
	} else {
		// a single parameter in the list
		l->setAutoDelete(false);
		KviKvsTreeNodeData * p = l->first();
		delete l;
		return p;
	}
}




//777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777

#if 0


<comment> ::= '#' <anything up to a newline or buffer end>

<operation> ::= <operation_leftside> [<operator> <operation_rightside>]

<operation_leftside> ::= <data_container> | <function_call>


<command> ::= <label> | <control command> | <simple command> | <callback command>

<label> ::= <identifier>[<space>]':'
<simple command> ::= <identifier> <switch_list> <parameter list> <terminator>
<callback command> ::= <identifier> <switch_list> '('<parameter_list>')' <instruction block>
<control command> ::= <if> | <while> | <for> | do | <switchcmd>

<if> ::= 'if'[<space>]'('<expression>')'<instruction aggregate>
         ['else' <instruction aggregate>]

<while> ::= 'while'[<space>]'('<expression>')'<instruction aggregate>

<switchcmd> ::= 'switch'[<space>]'('<expression>')'[<whitespace>]'{'
				{<switchlabel>}
			'}'

<do> ::= 'do'[<space>]<instruction aggregate>'while'[<space>]'('<expression>')'


<for> ::= 'for'[<space>]'('<simple command>';'

<identifier> ::= <letter> {<letter_or_digit>}

<switch_list> ::= <switch> [ <switch_list> ]
<switch> ::= '-'<letter_or_digit>[space]['='[space] <rvalue> ]

<rvalue> ::= <literal_string>[<rvalue>] |
				<data_container>[<rvalue>] |
				<function_call>[<rvalue>] |
				<quoted_rvalue>[<rvalue>]

<data_container> ::= <variable> |
				<array> |
				<hash>

<variable> ::= '%'<identifier>
<array> ::= <variable>'['[<rvalue>]']'
<hash> ::= <variable>'{'[<rvalue>]'}'


<function_call> ::= '$'<identifier> ['('[<function_parameter_list>]')']

<function_parameter_list> ::= <rvalue_or_space>[',' <rvalue_or_space>]

<rvalue_or_space> ::= ' '[<rvalue_or_space>] | <rvalue>

<quoted_rvalue> ::= '"'[<rvalue_or_space>]'"'

<literal_string> ::= <non_separator>[<literal_string>]

<non_separator> ::= all char sequences but <terminator> and <space>

<parameter_list> ::= <rvalue>[<space><parameter_list>];

#endif
