//=============================================================================
//
//   File : libkvichan.cpp
//   Creation date : Sun Feb 02 2002 05:27:11 GMT by Szymon Stefanek
//
//   This chan is part of the KVirc irc client distribution
//   Copyright (C) 2002-2004 Szymon Stefanek (pragma@kvirc.net)
//   Copyright (C) 2002-2004 Juanjo Alvarez  (juanjux@yahoo.es)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#include "kvi_module.h"
#include "kvi_string.h"
#include "kvi_app.h"
#include "kvi_channel.h"
#include "kvi_uparser.h"
#include "kvi_command.h"
#include "kvi_locale.h"
#include "kvi_topicw.h"
#include "kvi_datacontainer.h"
#include "kvi_ircmask.h"
#include "kvi_maskeditor.h"

static KviChannel * chan_module_find_channel(KviCommand *c,KviParameterList * parms,bool bNext = false)
{
	KviStr szChan = bNext ? parms->safeNextParam() : parms->safeFirstParam();
	if(szChan.isEmpty())
	{
		if(c->window()->type() == KVI_WINDOW_TYPE_CHANNEL)return (KviChannel *)(c->window());
		if(c->window()->type() == KVI_WINDOW_TYPE_DEADCHANNEL)return (KviChannel *)(c->window());
		c->warning(__tr("The current window is not a channel"));
	} else {
		KviWindow * w = g_pApp->findWindow(szChan.ptr());
		if(!w)
		{
			c->warning(__tr("Can't find the window with id '%s'"),szChan.ptr());
			return 0;
		}
		if(w->type() == KVI_WINDOW_TYPE_CHANNEL)return (KviChannel *)w;
		c->warning(__tr("The specified window (%s) is not a channel"),szChan.ptr());
	}
	return 0;
}

/*
	@doc: chan.name
	@type:
		function
	@title:
		$chan.name
	@short:
		Returns the name of a channel
	@syntax:
		$chan.name
		$chan.name(<window_id>)
	@description:
		The first form returns the name of the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the name of the channel specified by <window_id>.[br]
		This function works also on dead channels.[br]
*/


static bool chan_module_fnc_name(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::name");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(ch->windowName());
	return c->leaveStackFrame();
}


/*
	@doc: chan.isDead
	@type:
		function
	@title:
		$chan.isDead
	@short:
		Checks if a channel is dead
	@syntax:
		$chan.isDead
		$chan.isDead(<window_id>)
	@description:
		Returns 1 if the channel specified by <window_id> is a dead channel and 0 otherwise.[br]
		The form without parameters works on the current window.[br]
*/


static bool chan_module_fnc_isDead(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::isDead");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append((ch->type() == KVI_WINDOW_TYPE_DEADCHANNEL) ? '1' : '0');
	return c->leaveStackFrame();
}

/*
	@doc: chan.topic
	@type:
		function
	@title:
		$chan.topic
	@short:
		Returns the topic of a channel
	@syntax:
		$chan.topic
		$chan.topic(<window_id>)
	@description:
		The first form returns the topic of the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the topic of the channel specified by <window_id>.[br]
		The topic is returned as it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get an empty string (the topic is not yet known).[br]
		This function works also on dead channels altough the topic returned is the last
		topic seen while the channel wasn't dead.[br]
*/


static bool chan_module_fnc_topic(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::topic");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(ch->topicWidget()->topic());
	return c->leaveStackFrame();
}


/*
	@doc: chan.topicsetby
	@type:
		function
	@title:
		$chan.topicsetby
	@short:
		Returns the author of the topic of a channel
	@syntax:
		$chan.topicsetby
		$chan.topicsetby(<window_id>)
	@description:
		The first form returns the author of the topic of the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the author of the topic of the channel specified by <window_id>.[br]
		The topic author nickname is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get an empty string (the topic is not yet known).[br]
		This function works also on dead channels altough the information returned is the last
		information seen while the channel wasn't dead.[br]
*/


static bool chan_module_fnc_topicsetby(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::topicsetby");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(ch->topicWidget()->topicSetBy());
	return c->leaveStackFrame();
}


/*
	@doc: chan.topicsetat
	@type:
		function
	@title:
		$chan.topicsetat
	@short:
		Returns the set time of the topic of a channel
	@syntax:
		$chan.topicsetat
		$chan.topicsetat(<window_id>)
	@description:
		The first form returns the set time of the topic of the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the set time of the topic of the channel specified by <window_id>.[br]
		The topic set time is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get an empty string (the topic is not yet known).[br]
		This function works also on dead channels altough the information returned is the last
		information seen while the channel wasn't dead.[br]
*/


static bool chan_module_fnc_topicsetat(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::topicsetat");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(ch->topicWidget()->topicSetAt());
	return c->leaveStackFrame();
}



/*
	@doc: chan.usercount
	@type:
		function
	@title:
		$chan.usercount
	@short:
		Returns the number of users on a channel
	@syntax:
		$chan.usercount
		$chan.usercount(<window_id>)
	@description:
		The first form returns the number of users on the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the number of users on the channel specified by <window_id>.[br]
		The number of users is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_usercount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::usercount");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%u",ch->count());
	return c->leaveStackFrame();
}

/*
	@doc: chan.ownercount
	@type:
		function
	@title:
		$chan.ownercount
	@short:
		Returns the number of channel owner users on a channel
	@syntax:
		$chan.ownercount
		$chan.ownercount(<window_id>)
	@description:
		The first form returns the number of owners users on the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the number of owners users on the channel specified by <window_id>.[br]
		The number of owners is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_ownercount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::ownercount");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%d",ch->chanOwnerCount());
	return c->leaveStackFrame();
}

/*
	@doc: chan.admincount
	@type:
		function
	@title:
		$chan.admincount
	@short:
		Returns the number of channel admin users on a channel
	@syntax:
		$chan.admincount
		$chan.admincount(<window_id>)
	@description:
		The first form returns the number of administrator users on the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the number of administrator users on the channel specified by <window_id>.[br]
		The number of administrators is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_admincount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::admincount");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%d",ch->chanAdminCount());
	return c->leaveStackFrame();
}



/*
	@doc: chan.opcount
	@type:
		function
	@title:
		$chan.opcount
	@short:
		Returns the number of op users on a channel
	@syntax:
		$chan.opcount
		$chan.opcount(<window_id>)
	@description:
		The first form returns the number of op users on the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the number of op users on the channel specified by <window_id>.[br]
		The number of ops is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_opcount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::opcount");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%d",ch->opCount());
	return c->leaveStackFrame();
}


/*
	@doc: chan.voicecount
	@type:
		function
	@title:
		$chan.voicecount
	@short:
		Returns the number of voiced users on a channel
	@syntax:
		$chan.voicecount
		$chan.voicecount(<window_id>)
	@description:
		The first form returns the number of voiced users on the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the number of voiced users on the channel specified by <window_id>.[br]
		The number of voiced users is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_voicecount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::voicecount");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%d",ch->voiceCount());
	return c->leaveStackFrame();
}

/*
	@doc: chan.halfopcount
	@type:
		function
	@title:
		$chan.halfopcount
	@short:
		Returns the number of halfop users on a channel
	@syntax:
		$chan.halfOpCount
		$chan.halfOpCount(<window_id>)
	@description:
		The first form returns the number of half-operator users on the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the number of half-operator users on the channel specified by <window_id>.[br]
		The number of half-operator users is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_halfopcount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::halfopcount");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%d",ch->halfOpCount());
	return c->leaveStackFrame();
}

/*
	@doc: chan.useropcount
	@type:
		function
	@title:
		$chan.useropcount
	@short:
		Returns the number of userop users on a channel
	@syntax:
		$chan.userOpCount
		$chan.userOpCount(<window_id>)
	@description:
		The first form returns the number of user-operator users on the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and an empty string is returned.[br]
		The second form returns the number of user-operator users on the channel specified by <window_id>.[br]
		The number of user-operator users is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_useropcount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::useropcount");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%d",ch->userOpCount());
	return c->leaveStackFrame();
}

/*
	@doc: chan.bancount
	@type:
		function
	@title:
		$chan.bancount
	@short:
		Returns the number of entries in the channel ban list
	@syntax:
		$chan.bancount
		$chan.bancount(<window_id>)
	@description:
		The first form returns the number of entries in the ban list of the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and 0 is returned.[br]
		The second form returns the number entries in the ban list of the channel specified by <window_id>.[br]
		The number of ban list entries is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_bancount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::bancount");
	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%d",ch->banCount());
	return c->leaveStackFrame();
}

/*
	@doc: chan.banexceptioncount
	@type:
		function
	@title:
		$chan.banexceptioncount
	@short:
		Returns the number of entries in the channel ban exception list
	@syntax:
		$chan.banexceptioncount
		$chan.banexceptioncount(<window_id>)
	@description:
		The first form returns the number of entries in the ban exception list of the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and 0 is returned.[br]
		The second form returns the number entries in the ban exception list of the channel specified by <window_id>.[br]
		The number of ban exception list entries is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_banexceptioncount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::banexceptioncount");
	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%d",ch->banExceptionCount());
	return c->leaveStackFrame();
}


/*
	@doc: chan.invitecount
	@type:
		function
	@title:
		$chan.invitecount
	@short:
		Returns the number of entries in the channel invite list
	@syntax:
		$chan.invitecount
		$chan.invitecount(<window_id>)
	@description:
		The first form returns the number of entries in the invite list of the current channel (assuming that the current window
		is a channel at all). If the current window is not a channel, a warning is printed
		and 0 is returned.[br]
		The second form returns the number entries in the invite list of the channel specified by <window_id>.[br]
		The number of invite list entries is returned if it is known form at the call time: this means that 
		if the channel is not synchronized with the server (as just after the join, for example)
		you might get a number that is actually smaller.[br]
*/


static bool chan_module_fnc_invitecount(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::invitecount");
	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(KviStr::Format,"%d",ch->inviteCount());
	return c->leaveStackFrame();
}




/*
	@doc: chan.ison
	@type:
		function
	@title:
		$chan.ison
	@short:
		Checks if an user is on a channel
	@syntax:
		$chan.ison(<nickname>[,<window_id>])
	@description:
		Returns 1 if <nickname> is on the channel identified by <window_id>, 0 otherwise.[br]
		If <window_id> is not specified the current window is used (assuming that it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
*/






static bool chan_module_fnc_ison(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::ison");
	KviStr szNick = parms->safeFirstParam();
	KviChannel * ch = chan_module_find_channel(c,parms,true);
	if(ch)buffer.append(ch->isOn(szNick.ptr()) ? '1' : '0');
	return c->leaveStackFrame();
}

/*
	@doc: chan.getflag
	@type:
		function
	@title:
		$chan.getflag
	@short:
		Returns the channel-user mode flag of an user
	@syntax:
		$chan.getflag(<nickname>[,<window_id>])
	@description:
		Returns the channel user mode flag of an user on the channel specified by <window_id>.[br]
		If <window_id> is not passed, the current window is used.[br]
		If the specified window is not a channel, a warning is printed and an empty string is returned.[br]
		If the specified user is not on the channel identified by <window_id>, and empty string is returned.[br]
		The possible user flags are:[br]
			'!' for channel administrators[br]
			'@' for ops[br]
			'%' for halfops[br]
			'+' for voiced users[br]
			'-' for userops[br]
		If the user has more than one flag then the highest one is returned.[br]
*/


static bool chan_module_fnc_getflag(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::getflag");
	KviStr szNick = parms->safeFirstParam();
	KviChannel * ch = chan_module_find_channel(c,parms,true);
	if(ch)
	{
		char flag = ch->userListView()->getUserFlag(szNick.ptr());
		if(flag)buffer.append(flag);
	}
	return c->leaveStackFrame();
}


#define IS_FUNC(__clbkname,__stackframe,__chanfunc) \
static bool __clbkname(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer) \
{ \
	ENTER_STACK_FRAME(c,__stackframe); \
	KviStr szNick = parms->safeFirstParam(); \
	KviChannel * ch = chan_module_find_channel(c,parms,true); \
	if(ch)buffer.append(ch->__chanfunc(szNick.ptr()) ? '1' : '0'); \
	return c->leaveStackFrame(); \
}


/*
	@doc: chan.isowner
	@type:
		function
	@title:
		$chan.isowner
	@short:
		Checks if an user is at least a channel owner
	@syntax:
		$chan.isowner(<nickname>[,<window_id>])
	@description:
		Returns 1 if <nickname> is at least an owner on the channel identified by <window_id>, 0 otherwise.[br]
		If <window_id> is not specified the current window is used (assuming that it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		Note that if the user is not on the channel at all, you will get 0 as return value.[br]
*/

IS_FUNC(chan_module_fnc_isowner,"chan::isowner",isChanOwner)

/*
	@doc: chan.isadmin
	@type:
		function
	@title:
		$chan.isadmin
	@short:
		Checks if an user is at least channel administrator
	@syntax:
		$chan.isadmin(<nickname>[,<window_id>])
	@description:
		Returns 1 if <nickname> is at least an administrator on the channel identified by <window_id>, 0 otherwise.[br]
		If <window_id> is not specified the current window is used (assuming that it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		Note that if the user is not on the channel at all, you will get 0 as return value.[br]
*/

IS_FUNC(chan_module_fnc_isadmin,"chan::isadmin",isChanAdmin)

/*
	@doc: chan.isop
	@type:
		function
	@title:
		$chan.isop
	@short:
		Checks if an user is at least an op on a channel
	@syntax:
		$chan.isop(<nickname>[,<window_id>])
	@description:
		Returns 1 if <nickname> is at least an operator on the channel identified by <window_id>, 0 otherwise.[br]
		If <window_id> is not specified the current window is used (assuming that it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		Note that if the user is not on the channel at all, you will get 0 as return value.[br]
*/

IS_FUNC(chan_module_fnc_isop,"chan::isop",isOp)

/*
	@doc: chan.isvoice
	@type:
		function
	@title:
		$chan.isvoice
	@short:
		Checks if an user is at least voiced on a channel
	@syntax:
		$chan.isop(<nickname>[,<window_id>])
	@description:
		Returns 1 if <nickname> is at least voiced on the channel identified by <window_id>, 0 otherwise.[br]
		If <window_id> is not specified the current window is used (assuming that it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		Note that if the user is not on the channel at all, you will get 0 as return value.[br]
*/

IS_FUNC(chan_module_fnc_isvoice,"chan::isvoice",isVoice)

/*
	@doc: chan.ishalfop
	@type:
		function
	@title:
		$chan.ishalfop
	@short:
		Checks if an user is at least halfop on a channel
	@syntax:
		$chan.ishalfop(<nickname>[,<window_id>])
	@description:
		Returns 1 if <nickname> is at least a half-operator on the channel identified by <window_id>, 0 otherwise.[br]
		If <window_id> is not specified the current window is used (assuming that it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		Note that if the user is not on the channel at all, you will get 0 as return value.[br]
*/

IS_FUNC(chan_module_fnc_ishalfop,"chan::ishalfop",isHalfOp)

/*
	@doc: chan.isuserop
	@type:
		function
	@title:
		$chan.isuserop
	@short:
		Checks if an user is at least an userop on a channel
	@syntax:
		$chan.isuserop(<nickname>[,<window_id>])
	@description:
		Returns 1 if <nickname> is at least an user-operator on the channel identified by <window_id>, 0 otherwise.[br]
		If <window_id> is not specified the current window is used (assuming that it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		Note that if the user is not on the channel at all, you will get 0 as return value.[br]
*/

IS_FUNC(chan_module_fnc_isuserop,"chan::isuserop",isUserOp)

/*
	@doc: chan.ismeowner
	@type:
		function
	@title:
		$chan.isMeOwner
	@short:
		Checks if the current user is at least an owner on a channel
	@syntax:
		$chan.isMeOwner
		$chan.isMeOwner(<window_id>)
	@description:
		Returns 1 if the current user is at least an owner on the channel specified by <window_id>, 0 otherwise.[br]
		If <window_id> is not passed, the current window is used (assuming it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		This function is a "shortcut" for [fnc]$chan.isowner[/fnc]([fnc]$me[/fnc]).[br]
*/

#define IS_ME_FUNC(__clbkname,__stackframe,__chanfunc) \
static bool __clbkname(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer) \
{ \
	ENTER_STACK_FRAME(c,__stackframe); \
	KviChannel * ch = chan_module_find_channel(c,parms,false); \
	if(ch)buffer.append(ch->__chanfunc(true) ? '1' : '0'); \
	return c->leaveStackFrame(); \
}


IS_ME_FUNC(chan_module_fnc_ismeowner,"chan::ismeowner",isMeChanOwner)


/*
	@doc: chan.ismeadmin
	@type:
		function
	@title:
		$chan.isMeAdmin
	@short:
		Checks if the current user is at least an administrator on a channel
	@syntax:
		$chan.isMeAdmin
		$chan.isMeAdmin(<window_id>)
	@description:
		Returns 1 if the current user is at least an administrator on the channel specified by <window_id>, 0 otherwise.[br]
		If <window_id> is not passed, the current window is used (assuming it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		This function is a "shortcut" for [fnc]$chan.isadmin[/fnc]([fnc]$me[/fnc]).[br]
*/


IS_ME_FUNC(chan_module_fnc_ismeadmin,"chan::ismeadmin",isMeChanAdmin)

/*
	@doc: chan.ismeop
	@type:
		function
	@title:
		$chan.isMeOp
	@short:
		Checks if the current user is at least op on a channel
	@syntax:
		$chan.isMeOp
		$chan.isMeOp(<window_id>)
	@description:
		Returns 1 if the current user is at least op on the channel specified by <window_id>, 0 otherwise.[br]
		If <window_id> is not passed, the current window is used (assuming it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		This function is a "shortcut" for [fnc]$chan.isop[/fnc]([fnc]$me[/fnc]).[br]
*/

IS_ME_FUNC(chan_module_fnc_ismeop,"chan::ismeop",isMeOp)

/*
	@doc: chan.ismehalfop
	@type:
		function
	@title:
		$chan.isMeHalfOp
	@short:
		Checks if the current user is at least an half operator on a channel
	@syntax:
		$chan.isMeHalfOp
		$chan.isMeHalfOp(<window_id>)
	@description:
		Returns 1 if the current user is at least an half operator on the channel specified by <window_id>, 0 otherwise.[br]
		If <window_id> is not passed, the current window is used (assuming it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		This function is a "shortcut" for [fnc]$chan.ishalfop[/fnc]([fnc]$me[/fnc]).[br]
*/

IS_ME_FUNC(chan_module_fnc_ismehalfop,"chan::ismehalfop",isMeHalfOp)

/*
	@doc: chan.ismevoice
	@type:
		function
	@title:
		$chan.isMeVoice
	@short:
		Checks if the current user is at least voice on a channel
	@syntax:
		$chan.isMeVoice
		$chan.isMeVoice(<window_id>)
	@description:
		Returns 1 if the current user is at least voice on the channel specified by <window_id>, 0 otherwise.[br]
		If <window_id> is not passed, the current window is used (assuming it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		This function is a "shortcut" for [fnc]$chan.isvoice[/fnc]([fnc]$me[/fnc]).[br]
*/

IS_ME_FUNC(chan_module_fnc_ismevoice,"chan::ismevoice",isMeVoice)

/*
	@doc: chan.ismeuserop
	@type:
		function
	@title:
		$chan.isMeUserOp
	@short:
		Checks if the current user is at least an user operator on a channel
	@syntax:
		$chan.isMeUserOp
		$chan.isMeUserOp(<window_id>)
	@description:
		Returns 1 if the current user is at least an user operator on the channel specified by <window_id>, 0 otherwise.[br]
		If <window_id> is not passed, the current window is used (assuming it is a channel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		This function is a "shortcut" for [fnc]$chan.isuserop[/fnc]([fnc]$me[/fnc]).[br]
*/

IS_ME_FUNC(chan_module_fnc_ismeuserop,"chan::ismeuserop",isMeUserOp)

/*
	@doc: chan.mode
	@type:
		function
	@title:
		$chan.mode
	@short:
		Returns the mode string of a channel
	@syntax:
		$chan.mode
		$chan.mode(<window_id>)
	@description:
		Returns the mode string of the channel identified by <window_id>.[br]
		If no <window_id> is passed, the current channel mode string is returned (assuming that
		the current window is a chnannel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
*/


static bool chan_module_fnc_mode(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::mode");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)
	{
		QString chMode;
		ch->getChannelModeString(chMode);
		buffer.append(chMode);
	}
	return c->leaveStackFrame();
}

/*
	@doc: chan.key
	@type:
		function
	@title:
		$chan.key
	@short:
		Returns the key of a channel
	@syntax:
		$chan.key
		$chan.key(<window_id>)
	@description:
		Returns the key of the channel identified by <window_id>.[br]
		If no <window_id> is passed, the current channel key is returned (assuming that
		the current window is a chnannel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		If the channel has no key set, an empty string is returned.[br]
*/


static bool chan_module_fnc_key(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::key");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)buffer.append(ch->channelKey());
	return c->leaveStackFrame();
}

/*
	@doc: chan.limit
	@type:
		function
	@title:
		$chan.limit
	@short:
		Returns the limit of a channel
	@syntax:
		$chan.limit
		$chan.limit(<window_id>)
	@description:
		Returns the limit of the channel identified by <window_id>.[br]
		If no <window_id> is passed, the current channel limit is returned (assuming that
		the current window is a chnannel at all).[br]
		If the window is not a channel, a warning is printed and an empty string is returned.[br]
		If the channel has no limit set, "0" is returned.[br]
*/


static bool chan_module_fnc_limit(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::limit");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(ch)
	{
		if(ch->hasChannelLimit())buffer.append(ch->channelLimit());
		else buffer.append('0');
	}
	return c->leaveStackFrame();
}


/*
	@doc: chan.users
	@type:
		function
	@title:
		$chan.users
	@short:
		Returns an array of channel user nicknames
	@syntax:
		$chan.users([window_id],[mask],[flags])
	@description:
		Returns an array of nicknames on the channel specified by [window_id].[br]
		If [window_id] is empty, the current window is used.[br]
		If the window designated by [window_id] is not a channel a warning is printed and an empty array is returned.[br]
		If [mask] is given, each user is added to the array only
		if it matches the [mask].[br]
		[flags] may contain a subset of the letters "aovhnmi":[br]
		"ovhn" are mode flags: the users are added to the array only if they are channel administrators ('a'), operators ('o'),
		voiced users ('v'), half-operators ('h'), user-operators ('u') or unflagged ('n') users. (Unflagged means not operators, not
		voiced and not half-operators). If none of the "ovhun" flags are used, KVIrc behaves like all five were passed.[br]
		The flag 'm' causes the entire user masks to be added to the 
		array entries, as known by KVIrc at the moment of this function call.[br]
		The flag 'i' causes KVIrc to invert the match and add only the users that do NOT match [mask].[br]
		Please note that on really large channels this function may be time consuming (especially if [mask] is used):
		use with care.[br]
	@example:
		[example]
			[comment]# Get the nickname list[/comment]
			%test[] = $chan.users
			[comment]# And loop thru the items[/comment]
			%i = 0
			[comment]# %test[]# returns the number of elements in the array[/comment]
			%count = %test[]#
			while(%i < %count)
			{
				echo "User %i : %test[%i]"
				%i++
			}
			[comment]# Another way of looping[/comment]
			foreach(%tmp,%test[])echo "User %tmp"
			[comment]# Find all the users that come from the *.org domain[/comment]
			%test[]=$chan.users(,*!*@*.org)
			echo %test[]
			[comment]# This will work too![/comment]
			echo $chan.users(,*!*@*.org)
			[comment]# Find all the channel operators[/comment]
			%test[] = $chan.users(,,o)
			echo %test[]
			[comment]# Find all the voiced users that do NOT come from *.edu[/comment]
			[comment]# See also their whole mask[/comment]
			%test[] = $chan.users(,*!*@*.edu,vim)
			echo %test[]
		[/example]
		
*/


static bool chan_module_fnc_users(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan_module_fnc_users");

	c->beginListArrayOrDictionaryReturnIdentifier();

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(!ch)return c->leaveStackFrame();

	KviStr * szMask = parms->safeNext();
	KviStr * szFlags = parms->safeNext();

//	QAsciiDictIterator<KviUserListEntry> it(*(((KviChannel *)(c->window()))->entryDict()));
	KviUserListEntry * e = ch->userListView()->firstItem(); // Thnx Julien :)

	bool bCheckMask = szMask->hasData();
	bool bOp = szFlags->contains('o');
	bool bVoice = szFlags->contains('v');
	bool bHalfOp = szFlags->contains('h');
	bool bChanAdmins = szFlags->contains('a');
	bool bUserOp = szFlags->contains('u');
	bool bNone = szFlags->contains('n');
	bool bCheckFlags = bOp || bVoice || bHalfOp || bNone || bChanAdmins || bUserOp;
	bool bAddMask = szFlags->contains('m');

	int idx = 0;

	if(bAddMask || bCheckFlags || bCheckMask)
	{
		bool bMaskMustMatch = !(szFlags->contains('i'));
		KviIrcMask mask(szMask->ptr());

		while(e)
		{
			if(bCheckFlags)
			{
				if(bChanAdmins)
				{
					if(e->flags() & KVI_USERFLAG_CHANADMIN)goto check_mask;
				}
				if(bOp)
				{
					if(e->flags() & KVI_USERFLAG_OP)goto check_mask;
				}
				if(bVoice)
				{
					if(e->flags() & KVI_USERFLAG_VOICE)goto check_mask;
				}
				if(bHalfOp)
				{
					if(e->flags() & KVI_USERFLAG_HALFOP)goto check_mask;
				}
				if(bUserOp)
				{
					if(e->flags() & KVI_USERFLAG_USEROP)goto check_mask;
				}
				if(bNone)
				{
					if(!(e->flags() & KVI_USERFLAG_MASK))goto check_mask;
				}
				goto next_item;
			}
check_mask:
			if(bCheckMask)
			{
				if(mask.matchesFixed(e->nick(),e->globalData()->user(),e->globalData()->host()) == bMaskMustMatch)goto add_item;
				goto next_item;
			}
add_item:
			if(bAddMask)
			{
				KviStr x(e->nick());
				x.append('!');
				x.append(e->globalData()->user());
				x.append('@');
				x.append(e->globalData()->host());
				c->addListArrayOrDictionaryReturnValue(idx,x,buffer);
			} else {
				c->addListArrayOrDictionaryReturnValue(idx,e->nick().utf8().data(),buffer);
			}
			idx++;
next_item:
			e = e->next();
		}
	} else {
		while(e)
		{
			c->addListArrayOrDictionaryReturnValue(idx,e->nick().utf8().data(),buffer);
			idx++;
			e = e->next();
		}
	}
	return c->leaveStackFrame();

}

/*
	@doc: chan.banlist
	@type:
		function
	@title:
		$chan.banlist
	@short:
		Returns an array of channel ban masks
	@syntax:
		$chan.banlist([window_id])
	@description:
		Returns an array of ban masks set ont the channel identified by [window_id].[br]
		If [window_id] is empty, the current window is used.[br]
		If the window designated by [window_id] is not a channel a warning is printed and an empty array is returned.[br]
*/


static bool chan_module_fnc_banlist(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::banlist");

	c->beginListArrayOrDictionaryReturnIdentifier();

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(!ch)return c->leaveStackFrame();

	int idx = 0;

	KviPtrList<KviMaskEntry> * l = ((KviChannel *)(c->window()))->banList();

	for(KviMaskEntry * e = l->first();e;e = l->next())
	{
		c->addListArrayOrDictionaryReturnValue(idx,e->szMask.utf8().data(),buffer);
		idx++;
	}
	return c->leaveStackFrame();
}


/*
	@doc: chan.banexceptionlist
	@type:
		function
	@title:
		$chan.banexceptionlist
	@short:
		Returns an array of channel ban exception masks
	@syntax:
		$chan.banexceptionlist([window_id])
	@description:
		Returns an array of ban exception masks set ont the channel identified by [window_id].[br]
		If [window_id] is empty, the current window is used.[br]
		If the window designated by [window_id] is not a channel a warning is printed and an empty array is returned.[br]
*/


static bool chan_module_fnc_banexceptionlist(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::banexceptionlist");

	c->beginListArrayOrDictionaryReturnIdentifier();

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(!ch)return c->leaveStackFrame();

	int idx = 0;

	KviPtrList<KviMaskEntry> * l = ((KviChannel *)(c->window()))->banExceptionList();

	for(KviMaskEntry * e = l->first();e;e = l->next())
	{
		c->addListArrayOrDictionaryReturnValue(idx,e->szMask.utf8().data(),buffer);
		idx++;
	}
	return c->leaveStackFrame();
}


/*
	@doc: chan.invitelist
	@type:
		function
	@title:
		$chan.invitelist
	@short:
		Returns an array of channel invite masks
	@syntax:
		$chan.banexceptionlist([window_id])
	@description:
		Returns an array of invite masks set ont the channel identified by [window_id].[br]
		If [window_id] is empty, the current window is used.[br]
		If the window designated by [window_id] is not a channel a warning is printed and an empty array is returned.[br]
*/


static bool chan_module_fnc_invitelist(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::invitelist");

	c->beginListArrayOrDictionaryReturnIdentifier();

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(!ch)return c->leaveStackFrame();

	int idx = 0;

	KviPtrList<KviMaskEntry> * l = ((KviChannel *)(c->window()))->inviteList();

	for(KviMaskEntry * e = l->first();e;e = l->next())
	{
		c->addListArrayOrDictionaryReturnValue(idx,e->szMask.utf8().data(),buffer);
		idx++;
	}
	return c->leaveStackFrame();
}

/*
	@doc: chan.matchban
	@type:
		function
	@title:
		$chan.matchban
	@short:
		Matches a mask agains the channel ban list
	@syntax:
		$chan.matchban([window_id],<complete_mask>)
	@description:
		Returns the ban mask that matches <complete_mask> on channel identified by [window_id].[br]
		If no ban mask matches <complete_mask> an empty string is returned.[br]
		If [window_id] is empty, the current window is used.[br]
		If the window designated by [window_id] is not a channel a warning is printed and an empty string is returned.[br]
		This function is useful to determine if a ban set on the channel matches an user.[br]
*/


static bool chan_module_fnc_matchban(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::matchban");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(!ch)return c->leaveStackFrame();

	KviStr * szMask = parms->safeNext();

	KviPtrList<KviMaskEntry> * l = ((KviChannel *)(c->window()))->banList();

	for(KviMaskEntry * e = l->first();e;e = l->next())
	{
		if(kvi_matchString(e->szMask.utf8().data(),szMask->ptr()))
		{
			buffer.append(e->szMask);
			return c->leaveStackFrame();
		}
	}
	return c->leaveStackFrame();
}



/*
	@doc: chan.matchbanexception
	@type:
		function
	@title:
		$chan.matchbanexception
	@short:
		Matches a mask agains the channel ban exception list
	@syntax:
		$chan.matchbanexception([window_id],<complete_mask>)
	@description:
		Returns the ban exception mask that matches <complete_mask> on channel identified by [window_id].[br]
		If no ban exception mask matches <complete_mask> an empty string is returned.[br]
		If [window_id] is empty, the current window is used.[br]
		If the window designated by [window_id] is not a channel a warning is printed and an empty string is returned.[br]
		This function is useful to determine if a ban exception set on the channel matches an user.[br]
*/


static bool chan_module_fnc_matchbanexception(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::matchbanexception");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(!ch)return c->leaveStackFrame();

	KviStr * szMask = parms->safeNext();

	KviPtrList<KviMaskEntry> * l = ((KviChannel *)(c->window()))->banExceptionList();

	for(KviMaskEntry * e = l->first();e;e = l->next())
	{
		if(kvi_matchString(e->szMask.utf8().data(),szMask->ptr()))
		{
			buffer.append(e->szMask);
			return c->leaveStackFrame();
		}
	}
	return c->leaveStackFrame();
}


/*
	@doc: chan.matchinvite
	@type:
		function
	@title:
		$chan.matchinvite
	@short:
		Matches a mask agains the channel invite list
	@syntax:
		$chan.matchban([window_id],<complete_mask>)
	@description:
		Returns the invite mask that matches <complete_mask> on channel identified by [window_id].[br]
		If no invite mask matches <complete_mask> an empty string is returned.[br]
		If [window_id] is empty, the current window is used.[br]
		If the window designated by [window_id] is not a channel a warning is printed and an empty string is returned.[br]
		This function is useful to determine if a invite set on the channel matches an user.[br]
*/


static bool chan_module_fnc_matchinvite(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::matchinvite");

	KviChannel * ch = chan_module_find_channel(c,parms);
	if(!ch)return c->leaveStackFrame();

	KviStr * szMask = parms->safeNext();

	KviPtrList<KviMaskEntry> * l = ((KviChannel *)(c->window()))->inviteList();

	for(KviMaskEntry * e = l->first();e;e = l->next())
	{
		if(kvi_matchString(e->szMask.utf8().data(),szMask->ptr()))
		{
			buffer.append(e->szMask);
			return c->leaveStackFrame();
		}
	}
	return c->leaveStackFrame();
}



/*
	@doc: chan.usermodelevel
	@type:
		function
	@title:
		$chan.usermodelevel
	@short:
		Returns the channel user-mode level
	@syntax:
		$chan.userModeLevel(<nickname>[,<window_id>])
	@description:
		Returns an integer identifying the specified user's channel mode on the channel specified by <window_id>.[br]
		If <window_id> is not passed, the current window is used.[br]
		If the specified window is not a channel, a warning is printed and '0' is returned.[br]
		This number can be useful to implement comparison functions between
		users in order to determine the actions they can issue between each other.[br]
		For example it is granted that an op will have userModeLevel greater than
		a voiced user or that a simple "modeless" user will have
		an userModeLevel lower than a halfop.[br]
		IRC allows multiple modes to be applied to a single user on a channel,
		in that case this function will return the level of the highest mode
		applied to the user.[br]
		Note: Don't rely on this number being any particular exact value except
		for the completely modeless users (in which case this function will return always '0').
*/

static bool chan_module_fnc_usermodelevel(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::usermodelevel");
	KviStr szNick = parms->safeFirstParam();
	KviChannel * ch = chan_module_find_channel(c,parms,true);
	if(ch)
		buffer.append(KviStr::Format,"%d",ch->userListView()->getUserModeLevel(szNick.ptr()));
	else
		buffer.append('0');
	return c->leaveStackFrame();
}


/*
	@doc: chan.userjointime
	@type:
		function
	@title:
		$chan.userJoinTime
	@short:
		Returns the time that an user has joined the channel
	@syntax:
		$chan.userJoinTime(<nickname>[,<window_id>])
	@description:
		Returns the unix time at which the user specified by <nickname> has joined the channel specified by <window_id>.[br]
		If <window_id> is not passed, the current window is used.[br]
		If the specified window is not a channel, a warning is printed and '0' is returned.[br]
		'0' is also returned when the user's join time is unknown: this is true
		for all the users that were on the channel before the local user has joined it (e.g.
		you know the join time only for users that YOU see joining).[br]
	@seealso:
		[fnc]$unixtime[/fnc], [fnc]$chan.userLastActionTime[/fnc]()
*/

static bool chan_module_fnc_userjointime(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::userjointime");
	KviStr szNick = parms->safeFirstParam();
	KviChannel * ch = chan_module_find_channel(c,parms,true);
	if(ch)
		buffer.append(KviStr::Format,"%d",ch->userListView()->getUserJoinTime(szNick.ptr()));
	else
		buffer.append('0');
	return c->leaveStackFrame();
}


/*
	@doc: chan.userlastactiontime
	@type:
		function
	@title:
		$chan.userLastActionTime
	@short:
		Returns the time that an user has last performed some kind of action on the channel
	@syntax:
		$chan.userLastActionTime(<nickname>[,<window_id>])
	@description:
		Returns the unix time at which the user specified by <nickname> has performed
		some kind of action on the channel specified by <window_id>.[br]
		If <window_id> is not passed, the current window is used.[br]
		If the specified window is not a channel, a warning is printed and '0' is returned.[br]
		'0' is also returned when the user's last action time is unknown: this is true
		for all the users that were on the channel before the local user has joined it
		and have performed no actions since that moment.
	@seealso:
		[fnc]$unixtime[/fnc](), [fnc]$chan.userJoinTime[/fnc]()
*/

static bool chan_module_fnc_userlastactiontime(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"chan::userlastactiontime");
	KviStr szNick = parms->safeFirstParam();
	KviChannel * ch = chan_module_find_channel(c,parms,true);
	if(ch)
		buffer.append(KviStr::Format,"%d",ch->userListView()->getUserLastActionTime(szNick.ptr()));
	else
		buffer.append('0');
	return c->leaveStackFrame();
}


static bool chan_module_init(KviModule * m)
{
	m->registerFunction("name",chan_module_fnc_name);

	m->registerFunction("topic",chan_module_fnc_topic);
	m->registerFunction("topicsetby",chan_module_fnc_topicsetby);
	m->registerFunction("topicsetat",chan_module_fnc_topicsetat);

	m->registerFunction("usercount",chan_module_fnc_usercount);

	m->registerFunction("ownercount",chan_module_fnc_ownercount);
	m->registerFunction("admincount",chan_module_fnc_admincount);
	m->registerFunction("opcount",chan_module_fnc_opcount);
	m->registerFunction("voicecount",chan_module_fnc_voicecount);
	m->registerFunction("halfopcount",chan_module_fnc_halfopcount);
	m->registerFunction("useropcount",chan_module_fnc_useropcount);

	m->registerFunction("ison",chan_module_fnc_ison);

	m->registerFunction("isowner",chan_module_fnc_isowner);
	m->registerFunction("isadmin",chan_module_fnc_isadmin);
	m->registerFunction("isop",chan_module_fnc_isop);
	m->registerFunction("isvoice",chan_module_fnc_isvoice);
	m->registerFunction("ishalfop",chan_module_fnc_ishalfop);
	m->registerFunction("isuserop",chan_module_fnc_isuserop);

	m->registerFunction("ismeuserop",chan_module_fnc_ismeuserop);
	m->registerFunction("ismevoice",chan_module_fnc_ismevoice);
	m->registerFunction("ismehalfop",chan_module_fnc_ismehalfop);
	m->registerFunction("ismeop",chan_module_fnc_ismeop);
	m->registerFunction("ismeadmin",chan_module_fnc_ismeadmin);
	m->registerFunction("ismeowner",chan_module_fnc_ismeowner);

	m->registerFunction("isDead",chan_module_fnc_isDead);

	m->registerFunction("getflag",chan_module_fnc_getflag);
	
	m->registerFunction("usermodelevel",chan_module_fnc_usermodelevel);
	m->registerFunction("userjointime",chan_module_fnc_userjointime);
	m->registerFunction("userlastactiontime",chan_module_fnc_userlastactiontime);

	m->registerFunction("mode",chan_module_fnc_mode);

	m->registerFunction("key",chan_module_fnc_key);
	m->registerFunction("limit",chan_module_fnc_limit);

	m->registerFunction("users",chan_module_fnc_users);

	m->registerFunction("bancount",chan_module_fnc_bancount);
	m->registerFunction("banexceptioncount",chan_module_fnc_banexceptioncount);
	m->registerFunction("invitecount",chan_module_fnc_invitecount);

	m->registerFunction("banlist",chan_module_fnc_banlist);
	m->registerFunction("banexceptionlist",chan_module_fnc_banexceptionlist);
	m->registerFunction("invitelist",chan_module_fnc_invitelist);

	m->registerFunction("matchban",chan_module_fnc_matchban);
	m->registerFunction("matchbanexception",chan_module_fnc_matchbanexception);
	m->registerFunction("matchinvite",chan_module_fnc_matchinvite);

	return true;
}

static bool chan_module_cleanup(KviModule *m)
{
	return true;
}


KVIRC_MODULE(
	"Chan",                                                  // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2002 Szymon Stefanek (pragma at kvirc dot net)"\
	"          (C) 2002 Juanjo Alvarez (juanjux at yahoo dot es)",
	"Scripting interface for the channel management",
	chan_module_init,
	0,
	0,
	chan_module_cleanup
)
