//vim: ts=8
//   File : class_listbox.cpp
//   Creation date : Sat Oct 2 03:40:28 CET 2004 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2004 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_command.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_debug.h"
#include "kvi_scriptobject.h"

#include <qlistbox.h>

#include "class_list.h"
#include "class_listbox.h"

static KviScriptObjectClass * g_pListBoxClass = 0;


static KviScriptObject * listBoxClassCreateInstance(KviScriptObjectClass * c,
	KviScriptObject * p, const char * n)
{
	return new KviScriptListBoxObject(c, p, n);
}

KviScriptListBoxObject::KviScriptListBoxObject(KviScriptObjectClass * c,
	KviScriptObject * p, const char * n) : KviScriptWidgetObject(c,p,n)
{
}

KviScriptListBoxObject::~KviScriptListBoxObject()
{
}

/*
	@doc: listbox
	@keyterms:
		listbox object class
	@title:
		listbox class
	@type:
		class
	@short:
		A widget displaying a list of items
	@inherits:
		[class]object[/class]
		[class]widget[/class]
	@description:
		The listbox class is a widget displaying a list of string items.[br]
		The listbox can operate in four selection modes: single, multi, extended
		and none. In the "single" mode only one item can be selected at a time
		and that item is also the current one. In the "multi" mode multiple
		items can be selected and unselected and the current item may or may not
		be one of the selected items. In the "extended" mode multiple items
		can be selected but they must be selected in a single mouse action
		or by keeping the shift button pressed while clicking on the item.
		In this mode the current item is always one of the selected items.[br]
		In the "none" mode no items can be selected and the current item
		may be any item in the list.[br]
		In most situations you will operate the listbox in "single" mode
		(and this is the default mode set at widget creation).
	@functions:
		!fn: $setSelectionMode(<mode>)
		Sets the current selection mode specified by the parameter <mode>
		that can be one of the following string:[br]
		"single" : only one item can be selected at a time[br]
		"multi" : multiple items can be selected at a time[br]
		"extended" : multiple items can be selected but only in a single mouse interaction[br]
		"none" : no items can be selected[br]
		The default mode is "single"
		!fn: $selectionMode()
		Returns the current selection mode
		!fn: $insertList(<list>, <index>)
		Inserts the items present in the <list> object starting from 
		index <index>, which is -1 by default (which means the items 
		will be appended).
		!fn: $insertItem(<str>, <index>)
		Inserts a text item at position <index>. If index is negative or
		not specified the item is appended.
		!fn: $changeItem(<str>, <index>)
		Changes text of item at <index> to <str>.
		!fn: $removeItem(<index>)
		Removes item at given index.
		!fn: $count()
		Returns number of items in the widget.
		!fn: $currentText()
		Returns the text of the currently selected item.
		!fn: $currentItem()
		Returns index of current item or -1 if no item is current.
		!fn: $textAt(<index>)
		Returns item at given index.
		!fn: $setCurrentItem(<index>)
		Sets the current listbox item.
		!fn: $clear()
		Removes all the items
		!fn: $isSelected(<index>)
		Returns 1 if the item at <index> is selected and 0 otherwise
		!fn: $setSelected(<index>,<bool>)
		Changes the selection status of the item at <index> to the value
		specified by <bool>. If <bool> is 1 the item is selected,
		if <bool> is 0 the item is unselected.
		!fn: $selectionChangeEvent()
		This function is called by KVIrc when the selection in the listbox changes.
		!fn: $currentItemChangeEvent()
		This function is called by KVIrc when the current item changes.
*/

bool KviScriptListBoxObject::init(KviCommand *)
{
	QListBox * b = new QListBox(parentScriptWidget(), name());
	b->setSelectionMode(QListBox::Single);
	connect(b,SIGNAL(selectionChanged()),this,SLOT(selectionChanged()));
	connect(b,SIGNAL(currentChanged(QListBoxItem *)),this,SLOT(currentItemChanged(QListBoxItem *)));
	setObject(b,true);
	return true;
}

void KviScriptListBoxObject::selectionChanged()
{
	callEventFunction("selectionChangeEvent",0,0);
}

void KviScriptListBoxObject::currentItemChanged(QListBoxItem *)
{
	callEventFunction("currentItemChangeEvent",0,0);

	//callEventFunction("connectFailedEvent",0,new KviParameterList(
	//		new KviStr(KviStr::Format,__tr("Invalid ip address (%s)"),m_szRemoteIp.ptr())));
}

#define listboxFuncReg(__nam, __func) \
        g_pListBoxClass->registerFunctionHandler(__nam, \
        (KviScriptObjectFunctionHandlerProc)( \
		KVI_PTR2MEMBER(KviScriptListBoxObject::__func)), 0, true);

void KviScriptListBoxObject::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController-> \
		lookupClass("widget");
	__range_valid(base);

	g_pListBoxClass = new KviScriptObjectClass(base, "listbox",
		listBoxClassCreateInstance, true);

	listboxFuncReg("insertList", functionInsertList);
	listboxFuncReg("insertItem", functionInsertItem);
	listboxFuncReg("changeItem", functionChangeItem);
	listboxFuncReg("removeItem", functionRemoveItem);
	listboxFuncReg("count", functionCount);
	listboxFuncReg("currentText", functionCurrentText);
	listboxFuncReg("currentItem", functionCurrentItem);
	listboxFuncReg("textAt", functionTextAt);
	listboxFuncReg("setCurrentItem", functionSetCurrentItem);
	listboxFuncReg("clear",functionClear);
	listboxFuncReg("setSelectionMode",functionSetSelectionMode);
	listboxFuncReg("selectionMode",functionSelectionMode);
	listboxFuncReg("setSelected",functionSetSelected);
	listboxFuncReg("isSelected",functionIsSelected); //Noldor

	g_pListBoxClass->registerEmptyFunctionHandler("currentItemChangeEvent");
	g_pListBoxClass->registerEmptyFunctionHandler("selectionChangeEvent");
}

void KviScriptListBoxObject::unregisterSelf()
{
	delete g_pListBoxClass;
    g_pListBoxClass = 0;
}

bool KviScriptListBoxObject::functionInsertList(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "listbox::insertList");

	if(!widget())
		return c->leaveStackFrame();
		
	if(!p->count())
		return c->error(KviError_notEnoughParameters);
	
	KviScriptListObject * pList = (KviScriptListObject *)g_pScriptObjectController-> \
		lookupObject(p->first()->ptr());
		
	bool bOk = true;
	int idx = (p->count() == 1 ? -1 : p->getInt(&bOk));

	if(!bOk)
		return c->error(KviError_integerParameterExpected);

	if(!pList)
		return c->error(KviError_noSuchObject);

	for(KviStr * str = pList->dataList()->first(); str; str = \
		pList->dataList()->next())
		((QListBox *)widget())->insertItem(str->ptr());
	
	return c->leaveStackFrame();
}

bool KviScriptListBoxObject::functionInsertItem(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "listbox::insertItem");
	
	if(!widget())
		return c->leaveStackFrame();

	if(!p->count())
		return c->error(KviError_notEnoughParameters);
		
	KviStr * pStr = p->first();
	KviStr * pNum = p->next();
	bool bOk = true;
	int idx = (pNum ? pNum->toInt(&bOk) : -1);

	if(!bOk)
		return c->error(KviError_integerParameterExpected);

	((QListBox *)widget())->insertItem(pStr->ptr(), idx);
	
	return c->leaveStackFrame();
}

bool KviScriptListBoxObject::functionChangeItem(KviCommand * c, KviParameterList * p,
	KviStr &)
{
	ENTER_STACK_FRAME(c, "listbox::changeItem");
	if(!widget())
		return c->leaveStackFrame();

	if(!p->count())
		return c->error(KviError_notEnoughParameters);

	KviStr * pStr = p->first();
	KviStr * pNum = p->next();
	if(!pNum)
		return c->error(KviError_integerParameterExpected);

	bool bOk = true;
	int idx = (pNum ? pNum->toInt(&bOk) : -1);

	if(!bOk)
		return c->error(KviError_integerParameterExpected);

	int cnt;
	if(idx >= (cnt = ((QListBox *)widget())->count()))
	{
		c->warning("Item index [%d] is too big - defaulting to " \
			"$count() - 1 [%d]", idx, cnt);
		idx = cnt - 1;
	}

	((QListBox *)widget())->changeItem(pStr->ptr(), idx);
		
	return c->leaveStackFrame();
}

bool KviScriptListBoxObject::functionRemoveItem(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "listbox::removeItem");

	if(!widget())
		return c->leaveStackFrame();

	bool bOk;
	int idx = p->getInt(&bOk);

	if(!bOk)
		return c->error(KviError_integerParameterExpected);

	int cnt;
	if(idx >= (cnt = ((QListBox *)widget())->count()))
		c->warning("Item index [%d] is too big - defaulting to " \
			"$count() - 1 [%d]", idx, cnt);

	((QListBox *)widget())->removeItem(idx);

	return c->leaveStackFrame();
}


bool KviScriptListBoxObject::functionCount(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d",((QListBox *)widget())->count());

	return true;
}

bool KviScriptListBoxObject::functionCurrentText(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%s", ((QListBox *)widget())-> \
			currentText().local8Bit().data());
	return true;
}

bool KviScriptListBoxObject::functionCurrentItem(KviCommand * c, \
	KviParameterList * p, KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", ((QListBox *)widget())-> \
			currentItem());
	return true;	
}


bool KviScriptListBoxObject::functionTextAt(KviCommand * c, KviParameterList * p, \
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "listbox::textAt");
	
	if(widget())
	{
		bool bOk;
		int idx = p->getInt(&bOk);
		if(!bOk)
			return c->error(KviError_integerParameterExpected);
			
		b.append(((QListBox *)widget())->text(idx));
	}

	return c->leaveStackFrame();
}

bool KviScriptListBoxObject::functionSetCurrentItem(KviCommand * c, \
	KviParameterList * p, KviStr & b)
{
	ENTER_STACK_FRAME(c, "listbox::setCurrentItem");
	
	if(widget())
	{
		bool bOk;
		int idx = p->getInt(&bOk);
		if(!bOk)
			return c->error(KviError_integerParameterExpected);
			
		((QListBox *)widget())->setCurrentItem(idx);
	}

	return c->leaveStackFrame();
}

bool KviScriptListBoxObject::functionClear(KviCommand * c, \
	KviParameterList * p, KviStr & b)
{
	ENTER_STACK_FRAME(c, "listbox::clear");
	
	if(widget())
	{
		((QListBox *)widget())->clear();
	}

	return c->leaveStackFrame();
}

bool KviScriptListBoxObject::functionSetSelectionMode(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "listbox::setSelectionMode");
	
	if(!widget())
		return c->leaveStackFrame();

	if(!p->count())
		return c->error(KviError_notEnoughParameters);
		
	KviStr * pStr = p->first();
	QListBox::SelectionMode iMode = QListBox::Single;
	if(kvi_strEqualCI(pStr->ptr(),"single"))iMode = QListBox::Single;
	else if(kvi_strEqualCI(pStr->ptr(),"multi"))iMode = QListBox::Multi;
	else if(kvi_strEqualCI(pStr->ptr(),"extended"))iMode = QListBox::Extended;
	else if(kvi_strEqualCI(pStr->ptr(),"none"))iMode = QListBox::NoSelection;
	else c->warning("Invalid selection mode \"%s\": assuming \"single\"",pStr->ptr());

	((QListBox *)widget())->setSelectionMode(iMode);
	
	return c->leaveStackFrame();
}

bool KviScriptListBoxObject::functionSelectionMode(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "listbox::selectionMode");
	
	if(!widget())
		return c->leaveStackFrame();
		
	switch(((QListBox *)widget())->selectionMode())
	{
		case QListBox::Single: b.append("single"); break;
		case QListBox::Multi: b.append("multi"); break;
		case QListBox::Extended: b.append("extended"); break;
		case QListBox::NoSelection: b.append("none"); break;
		default: b.append("single"); break;
	}
	
	return c->leaveStackFrame();
}

bool KviScriptListBoxObject::functionIsSelected(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "listbox::isSelected");

	if(!widget())
		return c->leaveStackFrame();

	bool bOk;
	int idx = p->getInt(&bOk);

	if(!bOk)
		return c->error(KviError_integerParameterExpected);
	b.append(((QListBox *)widget())->isSelected(idx) ? '1' : '0');
	return c->leaveStackFrame();
}

bool KviScriptListBoxObject::functionSetSelected(KviCommand * c, KviParameterList * p,
	KviStr & b)
{
	ENTER_STACK_FRAME(c, "listbox::setSelected");

	if(!widget())
		return c->leaveStackFrame();

	bool bOk;
	int idx = p->getInt(&bOk);
	if(!bOk)
		return c->error(KviError_integerParameterExpected);
	int boo = p->getInt(&bOk);
	if(!bOk)
		return c->error(KviError_integerParameterExpected);
	((QListBox *)widget())->setSelected(idx,boo);
	return c->leaveStackFrame();
}



#include "m_class_listbox.moc"
