//
//   File : libkvioptions.cpp
//   Creation date : Fri Aug 18 2000 15:04:09 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_options.h"
#include "kvi_app.h"
#include "kvi_frame.h"
#include "kvi_window.h"
#include "kvi_locale.h"
#include "kvi_mirccntrl.h"
#include "kvi_out.h"

#include "container.h"
#include "instances.h"

#include "dlg_options.h"

#include <qsplitter.h>


KviGeneralOptionsDialog * g_pGeneralOptionsDialog = 0;
KviOptionsInstanceManager * g_pOptionsInstanceManager = 0;

extern int g_iOptionWidgetInstances;

/*
	@doc: options.save
	@type:
		command
	@title:
		options.save
	@short:
		Saves the options to disk
	@syntax:
		options.save
	@description:
		Saves the options to disk; this includes aliases , popups ,
		events and user preferences.
*/


static bool options_module_cmd_save(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"options_module_cmd_save");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;
	g_pApp->saveOptions();
	return c->leaveStackFrame();
}


/*
	@doc: options.dialog
	@type:
		command
	@title:
		options.dialog
	@short:
		Shows the "options dialog"
	@syntax:
		options.dialog [-t]
	@description:
		Shows the KVIrc "general options dialog". If the dialog
		is already open it is simply reparented to the current
		frame.[br]
		If the [-t] switch is used , the dialog is opened as toplevel window,
		otherwise it is opened as part of the current frame window.[br]
		This command is exported by the "options" module.
	@seealso:
		[fnc]$options.isdialog[/fnc]
*/


static bool options_module_cmd_dialog(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"options_module_cmd_dialog");

	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	if(g_pGeneralOptionsDialog)
	{
		if(c->hasSwitch('t'))
		{
			if(g_pGeneralOptionsDialog->parent())
			{
				g_pGeneralOptionsDialog->reparent(0,QPoint(0,0),true);
			}
		} else {
			if(g_pGeneralOptionsDialog->parent() != c->window()->frame()->splitter())
			{
				g_pGeneralOptionsDialog->reparent(c->window()->frame()->splitter(),QPoint(0,0),true);
			}
		}
	} else {
		if(c->hasSwitch('t'))
		{
			g_pGeneralOptionsDialog = new KviGeneralOptionsDialog(0);
		} else {
			g_pGeneralOptionsDialog = new KviGeneralOptionsDialog(c->window()->frame()->splitter());
		}
	}

	g_pGeneralOptionsDialog->show();
	g_pGeneralOptionsDialog->raise();
	g_pGeneralOptionsDialog->setFocus();

	return c->leaveStackFrame();
}


/*
	@doc: options.pages
	@type:
		command
	@title:
		options.pages
	@short:
		Lists the option pages
	@syntax:
		options.pages
	@description:
		Lists the option pages available for editing by the means of [cmd]options.edit[/cmd].
	@seealso:
*/


static void options_module_print_pages(KviCommand * c,KviOptionsWidgetInstanceEntry * e,const char * prefix)
{
	c->window()->output(KVI_OUT_SYSTEMMESSAGE,"%s%c%s%c  (%Q)",prefix,KVI_TEXT_BOLD,e->szClassName,KVI_TEXT_BOLD,&(e->szName));
	KviStr szPre = prefix;
	szPre.append("  ");
	if(e->pChildList)
	{
		for(KviOptionsWidgetInstanceEntry * ex = e->pChildList->first();ex;ex = e->pChildList->next())
		{
			options_module_print_pages(c,ex,szPre.ptr());
		}
	}
}

static bool options_module_cmd_pages(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"options_module_cmd_pages");
	KviStr dummy;
	if(!g_pUserParser->parseCmdFinalPart(c,dummy))return false;

	KviPtrList<KviOptionsWidgetInstanceEntry> * l = g_pOptionsInstanceManager->instanceEntryTree();

	for(KviOptionsWidgetInstanceEntry * e = l->first();e;e = l->next())
	{
		options_module_print_pages(c,e,"");
	}

	return c->leaveStackFrame();
}


/*
	@doc: options.edit
	@type:
		command
	@title:
		options.edit
	@short:
		Shows a single options page
	@syntax:
		options.edit <"options page class name">
	@description:
		Shows an options page as toplevel dialog.
		The available option pages can be listed by using [cmd]options.pages[/cmd].
	@seealso:
*/


static bool options_module_cmd_edit(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"options_module_cmd_edit");

	KviStr clName;
	if(!g_pUserParser->parseCmdFinalPart(c,clName))return false;

	KviOptionsWidgetInstanceEntry * e = g_pOptionsInstanceManager->findInstanceEntry(clName.ptr());
	if(!e)
	{
		c->warning(__tr("No such options page class name"));
		return c->leaveStackFrame();
	}

	if(e->pWidget)
	{
		//c->warning(__tr("The editor page is already open"));
		e->pWidget->raise();
		e->pWidget->setActiveWindow();
		e->pWidget->setFocus();
		return c->leaveStackFrame();
	}

	KviOptionsWidgetContainer * wc = new KviOptionsWidgetContainer(c->window()->frame());

	wc->setup(g_pOptionsInstanceManager->getInstance(e,wc));

	wc->show();
	wc->raise();

	return c->leaveStackFrame();
}

/*
	@doc: options.isdialog
	@type:
		function
	@title:
		$options.isdialog
	@short:
		Returns the options dialog state
	@syntax:
		$options.isdialog
	@description:
		Returns '1' if the general options dialog is open, '0' otherwise.[br]
		This command is exported by the "options" module.
	@seealso:
		[cmd]options.dialog[/cmd]
*/

static bool options_module_fnc_isdialog(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"options_module_fnc_isdialog");
	buffer.append(g_pGeneralOptionsDialog ? '1' : '0');
	return c->leaveStackFrame();
}

static bool options_module_init(KviModule * m)
{
	g_pOptionsInstanceManager = new KviOptionsInstanceManager();

	m->registerCommand("dialog",options_module_cmd_dialog);
	m->registerCommand("save",options_module_cmd_save);
	m->registerFunction("isDialog",options_module_fnc_isdialog);
	m->registerCommand("pages",options_module_cmd_pages);
	m->registerCommand("edit",options_module_cmd_edit);
	return true;
}

static bool options_module_cleanup(KviModule *m)
{
	if(g_pGeneralOptionsDialog)delete g_pGeneralOptionsDialog;
    g_pGeneralOptionsDialog = 0;
    
	g_pOptionsInstanceManager->cleanup(m);

	delete g_pOptionsInstanceManager;
    g_pOptionsInstanceManager = 0;

	m->unregisterMetaObject("KviMessageListView");
	m->unregisterMetaObject("KviServerDetailsWidget");
	m->unregisterMetaObject("KviOptionsInstanceManager");
	m->unregisterMetaObject("KviOptionsWidgetContainer");

	return true;
}

static bool options_module_can_unload(KviModule *m)
{
	return ((!g_pGeneralOptionsDialog) && (g_iOptionWidgetInstances == 0));
}

KVIRC_MODULE(
	"Options",                                              // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2000 Szymon Stefanek (pragma at kvirc dot net)", // author & (C)
	"Options Dialog",
	options_module_init,
	options_module_can_unload,
	0,
	options_module_cleanup
)
