//
//   File : libkvistr.cpp
//   Creation date : Thu Dec 27 2001 17:13:12 GMT by Szymon Stefanek
//
//   This str is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

//#warning: FIXME: Incomplete documentation ('seealso', 'example', etc)

#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_locale.h"
#include "kvi_string.h"
#include "kvi_debug.h"

/*
    @doc: str.len
    @type:
        function
    @title:
        $str.len
    @short:
        Returns the lenght of the given string
    @syntax:
        $str.len(string)
    @description:
		Returns the lenght (that is, number of characters) of the given string.
*/

/*
KVSM_FUNCTION(str,len)
{
	QString tmp;
	pParams->firstAsString(tmp);
	KVSM_FIRST_PARAMETER_AS_STRING(tmp);
	KVSM_FUNCTION_RESULT()->setInteger(tmp.length());
	return KVSM_SUCCESS;
}
*/

static bool str_module_fnc_len(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	buffer.append(KviStr::Format,"%d",(*(parms->safeFirst())).len());
	return true;
}

/*
    @doc: str.lowcase
    @type:
        function
    @title:
        $str.lowcase
    @short:
        Returns the given string with all characters turned to lower case
    @syntax:
        $str.lowcase(string)
    @description:
		Returns the given string with all characters turned to lower case.
*/

static bool str_module_fnc_lowcase(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	buffer.append((*(parms->safeFirst())).lower());
	return true;
}

/*
	@doc: str.upcase
	@type:
		function
	@title:
		$str.upcase
	@short:
		Returns the given string with all characters turned to upper case
	@syntax:
		$str.lowcase(string)
	@description:
		Returns the given string with all characters turned to lower case.
*/

static bool str_module_fnc_upcase(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	buffer.append((*(parms->safeFirst())).upper());
	return true;
}

/*
    @doc: str.isnumber
    @type:
        function
    @title:
        $str.isnumber
    @short:
        Returns 1 if the given string represents a number
    @syntax:
        $str.isnumber(string)
    @description:
		Returns 1 if the given string represents a number, 0 if not.
*/

static bool str_module_fnc_isnumber(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	bool isNum;
	parms->getInt(&isNum);
	buffer.append(isNum ? '1' : '0');
	return true;
}

/*
    @doc: str.isunsignednumber
    @type:
        function
    @title:
        $str.isunsignednumber
    @short:
        Returns 1 if the given string represents an unsigned number
    @syntax:
        $str.isunsignednumber(string)
    @description:
		Returns 1 if the given string represents an unsigned number, 0 if not.
*/

static bool str_module_fnc_isunsignednumber(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	bool isUNum;
	parms->getUInt(&isUNum);
	buffer.append(isUNum ? '1' : '0');
	return true;
}
/*
    @doc: str.isempty
    @type:
        function
    @title:
        $str.isempty
    @short:
        Returns 1 if the given string don't have any character.
    @syntax:
        $str.isnumber(string)
    @description:
		Returns 1 if the given string don't have any character (that is, is empty).
*/

static bool str_module_fnc_isempty(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	buffer.append((*(parms->safeFirst())).isEmpty() ? '1' : '0');
	return true;
}
/*
    @doc: str.contains
    @type:
        function
    @title:
        $str.contains
    @short:
        Returns 1 if the first parameter contains the second
    @syntax:
        $str.contains(string, substring)
    @description:
		Returns 1 if the first string parameter contains the second string parameter. 
		This function is case sensitive.
	@seealso:
		[fnc]$str.containsnocase[/fnc]()
*/

static bool str_module_fnc_contains(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(s1->contains(s2->ptr()) ? '1' : '0');
	return true;
}

/*
    @doc: str.containsnocase
    @type:
        function
    @title:
        $str.containsnocase
    @short:
		Returns 1 if the first parameter contains the second, case insensitive
    @syntax:
        $str.containsnocase(string,substring)
    @description:
        Returns 1 if the first string parameter contains the second string parameter 
		whithout taking in consideration the case of the characters in the string.
	@seealso:
		[fnc]$str.contains[/fnc]
*/

static bool str_module_fnc_containsnocase(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(s1->contains(s2->ptr(),false) ? '1' : '0');
	return true;
}

/*
    @doc: str.equal
    @type:
        function
    @title:
        $str.equal
    @short:
        Returns 1 if the two string parameters are equal
    @syntax:
        $str.equal(string1,string2)
    @description:
		Returns 1 if the two string parameters are equal. This function is case sensitive.
	@seealso:
		[fnc]$str.equalnocase[/fnc]()
*/

static bool str_module_fnc_equal(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	//buffer.append(kvi_strEqualCS( (*(parms->safeFirst())).ptr(),(*(parms->safeNext())).ptr() ) ? '1':'0');
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(s1->equalsCS(s2->ptr()) ? '1' : '0');
	return true;
}

/*
    @doc: str.equalnocase
    @type:
        function
    @title:
        $str.equalnocase
    @short:
        Returns 1 if the two string parameters are equal, case insensitive
    @syntax:
        $str.equalnocase(string1,string2)
    @description:
		Returns 1 if the two strngs parameters are equal, without taking the case of the 
		characters in consideration.
	@seealso:
		[fnc]$str.equal[/fnc]()
*/

static bool str_module_fnc_equalnocase(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(s1->equalsCI(s2->ptr()) ? '1' : '0');
//	buffer.append(kvi_strEqualCI( (*(parms->safeFirst())).ptr(),(*(parms->safeNext())).ptr() ) ? '1':'0');
	return true;
}

/*
    @doc: str.cmp
    @type:
        function
    @title:
        $str.cmp
    @short:
        Compare two strings alphabetically
    @syntax:
        $str.cmp(string1,string2)
    @description:
		This function compares two strings alphabetically. If the first string is 'greater'
		than the second, it will return a positive number, a negative number is the second is
		greater and 0 if the two strings are equal.
	@seealso:
		[fnc]$str.cmpnocase[/fnc]()
*/

static bool str_module_fnc_cmp(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(KviStr::Format,"%d",kvi_strcmpCS(s1->ptr(),s2->ptr()));
	return true;
}
/*
    @doc: str.cmpnocase
    @type:
        function
    @title:
        $str.cmpnocase
    @short:
        Compare two strings alphabetically, case insensitive.
    @syntax:
        $str.cmp(string1,string2)
    @description:
        This function compares two strings alphabetically. If the first string is 'greater'
        than the second, it will return a positive number, a negative number is the second is
        greater and 0 if the two strings are equal. This function is case insensitive.
	@seealso:
		[fnc]$str.cmp[/fnc]()
*/
static bool str_module_fnc_cmpnocase(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(KviStr::Format,"%d",kvi_strcmpCI(s1->ptr(),s2->ptr()));
	return true;
}

/*
    @doc: str.find
    @type:
        function
    @title:
        $str.find
    @short:
        Find the index of the nth ocurrence of a substring in a string
    @syntax:
        $str.find(string,substring,ocurrence)
    @description:
		This function search in the string given as the first parameter for the string
		given as his second parameter, and will return the index where the nth ocurrence 
		given as the third parameter is found or -1 if it's not located. It starts 
		counting at 0.
*/

static bool str_module_fnc_find(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr Str(parms->safeFirstParam());
	KviStr match(parms->safeNextParam());
	int pos = (*(parms->safeNext())).toInt();
	if (pos<1)
	{
		buffer.append("-1");
		return true;
	}
	int cnt = 1;
	int idx;
	int totalIdx = 0;

	while (cnt<=pos)
	{
	 	idx = Str.right(Str.len() - totalIdx).findFirstIdx(match.ptr());
		if(idx == -1)
		{
			buffer.append("-1");
			return true;
		}
		//Idx only gives the index until the pos _before_ the matched string so if this is 
		//not the match we want (cont != 0) we skip it
		totalIdx += (idx + (cnt == pos ? 0 : match.len()));
		++cnt;
	}
	buffer.append(KviStr::Format,"%d",totalIdx);
	return true;
}

/*
    @doc: str.findfirst
    @type:
        function
    @title:
        $str.findfirst
    @short:
        Find the index of a substring in a string
    @syntax:
        $str.findfirst(string1,string2)
    @description:
		This function search in the string given as the first parameter for the string
		given as his second parameter, and will return the index where is first located or
		-1 if it's not located. It starts counting at 0.
*/

static bool str_module_fnc_findfirst(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(KviStr::Format,"%d",s1->findFirstIdx(s2->ptr()));
	return true;
}

/*
    @doc: str.findfirstnocase
    @type:
        function
    @title:
        $str.findfirstnocase
    @short:
        Find the index of a substring in a string, case insensitive
    @syntax:
        $str.findfirstnocase(string1,string2)
    @description:
        This function search in the string given as the first parameter for the string
        given as his second parameter, and will return the index where is first located or
        -1 if it's not located. This function is case insensitive.
*/
static bool str_module_fnc_findfirstnocase(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{	
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(KviStr::Format,"%d",s1->findFirstIdx(s2->ptr(),false));
	return true;
}

/*
    @doc: str.findlast
    @type:
        function
    @title:
        $str.findlast
    @short:
        Find the last index of a substring in a string
    @syntax:
        $str.findlast(string1,string2)
    @description:
        This function search in the string given as the first parameter for the string
        given as his second parameter, and will return the index where is last located or
        -1 if it's not located.
*/

static bool str_module_fnc_findlast(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(KviStr::Format,"%d",s1->findLastIdx(s2->ptr()));
	return true;
}

/*
    @doc: str.findlastnocase
    @type:
        function
    @title:
        $str.findlastnocase
    @short:
        Find the last index of a substring in a string, case insensitive
    @syntax:
        $str.findlastnocase(string1,string2)
    @description:
        This function search in the string given as the first parameter for the string
        given as his second parameter, and will return the index where is last located or
        -1 if it's not located. This function is case insensitive.
*/

static bool str_module_fnc_findlastnocase(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(KviStr::Format,"%d",s1->findLastIdx(s2->ptr(),false));
	return true;
}

/*
    @doc: str.left
    @type:
        function
    @title:
        $str.left
    @short:
		Returns a substring starting from the left until the given index.
    @syntax:
        $str.left(string,nchars)
    @description:
		This function returns a substring of the first string parameter which is the
		string starting from the left until the index specified in the second parameter.
*/

static bool str_module_fnc_left(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(s1->left(s2->toInt()));
	return true;
}
/*
    @doc: str.right
    @type:
        function
    @title:
        $str.right
    @short:
		Returns a substring starting from the right until the given index.
    @syntax:
        $str.right(string,nchars)
    @description:
		This function returns a substring of the first string parameter which is the
		string starting from the right until the index specified in the second parameter.
		The index start counting at the last character and increase until the first.
*/

static bool str_module_fnc_right(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(s1->right(s2->toInt()));
	return true;
}
/*
    @doc: str.mid
    @type:
        function
    @title:
        $str.mid
    @short:
		Returns a substring starting from a given index.
    @syntax:
        $str.mid(string,startidx,nchars)
    @description:
		This function returns a substring of the first string parameter wich is the
		string starting at the (numeric) index given with startidx and counting nchars 
		forward.
*/

static bool str_module_fnc_mid(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{ 
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	KviStr * s3 = parms->safeNext();
	buffer.append(s1->middle(s2->toInt(),s3->toInt()));
	return true;
}
/*
    @doc: str.append
    @type:
        function
    @title:
        $str.append
    @short:
		Append one string to another.
    @syntax:
        $str.append(string1,string2)
    @description:
		This function returns a string created appending the second string parameter
		to the end of the first string parameter. 
*/

static bool str_module_fnc_append(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	buffer.append(*(parms->safeFirst()));
	buffer.append(*(parms->safeNext()));
	return true;
}
/*
    @doc: str.prepend
    @type:
        function
    @title:
        $str.prepend
    @short:
		Prepend one string to another.
    @syntax:
        $str.prepend(string1,string2)
    @description:
		This function returns a string created prepending the second string parameter
		to the start of the first string parameter. 
*/

static bool str_module_fnc_prepend(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	buffer.append(*(parms->safeNext()));
	buffer.append(*s1);
	return true;
}

/*
    @doc: str.insert
    @type:
        function
    @title:
        $str.insert
    @short:
        Inserts a substring in a string at a given index
    @syntax:
        $str.insert(string,substring,index)
    @description:
		Inserts the substring given in the second parameter in the string given in the
		first parameter at the index given in the third parameter, then returns the 
		resulting string.
*/

static bool str_module_fnc_insert(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	KviStr * s3 = parms->safeNext();
	buffer.append(s1->insert(s3->toInt(),*(s2)));
	return true;
}
/*
    @doc: str.strip
    @type:
        function
    @title:
        $str.strip
    @short:
		Returns a whitespace stripped string
    @syntax:
        $str.strip(string)
    @description:
		Returns a left and right whitespace stripped version of the string given as the
		first parameter.
*/

static bool str_module_fnc_strip(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr tmp = *(parms->safeFirst());
	tmp.stripWhiteSpace();
	buffer.append(tmp);
	return true;
}
/*
    @doc: str.stripleft
    @type:
        function
    @title:
        $str.stripleft
    @short:
		Returns a left whitespace stripped string
    @syntax:
        $str.stripright(string)
    @description:
		Returns a left whitespace stripped version of the string given as the
		first parameter.
*/

static bool str_module_fnc_stripleft(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr tmp = *(parms->safeFirst());
	tmp.stripLeftWhiteSpace();
	buffer.append(tmp);
	return true;
}
/*
    @doc: str.stripright
    @type:
        function
    @title:
        $str.stripright
    @short:
		Returns a right whitespace stripped string
    @syntax:
        $str.stripright(string)
    @description:
		Returns a right whitespace stripped version of the string given as the
		first parameter.
*/

static bool str_module_fnc_stripright(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr tmp = *(parms->safeFirst());
	tmp.stripRightWhiteSpace();
	buffer.append(tmp);
	return true;
}
/*
    @doc: str.replace
    @type:
        function
    @title:
        $str.replace
    @short:
		Append one string to another.
    @syntax:
        $str.replace(string,newstr,toreplace)
    @description:
		This function returns a string created replacing all ocurrences of the third parameter
		('toreplace') in the string given as the first parameter ('string') with the string 
		given as the second parameter ('newstr').
		The string replacement is case sensitive!.
*/

static bool str_module_fnc_replace(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr s1 = *(parms->safeFirst());
	KviStr * s2 = parms->safeNext();
	KviStr * s3 = parms->safeNext();
	s1.replaceAll(s3->ptr(),s2->ptr());
	buffer.append(s1);
	return true;
}


/*
    @doc: str.replacenocase
    @type:
        function
    @title:
        $str.replacenocase
    @short:
		Append one string to another.
    @syntax:
        $str.replacenocase(string,newstr,toreplace)
    @description:
		This function returns a string created replacing all ocurrences of the third parameter
		('toreplace') in the string given as the first parameter ('string') with the string 
		given as the second parameter ('newstr').[br]
		The replacement is case insensitive.[br]
*/

static bool str_module_fnc_replacenocase(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr s1 = *(parms->safeFirst());
	KviStr * s2 = parms->safeNext();
	KviStr * s3 = parms->safeNext();
	s1.replaceAll(s3->ptr(),s2->ptr(),false);
	buffer.append(s1);
	return true;
}

/*
    @doc: str.lefttofirst
    @type:
        function
    @title:
        $str.lefttofirst
    @short:
	 	Returns the left part of a string until a given substring
    @syntax:
        $str.lefttofirst(string,substring)
    @description:
		This function returns the left part of the string given as the first parameter
		from the start until the string given as the second parameter is found. It don't 
		include the substring of the second parameter in the returned value. If the second 
		parameter is not found, the entire string is returned.
		The match is case insensitive.
*/

static bool str_module_fnc_lefttofirst(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr Str(*(parms->safeFirst()));
	int where = Str.findFirstIdx(*(parms->safeNext()),false);
	if (where != -1)buffer.append(Str.left(where));
	else buffer.append(Str);
	return true;
}
/*
    @doc: str.lefttolast
    @type:
        function
    @title:
        $str.lefttolast
    @short:
	 	Returns the left part of a string until the last ocurrence of a given substring
    @syntax:
        $str.lefttolast(string,substring)
    @description:
		This function returns the left part of the string given as the first parameter
		from the start until the last ocurrence of the string given as the second parameter 
		is found. It don't include the substring of the second parameter in the returned value. 
		If the second parameter is not found, the entire string is returned.
		The match is case insensitive
*/

static bool str_module_fnc_lefttolast(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr Str(*(parms->safeFirst()));
	int where = Str.findLastIdx(*(parms->safeNext()),false);
	if (where != -1)buffer.append(Str.left(where));
	else buffer.append(Str);
	return true;
}
/*
    @doc: str.rightfromfirst
    @type:
        function
    @title:
        $str.rightfromfirst
    @short:
	 	Returns the right part of a string from the first ocurrence of a given substring
    @syntax:
        $str.rightfromfirst(string,substring)
    @description:
		This function returns the right part of the string given as the first parameter
		from the position where the first ocurrence of the string given as the second parameter 
		is found. It don't include the substring of the second parameter in the returned value. 
		If the second parameter is not found, an empty string is returned..
		The match is case insensitive
*/

static bool str_module_fnc_rightfromfirst(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr Str(*(parms->safeFirst()));
	KviStr match(*(parms->safeNext()));
	int idx = Str.findFirstIdx(match.ptr(),false);
	if (idx != -1)buffer.append(Str.right(Str.len()-(idx+match.len())));
	else buffer.append("");
	return true;
}
/*
    @doc: str.rightfromlast
    @type:
        function
    @title:
        $str.rightfromlast
    @short:
	 	Returns the right part of a string from the last ocurrence of a given substring
    @syntax:
        $str.rightfromlast(string,substring)
    @description:
		This function returns the right part of the string given as the first parameter
		from the position where the last ocurrence of the string given as the second parameter 
		is found. It don't include the substring of the second parameter in the returned value. 
		If the second parameter is not found, an empty string is returned..
		The match is case insensitive.
*/

static bool str_module_fnc_rightfromlast(KviModule *m, KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr Str(*(parms->safeFirst()));
	KviStr match(*(parms->safeNext()));
	int idx = Str.findLastIdx(match.ptr(),true);
	if (idx != -1)buffer.append(Str.right(Str.len()-(idx+match.len())));
	else buffer.append("");
	return true;
}			

/*
	@doc: str.match
	@type:
		function
	@title:
		$str.match
	@short:
		Matches a fixed string agains a wildcard expression
	@syntax:
		$str.match(<wildcard_expression>,<string>)
	@description:
		Returns 1 if the fixed <string> matches the <wildcard_expression>, 0 otherwise.[br]
		The <wildcard_expression> is a string containing the classic '?' and '*' wildcards.[br]
		The match is case sensitive.[br]
	@examples:
		[example]
			%test = "Hello! My nickname is Pragma"
			[cmd]if[/cmd]($str.match(Pragma*,%test))[cmd]echo[/cmd] "Matches Pragma*"
			[cmd]if[/cmd]($str.match(*Pragma,%test))[cmd]echo[/cmd] "Matches *Pragma"
			[cmd]if[/cmd]($str.match(H*y*i?k*a,%test))[cmd]echo[/cmd] "Matches H*y*i?k*a"
			[cmd]if[/cmd]($str.match(H*y*i?K*a,%test))[cmd]echo[/cmd] "Matches H*y*i?K*a"
		[/example]
	@seealso:
		[fnc]$str.matchnocase[/fnc]()
*/

//First parameter is the wildcard, the second is the string
static bool str_module_fnc_match(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(kvi_matchStringCS(s1->ptr(),s2->ptr()) ? '1' : '0');
	return true;
}

/*
	@doc: str.matchnocase
	@type:
		function
	@title:
		$str.matchnocase
	@short:
		Matches a fixed string agains a wildcard expression (case insensitive)
	@syntax:
		$str.matchnocase(<wildcard_expression>,<string>)
	@description:
		Returns 1 if the fixed <string> matches the <wildcard_expression>, 0 otherwise.[br]
		The <wildcard_expression> is a string containing the classic '?' and '*' wildcards.[br]
		The match is case insensitive.[br]
	@examples:
		[example]
			%test = "Hello! My nickname is Pragma"
			[cmd]if[/cmd]($str.match(pragma*,%test))[cmd]echo[/cmd] "Matches pragma*"
			[cmd]if[/cmd]($str.match(*pragma,%test))[cmd]echo[/cmd] "Matches *pragma"
			[cmd]if[/cmd]($str.match(H*y*i?k*a,%test))[cmd]echo[/cmd] "Matches H*y*i?k*a"
			[cmd]if[/cmd]($str.match(H*y*i?K*a,%test))[cmd]echo[/cmd] "Matches H*y*i?K*a"
		[/example]
	@seealso:
		[fnc]$str.match[/fnc]()
*/

static bool str_module_fnc_matchnocase(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	KviStr * s1 = parms->safeFirst();
	KviStr * s2 = parms->safeNext();
	buffer.append(kvi_matchStringCI(s1->ptr(),s2->ptr()) ? '1' : '0');
	return true;
}

/*
	@doc: str.word
	@type:
		function
	@title:
		$str.word
	@short:
		Returns the nth word in a string
	@syntax:
		$str.word(<n>,<string>)
	@description:
		Returns the nth word inside the <string> (with n starting from 0!)[br]
		A word is a substring not containing spaces (ASCII chars 32, carriage returns , tabs etc...).[br]
		If the string contains less than n+1 words then an empty string is returned.[br]
		This function is faster than a call to [fnc]split[/fnc]() and array indexing
		if you need a single word to be extracted from a complex string.[br]
		If you need to manage more than one word from the string then the [fnc]split[/fnc]()
		method is more efficient.[br]
		This function is a special case of [fnc]$str.token[/fnc]() and it runs a bit faster.
	@examples:
		[example]
			%test = "this is  a  string   full of  words"
			[cmd]echo[/cmd] $str.word(4,%test)
		[/example]
	@seealso:
		[fnc]$str.token[/fnc]()
*/
static bool str_module_fnc_word(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"str::word");
	unsigned int uN;

	KviStr * n = parms->safeFirst();

	bool bOk;
	uN = n->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr_ctx("The first parameter must be an unsigned number","str"));
		return c->leaveStackFrame();
	}

	KviStr * s = parms->safeNext();

	const char * p = s->ptr();
	const char * begin;

	unsigned int cnt = 0;

	while(*p)
	{
		while(isspace(*p))p++;
		// beginning of a token
		begin = p;
		while(*p && !isspace(*p))p++;
		if(cnt == uN)
		{
			buffer.append(begin,p - begin);
			return c->leaveStackFrame();
		}
		cnt++;
	}

	return c->leaveStackFrame();
}



/*
	@doc: str.token
	@type:
		function
	@title:
		$str.token
	@short:
		Returns the nth token in a string
	@syntax:
		$str.token(<n>,<separators>,<string>)
	@description:
		Returns the nth token inside the <string> (with n starting from 0!)[br]
		A token is a substring not containing the characters listed in <separators>.[br]
		If the string contains less than n+1 tokens then an empty string is returned.[br]
		[fnc]$str.word[/fnc]() is an optimized function dedicated to the special case
		in that <separators> is a string containing all the whitespace characters.[br]
		Please note that this function is SLOW. You might want to take a look at [fnc]$str.word[/fnc]()
		or even better to [fnc]$split[/fnc]().[br]
		This function is case sensitive: you need to specify both cases in the <separators> string
		if you want to do a case insensitive tokenization.[br]
	@examples:
		[example]
			%test = "21 hours 10 minutes 15 seconds"
			[cmd]echo[/cmd] $str.token(1," hoursmintecd",%test)
		[/example]
	@seealso:
		[fnc]$str.word[/fnc]()[br]
		[fnc]$split[/fnc]()[br]
*/


static bool str_module_fnc_token(KviModule *m,KviCommand *c,KviParameterList *parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"str::token");
	unsigned int uN;

	KviStr * n = parms->safeFirst();

	bool bOk;
	uN = n->toUInt(&bOk);
	if(!bOk)
	{
		c->warning(__tr_ctx("The first parameter must be an unsigned number","str"));
		return c->leaveStackFrame();
	}

	KviStr * sep = parms->safeNext();
	KviStr * s = parms->safeNext();

	if(sep->isEmpty())
	{
		buffer.append(*s);
		return c->leaveStackFrame();
	}

	const char * p = s->ptr();
	const char * begin;

	unsigned int cnt = 0;

	while(*p)
	{
		while(sep->contains(*p))p++;
		// beginning of a token
		begin = p;
		while(*p && !sep->contains(*p))p++;
		if(cnt == uN)
		{
			buffer.append(begin,p - begin);
			return c->leaveStackFrame();
		}
		cnt++;
	}

	return c->leaveStackFrame();
}



/*
    @doc: str.charsum
    @type:
        function
    @title:
        $str.charsum
    @short:
		Returns the sum of the character codes of the string
    @syntax:
        $str.charsum(<string>)
    @description:
		Returns the sum of the character codes of the parameter <string>.
		The sum is suitable for implementing a simple hashing algorithm.[br]
*/

static bool str_module_fnc_charsum(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	KviStr * str = parms->safeFirst();
	char * s = str->ptr();
	unsigned int sum = 0;
	while(*s)
	{
		sum += *s;
		s++;
	}
	buffer.append(KviStr::Format,"%u",sum);
	return true;
}

	
/*********************************************************************/
//              Module stuff
/********************************************************************/

static bool str_module_init(KviModule * m)
{
	m->registerFunction("len", str_module_fnc_len);
	m->registerFunction("lowcase", str_module_fnc_lowcase);
	m->registerFunction("upcase", str_module_fnc_upcase);
	m->registerFunction("isnumber", str_module_fnc_isnumber);
	m->registerFunction("isunsignednumber", str_module_fnc_isunsignednumber);
	m->registerFunction("isempty", str_module_fnc_isempty);
	m->registerFunction("contains", str_module_fnc_contains);
	m->registerFunction("containsnocase", str_module_fnc_containsnocase);
	m->registerFunction("equal", str_module_fnc_equal);
	m->registerFunction("equalnocase", str_module_fnc_equalnocase);
	m->registerFunction("cmp", str_module_fnc_cmp); 
	m->registerFunction("cmpnocase", str_module_fnc_cmpnocase); 
	m->registerFunction("find", str_module_fnc_find);
	m->registerFunction("findfirst", str_module_fnc_findfirst); 
	m->registerFunction("findfirstnocase", str_module_fnc_findfirstnocase); 
	m->registerFunction("findlast", str_module_fnc_findlast); 
	m->registerFunction("findlastnocase", str_module_fnc_findlastnocase);
	m->registerFunction("left", str_module_fnc_left); 
	m->registerFunction("right", str_module_fnc_right);
	m->registerFunction("mid", str_module_fnc_mid);
	m->registerFunction("append", str_module_fnc_append);
	m->registerFunction("prepend", str_module_fnc_prepend);
	m->registerFunction("insert", str_module_fnc_insert);
	m->registerFunction("strip", str_module_fnc_strip);
	m->registerFunction("stripleft", str_module_fnc_stripleft);
	m->registerFunction("stripright", str_module_fnc_stripright);
	m->registerFunction("replace", str_module_fnc_replace);
	m->registerFunction("replacenocase", str_module_fnc_replacenocase);
	m->registerFunction("lefttofirst", str_module_fnc_lefttofirst); 
	m->registerFunction("lefttolast", str_module_fnc_lefttolast); 
	m->registerFunction("rightfromfirst", str_module_fnc_rightfromfirst); 
	m->registerFunction("rightfromlast", str_module_fnc_rightfromlast); 
	m->registerFunction("match", str_module_fnc_match);
	m->registerFunction("matchnocase", str_module_fnc_matchnocase);
	m->registerFunction("word",str_module_fnc_word);
	m->registerFunction("token",str_module_fnc_token);
	m->registerFunction("charsum",str_module_fnc_charsum);

//
// str.findFirstWild, str.findLastWild , str.findWild
// str.findFirstRegex , str.findLastRegex , str.findRegex
// 
// str.transliterate
//
// str.leftToFirstWild str.leftToFirstRegex
// str.leftToLastWild str.leftToLastRegex
// str.rightFromFirstWild str.rightFromFirstRegex
// str.rightFromLastWild str.rightFromLastRegex
//
// str.replaceWild str.replaceRegexp
// str.replaceFirst str.replaceFirstWild str.replaceFirstRegex
// str.replaceLast str.replaceLastWild str.replaceLastRegex
//
// str.split str.splitWild str.splitRegexp
//
// str.containsRegexp

	return true;
}

static bool str_module_cleanup(KviModule *m)
{
	return true;
}

KVIRC_MODULE(
	"File",                                                 // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2002 Szymon Stefanek (pragma at kvirc dot net)"\
	"          (C) 2002 Juanjo Alvarez (juanjux@yahoo.es)", // author & (C)
	"Interface to the str system",
	str_module_init,
	0,
	0,
	str_module_cleanup
)
