//=============================================================================//
//
//   File : libkvitoolbar.cpp
//   Creation date : Wed 6 Nov 22:51:59 2002 GMT by Szymon Stefanek
//
//   This toolbar is part of the KVirc irc client distribution
//   Copyright (C) 2002-2004 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_locale.h"
#include "kvi_command.h"
#include "kvi_window.h"
#include "kvi_app.h"
#include "kvi_error.h"
#include "kvi_cmdformatter.h"
#include "kvi_customtoolbar.h"
#include "kvi_customtoolbardescriptor.h"
#include "kvi_customtoolbarmanager.h"
#include "kvi_actionmanager.h"
#include "kvi_kvs_useraction.h"





/*
	@doc: toolbar.define
	@type:
		command
	@title:
		toolbar.define
	@short:
		[DEPRECATED] Defines a toolbar (creates it)
	@syntax:
		toolbar.define [-m] (<tbname>,<label>)
		{
			button [-d] (<name>,<icon>,<text>)<callback code>
			separator <sepname>
			...
			...
			...
		}
	@description:
		This command is no longer supported and is provided only for
		backward compatibility. See [cmd]action.create[/cmd] and
		[cmd]toolbar.additem[/cmd] for the new method of creating toolbars.[br]
*/

static bool toolbar_module_cmd_define(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"toolbar::define");

	KviParameterList paramList;
	paramList.setAutoDelete(true);

	c->skipWhiteSpace();
	if(*(c->ptr()) != '(')return c->error(KviError_openParenthesisExpected);

	if(!g_pUserParser->extractFunctionParameters(c,&paramList))return false;

	c->skipWhiteSpace();

	if(*(c->ptr()) != '{')return c->error(KviError_openBraceExpected);


	KviStr toolbar = paramList.safeFirstParam();
	KviStr label   = paramList.safeNextParam();

	KviCustomToolBarDescriptor * def = KviCustomToolBarManager::instance()->find(toolbar.ptr());
	if(def)
	{
		if(!c->hasSwitch('m'))def->clear();
		def->rename(label.ptr());
	} else {
		def = KviCustomToolBarManager::instance()->create(toolbar.ptr(),label.ptr());
	}

	c->next();
	c->skipWhiteSpace();

	while(*(c->ptr()) && (*(c->ptr()) != '}'))
	{
		if(*(c->ptr()) == '#')g_pUserParser->skipComment(c);
		else {
			if(kvi_strEqualCIN(c->ptr(),"button",6))
			{
				c->skip(6);
				c->skipWhiteSpace();

				bool bDisabled = false;

				while(*(c->ptr()) == '-')
				{
					c->next();
					if(*(c->ptr()) == 'd')bDisabled = true;
					c->next();
					c->skipWhiteSpace();
				}

				if(*(c->ptr()) != '(')return c->error(KviError_openParenthesisExpected);

				paramList.clear();

				if(!g_pUserParser->extractFunctionParameters(c,&paramList))return false;

				c->skipWhiteSpace();

				//if(*(c->ptr()) != '{')return c->error(KviError_openBraceExpected);

				char * aux = c->ptr();
				if(!g_pUserParser->skipCommand(c))return false;

				KviStr body(aux,c->ptr());
				KviCommandFormatter::bufferFromBlock(body);

				KviStr btname = paramList.safeFirstParam();
				KviStr bticon = paramList.safeNextParam();
				KviStr bttext = paramList.safeNextParam();

				if(btname.isEmpty())
				{
					c->warning(__tr("No button name specified!"));
				} else {
					QString szName = btname.ptr();
					if(KviAction * old = KviActionManager::instance()->getAction(szName))
					{
						if(old->isKviUserActionNeverOverrideThis())
							delete old;
						else {
							c->warning(__tr2qs("The action \"%1\" is already defined as core action: choosing an alternate name").arg(szName));
							szName = KviActionManager::instance()->nameForAutomaticAction(btname.ptr());
						}
					}

					KviKvsUserAction * a = new KviKvsUserAction(KviActionManager::instance(),
											szName,
											body.ptr(),
											bttext.ptr(),
											__tr2qs("Backward compatibility action for toolbar.define"),
											"generic",
											bticon.ptr(),
											bticon.ptr(),
											0);
					KviActionManager::instance()->registerAction(a);
					def->addAction(szName);
				}

			} else if(kvi_strEqualCIN(c->ptr(),"separator",9))
			{
				c->skip(9);
				c->skipSpace();
				KviStr sname;
				if(!g_pUserParser->parseCmdFinalPart(c,sname))return false;

				if(sname.isEmpty())
				{
					c->warning(__tr("No separator name specified"));
				} else {
					def->addAction("kvirc.separator");
				}
			} else return c->error(KviError_unexpectedToken);
		}
		c->skipWhiteSpace();
	}

	if(!(*(c->ptr())))return c->error(KviError_missingClosingBrace);

	c->next();

	return c->leaveStackFrame();
}









/*
	@doc: toolbar.create
	@type:
		command
	@title:
		toolbar.create
	@keyterms:
		Creating toolbars
	@short:
		Creates a toolbar
	@syntax:
		toolbar.create [-p] <id> <label> [icon_id]
	@description:
		Creates a toolbar with the specified <id> and with the visible <label> (caption).[br]
		If a toolbar with the same <id> already exists then this command is equivalent to the
		sequence of [cmd]toolbar.hide[/cmd], [cmd]toolbar.clear[/cmd] and [cmd]toolbar.setlabel[/cmd] unless the -p
		switch is specified that causes the [cmd]toolbar.clear[/cmd] call to be skipped (contents preserved).[br]
		This command doesn't show the toolbar (you need to use [cmd]toolbar.show[/cmd]).[br]
		Toolbars are saved and preserved across KVIrc sessions thus you don't need to create them
		on application startup: you only create them once.
		If [icon_id] is specified then it is interpreted as an [doc:image_id]image identifier[/doc]
		for the small (16x16) icon that will be put in the toolbar listing popup menu.
		The toolbar will exist until [cmd]toolbar.destroy[/cmd] is called (you might want to add
		an "uninstall" feature to your script).[br]
	@seealso:
		[fnc]$toolbar.exists[/fnc]()[br]
		[cmd]toolbar.remove[/cmd][br]
		[cmd]toolbar.show[/cmd][br]
		[cmd]toolbar.destroy[/cmd]
*/

static bool toolbar_module_cmd_create(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"toolbar::create");
	KviStr szId;
	KviStr szLabel;
	KviStr szIconId;
	KviStr szDummy;
	if(!g_pUserParser->parseCmdSingleToken(c,szId))return false;
	const char * begin = c->ptr();
	if(!g_pUserParser->parseCmdSingleToken(c,szLabel))return false;
	szLabel = KviStr(begin,c->ptr() - begin);
	szLabel.stripWhiteSpace();
	if(!g_pUserParser->parseCmdSingleToken(c,szIconId))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szDummy))return false;

	if(szId.isEmpty())
	{
		c->warning(__tr2qs("No toolbar id specified"));
		return c->leaveStackFrame();
	}

	if(szLabel.isEmpty())szLabel = "$tr(Unnamed)";

	KviCustomToolBarDescriptor * d = KviCustomToolBarManager::instance()->find(szId.ptr());
	if(d)
	{
		if(!c->hasSwitch('p'))d->clear();
		d->rename(szLabel.ptr());
	} else {
		d = KviCustomToolBarManager::instance()->create(szId.ptr(),szLabel.ptr());
	}
	
	if(!szIconId.isEmpty())
		d->setIconId(szIconId.ptr());
	
	return c->leaveStackFrame();
}


/*
	@doc: toolbar.clear
	@type:
		command
	@title:
		toolbar.clear
	@short:
		Clears a toolbar
	@syntax:
		toolbar.clear [-q] <id>
	@description:
		Clears the contents of the toolbar specified by <id>. If the
		toolbar does not exist then a warning is printed unless the -q switch is used.[br]
	@seealso:
		[fnc]$toolbar.exists[/fnc]()[br]
		[cmd]toolbar.remove[/cmd][br]
		[cmd]toolbar.show[/cmd][br]
		[cmd]toolbar.destroy[/cmd]
*/

static bool toolbar_module_cmd_clear(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"toolbar::clear");
	KviStr szId;
	KviStr szLabel;
	if(!g_pUserParser->parseCmdSingleToken(c,szId))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szLabel))return false;

	if(szId.isEmpty())
	{
		c->warning(__tr2qs("No toolbar id specified"));
		return c->leaveStackFrame();
	}

	KviCustomToolBarDescriptor * d = KviCustomToolBarManager::instance()->find(szId.ptr());
	if(d)d->clear();
	else {
		if(!c->hasSwitch('q'))
			c->warning(__tr2qs("The specified toolbar doesn't exist"));
	}
	return c->leaveStackFrame();
}




/*
	@doc: toolbar.destroy
	@type:
		command
	@title:
		toolbar.destroy
	@keyterms:
		Destroying a toolbar
	@short:
		Destroys a toolbar
	@syntax:
		toolbar.destroy [-q] <id>
	@description:
		Destroys the toolbar with the specified <id>.[br]
		The toolbar definition is definitively destroyed with all the items contained.[br]
		If the -q switch is specified then this command runs in quiet mode and
		does not warn if the toolbar was not existing.
	@seealso:
		[fnc]$toolbar.exists[/fnc]()
		[cmd]toolbar.create[/cmd]
*/

static bool toolbar_module_cmd_destroy(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"toolbar::destroy");
	KviStr szId;
	if(!g_pUserParser->parseCmdFinalPart(c,szId))return false;
	if(!KviCustomToolBarManager::instance()->destroyDescriptor(szId.ptr()))
	{
		if(!c->hasSwitch('q'))
			c->warning(__tr2qs("The specified toolbar doesn't exist"));
	}
	return c->leaveStackFrame();
}


/*
	@doc: toolbar.show
	@type:
		command
	@title:
		toolbar.show
	@short:
		Shows a toolbar
	@syntax:
		toolbar.show [-q] <id>
	@description:
		Shows the toolbar with the specified <id>.[br]
		If -q is specified the command runs in quiet mode and does not complain if the
		specified toolbar doesn't exist.[br]
		Please note that you don't need to show the toolbar at application startup: KVIrc will
		take care of this for you (if the user chooses to keep the toolbar visible).[br]
		You should execute this command only if you create the toolbar after KVIrc has been
		started.[br]
	@seealso:
		[fnc]$toolbar.exists[/fnc]()
		[cmd]toolbar.create[/cmd]
*/

static bool toolbar_module_cmd_show(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"toolbar::show");
	KviStr szId;
	if(!g_pUserParser->parseCmdFinalPart(c,szId))return false;

	if(szId.isEmpty())
	{
		if(!c->hasSwitch('q'))c->warning(__tr2qs("No toolbar id specified"));
		return c->leaveStackFrame();
	}

	KviCustomToolBarDescriptor * d = KviCustomToolBarManager::instance()->find(szId.ptr());

	if(!d)
	{
		if(!c->hasSwitch('q'))c->warning(__tr2qs("The specified toolbar doesn't exist"));
		return c->leaveStackFrame();
	}

	if(!d->toolBar())
		d->createToolBar();
	return c->leaveStackFrame();
}


/*
	@doc: toolbar.hide
	@type:
		command
	@title:
		toolbar.hide
	@short:
		Hides a toolbar
	@syntax:
		toolbar.hide [-q] <id>
	@description:
		Hides the toolbar with the specified <id>.[br]
		If -q is specified the command runs in quiet mode and does not complain if the
		specified toolbar doesn't exist.[br]
		Please note that this command doesn't destroy the toolbar definition: the toolbar
		can be shown again by using [cmd]toolbar.show[/cmd].[br]
		If you want to completely remove the toolbar definition use [cmd]toolbar.destroy[/cmd].[br]
	@seealso:
		[fnc]$toolbar.exists[/fnc]()
		[cmd]toolbar.create[/cmd]
		[cmd]toolbar.destroy[/cmd]
*/


static bool toolbar_module_cmd_hide(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"toolbar::hide");
	KviStr szId;
	if(!g_pUserParser->parseCmdFinalPart(c,szId))return false;

	if(szId.isEmpty())
	{
		if(!c->hasSwitch('q'))c->warning(__tr2qs("No toolbar id specified"));
		return c->leaveStackFrame();
	}

	KviCustomToolBarDescriptor * d = KviCustomToolBarManager::instance()->find(szId.ptr());

	if(!d)
	{
		if(!c->hasSwitch('q'))c->warning(__tr2qs("The specified toolbar doesn't exist"));
		return c->leaveStackFrame();
	}

	if(d->toolBar())delete d->toolBar();
	return c->leaveStackFrame();
}



/*
	@doc: toolbar.exists
	@type:
		function
	@title:
		$toolbar.exists
	@short:
		Checks if a toolbar exists
	@syntax:
		$toolbar.exists(<id>)
	@description:
		Returns 1 if the toolbar with the specified <id> has been defined.[br]
		Note that this doesn't mean that the toolbar is visible!
	@seealso:
		[cmd]toolbar.create[/cmd]
*/

static bool toolbar_module_fnc_exists(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	KviStr szId = parms->safeFirstParam();
	buffer.append(KviCustomToolBarManager::instance()->find(szId.ptr()) ? '1' : '0');
	return true;
}


/*
	@doc: toolbar.isVisible
	@type:
		function
	@title:
		$toolbar.isVisible
	@short:
		Checks if a toolbar is visible
	@syntax:
		$toolbar.isVisible(<id>)
	@description:
		Returns 1 if the toolbar with the specified <id> is actually visible and 0 otherwise.[br]
		If the toolbar is not defined at all this function still returns 0.[br]
	@seealso:
		[cmd]toolbar.show[/cmd]
*/

static bool toolbar_module_fnc_isVisible(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	KviStr szId = parms->safeFirstParam();
	KviCustomToolBarDescriptor * d = KviCustomToolBarManager::instance()->find(szId.ptr());
	buffer.append(d ? (d->toolBar() ? '1' : '0') : '0');
	return true;
}

/*
	@doc: toolbar.list
	@type:
		function
	@title:
		$toolbar.list
	@short:
		Returns a list of defined toolbars
	@syntax:
		$toolbar.list()
	@description:
		Returns a list of defined toolbar identifiers.[br]
		You can use this function to loop thru all the toolbar definitions.[br]
	@seealso:
*/

static bool toolbar_module_fnc_list(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	QDictIterator<KviCustomToolBarDescriptor> it(*(KviCustomToolBarManager::instance()->descriptors()));

	c->beginListArrayOrDictionaryReturnIdentifier();
	int id = 0;

	while(KviCustomToolBarDescriptor * d = it.current())
	{
		KviStr hack = d->id();
		c->addListArrayOrDictionaryReturnValue(id,hack,buffer);
		id++;
		++it;
	}
	return true;
}


/*
	@doc: toolbar.removeitem
	@type:
		command
	@title:
		toolbar.removeitem
	@keyterms:
		Removing toolbar items
	@short:
		Removes a toolbar item
	@syntax:
		toolbar.removeitem [-q] [-i] <toolbarid> <action>
	@description:
		Removes the specified <action> from the toolbar with the specified <toolbarid>.[br]
		If no such item exists in the specified toolbar this command does nothing.[br]
		If the <toolbar> parameter doesn't identify an existing toolbar then
		a warning is printed unless the -q switch is used.[br]
		If the -i switch is specified the <action> is interpreted as zero based
		index of the action to be removed (instead of the action name).[br]
		Please note that this is an expensive operation if the toolbar is visible (basically
		the toolbar needs to be cleared and filled back again). It is a good idea to hide
		the toolbar before removing items.[br]
	@seealso:
*/

static bool toolbar_module_cmd_removeitem(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"toolbar::removeitem");
	KviStr szAction;
	KviStr szToolbar;
	if(!g_pUserParser->parseCmdSingleToken(c,szToolbar))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szAction))return false;

	KviCustomToolBarDescriptor * d = KviCustomToolBarManager::instance()->find(szToolbar.ptr());

	if(!d)
	{
		if(!c->hasSwitch('q'))c->warning(__tr2qs("The specified toolbar doesn't exist"));
		return c->leaveStackFrame();
	}

	if(c->hasSwitch('i'))
	{
		bool bOk;
		int iAction = szAction.toInt(&bOk);
		if(!bOk)
		{
			if(!c->hasSwitch('q'))c->warning(__tr2qs("The action parameter didn't evaluate to an index"));
		} else {
			if(!d->removeAction(iAction))
			{
				if(!c->hasSwitch('q'))c->warning(__tr2qs("The toolbar does not contain the specified item"));
			}
		}
	} else {
		if(!d->removeAction(szAction.ptr()))
		{
			if(!c->hasSwitch('q'))c->warning(__tr2qs("The toolbar does not contain the specified item"));
		}
	}
	return c->leaveStackFrame();
}


/*
	@doc: toolbar.additem
	@type:
		command
	@title:
		toolbar.additem
	@keyterms:
		Adding toolbar items
	@short:
		Adds a new item to a toolbar
	@syntax:
		toolbar.additem [-q] <toolbarid> <action>
	@description:
		Adds the specified <action> from the toolbar with the specified <toolbarid>.[br]
		If no such action exists this command does nothing (beside printing a warning unless -q is used).[br]
		If the <toolbar> parameter doesn't identify an existing toolbar then
		a warning is printed unless the -q switch is used.[br]
	@examples:
		[example]
			toolbar.create test $tr("My Toolbar")
			toolbar.additem test connect
			toolbar.additem test separator
			toolbar.additem test optionsdialog
			toolbar.show test
		[/example]
	@seealso:
*/

static bool toolbar_module_cmd_additem(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"toolbar::additem");
	KviStr szAction;
	KviStr szToolbar;
	if(!g_pUserParser->parseCmdSingleToken(c,szToolbar))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szAction))return false;

	KviCustomToolBarDescriptor * d = KviCustomToolBarManager::instance()->find(szToolbar.ptr());

	if(!d)
	{
		if(!c->hasSwitch('q'))c->warning(__tr2qs("The specified toolbar doesn't exist"));
		return c->leaveStackFrame();
	}

	if(!d->addAction(szAction.ptr()))
	{
		if(!c->hasSwitch('q'))c->warning(__tr2qs("The action \"%s\" doesn't exist"),szAction.ptr());
	}

	return c->leaveStackFrame();
}


/*
	@doc: toolbar.items
	@type:
		function
	@title:
		$toolbar.items
	@short:
		Returns the list of items in a specified toolbar
	@syntax:
		$toolbar.items(<toolbarid>)
	@description:
		Returns the array of items in the specified toolbar.
		If the toolbar doesn't exist then an empty list is returned.
	@seealso:
*/

static bool toolbar_module_fnc_items(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$toolbar.items");
	KviStr szId = parms->safeFirstParam();

	KviCustomToolBarDescriptor * d = KviCustomToolBarManager::instance()->find(szId.ptr());

	if(!d)
	{
		c->warning(__tr2qs("The specified toolbar doesn't exist"));
		return c->leaveStackFrame();
	}

	c->beginListArrayOrDictionaryReturnIdentifier();
	int id = 0;

	for(QString * s = d->actions()->first();s;s = d->actions()->next())
	{
		KviStr hack = *s;
		c->addListArrayOrDictionaryReturnValue(id,hack,buffer);
		id++;
	}
	return true;
}



static bool toolbar_module_init(KviModule * m)
{
	// setlabel , $label , $position , move , $itempos , $itemexists , $itemtype

	m->registerCommand("create",toolbar_module_cmd_create);
	m->registerCommand("show",toolbar_module_cmd_show);
	m->registerCommand("hide",toolbar_module_cmd_hide);
	m->registerCommand("destroy",toolbar_module_cmd_destroy);
	m->registerCommand("additem",toolbar_module_cmd_additem);
	m->registerCommand("removeitem",toolbar_module_cmd_removeitem);
	m->registerCommand("clear",toolbar_module_cmd_clear);
	// COMPAT
	m->registerCommand("define",toolbar_module_cmd_define);


	m->registerFunction("exists",toolbar_module_fnc_exists);
	m->registerFunction("isVisible",toolbar_module_fnc_isVisible);
	m->registerFunction("list",toolbar_module_fnc_list);
	m->registerFunction("items",toolbar_module_fnc_items);

	return true;
}

static bool toolbar_module_cleanup(KviModule *m)
{
	return true;
}

KVIRC_MODULE(
	"toolbar",                                                 // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2002 Szymon Stefanek (pragma at kvirc dot net)", // author & (C)
	"Interface to the scriptable toolbars",
	toolbar_module_init,
	0,
	0,
	toolbar_module_cleanup
)
