//
//   File : kvi_scriptobject.cpp
//   Creation date : Tue Sep 05 2000 15:31:12 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__
#define _KVI_SCRIPTOBJECT_CPP_

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_scriptobject.h"
#include "kvi_modulemanager.h"
#include "kvi_datacontainer.h"
#include "kvi_command.h"
#include "kvi_window.h"
#include "kvi_app.h"
#include "kvi_uparser.h"
#include "kvi_console.h"
#include "kvi_locale.h"
#include "kvi_error.h"
#include "kvi_out.h"
#include "kvi_mirccntrl.h"
#include "kvi_iconmanager.h"
#include "kvi_malloc.h"

#include <qmetaobject.h>
#include <qtimer.h>
#include <time.h>
#include <qiconset.h>

KviScriptObjectController * g_pScriptObjectController = 0;

KviScriptObjectClass::KviScriptObjectClass(KviScriptObjectClass * par,const char * name,
		KviScriptObjectAllocateInstanceProc proc,bool bBuiltin)
{
	m_pParentClass = par;
	if(m_pParentClass)m_pParentClass->registerChildClass(this);
	m_szName       = name;
	m_bBuiltin     = bBuiltin;
	m_pFunctionHandlers = new QAsciiDict<KviScriptObjectFunctionHandler>(17,false,true);
	m_pFunctionHandlers->setAutoDelete(true);
	m_pChildClasses = new KviPtrList<KviScriptObjectClass>;
	m_pChildClasses->setAutoDelete(false);
	m_allocProc    = proc ? proc : par->m_allocProc;

	// inherit everything from the class above
	if(par)
	{
		QAsciiDictIterator<KviScriptObjectFunctionHandler> it(*(par->m_pFunctionHandlers));
		while(it.current())
		{
			KviScriptObjectFunctionHandler * aux = new KviScriptObjectFunctionHandler;
			aux->szScriptHandler = it.current()->szScriptHandler;
			aux->iFlags = KVI_SCRIPT_OBJECT_FUNCTION_INHERITED;
			aux->proc = it.current()->proc;
			m_pFunctionHandlers->insert(it.currentKey(),aux);
			++it;
		}
	}
	// "object" class is automatically registered in the controller constructor
	if(g_pScriptObjectController)g_pScriptObjectController->registerClass(this);
}

KviScriptObjectClass::~KviScriptObjectClass()
{
//	debug("KviScriptObjectClass::~KviScriptObjectClass(%s)",name());
	g_pScriptObjectController->killAllObjectsWithClass(this);
//	debug("All objects with class %s killed",name());
	if(m_pParentClass)m_pParentClass->unregisterChildClass(this);
	g_pScriptObjectController->unregisterClass(this);
	delete m_pFunctionHandlers;
	while(m_pChildClasses->first())delete m_pChildClasses->first();
	delete m_pChildClasses;
}

void KviScriptObjectClass::registerFunctionHandler(const char * func,
		KviScriptObjectFunctionHandlerProc proc,const char * buffer,bool bBuiltin)
{
	KviScriptObjectFunctionHandler * h = new KviScriptObjectFunctionHandler;
	h->proc = proc;
	h->szScriptHandler = buffer ? buffer : "";
	KviScriptObjectFunctionHandler * old = m_pFunctionHandlers->find(func);
	if(old)h->iFlags = bBuiltin ? KVI_SCRIPT_OBJECT_FUNCTION_BUILTINOVERRIDE : KVI_SCRIPT_OBJECT_FUNCTION_OVERRIDE;
	else h->iFlags = bBuiltin ? KVI_SCRIPT_OBJECT_FUNCTION_BUILTIN : 0;
	m_pFunctionHandlers->replace(func,h);
}

void KviScriptObjectClass::registerEmptyFunctionHandler(const char * func)
{
	registerFunctionHandler(func,(KviScriptObjectFunctionHandlerProc)0,0,true);
}

KviScriptObject * KviScriptObjectClass::allocateInstance(KviScriptObject * par,const char * nam,KviCommand *c,KviParameterList * params)
{
	if(!m_allocProc)return 0;
	KviScriptObject * o = m_allocProc(this,par,nam);
	if(!o)return 0;

	if(!o->init(c,params))
	{
		// internal init failure : abort
		delete o;
		return 0;
	}

	KviStr ret;

	KviParameterList * copy = new KviParameterList();
	copy->setAutoDelete(true);
	while(KviStr * tmp = params->next())
	{
		KviStr * szCopy = new KviStr(*tmp);
		copy->append(szCopy);
	}

	if(!o->callEventFunction("constructor",&ret,copy))
	{
		// ops...constructor failed (script error!)
		delete o;
		return 0;
	} else {
		if(kvi_strEqualCI(ret.ptr(),"0"))
		{
			// implementation failure...
			delete o;
			return 0;
		}
	}

	return o;
}

void KviScriptObjectClass::registerChildClass(KviScriptObjectClass *c)
{
	m_pChildClasses->append(c);
}

void KviScriptObjectClass::unregisterChildClass(KviScriptObjectClass *c)
{
	m_pChildClasses->removeRef(c);
}


////////////////////////////////////////////////////////////////////////////////////////

static KviScriptObject * objectClassCreateInstance(KviScriptObjectClass * cls,KviScriptObject *par,const char * nam)
{
	return new KviScriptObject(cls,par,nam);
}

////////////////////////////////////////////////////////////////////////////////////////

KviScriptObjectController::KviScriptObjectController()
: QObject(0,"script_object_controller")
{
	m_pTopLevelObjectList = new KviPtrList<KviScriptObject>;
	m_pTopLevelObjectList->setAutoDelete(false);
	m_pObjectDict = new QAsciiDict<KviScriptObject>(31,true,true);
	m_pObjectDict->setAutoDelete(false);
	m_pClassDict = new QAsciiDict<KviScriptObjectClass>(17,true,true);
	m_pClassDict->setAutoDelete(false);

	// allocate the "object" builtin class
	// this is the only one that is always in core memory
	m_pObjectClass = new KviScriptObjectClass(0,"object",objectClassCreateInstance,true);
	m_pObjectClass->registerFunctionHandler("name",KVI_PTR2MEMBER(KviScriptObject::functionName),0,true);
	m_pObjectClass->registerFunctionHandler("startTimer",KVI_PTR2MEMBER(KviScriptObject::functionStartTimer),0,true);
	m_pObjectClass->registerFunctionHandler("killTimer",KVI_PTR2MEMBER(KviScriptObject::functionKillTimer),0,true);
	m_pObjectClass->registerFunctionHandler("killTimers",KVI_PTR2MEMBER(KviScriptObject::functionKillTimers),0,true);
	m_pObjectClass->registerFunctionHandler("className",KVI_PTR2MEMBER(KviScriptObject::functionClassName),0,true);
	m_pObjectClass->registerFunctionHandler("findChild",KVI_PTR2MEMBER(KviScriptObject::functionFindChild),0,true);
	m_pObjectClass->registerFunctionHandler("childCount",KVI_PTR2MEMBER(KviScriptObject::functionChildCount),0,true);
	m_pObjectClass->registerFunctionHandler("emit",KVI_PTR2MEMBER(KviScriptObject::functionEmit),0,true);
	m_pObjectClass->registerFunctionHandler("children",KVI_PTR2MEMBER(KviScriptObject::functionChildren),0,true);
	m_pObjectClass->registerFunctionHandler("signalSender",KVI_PTR2MEMBER(KviScriptObject::functionSignalSender),0,true);
	m_pObjectClass->registerFunctionHandler("signalName",KVI_PTR2MEMBER(KviScriptObject::functionSignalName),0,true);
	m_pObjectClass->registerFunctionHandler("destructor",KVI_PTR2MEMBER(KviScriptObject::functionDestructor),0,true);
	m_pObjectClass->registerFunctionHandler("parent",KVI_PTR2MEMBER(KviScriptObject::functionParent),0,true);

	m_pObjectClass->registerFunctionHandler("property",KVI_PTR2MEMBER(KviScriptObject::functionproperty),0,true);
	m_pObjectClass->registerFunctionHandler("setProperty",KVI_PTR2MEMBER(KviScriptObject::functionsetProperty),0,true);
	m_pObjectClass->registerFunctionHandler("listProperty",KVI_PTR2MEMBER(KviScriptObject::functionlistProperty),0,true);


	m_pObjectClass->registerEmptyFunctionHandler("constructor");
	m_pObjectClass->registerEmptyFunctionHandler("timerEvent");
	m_pClassDict->insert("object",m_pObjectClass);
}

KviScriptObjectController::~KviScriptObjectController()
{
	while(m_pTopLevelObjectList->first())delete m_pTopLevelObjectList->first();
	delete m_pTopLevelObjectList; // empty list
	delete m_pObjectDict; // empty dict
	m_pObjectDict = 0;
	delete m_pObjectClass; // delete the class tree
	delete m_pClassDict;  // empty dict
}

void KviScriptObjectController::killAllObjectsWithClass(KviScriptObjectClass * cl)
{
	if(!m_pObjectDict)return; // no more objects at all...
	KviPtrList<KviScriptObject> l;
	l.setAutoDelete(true);

	for(KviScriptObject * o = m_pTopLevelObjectList->first();o;o = m_pTopLevelObjectList->next())
	{
		if(o->getClass() == cl)l.append(o);
		else o->killAllChildrenWithClass(cl);
	}
}

void KviScriptObjectController::clearUserClasses()
{
	QAsciiDictIterator<KviScriptObjectClass> it(*m_pClassDict);
	KviPtrList<KviScriptObjectClass> l;
	l.setAutoDelete(true);
	while(it.current())
	{
		if(!(it.current()->isBuiltin()))l.append(it.current());
		++it;
	}
}

void KviScriptObjectController::clearInstances()
{
	while(m_pTopLevelObjectList->first())delete m_pTopLevelObjectList->first();
	delete m_pTopLevelObjectList; // empty list
	delete m_pObjectDict; // empty dict
	m_pTopLevelObjectList = new KviPtrList<KviScriptObject>;
	m_pTopLevelObjectList->setAutoDelete(false);
	m_pObjectDict = new QAsciiDict<KviScriptObject>(31,true,true);
	m_pObjectDict->setAutoDelete(false);
}

void KviScriptObjectController::registerClass(KviScriptObjectClass *c)
{
	m_pClassDict->insert(c->name(),c);
}

void KviScriptObjectController::unregisterClass(KviScriptObjectClass *c)
{
	m_pClassDict->remove(c->name());
}

void KviScriptObjectController::registerObject(KviScriptObject *o)
{
	m_pObjectDict->insert(o->m_szId.ptr(),o);
	if(o->parent() == 0)m_pTopLevelObjectList->append(o);
}

void KviScriptObjectController::unregisterObject(KviScriptObject *o)
{
	m_pObjectDict->remove(o->m_szId.ptr());
	if(o->parent() == 0)m_pTopLevelObjectList->removeRef(o);
}

KviScriptObjectClass * KviScriptObjectController::lookupClass(const char * cl)
{
	KviScriptObjectClass * pC = m_pClassDict->find(cl);
	if(!pC)
	{
		// maybe we need to load the object library ?
		KviModule * pModule = g_pModuleManager->getModule("objects");
		if(!pModule)
		{
			debug("ops...something wrong with the libkviobjects module!");
		} else pC = m_pClassDict->find(cl);
	}
	return pC;
};

bool KviScriptObjectController::objectExists(KviScriptObject * o)
{
	QAsciiDictIterator<KviScriptObject> it(*m_pObjectDict);
	while(it.current())
	{
		if(it.current() == o)return true;
		++it;
	}
	return false;
}

void KviScriptObjectController::deleteClass(KviScriptObjectClass * pClass){ delete pClass; }

///////////////////////////////////////////////////////////////////////////////////////

// This will overflow at 2^32 (4.294.967.296) , and when it will happen
// the lowest numbered objects will be SURELY dead
// (otherwise all the objects will occupy some terabytes of memory)
static unsigned int g_uScriptObjectId = 1;

KviScriptObject::KviScriptObject(KviScriptObjectClass * cla,KviScriptObject * par,const char * nam)
: QObject(par,nam)
{
	m_pObject            = 0;
	m_bObjectOwner       = true; // true by default

	m_pClass             = cla;
	// to be sure that it is unique just add the time string
	m_szId.sprintf("%u.%u",g_uScriptObjectId,(unsigned int)time(0));
	g_uScriptObjectId++;

	m_pChildList         = new KviPtrList<KviScriptObject>;
	m_pChildList->setAutoDelete(false);

	m_pDataContainer     = new KviDataContainer(false);

	m_pFunctionHandlers  = 0; // no local function handlers yet!

	m_bInDelayedDeath    = false;

	m_pSignalDict        = 0; // no signals connected to remote slots
	m_pConnectionList    = 0; // no local slots connected to remote signals

	if(par)par->registerChild(this);

	g_pScriptObjectController->registerObject(this);

//	debug("Hello world!");
//	[root@localhost cvs3]# kvirc
//	Hello world!
//	[root@localhost cvs3]# date
//	Tue Sep  5 21:53:54 CEST 2000
//	[root@localhost cvs3]#
}


KviScriptObject::~KviScriptObject()
{
	m_bInDelayedDeath = true;

	callEventFunction("destructor");

	while(m_pChildList->first())delete m_pChildList->first();
	delete m_pChildList;

	// Disconnect all the signals

	if(m_pSignalDict)
	{
		//debug("DISCONNECTING SIGNALS");
		QAsciiDictIterator<KviScriptObjectConnectionList> it(*m_pSignalDict);

		//debug("SIGNAL DICT HAS %d ENTRIES",m_pSignalDict->count());


		while(it.current())
		{
			//debug("WILL DISCONNECT it.current() =
			KviScriptObjectConnectionListIterator cit(*(it.current()));
			while(cit.current())
			{
				disconnectSignal(it.currentKey(),cit.current());
				// ++cit // NO!...we point to the next now!
			}
			// the iterator should automatically point to the next now
			//if(m_pSignalDict)++it;
		}

		__range_invalid(m_pSignalDict);
	}

	// Disconnect all the slots

	if(m_pConnectionList)
	{
		KviScriptObjectConnectionListIterator cit(*m_pConnectionList);
		while(cit.current())
		{
			KviStr szSig = cit.current()->szSignal;
			cit.current()->pSourceObject->disconnectSignal(szSig.ptr(),cit.current());
			//++cit;// NO!... we point to the next now!
		}
		__range_invalid(m_pConnectionList);
	}

	g_pScriptObjectController->unregisterObject(this);

	if(parentObject())parentObject()->unregisterChild(this);

	if(m_pObject)
	{
		disconnect(m_pObject,SIGNAL(destroyed()),this,SLOT(objectDestroyed()));
		if(m_bObjectOwner)delete m_pObject;
	}


	delete m_pDataContainer;
	if(m_pFunctionHandlers)delete m_pFunctionHandlers;
}

bool KviScriptObject::init(KviCommand *,KviParameterList *params)
{
	return true;
}


QWidget * KviScriptObject::parentScriptWidget()
{
	if(parentObject())
	{
		if(parentObject()->object())
		{
			if(parentObject()->object()->isWidgetType())
				return (QWidget *)(parentObject()->object());
		}
	}
	return 0;
}

void KviScriptObject::unregisterChild(KviScriptObject *c)
{
	m_pChildList->removeRef(c);
}

void KviScriptObject::registerChild(KviScriptObject *c)
{
	m_pChildList->append(c);
}

// SIGNAL/SLOT stuff

bool KviScriptObject::connectSignal(const char * sigName,KviScriptObject * target,const char * slotName)
{
	if(!target->lookupFunctionHandler(slotName))return false; // no such slot

	if(!m_pSignalDict)
	{
		m_pSignalDict = new QAsciiDict<KviScriptObjectConnectionList>(7,false,true);
		m_pSignalDict->setAutoDelete(true);
	}

	KviScriptObjectConnectionList * l = m_pSignalDict->find(sigName);
	if(!l)
	{
		l = new KviScriptObjectConnectionList;
		l->setAutoDelete(true);
		m_pSignalDict->insert(sigName,l);
	}

	KviScriptObjectConnection * con = new KviScriptObjectConnection;

	con->pSourceObject = this;
	con->pTargetObject = target;
	con->szSignal      = sigName;
	con->szSlot        = slotName;

	l->append(con);
	target->registerConnection(con);

	return true;
}

void KviScriptObject::registerConnection(KviScriptObjectConnection *con)
{
	if(!m_pConnectionList)
	{
		m_pConnectionList = new KviScriptObjectConnectionList;
		m_pConnectionList->setAutoDelete(false);
	}
	m_pConnectionList->append(con);
}

bool KviScriptObject::disconnectSignal(const char * sigName,KviScriptObject * target,const char * slotName)
{
	if(!m_pSignalDict)return false; //no such signal to disconnect

	KviScriptObjectConnectionList * l = m_pSignalDict->find(sigName);
	if(!l)return false;

	KviScriptObjectConnectionListIterator it(*l);

	while(KviScriptObjectConnection * sl = it.current())
	{
		if(sl->pTargetObject == target)
		{
			if(kvi_strEqualCI(sl->szSlot.ptr(),slotName))
			{
				target->unregisterConnection(sl);
				l->removeRef(sl);
				if(l->isEmpty())m_pSignalDict->remove(sigName);
				if(m_pSignalDict->isEmpty())
				{
					delete m_pSignalDict;
					m_pSignalDict = 0;
				}
				return true;
			}
		}
		++it;
	}
	return false;
}

bool KviScriptObject::disconnectSignal(const char * sigName,KviScriptObjectConnection * con)
{
	//debug("DISCONNECTING SIGNAL %s from connection %d",sigName,con);
	__range_valid(m_pSignalDict);
	if(!m_pSignalDict)return false;
	KviScriptObjectConnectionList * l = m_pSignalDict->find(sigName);
	__range_valid(l);
	if(!l)return false;
	//debug("CONNECTION LIST FOUND %d",l);
	con->pTargetObject->unregisterConnection(con);
	__range_valid(l->findRef(con) > -1);
	l->removeRef(con);
	if(l->isEmpty())m_pSignalDict->remove(sigName);
	//else debug("CONNECTION LIST STILL NOT EMPTY :%d ITEMS",l->count());
	if(m_pSignalDict->isEmpty())
	{
		//debug("SIGNAL DICT EMPTY : KILLING");
		delete m_pSignalDict;
		m_pSignalDict = 0;
	}// else debug("SIGNAL DICT NON EMPTY : still containing %d ITEMS",m_pSignalDict->count());
	return true;
}

bool KviScriptObject::unregisterConnection(KviScriptObjectConnection * con)
{
	if(!m_pConnectionList)return false;
	bool bOk = m_pConnectionList->removeRef(con); // no auto delete !
	if(!bOk)return false;
	if(m_pConnectionList->isEmpty())
	{
		delete m_pConnectionList;
		m_pConnectionList = 0;
	}
	return true;
}

void KviScriptObject::emitSignal(const char * sigName,KviStr * pBuffer,KviParameterList * params,KviCommand *c)
{
	if(!m_pSignalDict)return;

	KviScriptObjectConnectionList * l = m_pSignalDict->find(sigName);
	if(!l)return; // no slots registered

	if(!params)params = new KviParameterList;
	KviWindow * wnd = c ? c->window() : g_pApp->activeConsole();

	KviPtrList<KviScriptObjectConnection> dis;
	dis.setAutoDelete(false);

	KviScriptObjectConnectionListIterator it(*l);

	int emitted = 0;

	while(KviScriptObjectConnection * s = it.current())
	{
		KviScriptObjectFunctionHandler * h = s->pTargetObject->lookupFunctionHandler(s->szSlot.ptr(),0);
		if(h)
		{
			if(h->szScriptHandler.hasData() || (h->proc))
			{
				KviStr tmp(KviStr::Format,"builtinEmitSignal(%s,src:%s)",sigName,id());
				KviCommand cmd(tmp.ptr(),wnd,c);
				cmd.setScopeObject(s->pTargetObject);
				emitted ++;
				KviStr old = s->pTargetObject->signalSender();
				s->pTargetObject->setSignalSender(id());
				s->pTargetObject->setSignalName(sigName);
				if(!g_pUserParser->callObjectFunction(h,params,tmp,&cmd))
				{
					if(cmd.hasError())
					{
						g_pUserParser->printError(&cmd);
						if(g_pScriptObjectController->objectExists(cmd.scopeObject()) && it.current())
						{
							wnd->output(KVI_OUT_PARSERWARNING,
								__tr("Broken slot '%s' in target object '%s' while emitting signal '%s' from object '%s': disconnecting"),
								s->szSlot.ptr(),s->pTargetObject->id(),sigName,id());
							dis.append(s);
						} // else destroyed in the call! (already disconnected)
					}
				}
				if(g_pScriptObjectController->objectExists(cmd.scopeObject()))
				{
					cmd.scopeObject()->setSignalSender(old.ptr());
				}
			} // else empty handlers are OK
		} else {
			wnd->output(KVI_OUT_PARSERWARNING,
				__tr("No slot function '%s' exported by target object '%s' while emitting signal '%s' from object '%s': disconnecting"),
				s->szSlot.ptr(),s->pTargetObject->id(),sigName,id());
			dis.append(s);
		}
		++it;
	}

	for(KviScriptObjectConnection * con = dis.first();con;con = dis.next())disconnectSignal(sigName,con);

	delete params;

	if(pBuffer)pBuffer->append(KviStr::Format,"%d",emitted);
}

bool KviScriptObject::functionName(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(name());
	return true;
}

bool KviScriptObject::functionParent(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviScriptObject * o = parentObject();
	if(o)buffer.append(o->id());
	else buffer.append('0');
	return true;
}

bool KviScriptObject::functionClassName(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(getClass()->name());
	return true;
}

bool KviScriptObject::functionChildCount(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(KviStr::Format,"%u",m_pChildList->count());
	return true;
}

bool KviScriptObject::functionSignalSender(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(m_szSignalSender);
	return true;
}

bool KviScriptObject::functionSignalName(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	buffer.append(m_szSignalName);
	return true;
}

bool KviScriptObject::functionDestructor(KviCommand *c,KviParameterList *,KviStr &)
{
	ENTER_STACK_FRAME(c,"object::destructor");
	emitSignal("destroyed",0,0,c);
	return c->leaveStackFrame();
}

bool KviScriptObject::functionChildren(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	c->beginListArrayOrDictionaryReturnIdentifier();
	int id=0;
	for(KviScriptObject * o = m_pChildList->first();o;o = m_pChildList->next())
	{
		 c->addListArrayOrDictionaryReturnValue(id,o->id(),buffer);
		 id++;
	}
	return true;
}
bool KviScriptObject::functionFindChild(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	KviStr * pClass = params->safeFirst();
	KviStr * pName = params->safeNext();

	KviScriptObject * o = findChild(pClass->ptr(),pName->ptr());
	if(o)buffer.append(o->id());
	else buffer.append('0');

	return true;
}

bool KviScriptObject::functionEmit(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"object::emit");
	KviStr * pSignal = params->safeFirst();

	KviParameterList * l = new KviParameterList;
	while(KviStr * p = params->next())
	{
		l->append(new KviStr(*p));
	}

	emitSignal(pSignal->ptr(),&buffer,l,c);
	return c->leaveStackFrame();
}

bool KviScriptObject::functionStartTimer(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"object::startTimer");
	bool bOk;
	int timeout = params->getInt(&bOk);
	if(bOk)buffer.append(KviStr::Format,"%d",startTimer(timeout));
	else {
		c->warning(__tr("Invalid timeout"));
		buffer.append("-1");
	}
	return c->leaveStackFrame();
}

bool KviScriptObject::functionKillTimer(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"object::killTimer");
	bool bOk;
	int id = params->getInt(&bOk);
	if(bOk)killTimer(id);
	else c->warning(__tr("Invalid timer id"));
	return c->leaveStackFrame();
}

bool KviScriptObject::functionKillTimers(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	killTimers();
	return true;
}

bool KviScriptObject::functionlistProperty(KviCommand * cmd, KviParameterList * params, \
        KviStr & b)
{
	//if(!m_pObject)
	  //  return true;
	bool b_Flag=params->getBool();
	KviWindow *w = cmd->window();
	QString tmp_buff="";
	QString tmp_name;
	QString tmp_type;
	if (b_Flag)
	w->output(KVI_OUT_SYSTEMMESSAGE,
		__tr("%cListing Qt Properties for widget object %s (%s)"), KVI_TEXT_BOLD, name(), id()
	);

	QMetaObject *o = m_pObject->metaObject();
	do {
	if (b_Flag)
	w->output(69, __tr("Properties class: %c%s%c"), KVI_TEXT_BOLD, o->className(), KVI_TEXT_BOLD);
		QStrList l = o->propertyNames(false);
		int idx = 0;

		for( char *c = l.first(); c; c = l.next() ) {
			const QMetaProperty *p = o->property(idx);
			if( p )
			{
				KviStr tmp(KviStr::Format,
					__tr("Property: %c%s%c, type: %s"), KVI_TEXT_BOLD, p->name(), KVI_TEXT_BOLD, p->type()
				);
				if( p->isEnumType() )
				{
					tmp.append(__tr(", enum ("));
					QStrList le = p->enumKeys();
					int i = 0;
					for( char *c2 = le.first(); c2; c2 = le.next() )
					{
						if( i == 0 )
							i++;
						else
							tmp.append(", ");
						tmp.append(c2);
					}
					tmp.append(')');
				}
				if( p->isSetType() )
					tmp.append(__tr(", set"));
				if( p->writable() )
					tmp.append(__tr(", writeable"));
				if (b_Flag)
				w->output(69, tmp.ptr());
				tmp_name=p->name();
				tmp_type=p->type();

				tmp_buff=tmp_buff+tmp_name+","+tmp_type+"|";
			}
			else
			return cmd->warning(__tr("Oops... no such property: %s"), c);
			idx++;
		}
		o = o->superClass();
	} while( o );
	b.append(tmp_buff);
	return true;
}
bool KviScriptObject::functionsetProperty(KviCommand * c, KviParameterList * params, \
        KviStr & b)
{
	ENTER_STACK_FRAME(c,"object::setproperty");
	if( params->count()< 2)
		return c->warning(__tr("Not enough parameters"));
			    KviStr *pProperty = params->first();
                KviStr *pVal = params->next();
                if( pProperty )
				{
                        const QMetaProperty *p = m_pObject->metaObject()->property(
                                m_pObject->metaObject()->findProperty(pProperty->ptr(), true), true
                        );
                        if( !p )
						{
								c->warning(__tr("No such QT property (%s)"),pProperty->ptr());
								return c->leaveStackFrame();
						}
						KviStr type = p->type();
                        if( p->isEnumType() )
						{
                                if( !pVal )
								{
                                        c->warning(__tr("missing parameter"));
										return c->leaveStackFrame();
								}
								int val = p->keyToValue(pVal->ptr());
                                QVariant v(val);
                                m_pObject->setProperty(pProperty->ptr(), v);
                        }

						else if( kvi_strEqualCI("QString", type.ptr()) )
						{
                                QVariant v(QString(pVal ? pVal->ptr() : ""));
                                m_pObject->setProperty(pProperty->ptr(), v);
                        }
						else if( kvi_strEqualCI("QCString", type.ptr()) )
						{
                                QVariant v(QCString(pVal ? pVal->ptr() : ""));
                                m_pObject->setProperty(pProperty->ptr(), v);
                        }
						else if( kvi_strEqualCI("int", type.ptr()) )
						{
                                if( !pVal )
								{
                                    c->warning(__tr("missing parameter"));
									return c->leaveStackFrame();
								}
                                QVariant v(pVal ? pVal->toInt() : 0);
                                m_pObject->setProperty(pProperty->ptr(), v);
                        }
						else if( kvi_strEqualCI("uint", type.ptr()) )
						{
                                if( !pVal )
								{
                                    c->warning(__tr("missing parameter"));
									return c->leaveStackFrame();
								}
								QVariant v(pVal ? pVal->toUInt() : 0);
                                m_pObject->setProperty(pProperty->ptr(), v);
                        }
						else if( kvi_strEqualCI("bool", type.ptr()) ) {
                                if( !pVal )
								{
                                    c->warning(__tr("missing parameter"));
									return c->leaveStackFrame();
								}
								QVariant v(pVal ? pVal->toInt() : 0, 0);
                                m_pObject->setProperty(pProperty->ptr(), v);
                        }
						else if( kvi_strEqualCI("QPoint", type.ptr()) )
						{
								KviStr *pVal2 = params->next();
                                if( !pVal || !pVal2)
								{
                                    c->warning(__tr("missing parameter"));
									return c->leaveStackFrame();
								}
								QVariant v(QPoint(pVal->toInt(), pVal2->toInt()));
                                m_pObject->setProperty(pProperty->ptr(), v);
                        }
						else if( kvi_strEqualCI("QSize", type.ptr()) )
						{
								KviStr *pVal2 = params->next();
								if( !pVal || !pVal2)
								{
                                    c->warning(__tr("missing parameter"));
									return c->leaveStackFrame();
								}
								QVariant v(QSize(pVal->toInt(), pVal2->toInt()));
                                m_pObject->setProperty(pProperty->ptr(), v);
                        }
						else if( kvi_strEqualCI("QRect", type.ptr()) )
						{

								KviStr *pVal2 = params->next();
                                KviStr *pVal3 = params->next();
                                KviStr *pVal4 = params->next();

								if( !pVal || !pVal2 || !pVal3 || !pVal4 )
								{
                                    c->warning(__tr("missing parameter"));
									return c->leaveStackFrame();
								}

								QVariant v(QRect(pVal->toInt(), pVal2->toInt(), pVal3->toInt(), pVal4->toInt()));
                                m_pObject->setProperty(pProperty->ptr(), v);
                        }
						else if( kvi_strEqualCI("QColor", type.ptr()) )
						{
                                if( !pVal )
								return c->warning(__tr("missing parameter"));
                                char *buf = 0;
                                int len = pVal->hexToBuffer(&buf, false);
                                if( len == 3 ) {
                                        QVariant v(QColor(((unsigned char) buf[0]), ((unsigned char) buf[1]), ((unsigned char) buf[2])));
                                        m_pObject->setProperty(pProperty->ptr(), v);
                                        kvi_free(buf);
                                } else
								{
									c->warning(__tr("Invalid parameter (%s)"),pVal->ptr());
									return c->leaveStackFrame();
								}

                                if( len > 0 )
                                        kvi_free(buf);
                        }
						else if( kvi_strEqualCI("QPixmap", type.ptr()) )
						{
								QVariant pix;
								if( !pVal )
								return c->warning(__tr("missing parameter"));
								KviScriptObject * pixobject = g_pScriptObjectController->lookupObject(pVal->ptr());
								if(pixobject)
								{
									QVariant pix;
									if (!pixobject->inherits("KviScriptPixmapObject"))
									{
											c->warning(__tr("Pixmap Object required"));
											return c->leaveStackFrame();
									}
									else
									{
									pix=pixobject->property("pixmap");
									m_pObject->setProperty(pProperty->ptr(), pix);
									}
								}
								else
								{
								QPixmap *pix1=0;
								pix1=g_pIconManager->getImage(pVal->ptr());
								if (!pix1)
						  		{
										c->warning(__tr("Pixmap not found"));
										return c->leaveStackFrame();
								}
								else pix=*pix1;
								m_pObject->setProperty(pProperty->ptr(), pix);
								return c->leaveStackFrame();
								}

						}
						else if( kvi_strEqualCI("QIconset", type.ptr()) )
						{
							QPixmap *pix = 0;
							pix = g_pIconManager->getImage(pVal->ptr());
							if(!pix)
								{
										c->warning(__tr("Pixmap object required"));
										return c->leaveStackFrame();
								}
							QIconSet *icon=new QIconSet(*pix);
							QVariant v(*icon );
							m_pObject->setProperty(pProperty->ptr(), v);
							delete icon;
							return c->leaveStackFrame();
						}

						else
						{
							c->warning(__tr("Unsupported QT property (%s)"),pProperty->ptr());
							return c->leaveStackFrame();
						}
		        }

     return c->leaveStackFrame();
}

bool KviScriptObject::functionproperty(KviCommand * c, KviParameterList * params, \
        KviStr & buffer)
{
	//if(!m_pObject)

	ENTER_STACK_FRAME(c,"widget::property");

	if( params ) {
                KviStr *pS = params->first();
				KviStr * flag = params->safeNext();
				bool status=true;
				if(kvi_strEqualCS(flag->ptr(),"0")) status=false;
                if( pS )
				{

                        const QMetaProperty *p = m_pObject->metaObject()->property(
                                m_pObject->metaObject()->findProperty(pS->ptr(), true), true
                        );
                        if( !p )
						{
							buffer.append("NoQTProperty");
							if (status)
                                return c->warning(__tr("No such QT property (%s)"),pS->ptr());
							else  return c->leaveStackFrame();
						}
                        QVariant v = m_pObject->property(pS->ptr());
                        if( !v.isValid() )
						{
							buffer.append("NoQTProperty");
							if (status)
                                return c->warning(__tr("No such QT property (%s)"),pS->ptr());
							else  return c->leaveStackFrame();
						}
						if( p->isEnumType() ) {
                                const char *c = p->valueToKey(v.toInt());
                                buffer.append(c ? c : "");
                        } else switch( v.type() )
						{
                                case QVariant::Int: {
                                        KviStr tmp(KviStr::Format, "%d", v.toInt());
                                        buffer.append(tmp.ptr());
                                        break;
                                }
                                case QVariant::Point:
									{
                                        QPoint p = v.toPoint();
                                        KviStr tmp(KviStr::Format, "%d,%d", p.x(), p.y());
                                        buffer.append(tmp.ptr());
                                        break;
                                }
                                case QVariant::Size:
									{
                                        QSize s = v.toSize();
                                        KviStr tmp(KviStr::Format, "%d,%d", s.width(), s.height());
                                        buffer.append(tmp.ptr());
                                        break;
                                }
                                case QVariant::Rect:
									{
                                        QRect r = v.toRect();
                                        KviStr tmp(KviStr::Format, "%d,%d,%d,%d", r.x(), r.y(), r.width(), r.height());
                                        buffer.append(tmp.ptr());
                                        break;
                                }
                                case QVariant::Color:
									{
                                        QColor clr = v.toColor();
                                        char buf[3];
                                        buf[0] = clr.red();
                                        buf[1] = clr.green();
                                        buf[2] = clr.blue();
                                        KviStr tmp;
                                        tmp.bufferToHex(buf, 3);
                                        buffer.append(tmp);
                                        break;
                                }
                                case QVariant::UInt:
									{
                                        KviStr tmp(KviStr::Format, "%u", v.toUInt());
                                        buffer.append(tmp.ptr());
                                        break;
                                }
                                case QVariant::Bool:
									{
                                        KviStr tmp(KviStr::Format, "%d", v.toInt());
                                        buffer.append(tmp.ptr());
                                        break;
                                }
                                case QVariant::String:
                                        buffer.append(v.toString());
                                        break;
                                case QVariant::CString:
									{
                                        const char *c = v.toCString().data();
                                        if( c )
                                                buffer.append(c);
                                        break;
                                }
								case QVariant::Font:
									{

										QFont font=v.toFont();
										QString family=font.family();
										QString szSize;
										szSize.setNum(font.pointSize());

										QString szBold="";
										QString szUnderline="";
										QString szItalic="";
										QString szOverline="";
										QString szStrikeout="";
										QString szFixedpitch="";
										if (font.bold())
											szBold="Bold";
										if (font.underline())
											szUnderline="Underline,";
										if (font.italic())
											szItalic="Italic,";
										if (font.overline())
											szOverline="Overline,";
										if (font.strikeOut())
											szStrikeout="Strikeout,";
										if (font.fixedPitch())
											szFixedpitch="Fixedpitch,";


										QString type=","+szBold+szUnderline+szOverline+szStrikeout+szFixedpitch+szItalic;
										type.left(type.length()-1);
										buffer.append(family+","+szSize+type);
												break;
									}
                                default:
										buffer.append("Unsupported");
										if (status) return c->warning(__tr("Unsupported QT property (%s)"),pS->ptr());
                                        else return c->leaveStackFrame();
										break;
                        }
                  	return c->leaveStackFrame();
                }
        }
      return c->warning(__tr("Missing parameters"));
}



void KviScriptObject::killAllChildrenWithClass(KviScriptObjectClass *cl)
{
//	debug("object %s : killAllChildrenWithClass %s",id(),cl->name());
	KviPtrList<KviScriptObject> l;
	l.setAutoDelete(true);
	for(KviScriptObject * o=m_pChildList->first();o;o=m_pChildList->next())
	{
		if(o->getClass() == cl)
		{
			l.append(o);
//			debug("Killing child %s",o->id());
		} else o->killAllChildrenWithClass(cl);
	}
//	debug("Ok...all the children appended...destructors..");
}

KviScriptObjectClass * KviScriptObject::getClass(const char * classOverride)
{
	if(!classOverride)return m_pClass;
	KviScriptObjectClass * cl = m_pClass; // class override can be also THIS class
	// if object->$function() is a local override, class::object->$function()
	// is the class member function (not the local override)
	while(cl)
	{
		if(kvi_strEqualCI(cl->name(),classOverride))break;
		else cl = cl->m_pParentClass;
	}
	return cl;
}

KviScriptObjectFunctionHandler * KviScriptObject::lookupFunctionHandler(
		const char * funcName,const char * classOverride)
{
	KviScriptObjectFunctionHandler * h = 0;

	if(!classOverride && m_pFunctionHandlers)
	{
		// lookup the local overrides
		h = m_pFunctionHandlers->find(funcName);
	}

	if(!h)
	{
		// not a local override function... lookup in the class
		KviScriptObjectClass * cl = getClass(classOverride);
		if(cl)return cl->lookupFunctionHandler(funcName);
	}

	return h;
}


bool KviScriptObject::die()
{
	if(m_bInDelayedDeath)return false;
	m_bInDelayedDeath = true;
	QTimer::singleShot(0,this,SLOT(delayedDie()));
	return true;
}

bool KviScriptObject::dieNow()
{
	if(m_bInDelayedDeath)return false;
	m_bInDelayedDeath = true;
	delete this;
	return true;
}

void KviScriptObject::delayedDie()
{
	delete this; // byez!
}

void KviScriptObject::setObject(QObject * o,bool bIsOwned)
{
	__range_invalid(m_pObject);
	m_bObjectOwner = bIsOwned;
	m_pObject = o;
	o->installEventFilter(this);
	connect(m_pObject,SIGNAL(destroyed()),this,SLOT(objectDestroyed()));
}

void KviScriptObject::objectDestroyed()
{
	m_pObject = 0;
	die();
}

bool KviScriptObject::eventFilter(QObject *o,QEvent *e)
{
	return false; // do not stop
}

void KviScriptObject::timerEvent(QTimerEvent *e)
{
	callEventFunction("timerEvent",0,new KviParameterList(
			new KviStr(KviStr::Format,"%d",e->timerId())));
}


bool KviScriptObject::callEventFunction(const char * fncName,KviStr * pBuffer,
	KviParameterList * params,KviWindow * wnd)
{
	if(!params)params = new KviParameterList;

	KviScriptObjectFunctionHandler * h = lookupFunctionHandler(fncName,0);

	if(!h)
	{
		debug("No object event function handler %s",fncName);
		delete params;
		return true;
	}

/*
	Not only gcc spits out compiler errors:
	25.09.2001 , at this point in file

	c:\programmi\microsoft visual studio\myprojects\kvirc3\src\kvirc\uparser\kvi_scriptobject.cpp(1234) : fatal error C1001: INTERNAL COMPILER ERROR
    (compiler file 'E:\8168\vc98\p2\src\P2\main.c', line 494)
    Please choose the Technical Support command on the Visual C++
    Help menu, or open the Technical Support help file for more information
*/

	if(h->szScriptHandler.isEmpty() && (h->proc == ((KviScriptObjectFunctionHandlerProc)0)))
	{
		delete params;
		return true;
	}


	if(!wnd)wnd = g_pApp->activeConsole();

	KviStr tmp(KviStr::Format,"builtinTriggerObjectEvent(%s)",fncName);
	KviCommand c(tmp.ptr(),wnd);
	c.setScopeObject(this);

	if(!g_pUserParser->callObjectFunction(h,params,pBuffer ? *pBuffer : tmp,&c))
	{
		if(c.hasError())
		{
			g_pUserParser->printError(&c);
			delete params;
			return false;
		}
	}
	delete params;
	return true;
}

void KviScriptObject::registerPrivateImplementation(const char * fncName,const char * buffer)
{
	if(buffer)
	{
		if(!m_pFunctionHandlers)
		{
			m_pFunctionHandlers = new QAsciiDict<KviScriptObjectFunctionHandler>(7,false,true);
			m_pFunctionHandlers->setAutoDelete(true);
		}
		KviScriptObjectFunctionHandler * h = new KviScriptObjectFunctionHandler;
		h->proc = 0;
		h->szScriptHandler = buffer ? buffer : "";
		m_pFunctionHandlers->replace(fncName,h);
	} else {
		if(m_pFunctionHandlers)
		{
			m_pFunctionHandlers->remove(fncName);
			if(m_pFunctionHandlers->isEmpty())
			{
				delete m_pFunctionHandlers;
				m_pFunctionHandlers = 0;
			}
		}
	}
}


KviScriptObject * KviScriptObject::findChild(const char * cl,const char * n)
{
	__range_valid(cl && n);
	for(KviScriptObject * o = m_pChildList->first();o;o= m_pChildList->next())
	{
		if(*cl)
		{
			if(kvi_strEqualCI(cl,o->getClass()->name()))
			{
				if(*n)
				{
					// name must match
					if(kvi_strEqualCI(n,o->name()))return o;
				} else return o; //any name matches
			}
		} else {
			// any class matches
			if(*n)
			{
				// name must match
				if(kvi_strEqualCI(n,o->name()))return o;
			} else return o; // any name matches
		}
		KviScriptObject * c = o->findChild(cl,n);
		if(c)return c;
	}
	return 0;
}



#include "kvi_scriptobject.moc"
