/* KWlanInterface defines a class for configuring and monitoring an interface
   Parts of it's implemtation are based on KNemo by Percy Leonhardt <percy@eris23.de>
   
   (c) 2006 by Thomas Michel <tom.michel@arcor.de>

   KWlan is free software; you can redistribute it and/or modify
   it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   KWlan is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "kwlaninterface.h"        

#include "wlanlib.h"
#include "configuration.h"
#include "kstartsupplicantdlg.h"
#include "kwlansuprocess.h"


#include <kmessagebox.h>
#include <ktempfile.h>
#include <kstandarddirs.h>
#include <klocale.h>
#include <dirent.h>
#include <qtimer.h>
#include <kdebug.h>
#include <kprocess.h>
#include <qregexp.h>
#include <kio/global.h>
#include <kstandarddirs.h>

KWlanInterface::KWlanInterface()
{
    // initialise variables
    //Notifier = NULL;
    m_controlConnection = NULL;
    m_monitorConnection = NULL;
    m_controlInterfaceDir = strdup ("/var/run/wpa_supplicant");
    m_IpAddress = FALSE;
    m_associated = FALSE;
    m_wpaStarted = FALSE;
    m_userDataRequest = NULL;
    m_useDhclient = FALSE;
    m_useDhcpcd = FALSE;
    m_dhcpEnabled = TRUE;
    m_ifconfigProcess = 0L;
    m_iwconfigProcess = 0L;
    m_wpaWarned = FALSE;
    m_ipConfigured = FALSE;
    m_ssid = "";
}

void KWlanInterface::init(QString interface)
{
    //char buf[10];
    //size_t len;
    QStringList binDirs;
    m_controlInterface = strdup(interface.ascii());
    //m_controlInterfaceDir=QString("/var/run/wpa_supplicant");
    // check if wpa_config file for interface exists
    m_wpaConf = QString("kwlan.")+ interface; 
    if (locate("config",m_wpaConf).isEmpty())
    {
        KConfig *conf= new KConfig(m_wpaConf,false,false,"config");
        conf->writeEntry("update_config","1");
        conf->writeEntry("ctrl_interface",m_controlInterfaceDir);
        conf->writeEntry("ctrl_interface_group","20");
        delete conf;
    }
    m_wpaConf = locate("config",m_wpaConf);
    
    m_confFile = QString("kwlanrc.") + interface;

    //get path to wpa_supplicant
    m_wpaPath = getPath("wpa_supplicant");
    if (m_wpaPath.isEmpty()){
        KMessageBox::error(0,i18n("Could not find wpa_supplicant binary. Without it KWlan can't connect to encrypted networks.\n"
                "Please check your wpa_supplicant installation."));
    }
    
    //Check for dhcp client to use
    m_dhcpcdPath = getPath("dhcpcd");
    if (!m_dhcpcdPath.isEmpty()){
        m_useDhcpcd = TRUE;
    } else {
        m_dhclientPath = getPath("dhclient");
        if (!m_dhclientPath.isEmpty())
            m_useDhclient = TRUE;
    }
    if (! m_useDhclient && ! m_useDhcpcd)
        KMessageBox::sorry(0,i18n("Neither dhclient nor dhcpcd found. Dynamic IP configuration will not work."));
    // check for path to needed applications
    m_ifconfigPath = getPath("ifconfig");
    if (m_ifconfigPath.isEmpty())
        kdDebug() << i18n("ifconfig not found. Network statistics and static ip addresses will not work.");
    m_iwconfigPath = getPath("iwconfig");
    if (m_iwconfigPath.isEmpty())
        kdDebug() << i18n("iwconfig not found. Wireless statistics will not work.");
    
                
    //We need to find out if we have a wireless interface to see whether wpa_suuplicant 
    // needs to be started using configured driver or usind wired driver
    QStringList wirelessInterfaces = getInterfaces();
    if (wirelessInterfaces.findIndex(m_controlInterface)!=-1)
        m_interfaceData.wirelessDevice = TRUE;
    else m_interfaceData.wirelessDevice = FALSE;
    checkConnection();
    //Start WPA Supplicant if configured
    if (m_interfaceData.wirelessDevice)
    {
        if (!m_wpaStarted && m_configuration->m_startWpa){
            KStartSupplicantDlg *startSupplicant = new KStartSupplicantDlg();
            QString driver;
            m_configuration->readDriver(driver);
            if (startSupplicant ){
                startSupplicant->setData(&interface,&driver);
                if (m_configuration->m_startWpaNoQuestion&& !driver.isEmpty()){
                    startSupplicant->startSupplicant();
                }
                else {
                    startSupplicant->show();
                    startSupplicant->exec();
                }
                QString wpa_supplicant;
                if (startSupplicant->m_start){
                    KWlanSuProcess *startWpa= new KWlanSuProcess(NULL,"Start WPA Supplicant");
                    *startWpa << m_wpaPath << QString("-i%1").arg(QString(m_controlInterface)) << QString ("-D%1").arg(startSupplicant->getDriver()) << QString("-c")+ m_wpaConf << "-B";
                    startWpa->setDescription(i18n("Start WPA Supplicant"));
                    startWpa->start();
                    m_configuration->writeDriver(startSupplicant->getDriver());
                }
                delete startSupplicant;
                m_networkChange = true;
            }
        }
    }
    else {
        if (!m_wpaStarted && m_configuration->m_startWpa){
        //wired device, so drvier is wired
            KWlanSuProcess *startWpa= new KWlanSuProcess(NULL,"Start WPA Supplicant");
            *startWpa << m_wpaPath << QString("-i%1").arg(QString(m_controlInterface)) << QString ("-D%1").arg(QString("wired")) << QString("-c" ) + m_wpaConf <<"-B";
            startWpa->setDescription(i18n("Start WPA Supplicant"));
            startWpa->start();
        }
    }
    // Set Status update timer
    m_timer = new QTimer(this);
    if (m_timer){
        connect (m_timer, SIGNAL(timeout()),SLOT(checkConnection()));
        m_timer->start(5000,FALSE);
    }
    // activate last known profile if not wireless to allow fast configuration of wired interfaces
    //if (!m_interfaceData.wirelessDevice) profileActivate(m_configuration->m_lastSSID);
    // set ip data
    m_configuration->readLastSsid(m_ssid);
    getIpData();
    //profileActivate(m_ssid);
    emit (networksChanged());
    

}
        
KWlanInterface::~KWlanInterface()
{
}
         
        

void KWlanInterface::cleanTemporaryFiles()
{
    resolvconf->unlink();
    if (resolvconf) {
        delete resolvconf;
        resolvconf = 0;
    }
}
        
int KWlanInterface::openControlConnection()
{
    char *cfile;
    int flen;

    if (m_controlInterface == NULL)
        return -1;
    
    flen = strlen(m_controlInterfaceDir) + strlen(m_controlInterface) + 2;
    cfile = (char *) malloc(flen);
    if (cfile == NULL)
        return -1;
    snprintf(cfile, flen, "%s/%s", m_controlInterfaceDir, m_controlInterface);
    

    if (m_controlConnection) {
        wpa_ctrl_close(m_controlConnection);
        m_controlConnection = NULL;
    }

    if (m_monitorConnection) {
        delete m_messageNotifier;
        m_messageNotifier = NULL;
        wpa_ctrl_detach(m_monitorConnection);
        wpa_ctrl_close(m_monitorConnection);
        m_monitorConnection = NULL;
    }

    //printf("Trying to connect to '%s'\n", cfile);
    m_controlConnection = wpa_ctrl_open(cfile);
    if (m_controlConnection == NULL) {
        delete cfile;
        return -1;
    }
    m_wpaStarted = TRUE;
    kdDebug() << "Control connection established" << endl;
    emit (wpaStatusChanged(m_wpaStarted));
    emit (networksChanged());
    m_monitorConnection = wpa_ctrl_open(cfile);
    delete cfile;
    if (m_monitorConnection == NULL) {
        wpa_ctrl_close(m_controlConnection);
        kdDebug() << "Could not establish monitor connection." << endl;
        return -1;
    }
    if (wpa_ctrl_attach(m_monitorConnection)) {
        printf("Failed to attach to wpa_supplicant\n");
        wpa_ctrl_close(m_monitorConnection);
        m_monitorConnection = NULL;
        wpa_ctrl_close(m_controlConnection);
        m_controlConnection = NULL;
        return -1;
    }

    m_messageNotifier = new QSocketNotifier(wpa_ctrl_get_fd(m_monitorConnection),QSocketNotifier::Read, this);
    if (m_messageNotifier)
        connect(m_messageNotifier, SIGNAL(activated(int)), SLOT(receiveMsgs()));
    else kdDebug() << "Could not create Message Notifier" << endl;
    return 0;
    
}

void KWlanInterface::checkConnection()
{
    char buf[10];
    size_t len;
    QString status;
    len = sizeof(buf)-1;
    if (ctrlRequest("PING",buf,&len) <0){
        emit wpaStatusChanged ( FALSE);
        //kdDebug() << "PING failed" << endl;
        if (openControlConnection() >=0){
            //emit networksChanged();
        }
    }
    checkConfig();
}

int KWlanInterface::ctrlRequest(const char *command, char *buf, size_t *buflen)
{
    if (m_controlConnection == NULL)
        return -1;
    return wpa_ctrl_request(m_controlConnection, command, strlen(command), buf, buflen, kwlan_msg_cb);
}


QStringList KWlanInterface::listProfiles()
{
    QStringList wpaProfiles = listWpaProfiles();

    return wpaProfiles + listNonWpaProfiles(wpaProfiles);
}
        
QStringList KWlanInterface::listWpaProfiles()
{
    char buf[2048], *start, *end, *id, *ssid, *bssid, *flags;
    size_t len;
    QStringList networks;
    // Are we connected to wpa?
    if (!m_controlConnection) return networks;
    if (ctrlRequest("LIST_NETWORKS", buf, &len) < 0)
        return networks;
    
    buf[len] = '\0';
    start = strchr(buf, '\n');
    if (start == NULL)
        return networks;
    start++;
    while (*start) {
        bool last = false;
        end = strchr(start, '\n');
        if (end == NULL) {
            last = true;
            end = start;
            while (end[0] && end[1])
                end++;
        }
        *end = '\0';
	
        id = start;
        ssid = strchr(id, '\t');
        if (ssid == NULL)
            break;
        *ssid++ = '\0';
        bssid = strchr(ssid, '\t');
        if (bssid == NULL)
            break;
        *bssid++ = '\0';
        flags = strchr(bssid, '\t');
        if (flags == NULL)
            break;
        *flags++ = '\0';
	
        QString tmpNetwork(ssid);
        networks.append (tmpNetwork);
        if (last)
            break;
        start = end + 1;
    }

    return networks;
}        
        
QStringList KWlanInterface::listNonWpaProfiles(QStringList wpaProfiles)
{
    bool encrypted;
    QStringList profiles= m_configuration->getNetworks();
    // Now test if networks already esist from wpa config
    QStringList::Iterator it;
    it = profiles.begin();
    while (it !=profiles.end())
    {
        //kdDebug() << "Checking if " << *it << "is unencrypted" << endl;
        QStringList::Iterator index = wpaProfiles.find( *it );
        if ( !(*index).isEmpty() )
        {
            it = profiles.remove(it);
            continue;
        }
            // Check whether the Network can be used without encryption
        m_configuration->readEncryption(*it,encrypted);
        if (encrypted) {
            it = profiles.remove(it);
            continue;
        }
        it++;
    }
    
    return profiles;
}

int KWlanInterface::getWpaId(QString findSsid)
{
    char buf[2048], *start, *end, *id, *ssid, *bssid, *flags;
    size_t len = sizeof(buf);
    QStringList networks;
    // Are we connected to wpa?
    if (!m_controlConnection) {
        kdDebug() << "No control connection! " <<  endl;
        return  -1;
    }
    if (ctrlRequest("LIST_NETWORKS", buf, &len) < 0)
    {
        kdDebug() << "Getting list of networks failed! " <<  endl;
        return -1;
    }
    
    buf[len] = '\0';
    start = strchr(buf, '\n');
    if (start == NULL)
        return -1;
    start++;
    while (*start) {
        bool last = false;
        end = strchr(start, '\n');
        if (end == NULL) {
            last = true;
            end = start;
            while (end[0] && end[1])
                end++;
        }
        *end = '\0';
	
        id = start;
        ssid = strchr(id, '\t');
        if (ssid == NULL)
            break;
        *ssid++ = '\0';
        bssid = strchr(ssid, '\t');
        if (bssid == NULL)
            break;
        *bssid++ = '\0';
        flags = strchr(bssid, '\t');
        if (flags == NULL)
            break;
        *flags++ = '\0';
        QString tmpSsid =QString(ssid); 
        if (tmpSsid==findSsid)
        {
            return QString(id).toInt();
        }
        if (last)
            break;
        start = end + 1;
    }
    return -1;
}

bool KWlanInterface::profileActivateNonWpa(QString ssid)
{
    if (m_interfaceData.wirelessDevice)
    {
        KWlanSuProcess *iwconfig = new KWlanSuProcess(this);
        *iwconfig << m_iwconfigPath << m_controlInterface << QString("essid") << ssid;
        iwconfig->setDescription(i18n("Configure wlan interface using iwconfig"));
        iwconfig->start();
    }
   // store in config
    m_configuration->writeLastSsid( ssid);
    m_ssid = ssid;
    // set ip data
    getIpData();

    return TRUE;
    
}

bool KWlanInterface::profileActivateWpa(QString ssid)
{
        // select network using wpa_supplicant
    int ret;
    char reply[40], cmd[256];
    size_t reply_len = sizeof(reply);
    int id = getWpaId(ssid);
    if (id == -1 ){
        KMessageBox::sorry(0, i18n("Could not find network in wpa_supplicant\nconfiguration."));
        return FALSE;
    }
    snprintf(cmd, sizeof(cmd), "DISCONNECT");
    reply_len = sizeof(reply);
    ctrlRequest(cmd, reply, &reply_len);
 
    // Enable Network
    snprintf(cmd, sizeof(cmd), "SELECT_NETWORK %d", id);
    reply_len = sizeof(reply);
    ret=ctrlRequest(cmd, reply, &reply_len);
    if (ret==-1) {
        KMessageBox::sorry(0,"No control connection!");
        return FALSE;
    }
    if (strncmp(reply, "OK", 2) != 0) {
        KMessageBox::sorry(0, i18n("Failed to enable network in wpa_supplicant\nconfiguration."));
        return FALSE;
    } 
    // Connect
    snprintf(cmd, sizeof(cmd), "REASSOCIATE");
    reply_len = sizeof(reply);
    ctrlRequest(cmd, reply, &reply_len);
    // store in config
    m_configuration->writeLastSsid( ssid);
    // set ip data
    m_ssid = ssid;
    getIpData();
    
    checkConnection();
    return TRUE;   
}

bool KWlanInterface::profileActivate(QString ssid)
{
    // release any existing ip address
    releaseIpAddress();
    m_interfaceData.available = FALSE;
    m_interfaceData.connected = FALSE;
    if (m_controlConnection){
        return profileActivateWpa(ssid);
    }
    else {
        // select network using iwconfig
        return profileActivateNonWpa(ssid);
    }
}

void KWlanInterface::releaseIpAddress()
{
    if (!m_controlInterface)  return;
    m_ipConfigured = FALSE;
    KWlanSuProcess *dhclient = new KWlanSuProcess(this);    
    /*
    if  (!m_dhcpEnabled) {
        *dhclient << "ifconfig" << m_controlInterface << "0.0.0.0";
        
    } else {
        if (m_useDhclient)
        {
            *dhclient << m_dhclientPath << "-r" << m_controlInterface;
        } else if (m_useDhcpcd){
            *dhclient << m_dhcpcdPath << "-k" << m_controlInterface;
        }
}*/

    dhclient->setDescription(i18n("Release IP address"));
    *dhclient << m_ifconfigPath << m_controlInterface << "0.0.0.0";
    dhclient->start();
    emit ipChanged( QString(""));
}

void KWlanInterface::getIpAddress()
{
    m_ipConfigured = TRUE;
    if (!m_dhcpEnabled){
        kdDebug() << "set static ip address " << m_staticIpAddress << endl;
        if (m_staticIpAddress.isEmpty()) return;
	
        KWlanSuProcess *configurator = new KWlanSuProcess(this);
        configurator->setDescription(i18n("Configure the network"));	
        *configurator << m_ifconfigPath << m_controlInterface << m_staticIpAddress.ascii();	
	
        if (!m_staticNetmask.isEmpty())
            *configurator << "netmask" << m_staticNetmask.ascii();

	// We can to set up our new route if there is no route (hasGateway()==false) or we have but m_ovrgw is true ()
        bool m_hasGateway = hasGateway();
        if (!m_staticGateway.isEmpty() && (!m_dontOverrideGw || !m_hasGateway)) {
            *configurator << ";";
	    
	    // Delete default route if we already have one
            if (m_hasGateway)
                *configurator << getPath("route") << "del" << "default" << ";";
		
            *configurator << getPath("route") << "add" << "default" << "gw" << m_staticGateway.ascii();
        }    

        if (!m_dontOverrideDns || !hasNameserver()) {
	    // Create temporary file containing DNS configuration	    
            resolvconf = new KTempFile(locateLocal("tmp", "resolv"), ".conf", 0644);
            if (!m_staticDns1.isEmpty())
                *(resolvconf->textStream()) << QString("nameserver %1").arg(m_staticDns1) << endl;
            if (!m_staticDns2.isEmpty())
                *(resolvconf->textStream()) << QString("nameserver %1").arg(m_staticDns2) << endl;
            if (!m_domain.isEmpty())
                *(resolvconf->textStream()) << QString("search %1").arg(m_domain) << endl;
            resolvconf->close();
            connect(configurator, SIGNAL( destroyed() ), this, SLOT( cleanTemporaryFiles() ));

	    // And copy it to /etc fixing permissions (moving wont work if /etc/resolv.conf is a symlink)
            *configurator << ";";
            *configurator << getPath("cp") << resolvconf->name() << "/etc/resolv.conf";	    
            *configurator << ";";
            *configurator << getPath("chown") << "root:root" << "/etc/resolv.conf";
        }

        configurator->start();
    }
    else {
        KWlanSuProcess *Dhclient = new KWlanSuProcess(this);    
	
        if (m_useDhclient)
        {
            *Dhclient << getPath("dhclient") <<  "-1" << m_controlInterface;
            //*Dhclient << m_dhclientPath  << "-d" << m_controlInterface;
        } else if (m_useDhcpcd){
            *Dhclient << m_dhcpcdPath << "-nd" << m_controlInterface;
        }
	
        Dhclient->setDescription(i18n("Obtain IP address dynamically"));
        Dhclient->setRestart(true);
        Dhclient->start();
    }
}

void KWlanInterface::processMessage(char *message)
{
    char *pos = message, *pos2;
    int priority = 2;
    //kdDebug() << message << endl;
    if (*pos == '<') {
        /* skip priority */
        pos++;
        priority = atoi(pos);
        pos = strchr(pos, '>');
        if (pos)
            pos++;
        else
            pos = message;
    }

    WpaMsg wm(pos, priority);
    emit wpaEvent(wm);
    m_messages.append(wm);
    while (m_messages.count() > 100)
        m_messages.pop_front();
    
    /* Update last message with truncated version of the event */
    if (strncmp(pos, "CTRL-", 5) == 0) {
        pos2 = strchr(pos, str_match(pos, WPA_CTRL_REQ) ? ':' : ' ');
        if (pos2)
            pos2++;
        else
            pos2 = pos;
    } else
        pos2 = pos;
        QString lastmsg = pos2;
        lastmsg.truncate(40);
    //textLastMessage->setText(lastmsg);
    //pingsToStatusUpdate = 0;
        m_networkChange = true;
    
        if (str_match(pos, WPA_CTRL_REQ))
            processCtrlReq(pos + strlen(WPA_CTRL_REQ));
        if (str_match(pos,WPA_EVENT_CONNECTED)){
        // getIpAddress
        //    if (!m_associated) {
        //        getIpAddress();
        //    }
            m_associated = true;
            kdDebug() << "Connection established." << endl;
            //emit connectionStatusChanged( TRUE);
        }
        if (str_match(pos,WPA_EVENT_DISCONNECTED)){
        //if (m_dhcpEnabled && m_associated) releaseIpAddress();
        //m_ssid = "";
            m_associated = false;
            //emit connectionStatusChanged( false);
        }
}

void KWlanInterface::receiveMsgs()
{
    char buf[256];
    size_t len;
    if (!m_monitorConnection) return;
    while (wpa_ctrl_pending(m_monitorConnection)) {
        len = sizeof(buf) - 1;
        if (wpa_ctrl_recv(m_monitorConnection, buf, &len) == 0) {
            buf[len] = '\0';
            processMessage(buf);
        }
    }
}

void KWlanInterface::processCtrlReq( const char * request )
{
    // processe control request, for example ask for password for private key file.
    if (m_userDataRequest) {
        m_userDataRequest->close();
        delete m_userDataRequest;
    }
    m_userDataRequest = new KUserDataRequestDlg();
    if (m_userDataRequest == NULL)
        return;
    if (m_userDataRequest->setParams(NULL, request) < 0) {
        delete  m_userDataRequest;
        m_userDataRequest = NULL;
        return;
    }
    m_userDataRequest->show();
    m_userDataRequest->exec();
}

void KWlanInterface::startWpa(QString driver)
{
    if (m_wpaStarted) {
        kdDebug() << "wpa_supplicant is already started!" << endl;
        return;
    }
    KWlanSuProcess *startWpa = new KWlanSuProcess(this);
    *startWpa << m_wpaPath << QString("-i%1").arg(QString(m_controlInterface)) << QString ("-D%1").arg(driver) << QString("-c") +m_wpaConf << "-B";
    startWpa->setDescription(i18n("Start WPA Supplicant"));
    startWpa->start();
	//remember last interface driver
    m_configuration->writeDriver(driver);
    emit (wpaStatusChanged(TRUE));
    emit (networksChanged());
}

void KWlanInterface::stopWpa()
{
    char reply[100];
    size_t reply_len;
    if (!m_wpaStarted) {
        kdDebug() << "wpa_supplicant is not started!" << endl;
        return;
    }
    releaseIpAddress();
    reply_len = sizeof(reply);
    ctrlRequest("TERMINATE", reply, &reply_len);
    if (strncmp(reply,"OK",2) != 0) {
        KMessageBox::sorry(0,  i18n("Could not terminate WPA_Supplicant!"));
        return;
    }
    m_wpaStarted = FALSE;
    emit (wpaStatusChanged(FALSE));
    emit (networksChanged());
}

QString KWlanInterface::getInterfaceName()
{
    return QString(m_controlInterface);
}

void KWlanInterface::profileDelete(QString ssid)
{
    char cmd[256],reply[100];
    size_t reply_len = sizeof(reply);
    if (!m_controlConnection) {
        int result = KMessageBox::questionYesNo(0,"wpa_supplicant is not started. Only the non-wpa related network settings will be deleted.\n"
                "Do you want to delete these settings?");
        if (result == KMessageBox::Yes){
        }
        return; 
    }
    int id = getWpaId(ssid);
    snprintf(cmd, sizeof(cmd), "REMOVE_NETWORK %i", id);
    reply_len = sizeof(reply);
    ctrlRequest(cmd, reply, &reply_len);
    if (strncmp(reply, "OK", 2) != 0) {
        KMessageBox::sorry(0, i18n("Failed to remove network from wpa_supplicant\nconfiguration."));
    } else {
        reply_len = sizeof(reply);
        ctrlRequest("SAVE_CONFIG", reply, &reply_len);
        if (strncmp(reply,"OK",2) != 0) {
            KMessageBox::sorry(0, "Failed to save the  wpa_supplicant configuration.\nIs update_config=1 defined in wpa_supplicant.conf?");
        }
        // Delete the network from kwlan config file
        
        emit (networksChanged());
    }
}

void KWlanInterface::profileAdded()
{
    emit (networksChanged());
}

void KWlanInterface::checkConfig()
{
    if ( !m_ifconfigProcess )
    {
        m_ifconfigStdout = QString::null;
        m_ifconfigProcess = new KProcess();
        m_ifconfigProcess->setEnvironment( "LANG", "C" );
        m_ifconfigProcess->setEnvironment( "LC_ALL", "C" );
        *m_ifconfigProcess << m_ifconfigPath << "-a";
        connect( m_ifconfigProcess,  SIGNAL( receivedStdout( KProcess*, char*, int ) ),
                 this, SLOT( ifconfigProcessStdout( KProcess*, char*, int ) ) );
        connect( m_ifconfigProcess,  SIGNAL( processExited( KProcess* ) ),
                 this, SLOT( ifconfigProcessExited( KProcess* ) ) );

        if ( !m_ifconfigProcess->start( KProcess::NotifyOnExit, KProcess::Stdout ) )
        {
            delete m_ifconfigProcess;
            m_ifconfigProcess = 0L;
        }
    }
    if ( !m_iwconfigProcess )
    {
        m_iwconfigStdout = QString::null;
        m_iwconfigProcess = new KProcess();
        m_iwconfigProcess->setEnvironment( "LANG", "C" );
        m_iwconfigProcess->setEnvironment( "LC_ALL", "C" );
        *m_iwconfigProcess << m_iwconfigPath;
        connect( m_iwconfigProcess,  SIGNAL( receivedStdout( KProcess*, char*, int ) ),
                 this, SLOT( iwconfigProcessStdout( KProcess*, char*, int ) ) );
        connect( m_iwconfigProcess,  SIGNAL( receivedStderr( KProcess*, char*, int ) ),
                 this, SLOT( iwconfigProcessStdout( KProcess*, char*, int ) ) );
        connect( m_iwconfigProcess,  SIGNAL( processExited( KProcess* ) ),
                 this, SLOT( iwconfigProcessExited( KProcess* ) ) );

        if ( !m_iwconfigProcess->start( KProcess::NotifyOnExit, KProcess::AllOutput ) )
        {
            delete m_iwconfigProcess;
            m_iwconfigProcess = 0L;
        }
    }
}

void KWlanInterface::ifconfigProcessStdout( KProcess*, char* buffer, int buflen )
{
    m_ifconfigStdout += QString::fromLatin1( buffer, buflen );
}

void KWlanInterface::ifconfigProcessExited( KProcess* process )
{
    if ( process == m_ifconfigProcess )
    {
        delete m_ifconfigProcess;
        m_ifconfigProcess = 0L;
        parseIfconfigOutput();
    }
}


void KWlanInterface::parseIfconfigOutput()
{
    /* mIfconfigStdout contains the complete output of 'ifconfig' which we
    * are going to parse here.
    */
    bool tmpAvailable = m_interfaceData.available;
    bool tmpUp = m_interfaceData.up;
    QMap<QString, QString> configs;
    QStringList ifList = QStringList::split( "\n\n", m_ifconfigStdout );
    QStringList::Iterator it;
    for ( it = ifList.begin(); it != ifList.end(); ++it )
    {
        int index = ( *it ).find( ' ' );
        if ( index == -1 )
            continue;
        QString key = ( *it ).left( index );
        configs[key] = ( *it ).mid( index );
    }

    if ( configs.find( m_controlInterface) == configs.end() )
    {
        // The interface does not exist. Meaning the driver
        // isn't loaded and/or the interface has not been created.
        m_interfaceData.existing = false;
        m_interfaceData.available = false;
    }
    else{
        if (!configs[m_controlInterface].contains("UP"))  {
            // Interface is in down state
            m_interfaceData.up = FALSE;
            m_interfaceData.connected = FALSE;
        } else {
            m_interfaceData.up = TRUE;
            // interface is up, now check if RUNNING ( = connected)
            if ( !configs[m_controlInterface].contains( "inet " ) ||
                  !configs[m_controlInterface].contains( "RUNNING" ) )
            {
                // The interface is up or has an IP assigned but not both
                m_interfaceData.existing = true;
                if (configs[m_controlInterface].contains( "RUNNING" )) 
                    m_interfaceData.connected = TRUE;
                else m_interfaceData.connected = FALSE;
                m_interfaceData.available = false;
            }
            else
            {
        // ...determine the type of the interface
                if ( configs[m_controlInterface].contains( "Ethernet" ) )
                    m_type = ETHERNET;
                else
                    m_type = PPP;

        // Update the interface.
                m_interfaceData.existing = true;
                m_interfaceData.available = true;
                if (configs[m_controlInterface].contains( "RUNNING" )) 
                    m_interfaceData.connected = TRUE;
                else m_interfaceData.connected = FALSE;
                updateInterfaceData( configs[m_controlInterface], m_type );
            }
            if ( !configs[m_controlInterface].contains( "inet " ) && !m_ipConfigured){
                if (m_interfaceData.connected) getIpAddress();
            }
        }
        if (tmpAvailable != m_interfaceData.available) 
            emit (connectionStatusChanged (m_interfaceData.available));
        if (tmpUp != m_interfaceData.up)
            emit (interfaceUp(m_interfaceData.up ));
    }
}

void KWlanInterface::iwconfigProcessExited( KProcess* process )
{
    if ( process == m_iwconfigProcess )
    {
        delete m_iwconfigProcess;
        m_iwconfigProcess = 0L;
        parseIwconfigOutput();
    }
}

void KWlanInterface::iwconfigProcessStdout( KProcess*, char* buffer, int buflen )
{
    m_iwconfigStdout += QString::fromLatin1( buffer, buflen );
}

void KWlanInterface::parseIwconfigOutput()
{
    /* mIwconfigStdout contains the complete output of 'iwconfig' which we
    * are going to parse here.
    */
    QMap<QString, QString> configs;
    QStringList ifList = QStringList::split( "\n\n", m_iwconfigStdout );
    QStringList::Iterator it;
    QString key;
    for ( it = ifList.begin(); it != ifList.end(); ++it )
    {
        int index = ( *it ).find( ' ' );
        if ( index == -1 )
            continue;
        key = ( *it ).left( index );
        configs[key] = ( *it ).mid( index );
    }
    if ( configs.find( m_controlInterface) == configs.end() )
    {
        // The interface does not exist. Meaning the driver
        // isn't loaded and/or the interface has not been created.
        m_interfaceData.existing = false;
        m_interfaceData.available = false;
    }

    if ( configs[m_controlInterface].contains( "no wireless extensions" ) )
    {
        // The interface isn't a wireless device.
        m_interfaceData.wirelessDevice = false;
    }
    else
    {
        // Update the wireless data of the interface.
        m_interfaceData.wirelessDevice = true;
        updateWirelessData( configs[m_controlInterface] );
    }
}

void KWlanInterface::updateWirelessData( QString& config )
{
    QRegExp regExp( "ESSID:\"?([^\"]*)\"?" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.essid = regExp.cap( 1 );

    regExp.setPattern( "Mode:(\\w*)" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.mode = regExp.cap( 1 );

    regExp.setPattern( "Frequency:([\\w|\\.]*)" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.frequency = regExp.cap( 1 );
    else
    {
        regExp.setPattern( "Channel:(\\d*)" );
        if ( regExp.search( config ) > -1 )
            m_wirelessData.channel = regExp.cap( 1 );
    }

    regExp.setPattern( "Bit Rate[=:]([\\w/]*)" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.bitRate = regExp.cap( 1 );

    regExp.setPattern( "Signal level.(-?\\d+\\s*\\w+)" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.signal = regExp.cap( 1 );

    regExp.setPattern( "Noise level.(-?\\d+\\s*\\w+)" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.noise = regExp.cap( 1 );

    regExp.setPattern( "Link Quality[=:]([\\d/]*)" );
    if ( regExp.search( config ) > -1 )
        m_wirelessData.linkQuality = regExp.cap( 1 );
}

void KWlanInterface::updateInterfaceData( QString& config, int type )
{
    QRegExp regExp( ".*RX.*:(\\d+).*:\\d+.*:\\d+.*:\\d+" );
    if ( regExp.search( config ) > -1 )
        m_interfaceData.rxPackets = regExp.cap( 1 ).toULong();

    regExp.setPattern( ".*TX.*:(\\d+).*:\\d+.*:\\d+.*:\\d+" );
    if ( regExp.search( config ) > -1 )
        m_interfaceData.txPackets = regExp.cap( 1 ).toULong();

    regExp.setPattern( "RX bytes:(\\d+)\\s*\\(\\d+\\.\\d+\\s*\\w+\\)" );
    if ( regExp.search( config ) > -1 )
    {
        // We count the traffic on ourself to avoid an overflow after
        // 4GB of traffic.
        unsigned long currentRxBytes = regExp.cap( 1 ).toULong();
        if ( currentRxBytes < m_interfaceData.prevRxBytes )
        {
            // there was an overflow
            m_interfaceData.rxBytes += 0x7FFFFFFF - m_interfaceData.prevRxBytes;
            m_interfaceData.prevRxBytes = 0L;
        }
        if ( m_interfaceData.rxBytes == 0L )
        {
            // on startup set to currently received bytes
            m_interfaceData.rxBytes = currentRxBytes;
            // this is new: KNemo only counts the traffic transfered
            // while it is running. Important to not falsify statistics!
            m_interfaceData.prevRxBytes = currentRxBytes;
        }
        else
            // afterwards only add difference to previous number of bytes
            m_interfaceData.rxBytes += currentRxBytes - m_interfaceData.prevRxBytes;

        m_interfaceData.incomingBytes = currentRxBytes - m_interfaceData.prevRxBytes;
        m_interfaceData.prevRxBytes = currentRxBytes;
            m_interfaceData.rxString = KIO::convertSize( m_interfaceData.rxBytes );
    }

    regExp.setPattern( "TX bytes:(\\d+)\\s*\\(\\d+\\.\\d+\\s*\\w+\\)" );
    if ( regExp.search( config ) > -1 )
    {
        // We count the traffic on ourself to avoid an overflow after
        // 4GB of traffic.
        unsigned long currentTxBytes = regExp.cap( 1 ).toULong();
        if ( currentTxBytes < m_interfaceData.prevTxBytes )
        {
            // there was an overflow
            m_interfaceData.txBytes += 0x7FFFFFFF - m_interfaceData.prevTxBytes;
            m_interfaceData.prevTxBytes = 0L;
        }
        if ( m_interfaceData.txBytes == 0L )
        {
            // on startup set to currently transmitted bytes
            m_interfaceData.txBytes = currentTxBytes;
            // this is new: KNemo only counts the traffic transfered
            // while it is running. Important to not falsify statistics!
            m_interfaceData.prevTxBytes = currentTxBytes;
        }
        else
            // afterwards only add difference to previous number of bytes
            m_interfaceData.txBytes += currentTxBytes - m_interfaceData.prevTxBytes;

        m_interfaceData.outgoingBytes = currentTxBytes - m_interfaceData.prevTxBytes;
        m_interfaceData.prevTxBytes = currentTxBytes;
        m_interfaceData.txString = KIO::convertSize( m_interfaceData.txBytes );
    }
    

    regExp.setPattern( "inet\\s+\\w+:(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3})" );
    if ( regExp.search( config ) > -1 )
        m_interfaceData.ipAddress = regExp.cap( 1 );
    else m_interfaceData.ipAddress = "";
    regExp.setPattern( "(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}).*(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}).*(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3})" );
    if ( regExp.search( config ) > -1 )
    {
        m_interfaceData.broadcastAddress = regExp.cap( 2 );
        m_interfaceData.subnetMask = regExp.cap( 3 );
    }
    else {
        m_interfaceData.broadcastAddress = "";
        m_interfaceData.subnetMask = "";
    }
    

    if ( type == ETHERNET )
    {
        regExp.setPattern( "(.{2}:.{2}:.{2}:.{2}:.{2}:.{2})" );
        if ( regExp.search( config ) > -1 )
            m_interfaceData.hwAddress = regExp.cap( 1 );
    }
    else if (  type == PPP )
    {
        regExp.setPattern( "(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}).*(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}).*(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3})" );
        if ( regExp.search( config ) > -1 )
            m_interfaceData.ptpAddress = regExp.cap( 2 );
    }
}

void KWlanInterface::initLogViewer()
{
    emit (wpaEvents( m_messages));
}

void KWlanInterface::getIpData()
{
    QString ipAddress,netmask,dns1,dns2,gateway, domain;
    bool dontOverrideGw,dontOverrideDns, dhcpEnabled;
    int ret;
    ret = m_configuration->readIpSettings(m_ssid,dhcpEnabled, ipAddress, netmask, gateway, dns1, dns2,domain,dontOverrideGw,dontOverrideDns);
    if ((ret != 0)|| (dhcpEnabled)){
        m_dhcpEnabled = true; //default will be dhcp
        m_staticIpAddress="";
        m_staticNetmask = "";
        m_staticGateway="";
        m_staticDns1 = "";
        m_staticDns2="";
        m_dontOverrideDns="";
        m_dontOverrideGw="";
        m_domain = "";
    }
    else {
        // set static addresses;
        m_dhcpEnabled = dhcpEnabled;
        m_staticIpAddress= ipAddress;
        m_staticNetmask = netmask;
        m_staticGateway=gateway;
        m_staticDns1 = dns1;
        m_staticDns2=dns2;
        m_dontOverrideGw=dontOverrideGw;
        m_dontOverrideDns=dontOverrideDns;
        m_domain=domain;
    }
}

void KWlanInterface::enableInterface( bool enable)
{
    if (!m_controlInterface) return;
    KWlanSuProcess *ifup = new KWlanSuProcess();
    if (enable) {
        *ifup << m_ifconfigPath << m_controlInterface  << "up";
        ifup->setDescription( i18n("Enable interface"));
        kdDebug() << "Enabling interface " << m_controlInterface << endl;
    }
    else {
        *ifup << m_ifconfigPath << m_controlInterface << "down";
        ifup->setDescription( i18n("Disable interface"));
        kdDebug() << "Disabling interface " << m_controlInterface << endl;
    }
    ifup->start( );
}

