

#include "kscandlg.h"

#include "kprofileconfigdlg.h"
#include "kwlaninterface.h"

#include <qlistview.h>
#include <kmessagebox.h>
#include <kprocess.h>
#include <qregexp.h>
#include <kdebug.h>
#include <qtimer.h>

KScanDlg::KScanDlg(QWidget* parent, const char* name, bool modal, WFlags fl)
: KScan(parent,name, modal,fl)
{
    m_interface = 0L;
    m_iwconfigProcess = 0L;
    m_timer = 0L;
}

KScanDlg::~KScanDlg()
{
}

void KScanDlg::closeScanWindow()
{
    if (m_timer) delete m_timer;
    m_timer = 0L;
    close();
}

/*$SPECIALIZATION$*/
void KScanDlg::addSelected()
{
    QListViewItem *sel = scanResultsView->selectedItem();
    if (!sel) {
        KMessageBox::sorry(this,"kwlan","No Network selected!");
        return;
    }
    bssSelected( sel);
}

void KScanDlg::bssSelected( QListViewItem * sel )
{
    KProfileConfigDlg *nc = new KProfileConfigDlg();
    if (nc == NULL)
        return;
    nc->setInterface(m_interface);
    nc->paramsFromScanResults(sel);
    nc->show();
    nc->exec();
}

void KScanDlg::getResults()
{
    updateResults();
}

void KScanDlg::scanRequest()
{
    if (m_interface == NULL)
        return;
    if (m_interface->hasControlConnection())
    {
        char reply[10];
        size_t reply_len = sizeof(reply);
        m_interface->ctrlRequest("SCAN", reply, &reply_len);
    } else
    {
	//kdDebug() << "scanning using iwconfig" << endl;
        scanNonWpa();
    }
}

void KScanDlg::updateResults()
{
    char reply[8192];
    size_t reply_len;
    
    if (m_interface == NULL)
        return;

    reply_len = sizeof(reply) - 1;
    if (m_interface->ctrlRequest("SCAN_RESULTS", reply, &reply_len) < 0)
        return;
    reply[reply_len] = '\0';

    scanResultsView->clear();
    
    QString res(reply);
    QStringList lines = QStringList::split(QChar('\n'), res);
    bool first = true;
    for (QStringList::Iterator it = lines.begin(); it != lines.end(); it++) {
        if (first) {
            first = false;
            continue;
        }
        //kdDebug() << res << endl;
        QStringList cols = QStringList::split(QChar('\t'), *it, true);
        QString ssid, bssid, freq, signal, flags;
        bssid = cols.count() > 0 ? cols[0] : "";
        freq = cols.count() > 1 ? cols[1]: "";
        signal = cols.count() > 2 ? cols[2] : "";
        flags = cols.count() > 3 ? cols[3] : "";
        ssid = cols.count() > 4 ? cols[4] : "";
        //work around signal bug (it's about twice as what iwscan gives)
        signal = QString("%1").arg(signal.toInt()/2);
        new QListViewItem(scanResultsView, ssid, bssid, freq, signal, flags);
    }
}

void KScanDlg::setInterface( KwlanInterface *interface )
{
    m_interface  = interface;
    if (!m_interface) return; 
    
    if (m_interface->hasControlConnection())
    {
        // use wpaScanning
        //updateResults();
	scanRequest();
        m_timer = new QTimer(this);
        if (!m_timer) return;
        connect(m_timer, SIGNAL(timeout()), SLOT(getResults()));
        m_timer->start(5000, FALSE);
     } else { 
        scanNonWpa();
        m_timer = new QTimer(this);
        if (!m_timer) return;
        connect(m_timer, SIGNAL(timeout()), SLOT(scanNonWpa()));
        m_timer->start(3000, FALSE);
    }
}

void KScanDlg::scanNonWpa()
{
    if ( !m_iwconfigProcess )
    {
        m_iwconfigStdout = QString::null;
        m_iwconfigProcess = new KProcess();
        m_iwconfigProcess->setEnvironment( "LANG", "C" );
        m_iwconfigProcess->setEnvironment( "LC_ALL", "C" );
        //TODO: use path to iwconfig
        *m_iwconfigProcess << "iwlist" << m_interface->getInterfaceName() << "scanning";
        connect( m_iwconfigProcess,  SIGNAL( receivedStdout( KProcess*, char*, int ) ),
                 this, SLOT( iwconfigProcessStdout( KProcess*, char*, int ) ) );
        connect( m_iwconfigProcess,  SIGNAL( receivedStderr( KProcess*, char*, int ) ),
                 this, SLOT( iwconfigProcessStdout( KProcess*, char*, int ) ) );
        connect( m_iwconfigProcess,  SIGNAL( processExited( KProcess* ) ),
                 this, SLOT( iwconfigProcessExited( KProcess* ) ) );

        if ( !m_iwconfigProcess->start( KProcess::NotifyOnExit, KProcess::AllOutput ) )
        {
            delete m_iwconfigProcess;
            m_iwconfigProcess = 0L;
        }
    }
}

void KScanDlg::iwconfigProcessExited( KProcess* process )
{
    if ( process == m_iwconfigProcess )
    {
        delete m_iwconfigProcess;
        m_iwconfigProcess = 0L;
        parseIwconfigOutput();
    }
}
void KScanDlg::iwconfigProcessStdout( KProcess*, char* buffer, int buflen )
{

    //kdDebug() << buffer << endl;
    m_iwconfigStdout += QString::fromLatin1( buffer, buflen );
}

void KScanDlg::parseIwconfigOutput()
{
    QMap<QString, QString> configs;
    QStringList ifList = QStringList::split( "Cell", m_iwconfigStdout );
    // Now we have each network in a QString element;
    QStringList::Iterator it;
    QString key;
    scanResultsView->clear();
    for ( it = ifList.begin(); it != ifList.end(); ++it )
    {
        int index = ( *it ).find( ' ' );
        if ( index == -1 )
            continue;
        QString config = ( *it ).mid( index );
        addScannedNetwork(config);
    }
}

void KScanDlg::addScannedNetwork(QString config)
{
    QStringList::Iterator it;
    QString essid, bssid, channel, quality,speed,flags;
    QRegExp regExp( "ESSID:\"?([^\"]*)\"?" );
    if ( regExp.search( config ) > -1 )
        essid = regExp.cap( 1 );
    regExp.setPattern( "((\\d|[A-F]){2,2}\\:(\\d|[A-F]){2,2}\\:(\\d|[A-F]){2,2}\\:(\\d|[A-F]){2,2}\\:(\\d|[A-F]){2,2}\\:(\\d|[A-F]){2,2})" );
    if ( regExp.search( config ) > -1 )
        bssid = regExp.cap( 1 );
    regExp.setPattern( "Quality:(\\d*)" );
    if ( regExp.search( config ) > -1 )
        quality = regExp.cap( 1 );
    regExp.setPattern("Channel:(\\d{1,2})");
    if ( regExp.search( config ) > -1 )
        channel = regExp.cap( 1 );
    regExp.setPattern("Bit Rates:(\\w*)");
    if ( regExp.search( config ) > -1 )
        speed = regExp.cap( 1 );            
    QStringList encryptionList = QStringList::split( "IE:", config );
    // Now loop through the encryption modes
    for ( it = encryptionList.begin(); it != encryptionList.end(); ++it )
    {
        if ((*it).contains("WPA Version"))
        {
            flags +="[WPA";
            if ((*it).contains("PSK")) flags +="-PSK";
            else flags+="-EAP";
            if ( (*it).contains("TKIP")) flags +="-TKIP]";
            else flags += "-CCMP";
        }   
        if ((*it).contains("WPA2 Version"))
        {
            flags +="[WPA2";
            if ((*it).contains("PSK")) flags +="-PSK";
            else flags+="-EAP";
            if ( (*it).contains("TKIP")) flags +="-TKIP]";
            else flags += "-CCMP";
        }   
        if ((*it).contains("WEP")) flags += "[WEP]";
    }
    new QListViewItem(scanResultsView, essid, bssid, channel, quality, flags);
}

#include "kscandlg.moc"


