/* 
        (c) 2006 by Thomas Michel <tom.michel@arcor.de>

        Kwlan is free software; you can redistribute it and/or modify
        it under the terms of the GNU Library General Public License as
        published by the Free Software Foundation; either version 2 of
        the License, or (at your option) any later version.

        Kwlan is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU Library General Public License for more details.

        You should have received a copy of the GNU Library General Public License
        along with this library; see the file COPYING.LIB.  If not, write to
        the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
        Boston, MA 02110-1301, USA.
*/
#include <stdlib.h>

#include <qregexp.h>
#include <qstringlist.h>
#include <qtextstream.h>
#include <kdebug.h>
#include <kprocess.h>
#include <kglobal.h>
#include <kconfig.h>
#include <kstandarddirs.h>
#include <ktempfile.h>


#include "suprocessbase.h"
#include "config.h"
#include "globals.h"

class SuProcessBase::Private {
  public:
    KProcess *proc;
    int exitStatus, comm;
    bool passwordSent, authFailed, canRestart;
    State state;    
    RunMode runmode;
    
    QCString password;
    QString marker;
    QStringList args;
    QString user;
    
    KTempFile *wrapper;
};

SuProcessBase::SuProcessBase(QObject *parent, const char *name): 
  QObject(parent, name)
{
  d = new Private;
  // First check for su command (sudo or su)
  KConfig* config = KGlobal::config();
  config->setGroup("super-user-command");
  m_suCommand = config->readEntry("super-user-command", DEFAULT_SUPER_USER_COMMAND);
  if ( m_suCommand != "sudo" && m_suCommand != "su" ) {
      kdWarning() << "unknown super user command" << endl;
      m_suCommand = "su";
  }
  m_useSudo = FALSE;
  if (m_suCommand == "sudo") m_useSudo = TRUE;
  d->proc = new KProcess(this);
  //init start
  d->exitStatus = -1;
  d->passwordSent = false;
  d->authFailed = false;
  d->state = NotRunning; 
  d->canRestart = false;
  d->marker = QString("SPB_%1").arg(random(), 8, 16);
  d->wrapper=0;
  // init end
  d->runmode = NotifyOnExit;
  d->comm = 0;
}

SuProcessBase::~SuProcessBase()
{
  d->password.fill(0);
  if (d->wrapper) delete d->wrapper;
  delete d;
}

SuProcessBase& SuProcessBase::operator << (const QString &arg)
{
  d->args.append(arg);
  return *this;
}

void SuProcessBase::clearArguments()
{
    d->args.clear();
}

bool SuProcessBase::start(RunMode runmode, Communication comm)
{
  prepareStart();

  connect(d->proc, SIGNAL(wroteStdin(KProcess *)), 
    this, SLOT(slotWroteStdin(KProcess *))); 
  connect(d->proc, SIGNAL(receivedStdout(KProcess *, char *, int)), 
    this, SLOT(slotReceivedStdout(KProcess *, char *, int)));
  connect(d->proc, SIGNAL(receivedStderr(KProcess *, char *, int)), 
    this, SLOT(slotReceivedStderr(KProcess *, char *, int)));
  connect(d->proc, SIGNAL(processExited(KProcess *)), 
    this, SLOT(slotProcessExited(KProcess *)));
    
  d->proc->setUsePty(KProcess::Communication(KProcess::Stdin + KProcess::Stdout), false);    
  
  d->state = StartingUp;
  d->runmode = runmode;
  d->comm = comm;
  return d->proc->start(KProcess::NotifyOnExit, KProcess::All);
}

void SuProcessBase::restart()
{
  if (d->canRestart) {
    d->exitStatus = -1;
    d->passwordSent = false;
    d->authFailed = false;
    d->state = StartingUp; 
    d->canRestart = false;
  
    d->proc->start(KProcess::NotifyOnExit, KProcess::All);
  }
}

void SuProcessBase::prepareStart()
{
    //QString superUserCommand;
    bool exitCodeEchoed = FALSE;  
   /* First of all, we need to build a wrapper to run user specified command(s)
    * It generally looks like this:
    *
    * echo SPBxxxxxxxx:; #Password was okay, preparing to run the command
    * echo SPBxxxxxxxx: Starting; #Starting the command
    * user specified command 1;
    * SPBxxxxxxxx=$?; if [ ! $SPBxxxxxxxx -eq 0 ]; then echo -e \\nSPBxxxxxxxx: Exit $SPBxxxxxxxx; exit; fi # Check exit code and exit if it failed
    * user specified command 2;   
    * SPBxxxxxxxx=$?; if [ ! $SPBxxxxxxxx -eq 0 ]; then echo -e \\nSPBxxxxxxxx: Exit $SPBxxxxxxxx; exit; fi # Check exit code and exit if it failed
    * ...
    * user specified command N;      
    * echo -e \\nSPBxxxxxxxx: Exit $? # Show exit code for the last command
    *
   */
    d->wrapper = new KTempFile(locateLocal("tmp", "wrapper"), 0, 0755);
    d->wrapper->setAutoDelete(true);
    QTextStream &ts = *(d->wrapper->textStream());
    
    ts << "#!/bin/sh" << endl << endl;
    ts << "echo " << d->marker << ":" << endl;
    ts << "echo " << d->marker << ": Starting; " << endl;


  for(QStringList::Iterator it = d->args.begin(); it != d->args.end(); it++) {
      if (*it != ";") {
      ts << *it << " ";
      exitCodeEchoed = false;        
    }
    else {
      // ";" denotes special case: we need to be sure last command was successful before starting the next one
      ts << endl << d->marker + "=$?" << endl;
      ts << "if [ ! $" << d->marker << " -eq 0 ]; then echo -e \\\\n" << d->marker + ": Exit $" << d->marker + "; exit; fi; " << endl;
      exitCodeEchoed = true;
    }
  }
  if (!exitCodeEchoed)
     ts << endl << "echo -e \\\\n" << d->marker << ": Exit: $?" << endl;
      
  d->wrapper->close();
  KProcess &proc = *d->proc;
  proc << m_suCommand;
  if (!m_useSudo) {
      proc << "-";
      if (!d->user.isEmpty()) proc << d->user;
      proc << "-c";  
  }
  else    {
      if (!d->user.isEmpty()) proc << "-u" << d->user;
  }
  proc << d->wrapper->name();
}

void SuProcessBase::slotWroteStdin(KProcess *)
{
  if (d->passwordSent)
    d->password.fill(0);
}

void SuProcessBase::slotReceivedStdout(KProcess *, char *buffer, int buflen)
{
  handleOutput(TypeStdout, buffer, buflen);
}

void SuProcessBase::slotReceivedStderr(KProcess *, char *buffer, int buflen)
{
  handleOutput(TypeStderr, buffer, buflen);
}

void SuProcessBase::slotProcessExited(KProcess *)
{
  d->canRestart = true;
  if (d->runmode != DontCare)
    emit processExited(this);
}

void SuProcessBase::handleOutput(OutputType type, char *buffer, int buflen)
{
  // Split data into separate lines first, because even when the app is
  // running we need to detect out-of-band data, which is line-based
  // rather than on raw data streams
  QString rawBuffer = QString::fromLocal8Bit( buffer, buflen );
  QStringList lines = QStringList::split( QRegExp( "\n|\n\r|\r\n" ), rawBuffer, true );
  QStringList::Iterator it = lines.begin();    
  bool removeIt = false;    
  
  while ( it != lines.end() ) {
    QString curLine = *it;
    
    if (debugOutput)   kdDebug() << curLine << endl;
	
    switch ( d->state ) {
      case NotRunning:
        kdWarning() << k_funcinfo << "Received output while process is not known as running?" << endl;
        break;
	
      // Here we go. The only allowed line is "Password:" from su or marker wich means su succeeded
      case StartingUp:
        if ( curLine.startsWith( QString( "%1:" ).arg( d->marker ) ) ) {
	  d->state = Launching;
        }
        else if ( curLine.isEmpty() ) {
          // Ignore stray newlines, those happen after e.g. sending
          // the pass and are thus to be expected
        }
        else {
	  bool result = handlePasswordRequest(curLine);
	  //if (!result)
	  //  kdWarning() << k_funcinfo << "Unhandled out-of-band data received: " << curLine << endl;	  
	}
	    
	removeIt = true;
        break;
	
      // At this point we've entered the password and waiting for "Starting" line to appear.
      case Launching:
          if ( curLine.startsWith( QString( "%1: Starting" ).arg( d->marker ) ) ) {
            d->state = Running;
	  }  

          removeIt = true;
          break;

      // At this point the program is running so we expect an arbitrary text output from it.
      // The only special case is "$marker: Exit: $exitcode" line wich means the program is exiting.
      case Running:
        //kdDebug() << curLine << endl;
        if ( curLine.startsWith( QString( "%1: " ).arg( d->marker ) ) ) {
          int pos;
	  
          if ( ( pos = curLine.findRev( "Exit: " ) ) != -1 ) {
            d->state = ShuttingDown;
            d->exitStatus = curLine.mid( pos + 6 ).toInt();
          }    
          //else
            //kdWarning() << k_funcinfo << "Unhandled out-of-band data received: " << curLine << endl;
		
          removeIt = true;
        }
        else
          removeIt = false;
	    
        break;

      // The program is shutting down.
      // Nothing special. Just send the contents of flushed buffers to client	
      // FIXME: Do we really need to send it to the client?      
      case ShuttingDown:
        removeIt = false;
        break;
	
      default:
        kdWarning() << k_funcinfo << "Process is in state " << d->state << ", which is not defined!" << endl;
        break;
    } // case
	
    if ( removeIt )
      it = lines.remove( it );
    else
      it++;
  }	

  if ( !lines.isEmpty() ) {
    // FIXME: The join removes any distinction between \n and \r. This is
    //        ok for most console applications, but for apps that use real
    //        terminal I/O, like curses apps, this is not good.
    //        Rewrite the algorithm to preserve the used type of newline.
    QCString newBuffer = lines.join( "\r\n" ).local8Bit();

    // FIXME: The const_cast is ugly, to say the least. Either we need to
    //        divert from KProcess compatibility and make the buffer
    //        argument const, or we need to allocate a temporary local
    //        buffer.
    if ( type == TypeStdout ) {
      if (d->comm & Stdout)
        emit receivedStdout( this, const_cast<char *>( ( const char * )( newBuffer ) ), newBuffer.length() );
    }	
    else {
      if (d->comm & Stderr)
        emit receivedStderr( this, const_cast<char *>( ( const char * )( newBuffer ) ), newBuffer.length() );
    }	
  }

}

bool SuProcessBase::handlePasswordRequest(const QString &buffer)
{
  if ( buffer.startsWith( "Password:" ) ) {
    if (d->password.isEmpty() && !getPassword(d->password))
      // From our point of view, it's a failure: it's up to calling program to distinguish
      // "wrong password" and "no password entered" cases and suppress error dialogs as needed
      incorrectPassword();

      d->password.append( "\n" );
      d->proc->writeStdin( d->password, d->password.length() );
      d->passwordSent = true;
      //m_state = Launching;
      
      return true;
  }

  // We've entered the password and are still in authorization phase. It means
  // that password was was incorrect since su never writes anything upon successful login
  if ( d->passwordSent && d->state == StartingUp) {
    incorrectPassword();
    return true;
  }    
  
  return false;
}

void SuProcessBase::incorrectPassword() 
{
  d->authFailed = true;
  d->proc->kill();
  return;
}

int SuProcessBase::exitStatus() const
{
  return d->exitStatus;
}

bool SuProcessBase::isRunning() const
{
  return (d->state == Running);
}

bool SuProcessBase::authFailed() const
{
  return d->authFailed;
}

QString SuProcessBase::quote(const QString &arg)
{
  return KProcess::quote(arg);
}
