/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qpixmap.h>
#include <qpainter.h>
#include <qptrlist.h>
#include <qtabwidget.h>
#include <qpushbutton.h>
#include <qcheckbox.h>
#include <qprogressdialog.h>
#include <qtoolbutton.h>
#include <knuminput.h>
#include <klineedit.h>
#include <kapplication.h>
#include <klocale.h>
#include <kstandarddirs.h>
#include "configuration.h"
#include "importimagedlg.h"
#include "kxstitch.h"
#include "flossscheme.h"
#include "floss.h"

ImportImageDlg::ImportImageDlg(const Magick::Image& image, QWidget *parent, const char *name, bool modal)
  : ImportImage(parent,name,modal),
    m_pixmap(new QPixmap),
    m_orig(image)
{
  int w = m_orig.columns();
  int h = m_orig.rows();

  // get the users preferred pattern size in stitches
  double cc = KXSConfig().NewPattern_ClothCount;
  QString ccu = KXSConfig().NewPattern_ClothCountUnits;
  Import_ClothCount->setSuffix(ccu);
  Import_ClothCount->setRange(1,55,(ccu == "/cm")?0.01:1.0,false);
  Import_ClothCount->setValue(cc);
  int pw = KXSConfig().NewPattern_Width;
  int ph = KXSConfig().NewPattern_Height;
  int sw = pw*100/w;
  int sh = ph*100/h;
  int s = sw<?sh; // pick smallest scale

  Import_FlossScheme->insertStringList(((KXStitchApplication*)kapp)->schemes());
  Configuration::setComboBoxItem(Import_FlossScheme,KXSConfig().NewPattern_FlossScheme);
  createImageMap();
  Import_PatternScale->setValue(s);
  Import_UseMaxColors->setChecked(KXSConfig().Import_UseMaxColors);
  Import_MaxColors->setValue(KXSConfig().Import_MaxColors);

  calcSizes();

  connect(Import_FlossScheme, SIGNAL(activated(int)), this, SLOT(schemeChanged(int)));
  connect(Import_UseMaxColors, SIGNAL(stateChanged(int)), this, SLOT(maxColorsChanged(int)));
  connect(Import_MaxColors, SIGNAL(valueChanged(int)), this, SLOT(maxColorsChanged(int)));
  connect(Import_ClothCount, SIGNAL(valueChanged(double)), this, SLOT(clothCountChanged(double)));
  connect(Import_PatternScale, SIGNAL(valueChanged(int)), this, SLOT(scaleChanged(int)));
  connect(ResetButton, SIGNAL(clicked()), this, SLOT(resetImage()));
  connect(HelpButton, SIGNAL(clicked()), this, SLOT(contextHelp()));

  renderPixmap();
}

ImportImageDlg::~ImportImageDlg()
{
  delete m_pixmap;
}

void ImportImageDlg::resetImage()
{
  m_image = m_orig;
  calcSizes();
  renderPixmap();
}

void ImportImageDlg::schemeChanged(int)
{
  createImageMap();
  calcSizes();
  renderPixmap();
}

void ImportImageDlg::calcSizes()
{
  m_image = m_orig;
  Magick::Geometry geom(Import_PatternScale->value(), Import_PatternScale->value());
  geom.percent(true);
  m_image.sample(geom);
  int w = m_image.columns();
  int h = m_image.rows();
  m_size = QSize(w,h);
  clothCountChanged(Import_ClothCount->value());
}

void ImportImageDlg::maxColorsChanged(int)
{
  calcSizes();
  renderPixmap();
}

void ImportImageDlg::clothCountChanged(double n)
{
  double wi = m_size.width();
  double hi = m_size.height();
  QString suffix;
  QString units = KXSConfig().NewPattern_Units;
  if (units != "Stitches")
  {
    wi /= n;
    hi /= n;
    suffix = Import_ClothCount->suffix().right(2);
  }
  PatternSize->setText(QString(i18n("%1%2 x %3%4")).arg(wi,0,'g',3).arg(suffix).arg(hi,0,'g',3).arg(suffix));
}

void ImportImageDlg::scaleChanged(int)
{
  calcSizes();
  renderPixmap();
}

void ImportImageDlg::createImageMap()
{
  FlossScheme *scheme = ((KXStitchApplication*)kapp)->scheme(Import_FlossScheme->currentText());
  QPtrListIterator<Floss> it = scheme->flossIterator();
  char *pixels = new char[it.count()*3];
  char *pixel = pixels;
  Floss *f;
  for ( ; (f = it.current()) ; ++it)
  {
    *pixel++ = (char)(f->color.red());
    *pixel++ = (char)(f->color.green());
    *pixel++ = (char)(f->color.blue());
  }
  m_map = Magick::Image(it.count(), 1, "RGB", MagickLib::CharPixel, pixels);
  delete pixels;
}

void ImportImageDlg::renderPixmap()
{
  CropImageLabel->setCursor(waitCursor);
  m_image.modifyImage();
  m_pixmap->resize(m_size);
  m_pixmap->fill();
  if (Import_UseMaxColors->isChecked())
  {
    // TODO add noise reduction routines to clean up converted images
    m_image.quantizeColorSpace(Magick::RGBColorspace);
    m_image.quantizeColors(Import_MaxColors->value());
    m_image.quantize();
  }
  m_image.map(m_map);
  m_image.modifyImage();

  QPainter p;
  p.begin(m_pixmap);
  int w = m_size.width();
  int h = m_size.height();
  int pixelCount = w*h;

  QProgressDialog progress(i18n("Rendering preview..."), i18n("Cancel"), pixelCount, this, i18n("Progress"), true);
  progress.setMinimumDuration(100);

  Magick::Pixels cache(m_image);
  const Magick::PixelPacket *pixels = cache.get(0,0,w,h);
  for (int cy = 0 ; cy < h ; cy++)
  {
    kapp->processEvents();
    progress.setProgress(cy*w);
    if (progress.wasCancelled()) break;
    for (int cx = 0 ; cx < w ; cx++)
    {
      Magick::PixelPacket packet = *pixels++;
      Magick::Color c = Magick::Color(packet.red, packet.green, packet.blue);
      if (!(packet.opacity))
      {
        QColor c(packet.red/257, packet.green/257, packet.blue/257);
        p.setPen(QPen(c));
        p.drawPoint(cx,cy);
      }
    }
  }
  p.end();
  CropImageLabel->setPixmap(*m_pixmap);
  CropImageLabel->setCursor(arrowCursor);
}

void ImportImageDlg::accept()
{
  if (Import_SetDefault->isChecked())
  {
    KXSConfig().Import_UseMaxColors = Import_UseMaxColors->isChecked();
    KXSConfig().Import_MaxColors = Import_MaxColors->value();
    KXSConfig().writeSettings();
  }
  QDialog::accept();
}

void ImportImageDlg::contextHelp()
{
  kapp->invokeHelp("ImportImageDlg");
}
