/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qdir.h>
#include <qwidget.h>
#include <qradiobutton.h>
#include <qspinbox.h>
#include <qlineedit.h>
#include <qpixmap.h>
#include <qcheckbox.h>
#include <qmultilineedit.h>
#include <qpainter.h>
#include <qprogressdialog.h>
#include <qimage.h>
#include <qbuffer.h>
#include <qclipboard.h>
#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kconfig.h>
#include <kcolorbutton.h>
#include <kio/job.h>
#include <kio/netaccess.h>
#include <kdebug.h>
#include <kfiledialog.h>
#include <kscan.h>
#include "kxstitchdoc.h"
#include "kxstitch.h"
#include "kxstitchview.h"
#include "kxstitchmimesource.h"
#include "patternpropertiesdialog.h"
#include "flosslistboxitem.h"
#include "floss.h"
#include "flossscheme.h"
#include "patterncanvas.h"
#include "flosspalette.h"
#include "importimagedlg.h"
#include "configuration.h"
#include "palettemanagerdialog.h"

#define FILE_FORMAT_VERSION 6

QPtrList<KXStitchView> *KXStitchDoc::viewList = 0L;

KXStitchDoc::KXStitchDoc(QWidget *parent, const char *name)
  : QObject(parent, name),
    m_scanDialog(0)
{
  if(!viewList)
  {
    viewList = new QPtrList<KXStitchView>();
  }
  viewList->setAutoDelete(true);

  m_propertiesDialog = new PatternPropertiesDialog(parent);  // will initialise the properties to the users defaults
  m_palette = new FlossPalette();
  m_canvas = new PatternCanvas();
  m_preview = new QPixmap();
  m_docURL.setFileName(i18n("Untitled"));
  setModified(false);
}

KXStitchDoc::~KXStitchDoc()
{
  delete m_palette;
  delete m_canvas;
  delete m_preview;
  delete m_propertiesDialog;
}

void KXStitchDoc::resize(int w, int h)
{
  m_canvas->resize(w, h);
  m_preview->resize(w*2, h*2);
}

void KXStitchDoc::extendPattern(int l, int t, int r, int b)
{
  if (l) m_canvas->extendLeft(l);
  if (t) m_canvas->extendTop(t);
  if (r) m_canvas->extendRight(r);
  if (b) m_canvas->extendBottom(b);
  m_propertiesDialog->setPatternSize(m_canvas->patternWidth(),m_canvas->patternHeight());
  renderPixmap();
  setModified(true);
}

void KXStitchDoc::centerPattern()
{
  m_canvas->centerPattern();
  renderPixmap();
  setModified(true);
}

void KXStitchDoc::cropCanvasToPattern()
{
  cropCanvasToRect(m_canvas->patternBoundingRect());
}

void KXStitchDoc::cropCanvasToRect(QRect rect)
{
  m_canvas->cropCanvasToRect(rect);
  m_propertiesDialog->setPatternSize(m_canvas->patternWidth(),m_canvas->patternHeight());
  renderPixmap();
  setModified(true);
}

void KXStitchDoc::addView(KXStitchView *view)
{
  viewList->append(view);
}

void KXStitchDoc::removeView(KXStitchView *view)
{
  viewList->remove(view);
}

void KXStitchDoc::setURL(const KURL &url)
{
  if (url.fileName().right(4).lower() == ".pat")  // this looks like a PCStitch file
    m_docURL.setFileName(i18n("Untitled"));        // so that the user doesn't overwrite a PCStitch
                                                  // file with a KXStitch file
  else
    m_docURL=url;
}

const KURL& KXStitchDoc::URL() const
{
  return m_docURL;
}

void KXStitchDoc::slotResetAllViews()
{
  KXStitchView *w;
  if(viewList)
  {
    for(w=viewList->first(); w!=0; w=viewList->next())
    {
      w->resetView();
    }
  }
}

bool KXStitchDoc::saveModified()
{
  bool completed=true;
  if(m_modified)
  {
    KXStitchApp *win=(KXStitchApp *) parent();
    int want_save = KMessageBox::warningYesNoCancel(win,
                                        i18n("The current file has been modified.\n"
                                              "Do you want to save it?"),
                                        i18n("Warning"));
    switch(want_save)
    {
      case KMessageBox::Yes:
          if (m_docURL.fileName() == i18n("Untitled"))
          {
            win->slotFileSaveAs();
          }
          else
          {
            saveDocument(URL());
          }
          deleteContents();
          completed=true;
          break;

      case KMessageBox::No:
          setModified(false);
          completed=true;
          break;

      case KMessageBox::Cancel:
          completed=false;
          break;

      default:
          completed=false;
          break;
    }
  }
  return completed;
}

void KXStitchDoc::closeDocument()
{
  deleteContents();
}

QString KXStitchDoc::title()
{
  return m_propertiesDialog->title();
}

QString KXStitchDoc::author()
{
  return m_propertiesDialog->author();
}

QString KXStitchDoc::copyright()
{
  return m_propertiesDialog->copyright();
}

QString KXStitchDoc::instructions()
{
  return m_propertiesDialog->instructions();
}

QColor KXStitchDoc::canvasColor()
{
  return m_propertiesDialog->fabricColor();
}

void KXStitchDoc::fileProperties()
{
  if (m_propertiesDialog->exec() == QDialog::Accepted)
  {
    QSize requestedSize = m_propertiesDialog->patternSize();
    if (requestedSize.width() != m_canvas->patternWidth() || requestedSize.height() != m_canvas->patternHeight())
    {
      /** size of canvas has changed */
      int l = 0;
      int r = 0;
      int t = 0;
      int b = 0;
      QRect currentRect = m_canvas->patternBoundingRect();
      if (currentRect.isValid())
      {
        if (requestedSize.width() < currentRect.width())
          requestedSize.setWidth(currentRect.width());
        if (requestedSize.height() < currentRect.height())
          requestedSize.setHeight(currentRect.height());
        if (requestedSize.width() > currentRect.width())
        {
          l = (requestedSize.width()-currentRect.width())<?currentRect.left();
          r = requestedSize.width()-(currentRect.width()+l);
        }
        if (requestedSize.height() > currentRect.height())
        {
          t = (requestedSize.height()-currentRect.height())<?currentRect.top();
          b = requestedSize.height()-(currentRect.height()+t);
        }
        cropCanvasToPattern();
        extendPattern(l,t,r,b);
      }
      else
      {
        resize(requestedSize.width(),requestedSize.height());
      }
    }
    if (m_propertiesDialog->scheme() != m_palette->getScheme())
    {
      /** floss scheme changed */
      if (m_canvas->isEmpty())
      {
        /** its ok to change the palette as there are no existing stitches */
        m_palette->setScheme(m_propertiesDialog->scheme());
      }
      else
      {
        /** there are stitches based on the current palette
            try to change the scheme, but if this fails reset the dialog box value to current scheme
          */
        if (m_palette->changeScheme(m_propertiesDialog->scheme()) == false)
        {
          QString currentScheme = m_palette->getScheme();
          m_propertiesDialog->setScheme(currentScheme);
        }
      }
    }
    renderPixmap();
  }
}

bool KXStitchDoc::newDocument()
{
  m_propertiesDialog->reset();
  m_docURL.setFileName(i18n("Untitled"));
  m_palette->clear();
  m_palette->setScheme(m_propertiesDialog->scheme());
  m_canvas->clear();
  resize(m_propertiesDialog->patternWidth(),m_propertiesDialog->patternHeight());
  m_preview->fill();
  m_modified = false;
  return true;
}

void KXStitchDoc::renderPixmap()
{
  int w = m_canvas->patternWidth();
  int h = m_canvas->patternHeight();
  if (w>0 && h>0)
  {
    if ((w*2) != m_preview->width() || (h*2) != m_preview->height())
      m_preview->resize(w*2,h*2);
    updatePixmap(QRect(0,0,w,h));
  }
}

void KXStitchDoc::updatePixmap()
{
  updatePixmap(QRect(0,0,m_canvas->patternWidth(),m_canvas->patternHeight()));
}

void KXStitchDoc::updatePixmap(QPoint cell)
{
  updatePixmap(QRect(cell.x(),cell.y(),1,1));
}

void KXStitchDoc::updatePixmap(QRect rect)
{ // rect is the area of cells that are to be updated
  QPainter *p = new QPainter();
  p->begin(m_preview);

  int x = rect.left();
  int x2 = x*2;
  int y = rect.top();
  int y2 = y*2;
  int w = rect.width();
  int w2 = w*2;
  int h = rect.height();
  int h2 = h*2;

  for (int dy = y ; dy < y+h ; dy++)
  {
    for (int dx = x ; dx < x+w ; dx++)
    {
      if (Stitch::Queue *pStitchQueue = m_canvas->stitchAt(QPoint(dx,dy)))
      {
        int stitches = pStitchQueue->count();
        while (stitches--)
        {
          int dx2 = dx*2;
          int dy2 = dy*2;
          Stitch *pStitch = pStitchQueue->dequeue();
          Stitch::Type t = pStitch->type;
          int f = pStitch->floss;
          Floss *pF = m_palette->flossAt(f);
          p->setPen(pF->color);
          if (t & Stitch::TLQtr)
            p->drawPoint(dx2,dy2);
          if (t & Stitch::TRQtr)
            p->drawPoint(dx2+1,dy2);
          if (t & Stitch::BLQtr)
            p->drawPoint(dx2,dy2+1);
          if (t & Stitch::BRQtr)
            p->drawPoint(dx2+1,dy2+1);
          pStitchQueue->enqueue(pStitch);
        }
      }
      else
        p->fillRect(dx*2,dy*2,2,2,canvasColor());
    }
  }
  QPtrListIterator<Knot> *itK = m_canvas->knots();
  for (Knot *k = 0 ; (k = itK->current()) ; ++(*itK))
  {
    if (QRect(x2,y2,w2,h2).contains(k->pos))
    {
      p->setPen(m_palette->flossAt(k->floss)->color);
      p->drawPoint(k->pos);
    }
  }
  QPtrListIterator<BackStitch> *it = m_canvas->backstitches();
  for (BackStitch *b = 0 ; (b = it->current()) ; ++(*it))
  {
    p->setPen(m_palette->flossAt(b->floss)->color);
    p->drawLine(b->start,b->end);
  }
  p->end();
}

bool KXStitchDoc::readKXStitchDocument(QDataStream& s)
{
  QString magic;
  Q_INT16 version;
  bool    validRead = false;
  QString title;
  QString author;
  QString fabric;
  QColor  fabricColor;
  Q_INT16 scheme;
  Q_INT16 inches;
  Q_INT32 width;
  Q_INT32 height;
  Q_INT32 clothCount;
  QString instructions;
  s >> magic;
  if (magic == "KXStitch")
  {
    newDocument();
    s >> version;
    switch (version)
    {
      case 6: // no change for this section
      case 5:
        s >> *m_propertiesDialog;
        validRead = m_palette->readKXStitchPalette(s,version) && m_canvas->readKXStitchCanvas(s,version);
        break;
      case 4:
      case 3:
      case 2:
        s >> title >> author >> fabric >> fabricColor >> scheme >> inches >> width >> height >> clothCount >> instructions;
        m_propertiesDialog->setTitle(title);
        m_propertiesDialog->setAuthor(author);
        m_propertiesDialog->setCopyright(QString());
        m_propertiesDialog->setFabric(fabric);
        m_propertiesDialog->setFabricColor(fabricColor);
        m_propertiesDialog->setUnits(bool(inches)?"Inches":"Stitches");
        m_propertiesDialog->setClothCount(double(clothCount));
        m_propertiesDialog->setClothCountUnits("/in");
        if (m_propertiesDialog->units() == "Inches")
        {
          width *= clothCount;
          height *= clothCount;
        }
        m_propertiesDialog->setPatternSize(width,height);
        m_propertiesDialog->setInstructions(instructions);
        validRead = m_palette->readKXStitchPalette(s,version);
        if (validRead)
        {
          m_propertiesDialog->setScheme(m_palette->getScheme());
          validRead = m_canvas->readKXStitchCanvas(s,version);
        }
        break;
      case 1:
      default:
        KMessageBox::sorry(0, i18n("The version of this pattern file is not supported by this version of the software."), i18n("Unsupported File Format"));
        break;
    }
  }
  return validRead;
}

bool KXStitchDoc::readPCStitch5Document(QDataStream& s)
{
  char*   buffer = new char[256];
  Q_INT16 width;
  Q_INT16 height;
  Q_INT16 clothCount;
  Q_INT32 unknown;
  QString instructions;
  int     index;

  s.setByteOrder(QDataStream::LittleEndian);
  s.readRawBytes(buffer,256);
  buffer[23] = '\0';
  if (QString(buffer) != "PCStitch 5 Pattern File")
  {
    delete [] buffer;
    return false;
  }
  delete [] buffer;
  s >> unknown;
  s >> unknown;
  s >> width;
  s >> height;
  m_propertiesDialog->setUnits("Stitches");
  s >> clothCount;
  m_propertiesDialog->setClothCount(clothCount);
  m_propertiesDialog->setClothCountUnits("/in");
  s >> clothCount; // this is a vertical cloth count and isn't used
  m_propertiesDialog->setPatternSize(width,height);
  m_propertiesDialog->setAuthor(readPCStitchString(s));
  m_propertiesDialog->setCopyright(readPCStitchString(s));
  m_propertiesDialog->setTitle(readPCStitchString(s));
  m_propertiesDialog->setFabric(readPCStitchString(s));
  instructions = readPCStitchString(s);
  index = instructions.find("}}");      // end of font defs
  index += 4;                           // skip paste }} and CR LF
  index = instructions.find('}',index); // end of color table defs
  index += 3;                           // skip paste } and CR LF
  index = instructions.find(' ',index); // find first space - end of text def
  index++;                              // and skip past it
  // index should now point to the first character of the instructions
  instructions = instructions.remove(0,index);
  instructions.truncate(instructions.length()-10);
  m_propertiesDialog->setInstructions(instructions);
  m_propertiesDialog->setFabricColor(white);
  if (!m_palette->readPCStitch5Palette(s))
    return false;
  m_propertiesDialog->setScheme(m_palette->getScheme());
  m_canvas->clear();
  m_canvas->resize(width,height);
  return m_canvas->readPCStitch5Canvas(s);
}

QString KXStitchDoc::readPCStitchString(QDataStream& s)
{
  char*   buffer;
  Q_INT16 stringSize;
  s >> stringSize;
  buffer = new char[stringSize+1];
  s.readRawBytes(buffer,stringSize);
  buffer[stringSize] = '\0';
  QString string(buffer);
  delete [] buffer;
  return string;
}

bool KXStitchDoc::openDocument(const KURL& url, const char*)
{
  Q_INT8  magic;
  QString tmpfile;
  bool    validRead = false;

  deleteContents();
  if (!url.isEmpty())
  {
    KIO::NetAccess::download(url,tmpfile);
    QFile f(tmpfile);
    if (f.open(IO_ReadOnly))
    {
      QDataStream s(&f);
      // Check first byte of file, which might indicate certain formats to try
      // 'P' may be a PCStitch file version 5 or later
      // 0 may be a KXStitch file
      s >> magic;
      f.reset();
      switch (magic)
      {
        case 'P':
          validRead = readPCStitch5Document(s);
          break;
        case '\0':
          validRead = readKXStitchDocument(s);
          break;
        default:
          validRead = false;
          break;
      }
    }
    f.close();
    KIO::NetAccess::removeTempFile( tmpfile );
  }
  if (validRead)
    setURL(url);
  else
  {
    KMessageBox::sorry(0,i18n("KXStitch could not open the file specified."),i18n("File open error"));
    newDocument();
  }
  renderPixmap();
  setModified(false);
  return validRead;
}

bool KXStitchDoc::importImage()
{
  newDocument();
  KURL url;
  QString tmpfile;

  url = KFileDialog::getImageOpenURL(QString::null,(KXStitchApp *)parent(),i18n("Import Image"));
  if (!url.path().isNull())
  {
    KIO::NetAccess::download(url,tmpfile);
    Magick::Image magickImage(tmpfile);
    convertImage(magickImage);
    m_propertiesDialog->setTitle(url.fileName());
    KIO::NetAccess::removeTempFile(tmpfile);
    return true;
  }
  return false;
}

bool KXStitchDoc::scanImage()
{
  newDocument();

  if (!m_scanDialog)
  {
    m_scanDialog = KScanDialog::getScanDialog(0, "scandialog");
    if (!m_scanDialog)
    {
      // no scanning support
      KMessageBox::sorry(0, i18n("Scanning does not seem to be available.\nIs SANE configured correctly and is your scanner switched on?"),  i18n("Scanning Not Available"));
      return false;
    }
    connect(m_scanDialog, SIGNAL(finalImage(const QImage&, int)), this, SLOT(fileImageScanned(const QImage&, int)));
  }
  if (m_scanDialog->setup())
    m_scanDialog->show();
  return true;
}

void KXStitchDoc::fileImageScanned(const QImage& image, int)
{
  // write the QImage to a memory buffer in PNG format so that it can be used to initialise the ImageMagick BLOB
  QByteArray data;
  QBuffer fileBuffer(data);
  fileBuffer.open(IO_WriteOnly);
  QImageIO io(&fileBuffer,"PNG");
  io.setImage(image);
  io.write();
  fileBuffer.close();

  Magick::Blob magickBlob(data.data(),data.size());
  Magick::Image magickImage(magickBlob);
  convertImage(magickImage);
  slotResetAllViews();
}

void KXStitchDoc::convertImage(const Magick::Image& magickImage)
{
  ImportImageDlg *importDlg = new ImportImageDlg(magickImage, (KXStitchApp *)parent(), "ImportImage", true);
  if (importDlg->exec() == QDialog::Accepted)
  {
    m_propertiesDialog->reset(); // initialise properties to users defaults
    m_propertiesDialog->setScheme(importDlg->flossScheme());
    Magick::Image image = importDlg->modifiedImage();
    int w = image.columns();
    int h = image.rows();
    m_palette->setScheme(m_propertiesDialog->scheme());
    int pixelCount = w*h;
    if (importDlg->useFractionals())
    {
      int pw = w/2;
      int ph = h/2;
      m_propertiesDialog->setPatternSize(pw, ph);
      resize(pw, ph);
      int i;
      QProgressDialog progress(i18n("Converting to stitches..."), i18n("Cancel"), pixelCount, (KXStitchApp *)parent(), i18n("Progress"), true);
      progress.setMinimumDuration(0);
      Magick::Pixels cache(image);
      const Magick::PixelPacket *pixels = cache.get(0,0,w,h);
      for (int cy = 0 ; cy < h ; cy++)
      {
        progress.setProgress(cy*w);
        KApplication::kApplication()->processEvents();
        if (progress.wasCancelled()) break;
        for (int cx = 0 ; cx < w ; cx++)
        {
          Magick::PixelPacket packet = *pixels++;
          if (!packet.opacity)
          {
            QColor c(packet.red/257, packet.green/257, packet.blue/257);
            i = m_palette->addColor(c);
            if (cx % 2)
            {
              if (cy % 2)
                addStitch(cx/2, cy/2, Stitch::BRQtr, i);
              else
                addStitch(cx/2, cy/2, Stitch::TRQtr, i);
            }
            else
            {
              if (cy % 2)
                addStitch(cx/2, cy/2, Stitch::BLQtr, i);
              else
                addStitch(cx/2, cy/2, Stitch::TLQtr, i);
            }
          }
        }
      }
    }
    else
    {
      m_propertiesDialog->setPatternSize(w, h);
      resize(w, h);
      int i;
      QProgressDialog progress(i18n("Converting to stitches..."), i18n("Cancel"), pixelCount, (KXStitchApp *)parent(), i18n("Progress"), true);
      progress.setMinimumDuration(0);
      Magick::Pixels cache(image);
      const Magick::PixelPacket *pixels = cache.get(0,0,w,h);
      for (int cy = 0 ; cy < h ; cy++)
      {
        progress.setProgress(cy*w);
        KApplication::kApplication()->processEvents();
        if (progress.wasCancelled()) break;
        for (int cx = 0 ; cx < w ; cx++)
        {
          Magick::PixelPacket packet = *pixels++;
          if (!packet.opacity)
          {
            QColor c(packet.red/257, packet.green/257, packet.blue/257);
            i = m_palette->addColor(c);
            addStitch(cx,cy,Stitch::Full,i);
          }
        }
      }
    }
    renderPixmap();
  }
  delete importDlg;
}

bool KXStitchDoc::saveDocument(const KURL& url, const char*)
{
  QFile f(url.path());
  if (f.open(IO_WriteOnly))
  {
    QDataStream s(&f);
    /** write data to file */
    writeKXStitchStream(s);
    f.flush();
    f.close();
  }
  m_modified=false;
  return true;
}

void KXStitchDoc::writeKXStitchStream(QDataStream &s)
{
  // write file magic number and format version
  s << QString("KXStitch");
  s << (Q_INT16)FILE_FORMAT_VERSION;
  // write contents of m_propertiesDialog using portable types
  s << *m_propertiesDialog;
  m_palette->writePalette(s);
  m_canvas->writeCanvas(s);
}

void KXStitchDoc::createSavePoint()
{
  QByteArray a;
  QDataStream s(a, IO_WriteOnly);
  writeKXStitchStream(s);
  QByteArray *b = new QByteArray();
  *b = qCompress(a);
  m_undoStack.push(b);
}

bool KXStitchDoc::undo()
{
  createSavePoint();
  m_redoStack.push(m_undoStack.pop());
  KURL url = m_docURL;
  m_propertiesDialog->reset();
  m_palette->clear();
  m_canvas->clear();
  QByteArray *a = m_undoStack.pop();
  QDataStream s(qUncompress(*a), IO_ReadOnly);
  readKXStitchDocument(s);
  renderPixmap();
  m_docURL = url;
  slotResetAllViews();
  return !m_undoStack.isEmpty();
}

bool KXStitchDoc::redo()
{
  createSavePoint();
  KURL url = m_docURL;
  m_propertiesDialog->reset();
  m_palette->clear();
  m_canvas->clear();
  QByteArray *a = m_redoStack.pop();
  QDataStream s(qUncompress(*a), IO_ReadOnly);
  readKXStitchDocument(s);
  renderPixmap();
  m_docURL = url;
  slotResetAllViews();
  return !m_redoStack.isEmpty();
}

void KXStitchDoc::deleteContents()
{
  m_docURL.setFileName("");
  if (m_canvas)
    m_canvas->clear();
  if (m_palette)
    m_palette->clear();
  if (m_preview)
    m_preview->fill();
  setModified(false);
}

PatternCanvas *KXStitchDoc::canvas()
{
  return m_canvas;
}

void KXStitchDoc::setModified(bool _m)
{
  m_modified=_m;
}

FlossPalette *KXStitchDoc::palette()
{
  return m_palette;
}

QPixmap *KXStitchDoc::pixmap()
{
  return m_preview;
}

void KXStitchDoc::addStitches(QPointArray cells, Stitch::Type type)
{
  QPoint p;
  for (uint i = 0 ; i < cells.count() ; i++)
  {
    p = cells.point(i);
    if (type == Stitch::FRKnot)
      addFrenchKnot(p*2+QPoint(1,1));
    else
      addStitch(p, type);
  }
}

void KXStitchDoc::addFrenchKnot(QPoint s)
{
  int color = m_palette->currentColor();
  if (color != -1)
  {
    m_canvas->addFrenchKnot(s,color);
    updatePixmap(QRect((s.x()-1)/2,(s.y()-1)/2,2,2));
    setModified();
  }
}

void KXStitchDoc::deleteFrenchKnot(QPoint p, bool colorMask)
{
  m_canvas->deleteFrenchKnot(p, colorMask?m_palette->currentColor():-1);
  updatePixmap(QRect((p.x()-1)/2,(p.y()-1)/2,2,2));
}

int KXStitchDoc::findColor(QPoint cell, Stitch::Type type=Stitch::Delete)
{
  return m_canvas->findColor(cell, type);
}

/*
bool KXStitchDoc::removeColors()
{
  return m_palette->removeColors(m_canvas->usedColors());
}
*/

bool KXStitchDoc::removeUnusedColors()
{
  return m_palette->removeUnused(m_canvas->usedColors());
}

void KXStitchDoc::swapColors(int a)
{
  m_palette->swapColors(a);
  renderPixmap();
}

void KXStitchDoc::replaceColor(int a)
{
  m_canvas->replaceColor(m_palette->currentColor(),a);
  renderPixmap();
}

bool KXStitchDoc::paletteManager()
{
  bool added = false;
  PaletteManagerDialog *pDlg = new PaletteManagerDialog(0, m_palette, m_canvas->usedColors());
  if (pDlg)
  {
    if (pDlg->exec() == QDialog::Accepted)
    {
      // do stuff
      added = true;
    }
    delete pDlg;
  }
  return added;
}

void KXStitchDoc::addStitch(int x, int y, Stitch::Type type, int i)
{
  if (i != -1)
    if (m_canvas->addStitch(QPoint(x,y), type, i))
    {
      updatePixmap(QPoint(x,y));
      setModified();
    }
}

void KXStitchDoc::addStitch(QPoint cell, Stitch::Type type)
{
  int color = m_palette->currentColor();
  if (color != -1)
    if (m_canvas->addStitch(cell, type, color))
    {
      updatePixmap(cell);
      setModified();
    }
}

void KXStitchDoc::deleteStitch(QPoint cell, Stitch::Type t, bool colorMask)
{
  if (m_canvas->deleteStitch(cell, t, colorMask?m_palette->currentColor():-1))
  {
    updatePixmap(cell);
    setModified();
  }
}

void KXStitchDoc::addBackstitch(QPoint start, QPoint end)
{
  m_canvas->addBackstitch(start, end, m_palette->currentColor());
  updatePixmap();
  setModified();
}

void KXStitchDoc::deleteBackstitch(BackStitch *bs)
{
  m_canvas->deleteBackstitch(bs);
  updatePixmap();
  setModified();
}

QByteArray KXStitchDoc::copySelection(QRect area, Stitch::Type stitchMask, bool colorMask, bool excludeBackstitches, bool excludeKnots, bool cut)
{
  QByteArray a;
  QDataStream stream(a,IO_WriteOnly);
  stream << m_palette->getScheme();
  stream << (Q_INT32)area.width() << (Q_INT32)area.height();
  for (int row = area.top() ; row <= area.bottom() ; row++)
  {
    for (int col = area.left() ; col <= area.right() ; col++)
    {
      Stitch::Queue *queue = m_canvas->stitchAt(QPoint(col,row));
      Stitch::Queue adding;
      if (queue)
      {
        for (int stitches = queue->count() ; stitches ; stitches--)
        {
          bool add = true;
          Stitch* s = queue->dequeue();
          if (colorMask && s->floss != m_palette->currentColor()) add = false; //skip this one as it isn't of the specified color
          if (stitchMask && s->type != stitchMask) add = false; //skip this one as it isn't of the specified type
          if (add)
          {
            adding.enqueue(new Stitch(*s));
          }
          if (add && cut)
          {
            delete s;
          }
          else
          {
            queue->enqueue(s);
          }
        }
        if (queue->isEmpty())
        {
          deleteStitch(QPoint(col,row));
        }
      }
      stream << (Q_INT8)adding.count();
      for (int stitches = adding.count() ; stitches ; stitches--)
      {
        Stitch* s = adding.dequeue();
        Floss* f = m_palette->flossAt(s->floss);
        stream << (Q_INT8)s->type << f->color;
        delete s;
      }
    }
  }
  int xs = area.left()*2;
  int ys = area.top()*2;
  int ws = area.width()*2+2;
  int hs = area.height()*2+2;
  QPoint tl(xs,ys);
  QRect snapRect(xs,ys,ws,hs);
  if (excludeBackstitches)
    stream << (Q_INT32)0; // no backstitches to copy
  else
  {
    QPtrListIterator<BackStitch>* bs = m_canvas->backstitches();
    QPtrList<BackStitch> addBS;
    for (BackStitch* b = 0 ; (b = bs->current()) ; ++(*bs))
      if (snapRect.contains(b->start) && snapRect.contains(b->end))
        addBS.append(b);
    stream << (Q_INT32)addBS.count();
    QPtrListIterator<BackStitch> addBSit(addBS);
    for (BackStitch* b = 0 ; (b = addBSit.current()) ; ++addBSit)
    {
      stream << (b->start-tl) << (b->end-tl) << (m_palette->flossAt(b->floss))->color;
      if (cut)
        m_canvas->deleteBackstitch(b);
    }
  }
  if (excludeKnots)
    stream << (Q_INT32)0; // no knots to copy
  else
  {
    QPtrListIterator<Knot>* knots = m_canvas->knots();
    QPtrList<Knot> addKnot;
    for (Knot* k = 0 ; (k = knots->current()) ; ++(*knots))
      if (snapRect.contains(k->pos))
        addKnot.append(k);
    stream << (Q_INT32)addKnot.count();
    QPtrListIterator<Knot> addKnotit(addKnot);
    for (Knot* k = 0 ; (k = addKnotit.current()) ; ++addKnotit)
    {
      stream << (k->pos-tl) << (m_palette->flossAt(k->floss))->color;
      if (cut)
        m_canvas->deleteFrenchKnot(k->pos);
    }
  }
  renderPixmap();
  return a;
}

void KXStitchDoc::pasteSelection(QPoint point, QByteArray data, bool merge)
{
  QDataStream stream(data, IO_ReadOnly);
  QString scheme;
  stream >> scheme;
  if (scheme == m_palette->getScheme())
  {
    Q_INT32 width;
    Q_INT32 height;
    Q_INT8  count;
    Q_INT8  type;
    Q_INT32 backstitches;
    Q_INT32 knots;
    QPoint  start;
    QPoint  end;
    QColor  color;
    QPoint  tl = point*2;
    stream >> width >> height;
    for (int dy = point.y() ; height ; height--, dy++)
    {
      for (int dx = point.x(), x = width ; x ; x--, dx++)
      {
        if (!merge)
          deleteStitch(QPoint(dx,dy));
        stream >> count;
        while (count--)
        {
          stream >> type >> color;
          int index = m_palette->addColor(color);
          addStitch(dx,dy,(Stitch::Type)type,index);
        }
      }
    }
    stream >> backstitches;
    while (backstitches--)
    {
      stream >> start >> end >> color;
      m_canvas->addBackstitch(start+tl,end+tl,m_palette->addColor(color));
    }
    stream >> knots;
    while (knots--)
    {
      stream >> start >> color;
      m_canvas->addFrenchKnot(start+tl,m_palette->addColor(color));
    }
    setModified();
    slotResetAllViews();
  }
  else
  {
    KMessageBox::sorry(0,i18n("The scheme for the clipboard contents\nis not the same as the current pattern.\nThe clipboard contents can not be pasted."),i18n("Incompatible Clipboard Contents"));
  }
  renderPixmap();
}

UsageMap KXStitchDoc::flossUsage()
{
  UsageMap usageMap = m_canvas->calculateUsage();
  UsageMap::Iterator it;
  double clothCount = m_propertiesDialog->clothCount();
  QString clothCountUnits = m_propertiesDialog->clothCountUnits();
  for (it = usageMap.begin() ; it != usageMap.end() ; ++it)
  {
    double stitchLength = it.data().stitchLength/clothCount;
    double backstitchLength = it.data().backstitchLength/clothCount;
    if (clothCountUnits == "/in")
    {
      stitchLength *= 2.54;      // convert to cm
      backstitchLength *= 2.54;  // convert to cm
    }
    stitchLength /= 100.0;       // convert to m
    backstitchLength /= 100.0;   // convert to m

    usageMap[it.key()].stitchLength = stitchLength;
    usageMap[it.key()].backstitchLength = backstitchLength;
  }
  return usageMap;
}
