// LabPlot : Plot2D.cc

#include <qsimplerichtext.h>
#include <klocale.h>
#include <kdebug.h>
#include <iostream>
#include "Plot2D.h"
#include "Plot2DSurface.h"
#include "defs.h"

using namespace std;

inline double MAX(double x,double y) {if (x>y) return x; else return y;}

// general 2D Plot class
Plot2D::Plot2D(Worksheet *p)
	: Plot(p) {
	if(p==0) kdDebug()<<"no Worksheet defined!"<<endl;
	QFont font = p->getMainWin()->defaultFont();
	font.setPointSize((int)(0.7*font.pointSize()));	// for axes label
	kdDebug()<<"Plot2D()"<<endl;
	axis[0].setLabel(new Label(i18n("x-Axis"),font));
	axis[1].setLabel(new Label(i18n("y-Axis"),font));
	axis[2].setLabel(new Label(i18n("y2-Axis"),font));
	axis[3].setLabel(new Label(i18n("x2-Axis"),font));
	
	// set default font
	font.setPointSize((int)(0.7*font.pointSize()));	// for tic label
	axis[0].setTicsFont(font);
	axis[1].setTicsFont(font);
	axis[2].setTicsFont(font);
	axis[3].setTicsFont(font);

	// grid & border
	for (int i=0;i<4;i++)
		borderenabled[i] = true;
	for (int i=0;i<8;i++)
		gridenabled[i] = false;
	for (int i=2;i<4;i++) {
		axis[i].enable(0);
		borderenabled[i] = false;
	}
	kdDebug()<<"Plot2D OK"<<endl;
}

void Plot2D::setActRange(LRange* r,int i) {
	kdDebug()<<"Plot2D::setActRange("<<i<<")"<<endl;
	LRange tmp;

	double offset=0;//(r->rMax()-r->rMin())/10;

	tmp = LRange(r->rMin()-offset,r->rMax()+offset);
	actrange[i]=tmp;
}

void Plot2D::setActRanges(LRange* r) {
	setActRange(&r[0],0);
	setActRange(&r[1],1);
}

void Plot2D::setBorder(int item, bool on) {
	kdDebug()<<"Plot2D::setBorder()"<<endl;
	const int unit = 5, numunit = 40, numunit2 = 20;
	int w = worksheet->width(), h = worksheet->height();

	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));

	if(item == 0) {
		if (on) {
			if (axis[0].label()->Title().length() > 0) ymax -= axis[0].label()->Font().pointSize();
			if (axis[0].MajorTicsEnabled())   ymax -= unit+numunit2;
			if (axis[0].MinorTicsEnabled())   ymax -= unit;
		}
		else  {
			if (axis[0].label()->Title().length() > 0) ymax += axis[0].label()->Font().pointSize();
			if (axis[0].MajorTicsEnabled())   ymax += unit+numunit2;
			if (axis[0].MinorTicsEnabled())   ymax += unit;
		}
	}
	if(item == 1) {
		if (on) {
			if (axis[1].label()->Title().length() > 0) xmin += axis[1].label()->Font().pointSize();
			if (axis[1].MajorTicsEnabled())   xmin += unit+numunit;
			if (axis[1].MinorTicsEnabled())   xmin += unit;
		}
		else {
			if (axis[1].label()->Title().length() > 0) xmin -= axis[1].label()->Font().pointSize();
			if (axis[1].MajorTicsEnabled())   xmin -= unit+numunit;
			if (axis[1].MinorTicsEnabled())   xmin -= unit;
		}
	}
	if(item == 2) {
		if (on) {
			if (axis[2].label()->Title().length() > 0) xmax -= axis[2].label()->Font().pointSize();
			if (axis[2].MajorTicsEnabled())   xmax -= unit+numunit;
			if (axis[2].MinorTicsEnabled())   xmax -= unit;
		}
		else {
			if (axis[2].label()->Title().length() > 0) xmax += axis[2].label()->Font().pointSize();
			if (axis[2].MajorTicsEnabled())   xmax += unit+numunit;
			if (axis[2].MinorTicsEnabled())   xmax += unit;

		}
	}
	if(item == 3) {
		if (on) {
			if (axis[3].label()->Title().length() > 0) ymin += axis[3].label()->Font().pointSize();
			if (axis[3].MajorTicsEnabled())   ymin += unit+numunit2;
			if (axis[3].MinorTicsEnabled())   ymin += unit;
		}
		else {
			if (axis[3].label()->Title().length() > 0) ymin -= axis[3].label()->Font().pointSize();
			if (axis[3].MajorTicsEnabled())   ymin -= unit+numunit2;
			if (axis[3].MinorTicsEnabled())   ymin -= unit;
		}
	}
	setXMin(xmin,w);
	setXMax(xmax,w);
	setYMin(ymin,h);
	setYMax(ymax,h);
}

void Plot2D::draw(QPainter *p,int w,int h) {
	kdDebug()<<"Plot2D::draw() w/h : "<<w<<' '<<h<<endl;
	kdDebug()<<"	TYPE = "<<type<<endl;
	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));

	kdDebug()<<"	XMIN/MXAX/YMIN/YMAX = "<<xmin<<','<<xmax<<','<<ymin<<','<<ymax<<endl;
	kdDebug()<<"	p1 = "<<p1.X()<<'/'<<p1.Y()<<" p2 = "<<p2.X()<<'/'<<p2.Y()<<endl;

	if (!transparent) {
		// background color
		p->setBrush(bgcolor);
		p->setPen(Qt::NoPen);
		p->drawRect((int)(w*position.X()),(int)(h*position.Y()),(int)(w*size.X()),(int)(h*size.Y()));

		// graph background color
		p->setBrush(gbgcolor);
		p->setPen(Qt::NoPen);
		p->drawRect(xmin,ymin,xmax-xmin,ymax-ymin);
	}

	kdDebug()<<"	title->draw() pos:"<<position.X()<<' '<<position.Y()<<endl;
	kdDebug()<<" 	size:"<<size.X()<<' '<<size.Y()<<endl;
	title->draw(worksheet,p,position,size,w,h,0);

	drawAxes(p,w,h);
	drawBorder(p,w,h);

	drawCurves(p,w,h);

	if (baseline_enabled) {
		double min = actrange[1].rMin();
		double max = actrange[1].rMax();
		int y = ymax - (int) ((baseline-min)/(max-min)*(double)(ymax-ymin));
		//kdDebug()<< "Y BASLINE @ "<<y<<endl;
		p->drawLine(xmin,y,xmax,y);
	}
	if (xbaseline_enabled) {
		double min = actrange[0].rMin();
		double max = actrange[0].rMax();
		int x = xmin + (int) ((xbaseline-min)/(max-min)*(double)(xmax-xmin));
		//kdDebug()<< "X BASLINE @ "<<y<<endl;
		p->drawLine(x,ymin,x,ymax);
	}

	if (region_enabled) {
		double min = actrange[0].rMin();
		double max = actrange[0].rMax();
		int minx = xmin + (int) ((region->rMin()-min)/(max-min)*(double)(xmax-xmin));
		int maxx = xmin + (int) ((region->rMax()-min)/(max-min)*(double)(xmax-xmin));

		//kdDebug()<<"REGION : "<<minx<<" "<<maxx<<endl;

		if(minx != maxx) {
			p->drawLine(minx,ymin,minx,ymax);
			p->drawLine(maxx,ymin,maxx,ymax);
		}

		if (maxx-minx > 20) {
			int y = (ymax+ymin)/2;
			p->drawLine(minx+5,y,maxx-5,y);
			p->drawLine(minx+5,y,minx+10,y+5);
			p->drawLine(minx+5,y,minx+10,y-5);
			p->drawLine(maxx-5,y,maxx-10,y+5);
			p->drawLine(maxx-5,y,maxx-10,y-5);
		}
	}

	if(marks_enabled) {
		double min = actrange[0].rMin();
		double max = actrange[0].rMax();
		int minx = xmin + (int) ((markx->rMin()-min)/(max-min)*(double)(xmax-xmin));
		int maxx = xmin + (int) ((markx->rMax()-min)/(max-min)*(double)(xmax-xmin));
		min = actrange[1].rMin();
		max = actrange[1].rMax();
		int miny = ymax - (int) ((marky->rMin()-min)/(max-min)*(double)(ymax-ymin));
		int maxy = ymax - (int) ((marky->rMax()-min)/(max-min)*(double)(ymax-ymin));
		
		p->setPen(QPen(Qt::gray,1,Qt::DashLine));
		p->setFont(QFont("Adobe Times",8));
		QPointArray a;
		if(minx>xmin-clipoffset&&minx<xmax+clipoffset) { 
			p->drawLine(minx,ymin,minx,ymax);	// mark x1
			a.setPoints(3,minx,ymax,minx+2,ymax+4,minx-2,ymax+4);
			p->drawPolygon(a);
			a.setPoints(3,minx,ymin,minx+2,ymin-4,minx-2,ymin-4);
			p->drawPolygon(a);
			p->drawText(minx+5,ymax+10,QString::number(markx->rMin()));
		}
		if(maxx>xmin-clipoffset&&maxx<xmax+clipoffset) { 
			p->drawLine(maxx,ymin,maxx,ymax);	// mark x2
			a.setPoints(3,maxx,ymax,maxx+2,ymax+4,maxx-2,ymax+4);
			p->drawPolygon(a);
			a.setPoints(3,maxx,ymin,maxx+2,ymin-4,maxx-2,ymin-4);
			p->drawPolygon(a);
			p->drawText(maxx+5,ymax+10,QString::number(markx->rMax()));
		}
		if(miny>ymin-clipoffset&&miny<ymax+clipoffset) {
			p->drawLine(xmin,miny,xmax,miny);	// mark y1
			a.setPoints(3,xmin,miny,xmin-4,miny+2,xmin-4,miny-2);
			p->drawPolygon(a);
			a.setPoints(3,xmax,miny,xmax+4,miny+2,xmax+4,miny-2);
			p->drawPolygon(a);
			
			p->save();
			p->translate(xmin-5,miny-5);
			p->rotate(-90);
			p->drawText(0,0,QString::number(marky->rMin()));
			p->restore();
		}
		if(maxy>ymin-clipoffset&&maxy<ymax+clipoffset) {
			p->drawLine(xmin,maxy,xmax,maxy);	// mark y2
			a.setPoints(3,xmin,maxy,xmin-4,maxy+2,xmin-4,maxy-2);
			p->drawPolygon(a);
			a.setPoints(3,xmax,maxy,xmax+4,maxy+2,xmax+4,maxy-2);
			p->drawPolygon(a);
			p->save();
			p->translate(xmin-5,maxy-5);
			p->rotate(-90);
			p->drawText(0,0,QString::number(marky->rMax()));
			p->restore();
		}

		// value + diff
		p->drawText((int)((minx+maxx)/2.0),ymax+10,i18n("dx=")+QString::number(markx->rMax()-markx->rMin()));

		p->save();
		p->translate(xmin-5,(int)((miny+maxy)/2.0));
		p->rotate(-90);
		p->drawText(0,0,i18n("dy=")+QString::number(marky->rMax()-marky->rMin()));
		p->restore();

		p->setPen(Qt::black);
	}

	if(legend.enabled()) {
		kdDebug()<<"	Legend enabled"<<endl;
		
		if (type == PSURFACE) {		// legend can't do this :-(
			if (legend.X() == 0.7 && legend.Y() == 0.05 ) // replace the legend for surface plots first time
				legend.setPosition(0.83,0.05);
			
			int x = (int) (w*(size.X()*legend.X()+position.X()));
			int y = (int) (h*(size.Y()*legend.Y()+position.Y()));

			Plot2DSurface *plot = (Plot2DSurface *)this;
			plot->drawLegend(p,x,y);
		}

		legend.draw(p,type,graphlist,position,size,w,h);
		// kdDebug()<<" drawing legend with pos = "<<position.X()<<' '<<position.Y()<<endl;
		// kdDebug()<<" 	size.X()*w/size.Y()*h = "<<size.X()*w<<' '<<size.Y()*h<<endl;
		
	}
	p->setPen(Qt::NoPen);
}

void Plot2D::drawBorder(QPainter *p,int w,int h) {
	kdDebug()<<"Plot2D::drawBorder()"<<endl;
	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));
	kdDebug()<<"	xmin/xmax ymin/ymax "<<xmin<<'/'<<xmax<<' '<<ymin<<'/'<<ymax<<endl;

	if (borderenabled[1]) {
		p->setPen(QPen(axis[1].BorderColor(),axis[1].borderWidth()));
		p->drawLine(xmin,ymin,xmin,ymax);
	}
	if (borderenabled[3]) {
		p->setPen(QPen(axis[3].BorderColor(),axis[3].borderWidth()));
		p->drawLine(xmin,ymin,xmax,ymin);
	}
	if (borderenabled[2]) {
		p->setPen(QPen(axis[2].BorderColor(),axis[2].borderWidth()));
		p->drawLine(xmax,ymin,xmax,ymax);
	}
	if (borderenabled[0]) {
		p->setPen(QPen(axis[0].BorderColor(),axis[0].borderWidth()));
		p->drawLine(xmin,ymax,xmax,ymax);
	}
}

void Plot2D::drawAxes(QPainter *p,int w, int h) {
	kdDebug()<<"Plot2D::drawAxes()"<<endl;
	const int unit = (int)(5*size.X());
	const int numunit = (int)(40*size.X()), numunit2 = (int)(20*size.X());

	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));
	kdDebug()<<"	xmin/xmax ymin/ymax : "<<xmin<<'/'<<xmax<<' '<<ymin<<'/'<<ymax<<endl;
	kdDebug()<<"	width/height : "<<w<<'/'<<h<<endl;

	//AxesLabel
	kdDebug()<<"	drawing axis label"<<endl;
	Label *label = axis[3].label();	// x2
	if (label->X()==0 && label->Y()==0)	// default
		label->setPosition((xmin+(xmax-xmin)/2)/(double)w,
			(ymin-(unit+3*numunit2)*axis[3].MajorTicsEnabled()-2*unit)/(double)h);
	if (axis[3].enabled())
		label->draw(worksheet,p,position,size,w,h,0);
	
	label = axis[0].label();		// x
	if (label->X()==0 && label->Y()==0)	// default
		label->setPosition((xmin+(xmax-xmin)/2)/(double)w,
			(ymax+(unit+numunit2)*axis[0].MajorTicsEnabled())/(double)h);
	if (axis[0].enabled())
		label->draw(worksheet,p,position,size,w,h,0);
	
	label = axis[1].label();		// y
	if (label->X()==0 && label->Y()==0)	// default
		label->setPosition(0.01, (ymin+(ymax-ymin)/2)/(double)h);
	if (axis[1].enabled()) {
		p->save();
		label->draw(worksheet,p,position,size,w,h,270);
		p->restore();
	}
	
	label = axis[2].label();		// y2
	if (label->X()==0 && label->Y()==0)	// default
		label->setPosition((xmax+(2*unit+numunit)*axis[2].MajorTicsEnabled())/(double)w,
			(ymin+(ymax-ymin)/2)/(double)h);
	if (axis[2].enabled()) {
		p->save();
		label->draw(worksheet,p,position,size,w,h,270);
		p->restore();
	}

	// axes tics and grid
	kdDebug()<<"	drawing axis tics and grid"<<endl;
	for (int k=0;k<4;k++) {
		if(k==0 || k==3) {	// x,x2
			kdDebug()<<"	X axis tics and grid"<<endl;
			int tmpi=0;
			(k==0)?tmpi=0:tmpi=4;

			if (axis[k].MajorTicsEnabled() && axis[k].enabled()) {
				int t=-1;		// number of major tics
				TScale scale = axis[k].Scale();
				double min = actrange[0].rMin();
				double max = actrange[0].rMax();
				switch (scale) {
				case LINEAR: case SQRT : case SX2: 
					t = axis[k].MajorTics(); 
					if(t==-1)
						t=autoTics(min,max);
					break;
				case LOG10: t = (int) log10(max/min)+2; break;
				case LOG2: t = (int) log2(max/min)+2; break;
				case LN: t = (int) log(max/min)+2; break;
				}
				if(t==0) t=-1;
//				kdDebug()<<"	T="<<t<<endl;
				
				for (int i=0;i <= t;i++) {
					int x1=0, x2=0;
					switch(scale) {
					case LINEAR: {
						x1 = xmin+i*(xmax-xmin)/t;
						x2 = xmin+(i+1)*(xmax-xmin)/t;
						} break;
					case LOG10: {
						double gap = 1.0-log10(pow(10,ceil(log10(min)))/min);	// fragment of decade to shift left
						double decade = (xmax-xmin)/(log10(max/min));		// width of decade
						x1 = xmin+(int)((i-gap)*decade);
						x2 = (int) (x1+decade+ceil(fabs(log10(max))));
						} break;
					case LOG2: {
						double gap = 1.0-log2(pow(2,ceil(log2(min)))/min);		// fragment of decade to shift left
						double decade = (xmax-xmin)/(log2(max/min));		// width of decade
						x1 = xmin+(int)((i-gap)*decade);
						x2 = (int) (x1+decade+ceil(fabs(log2(max))));
						} break;
					case LN: {
						double gap = 1.0-log(pow(M_E,ceil(log(min)))/min);	// fragment of decade to shift left
						double decade = (xmax-xmin)/(log(max/min));		// width of decade
						x1 = xmin+(int)((i-gap)*decade);
						x2 = (int) (x1+decade+ceil(fabs(log(max))));
						} break;
					case SQRT: {
						x1 = xmin+(int)(sqrt((double)i)*(xmax-xmin)/sqrt((double)t));
						x2 = xmin+(int)(sqrt((double)(i+1))*(xmax-xmin)/sqrt((double)t));
						} break;
					case SX2: {
						x1 = xmin+(i*i)*(xmax-xmin)/(t*t);
						x2 = xmin+(i+1)*(i+1)*(xmax-xmin)/(t*t);
						} break;
					}

					if(x1<=xmax+1 && x1>=xmin-1) { // major tics
						p->setPen(QPen(axis[k].TicsColor(),axis[k].majorTicsWidth()));
						if(k==0) {	// x
							switch(axis[k].TicsPos()) {
							case 0: p->drawLine(x1,ymax,x1,ymax+10); break;
							case 1: p->drawLine(x1,ymax-10,x1,ymax); break;
							case 2:p->drawLine(x1,ymax-10,x1,ymax+10); break;
							case 3: break;
							}
						}
						else {		// x2
							switch(axis[k].TicsPos()) {
							case 0: p->drawLine(x1,ymin-10,x1,ymin); break;
							case 1: p->drawLine(x1,ymin,x1,ymin+10); break;
							case 2: p->drawLine(x1,ymin-10,x1,ymin+10); break;
							case 3: break;
							}
						}

						if (gridenabled[tmpi]) {
							p->setPen(QPen(axis[k].majorGridColor(),axis[k].majorGridWidth(),axis[k].MajorGridType()));
							p->drawLine(x1,ymin,x1,ymax);
							p->setPen(Qt::SolidLine);
						}
					}
					if (graphlist->Number() > 0) {		// Numbers
						QColor c = axis[k].TicsLabelColor();
						QFont f = axis[k].TicsFont();
						double dx = max-min, value=0;
						
						switch(scale) {
						case LINEAR: case SQRT: case SX2: value = min + i*dx/t; break;
						case LOG10: value = pow(10,ceil(log10(min)))*pow(10.0,i-1); break;
						case LOG2: value = pow(2,ceil(log2(min)))*pow(2.0,i-1); break;
						case LN: value = pow(M_E,ceil(log(min)))*pow(M_E,i-1); break;
						}

						// apply scale and shift value
						value = value*axis[k].Scaling()+axis[k].Shift();

						TFormat atlf = axis[k].TicsLabelFormat();

						QString label = TicLabel(atlf,axis[k].TicsLabelPrecision(), axis[k].DateTimeFormat(),value);

						// apply prefix & suffix
						label.prepend(axis[k].TicsLabelPrefix());
						label.append(axis[k].TicsLabelSuffix());

						// draw tic label
						QFontMetrics fm(f);
						int y1;
						int gap = (int)(size.Y()*axis[k].TicsLabelGap());
						
						if(k==0)
							y1=ymax+gap+(int)(size.Y()*fm.ascent()/2);
						else
							y1=ymin-gap-(int)(size.Y()*fm.ascent()/2);
						
						p->save();
						p->translate(x1,y1);
						p->rotate(axis[k].TicsLabelRotation());
						f.setPointSize((int)(f.pointSize()*size.X()));	// resize tic label
						if (atlf == AUTO || atlf == NORMAL || atlf == SCIENTIFIC) {
							p->setPen(c);
							p->setFont(f);

							if(x1<=xmax+1 && x1>=xmin-1)
								p->drawText(-fm.width(label)/2,fm.ascent()/2,label);
						}
						else {	// rich text label
							QSimpleRichText *richtext = new QSimpleRichText(label,f);
							if(x1<=xmax+1 && x1>=xmin-1) {
								QColorGroup cg;
								cg.setColor(QColorGroup::Text,c);
								richtext->draw(p,-richtext->width()/4, -fm.ascent()/2,QRect(),cg);
							}
						}
						p->restore();
					}
					if (axis[k].MinorTicsEnabled() && i != t )
						for (int j=1;j <= axis[k].MinorTics()+1;j++) {
							int x=0;
							if(scale == LINEAR)
								x = x1+j*(x2-x1)/(axis[0].MinorTics()+1);
							else if (scale == LOG10)
								x=(int)(x1+(x2-x1)*log10((double)(j)));
							// TODO : SQRT minor tics ?
							// other scales have no minor tics

							if(x<=xmax+1 && x>=xmin-1) { // minor tics
								p->setPen(QPen(axis[k].TicsColor(),axis[k].minorTicsWidth()));
								if(k==0) {	// x
									switch(axis[k].TicsPos()) {
									case 0: p->drawLine(x,ymax,x,ymax+5); break;
									case 1: p->drawLine(x,ymax-5,x,ymax); break;
									case 2: p->drawLine(x,ymax-5,x,ymax+5); break;
									case 3: break;
									}
								}
								else {		// x2
									switch(axis[k].TicsPos()) {
									case 0: p->drawLine(x,ymin-5,x,ymin); break;
									case 1: p->drawLine(x,ymin,x,ymin+5); break;
									case 2: p->drawLine(x,ymin-5,x,ymin+5); break;
									case 3: break;
									}
								}
								if (gridenabled[tmpi+1] && j != axis[k].MinorTics()+1) {
									p->setPen(QPen(axis[k].minorGridColor(),
										axis[k].minorGridWidth(),axis[k].MinorGridType()));
									p->drawLine(x,ymin,x,ymax);
									p->setPen(Qt::SolidLine);
								}
							}
						}
				}
			}
		}
		else if(k==1 || k==2) {	// y,y2
			kdDebug()<<"	Y axis tics and grid"<<endl;
			int tmpi=0;
			(k==1)?tmpi=2:tmpi=6;

			if (axis[k].MajorTicsEnabled() && axis[k].enabled()) {
				int t=-1;		// number of major tics
				int scale = axis[k].Scale();
				double min = actrange[1].rMin();
				double max = actrange[1].rMax();
				switch(scale) {
				case LINEAR: case SQRT: case SX2: 
					t = axis[k].MajorTics(); 
					if(t==-1)
						t=autoTics(min,max);
					break;
				case LOG10: t = (int) log10(max/min)+2; break;
				case LOG2: t = (int) log2(max/min)+2; break;
				case LN: t = (int) log(max/min)+2; break;
				}
				if(t==0) t=-1;

				for (int i=0;i <= t;i++) {
					int y1=0,y2=0;

					switch(scale) {
					case LINEAR:
						y1 = ymax-i*(ymax-ymin)/t;
						y2 = ymax-(i+1)*(ymax-ymin)/t;
						break;
					case LOG10: {
						double gap = 1.0-log10(pow(10,ceil(log10(min)))/min);	// fragment of decade to shift left
						double decade = (ymax-ymin)/(log10(max/min));		// width of decade
						y1 = ymax-(int)((i-gap)*decade);
						y2 = (int) (y1-decade-ceil(fabs(log10(max))));
						} break;
					case LOG2: {
						double gap = 1.0-log2(pow(2,ceil(log2(min)))/min);	// fragment of decade to shift left
						double decade = (ymax-ymin)/(log2(max/min));		// width of decade
						y1 = ymax-(int)((i-gap)*decade);
						y2 = (int) (y1-decade-ceil(fabs(log2(max))));
						} break;
					case LN:{
						double gap = 1.0-log(pow(M_E,ceil(log(min)))/min);	// fragment of decade to shift left
						double decade = (ymax-ymin)/(log(max/min));		// width of decade
						y1 = ymax-(int)((i-gap)*decade);
						y2 = (int) (y1-decade-ceil(fabs(log(max))));
						} break;
					case SQRT:
						y1 = ymax-(int)(sqrt((double)i)*(ymax-ymin)/sqrt((double)t));
						y2 = ymax-(int)(sqrt((double)(i+1))*(ymax-ymin)/sqrt((double)t));
						break;
					case SX2:
						y1 = ymax-(i*i)*(ymax-ymin)/(t*t);
						y2 = ymax-(i+1)*(i+1)*(ymax-ymin)/(t*t);
						break;
					}

					if(y1<=ymax+1 && y1>=ymin-1) { // major tics
						p->setPen(QPen(axis[k].TicsColor(),axis[k].majorTicsWidth()));
						if(k==1) {	// y
							switch(axis[k].TicsPos()) {
							case 0: p->drawLine(xmin-10,y1,xmin,y1); break;
							case 1: p->drawLine(xmin,y1,xmin+10,y1); break;
							case 2: p->drawLine(xmin-10,y1,xmin+10,y1); break;
							case 3: break;
							}
						}
						else {		// y2
							switch(axis[k].TicsPos()) {
							case 0: p->drawLine(xmax,y1,xmax+10,y1); break;
							case 1: p->drawLine(xmax-10,y1,xmax,y1); break;
							case 2: p->drawLine(xmax-10,y1,xmax+10,y1); break;
							case 3: break;
							}
						}
						if (gridenabled[tmpi]) {
							p->setPen(QPen(axis[k].majorGridColor(),axis[k].majorGridWidth(),axis[k].MajorGridType()));
							p->drawLine(xmin,y1,xmax,y1);
							p->setPen(Qt::SolidLine);
						}
					}
					if (graphlist->Number() > 0) {
						QColor c = axis[k].TicsLabelColor();
						QFont f = axis[k].TicsFont();
						double dy = max-min, value=0;

						switch(scale) {
						case LINEAR: case SQRT: case SX2: value = min + i*dy/t; break;
						case LOG10: value = pow(10,ceil(log10(min)))*pow(10.0,i-1); break;
						case LOG2: value = pow(2,ceil(log2(min)))*pow(2.0,i-1); break;
						case LN: value = pow(M_E,ceil(log(min)))*pow(M_E,i-1); break;
						}

						// scale and shift value
						value = value*axis[k].Scaling()+axis[k].Shift();

						int atlf = axis[k].TicsLabelFormat();

						QString label = TicLabel(atlf,axis[k].TicsLabelPrecision(),
							axis[k].DateTimeFormat(),value);

						// apply prefix & suffix
						label.prepend(axis[k].TicsLabelPrefix());
						label.append(axis[k].TicsLabelSuffix());

						// draw tic label
						QFontMetrics fm(f);
						int x1;
						int gap = (int)(size.X()*axis[k].TicsLabelGap());
						if(k==1)
							x1=xmin-gap-(int)(size.X()*fm.ascent());
						else
							x1=xmax+gap+(int)(size.X()*fm.ascent());
							
						p->save();
						p->translate(x1,y1);
						p->rotate(axis[k].TicsLabelRotation());
						f.setPointSize((int)(f.pointSize()*size.X()));	// resize tic label
						if (atlf == AUTO || atlf == NORMAL || atlf == SCIENTIFIC) {
							p->setPen(c);
							p->setFont(f);

							if(y1<=ymax+1 && y1>=ymin-1)
								p->drawText(-fm.width(label)/2,fm.ascent()/2-1,label);
						}
						else {		// rich text label
							QSimpleRichText *richtext = new QSimpleRichText(label,f);
							if(y1<=ymax+1 && y1>=ymin-1) {
								QColorGroup cg;
								cg.setColor(QColorGroup::Text,c);
								richtext->draw(p,-richtext->width(),(int)(-richtext->height()/2.0-1), QRect(),cg);
							}
						}
						p->restore();
					}
					if (axis[k].MinorTicsEnabled() && i != t )
						for (int j=1;j <= axis[k].MinorTics()+1;j++) {
							int y=0;
							if(scale == LINEAR)
								y = y1+j*(y2-y1)/(axis[k].MinorTics()+1);
							else if (scale == LOG10)
								y=(int)(y1+(y2-y1)*log10((double)(j)));
							// all other scales have minor tics = 0
							
							if(y<=ymax+1 && y>=ymin-1) { // minor tics
								p->setPen(QPen(axis[k].TicsColor(),axis[k].minorTicsWidth()));
								if(k==1) {	// y
									switch(axis[k].TicsPos()) {
									case 0: p->drawLine(xmin-5,y,xmin,y); break;
									case 1: p->drawLine(xmin,y,xmin+5,y); break;
									case 2: p->drawLine(xmin-5,y,xmin+5,y); break;
									case 3: break;
									}
								}
								else {		// y2
									switch(axis[k].TicsPos()) {
									case 0: p->drawLine(xmax,y,xmax+5,y); break;
									case 1: p->drawLine(xmax-5,y,xmax,y); break;
									case 2: p->drawLine(xmax-5,y,xmax+5,y); break;
									case 3: break;
									}
								}
								if (gridenabled[tmpi+1] && j != axis[k].MinorTics()+1) {
									p->setPen(QPen(axis[k].minorGridColor(),
										axis[k].minorGridWidth(),axis[k].MinorGridType()));
									p->drawLine(xmin,y,xmax,y);
									p->setPen(Qt::SolidLine);
								}
							}
						}
					}
				}
			}
	}
}

void Plot2D::saveAxes(QTextStream *t) {
	for (int i = 0; i < 4; i++)
		saveAxis(t,&axis[i],gridenabled[2*i],borderenabled[i],gridenabled[2*i+1]);
}

void Plot2D::openAxes(QTextStream *t, int version) {
	for(int i = 0;i<4;i++)
		openAxis(t,version, &axis[i],&gridenabled[2*i],&borderenabled[i],&gridenabled[2*i+1]);
}

