//LabPlot : GraphM.cc

#include <stdlib.h>
#include <iostream>
#include <qimage.h>
#include <kdebug.h>
#include "GraphM.h"

using namespace std;

GraphM::GraphM(QString n, QString l, LRange r[3], LSource src, PType t, Style *st,
		Symbol *sy, double *a,int dimx, int dimy, bool s)
	: Graph(n,l,src,t,st,sy,dimx*dimy,s)
{
	if(r) {
		range[0].setMin(r[0].rMin());
		range[0].setMax(r[0].rMax());
		range[1].setMin(r[1].rMin());
		range[1].setMax(r[1].rMax());
		range[2].setMin(r[2].rMin());
		range[2].setMax(r[2].rMax());
	}

	nx = dimx;	// xrange
	ny = dimy;	// yrange

	array = new double[nx*ny];
	for (int i=0;i<nx;i++)
		for (int j=0;j<ny;j++)
			array[j+i*ny] = a[j+i*ny];

	/*for (int i=0;i<nx;i++) {
		for (int j=0;j<ny;j++) {
			kdDebug()<<' '<<array[j+ny*i]<<endl;
		}
		kdDebug()<<endl;
	}*/
}

GraphM *GraphM::Clone() {
	GraphM *newg = new GraphM(*this);
	Label *l = new Label();
	*l = *(label);
	newg->setLabel(l);	// set label
	double *data = new double[number];
	for(int i=0;i<number;i++)
		data[i] = array[i];
	newg->setData(data);	// set data

	LRange nrange[3];
	nrange[0] = range[0];
	nrange[1] = range[1];
	nrange[2] = range[2];
	newg->setRange(nrange);

	return newg;
}

void GraphM::saveXML(QDomDocument doc, QDomElement graphtag) {
	kdDebug()<<"GraphM::saveXML()"<<endl;
	QDomElement tag = doc.createElement( "Range" );
	tag.setAttribute("xmin",QString::number(range[0].rMin()));
	tag.setAttribute("xmax",QString::number(range[0].rMax()));
	tag.setAttribute("ymin",QString::number(range[1].rMin()));
	tag.setAttribute("ymax",QString::number(range[1].rMax()));
	tag.setAttribute("zmin",QString::number(range[2].rMin()));
	tag.setAttribute("zmax",QString::number(range[2].rMax()));
    	graphtag.appendChild( tag );

	tag = doc.createElement( "Dimension" );
	tag.setAttribute("x",QString::number(nx));
	tag.setAttribute("y",QString::number(ny));
    	graphtag.appendChild( tag );
	
	kdDebug()<<"	saving data nx/ny = "<<nx<<' '<<ny<<endl;
/*	old style
	for (int i=0;i< nx;i++) {
		kdDebug()<<"	x = "<<i<<endl;
		for (int j=0;j< ny;j++) {
			QDomElement dtag = doc.createElement( "Data" );
			dtag.setAttribute("x",QString::number(i));
			dtag.setAttribute("y",QString::number(j));
			dtag.setAttribute("v",QString::number(array[j+i*ny]));
    			graphtag.appendChild( dtag );
		}
	}
*/	
	for (int i=0;i< nx;i++) {
		QDomElement dtag = doc.createElement( "Data" );
		for (int j=0;j< ny;j++)
			dtag.setAttribute("v"+QString::number(j),QString::number(array[j+i*ny]));
    		graphtag.appendChild( dtag );
	}
	kdDebug()<<"	... done"<<endl;
}

void GraphM::save(QTextStream *t, QProgressDialog *progress) {
	saveGraph(t);
	*t<<nx<<' '<<ny<<endl;
       	*t<<range[0].rMin()<<' '<<range[0].rMax()<<endl;
       	*t<<range[1].rMin()<<' '<<range[1].rMax()<<endl;
       	*t<<range[2].rMin()<<' '<<range[2].rMax()<<endl;
	*t<<type<<' ';
	style->save(t);
	symbol->save(t);

	//dump data
	progress->setTotalSteps(nx);
	for (int i=0;i< nx;i++) {
		if(i%100 == 0) progress->setProgress(i);
		for(int j=0;j<ny;j++) {
			*t<<array[j+i*ny]<<' ';
		}
		*t<<endl;
	}
	progress->cancel();
}

void GraphM::open(QTextStream *t, int version, QProgressDialog *progress) {
	openGraph(t,version);
	
	*t>>nx>>ny;

	double zmin, zmax;
	*t>>zmin>>zmax;
	range[0].setMin(zmin); range[0].setMax(zmax);
	*t>>zmin>>zmax;
	range[1].setMin(zmin); range[1].setMax(zmax);
	*t>>zmin>>zmax;
	range[2].setMin(zmin); range[2].setMax(zmax);

	type = (PType) style->open(t,version);
	symbol->open(t,version);

	// read data
	array = new double[nx*ny];
	t->readLine();
	
	progress->setTotalSteps(number);
	for (int i=0;i<nx;i++) {
		if(i%100 == 0) progress->setProgress(i);
		QString line = t->readLine();

		//kdDebug()<<"line "<<i<<" : "<<line<<endl;

                line = line.simplifyWhiteSpace();
        	QStringList oneline = QStringList::split(QChar(' '), line );

		int j=0;
		for ( QStringList::Iterator it = oneline.begin(); it != oneline.end(); ++it ) {
			double z = (*it).toDouble();
			if (j==0 && i==0) {
				zmin=zmax=z;
			}
			else {
				z<zmin?zmin=z:0;
				z>zmax?zmax=z:0;
			}
			array[j+ny*i]=z;
			j++;
			//kdDebug()<<"array["<<j+ny*i<<"] = "<<array[j+ny*i]<<endl;
		}
	}
	progress->cancel();
}

void GraphM::openXML(QDomNode node) {
	int i=0;
	while(!node.isNull()) {
		QDomElement e = node.toElement();
//		kdDebug()<<"GRAPH TAG = "<<e.tagName()<<endl;
//		kdDebug()<<"GRAPH TEXT = "<<e.text()<<endl;

		openGraphXML(e);

		if(e.tagName() == "Dimension") {
			nx = e.attribute("x").toInt();
			ny = e.attribute("y").toInt();
			array = new double[nx*ny];
			kdDebug()<<"NX/NY "<<nx<<' '<<ny<<endl;
		}
		else if(e.tagName() == "Range") {
			range[0].setRange(e.attribute("xmin").toDouble(),e.attribute("xmax").toDouble());
			range[1].setRange(e.attribute("ymin").toDouble(),e.attribute("ymax").toDouble());
			range[2].setRange(e.attribute("zmin").toDouble(),e.attribute("zmax").toDouble());
		}
		else if(e.tagName() == "Data") {
			/* old style
			int i = e.attribute("x").toInt(), j = e.attribute("y").toInt();
			kdDebug()<<"i/j "<<i<<' '<<j<<endl;
			array[j+i*ny] = e.attribute("v").toDouble();
			*/
			for(int j=0;j<ny;j++)
				array[j+i*ny] = e.attribute("v"+QString::number(j)).toDouble();
			i++;
		}

		node = node.nextSibling();
	}
}

QStringList GraphM::Info(){
	kdDebug()<<"GraphM:Info()"<<endl;

	QStringList s;
	QString t;
	if(type==P2D)
		t=i18n("2D");
	else if (type==PSURFACE)
		t=i18n("Surface");
	else if (type==P3D)
		t=i18n("3D");
	else if (type==PQWT3D)
		t=i18n("QWT 3D");

	QString sh=i18n("NO");
	if (shown)
		sh=i18n("YES");

	s << label->simpleTitle() << t << sh;
	s << QString::number(nx) << QString::number(ny);
	s << QString::number(range[0].rMin())+ " .. " + QString::number(range[0].rMax());
	s << QString::number(range[1].rMin())+ " .. " + QString::number(range[1].rMax());
	s << QString::number(range[2].rMin())+ " .. " + QString::number(range[2].rMax());
	
	return s;
}

//! return matrix data as pixmap
QPixmap GraphM::Pixmap() {
	kdDebug()<<"GraphM:Pixmap()"<<endl;
	QPixmap pm(nx,ny);
	
	// 8 bit
	QImage *image = new QImage(nx,ny,8,256);
	for (int i=0;i<256;i++) {
		QColor c(i,i,i);
		image->setColor (i, c.rgb() );
	}
	
	for (int i=0;i<ny;i++) {
		for (int j=0;j<nx;j++) {
			unsigned int value =
				(int)((array[j+nx*i]-range[2].rMin())/(range[2].rMax()-range[2].rMin())*255.0);
			image->setPixel(j,i,value);
		}
	}
	pm.convertFromImage(*image);
	free(image);

	return pm;
}

void GraphM::setPixmap(QPixmap pm) {
	kdDebug()<<"GraphM:setPixmap()"<<endl;
	QImage image = pm.convertToImage();
	
	nx = pm.width();
	ny = pm.height();

	free(array);
	
	// set matrix data from pixmap
	array= new double[nx*ny];
	for (int i=0;i<ny;i++) {
		for (int j=0;j<nx;j++) {
			array[j+nx*i] = (double) qGray(image.pixel(j,i));
		}
	}
}
