/***************************************************************************
 *            aptcache.cc
 *
 *  Mon Nov 13 11:55:38 2006
 *  Copyright  2006-2009   Neil Williams <codehelp@debian.org>
 *  Copyright  1997, 1998, 1999 Jason Gunthorpe <jgg@debian.org>
 *  Copyright  1997, 1998, 1999 Matt Zimmerman <mdz@debian.org>
 *  Copyright  1997, 1998, 1999 Michael Piefel <piefel@debian.org>
 *  Copyright  1997-1999, 2009 Michael Vogt <mvo@debian.org>
 ****************************************************************************/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**	@file   aptcache.cc
	@brief  Query the apt cache
	@author Neil Williams <codehelp@debian.org>
*/
/** @{
*/

#include "config.h"
#include <iostream>
#include <string>
#ifdef __GNUG__
#pragma implementation "apt-pkg/cachefile.h"
#endif

#include <apt-pkg/init.h>
#include <apt-pkg/cachefile.h>
#include <apt-pkg/error.h>
#include <apt-pkg/sourcelist.h>
#include <apt-pkg/pkgcachegen.h>
#include <apt-pkg/pkgcache.h>
#include <apt-pkg/configuration.h>
#include <apt-pkg/policy.h>
#include <apt-pkg/pkgsystem.h>
#include <apt-pkg/debrecords.h>
#include <apt-pkg/cacheiterators.h>
#include <glib.h>
#include <glib/gi18n.h>
#include "langupdate.h"

/** @brief the Hash table of packages in the cache */
static GHashTable * apt_table = NULL;
/** @brief Hash table of source packages */
static GHashTable * src_table = NULL;
/** @brief support full quiet mode for apt */
static const gchar * apt_quiet = "-q -q";
static pkgCache *LangCache = 0;
static pkgSourceList *SrcList = 0;

static bool LoadPkgNames(void)
{
	pkgCache &Cache = *LangCache;
	pkgCache::PkgIterator I = Cache.PkgBegin();
	bool All = _config->FindB("APT::Cache::AllNames","false");
	gchar * str, * ver;
	apt_table = g_hash_table_new (g_str_hash, g_str_equal);
	for (;I.end() != true; I++)
	{
		if (All == false && I->VersionList == 0)
			continue;
		str = g_strdup(I.Name());
		/* force the first available version only
		This is OK as we control the sources list */
		for (pkgCache::VerIterator V = I.VersionList(); V.end() == false; V++)
		{
			ver = g_strdup(V.VerStr());
			break;
		}
		g_hash_table_insert(apt_table, str, ver);
	}
	if (g_hash_table_size (apt_table) > 0)
		return true;
	return false;
}

static bool LoadSourceNames (gint verbose)
{
	gchar *src, *bin;
	GList * p;

	pkgCache &Cache = *LangCache;
	pkgRecords pkgRecs(*LangCache);
	pkgRecords::Parser *Parse;
	pkgSrcRecords SrcRecs(*SrcList);

	src_table = g_hash_table_new (g_str_hash, g_str_equal);

	p = NULL;
	pkgCache::PkgIterator Pkg = Cache.PkgBegin();
	for (;Pkg.end() != true; Pkg++)
	{
		for (pkgCache::VerIterator Cur = Pkg.VersionList(); Cur.end() != true; Cur++)
		{
			pkgCache::VerFileIterator Vf = Cur.FileList();
			Parse = &pkgRecs.Lookup(Vf);

			string pkgstr;
			pkgstr = Parse->SourcePkg();
			src = g_strdup(pkgstr.data());
			pkgstr = Parse->Name();
			bin = g_strdup(pkgstr.data());
			g_hash_table_insert (src_table, bin, src);
		}
	}
	if (g_hash_table_size (src_table) > 0)
		return true;
	return false;
}

void
lu_clear_caches (void)
{
	g_hash_table_destroy (src_table);
}

gboolean
apt_init (const gchar * sourcelist, const gchar * suite_codename, 
		const gchar * prefix, gint verbose)
{
	FILE * fout = NULL;
	gchar * config_str, * path, *quiet, * apt, * out, *errstr;
	MMap *Map = 0;
	GError * em_gerr = NULL;

	out = errstr = NULL;
	path = g_build_filename (prefix, "/var/lib/"PACKAGE"/", suite_codename,
							 "/lists/partial", NULL);
	g_mkdir_with_parents (path, 0755);
	g_free (path);
	path = g_build_filename (prefix, "/var/lib/"PACKAGE"/", suite_codename,
							 "/archives/partial", NULL);
	g_mkdir_with_parents (path, 0755);
	g_free (path);
	path = g_build_filename (prefix, "/var/lib/"PACKAGE"/status", NULL);
	fout = fopen (path, "a");
	fclose (fout);
	g_free (path);
	path = g_build_filename (prefix, "/var/lib/"PACKAGE"/", suite_codename,
							 "/lists/lock", NULL);
	fout = fopen (path, "a");
	fclose (fout);
	g_free (path);
	path = g_build_filename (prefix, "/var/lib/"PACKAGE"/var/log/apt/", NULL);
	g_mkdir_with_parents (path, 0755);
	g_free (path);
	path = g_build_filename (prefix, "/var/lib/"PACKAGE"/", suite_codename,
							 "/pkgcache.bin", NULL);
	fout = fopen (path, "a");
	fclose (fout);
	g_free (path);
	if (verbose <= 1)
		quiet = g_strdup (apt_quiet);
	else if (verbose >= 2)
		quiet = g_strndup(apt_quiet, 2);
	else
		quiet = g_strdup ("");
	apt = g_strconcat ("apt-get ", quiet, NULL);
	config_str = g_strconcat(apt, lu_get_aptstring(), " update", NULL);
	g_free (quiet);
	g_free (apt);
	g_spawn_command_line_sync (config_str, &out, &errstr, NULL, &em_gerr);
	if (em_gerr)
	{
		g_critical (em_gerr->message);
		g_message (_("Unable to execute command: %s\n%s"), config_str, out);
		if (errstr)
			g_message (_("Error message was: %s"), errstr);
		g_clear_error (&em_gerr);
	}
	if (verbose >= 4)
		g_print ("%s: %s\n", PACKAGE, config_str);
	/* always initialise the config BEFORE changing values - init resets everything. */
	g_return_val_if_fail (pkgInitConfig(*_config), FALSE);
	g_return_val_if_fail (pkgInitSystem(*_config,_system), FALSE);
	if (verbose <= 1)
		_config->Set("quiet", 2);
	else
		_config->Set("quiet", 0);
	_config->Set("Apt::Architecture", HOST_CPU);
	_config->Set("help", true);
	_config->Set("APT::Get::List-Cleanup", "on");
	_config->Set("Apt::Install-Recommends","false");
	path = g_build_filename (prefix, "/var/lib/"PACKAGE, NULL);
	_config->Set("Dir",      path);
	_config->Set("Dir::Etc", path);
	g_free (path);
	path = g_build_filename (prefix, "/var/lib/"PACKAGE"/sources.list", NULL);
	_config->Set("Dir::Etc::SourceList", path);
	g_free (path);
	_config->Set("Dir::State", suite_codename);
	path = g_build_filename (prefix, "/var/lib/"PACKAGE"/status", NULL);
	_config->Set("Dir::State::Status", path);
	g_free (path);
	_config->Set("Dir::Cache", suite_codename);
	if (_config->FindB("APT::Cache::Generate",true) == false)
	{
		Map = new MMap(*new FileFd(_config->FindFile("Dir::Cache::pkgcache"),
				FileFd::ReadOnly),MMap::Public|MMap::ReadOnly);
	}
	else
	{
		// Open the cache file
		SrcList = new pkgSourceList;
		if (SrcList->Read(sourcelist) == false)
		{
			g_error ("%s: %s", PACKAGE, _("Failed to read sources"));
			return false;
		}
		
		// Generate it and map it
		OpProgress Prog;
		pkgMakeStatusCache(*SrcList,Prog,&Map,true);
	}

	// Print any errors or warnings found during parsing
	if (_error->PendingError())
	{
		_error->DumpErrors();
		return FALSE;
	}
	pkgCache Cache(Map);
	LangCache = &Cache;

	// Print any errors or warnings found during parsing
	if (_error->empty() == false)
	{
		bool Errors = _error->PendingError();
		_error->DumpErrors();
		return Errors == true?100:0;
	}
	LoadPkgNames();
	LoadSourceNames(verbose);
	return TRUE;
}

gboolean 
aptcache_lookup (const gchar * pkg)
{
	gchar * available = NULL;

	available = (gchar*)g_hash_table_lookup (apt_table, pkg);
	if (available)
		return TRUE;
	return FALSE;
}

gint
lu_check_pkg_version (const gchar * package, const gchar * version)
{
	gchar * verstr;
	verstr = g_strdup((gchar*)g_hash_table_lookup (apt_table, package));
	return g_strcmp0 (verstr, version);
}

gchar *
lu_get_sourcepkg (const gchar * binary)
{
	gchar * src;
	gint size;

	size = g_hash_table_size (src_table);
	src = g_strdup((gchar*)g_hash_table_lookup (src_table, binary));
	return src;
}

/** @} */
