/***************************************************************************
 *   Copyright (C) 2005 - 2006 by Christian Muehlhaeuser, Last.fm Ltd.     *
 *   chris@last.fm                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include "settings.h"
#include "md5/md5.h"
#include "rtaudio/RtAudio.h"

#ifdef Q_WS_MAC
#include <ApplicationServices/ApplicationServices.h>
#endif


Settings *Settings::s_instance = 0;

Settings::Settings()
    : QObject()
    , m_debug( false )
{
    s_instance = this;

    #ifdef WIN32
    config = new QSettings( "HKEY_CURRENT_USER\\Software\\Last.fm", QSettings::NativeFormat );
    config->setFallbacksEnabled( false );
    #endif
    #ifndef WIN32
    config = new QSettings( QSettings::IniFormat, QSettings::UserScope, "last.fm", "player", this );
    config->setFallbacksEnabled( false );

    qDebug() << "Settings filename: " << config->fileName();
    #endif
}


QString
Settings::MD5Digest( const char *token )
{
    md5_state_t md5state;
    unsigned char md5pword[16];

    md5_init( &md5state );
    md5_append( &md5state, (unsigned const char *)token, (int)strlen( token ) );
    md5_finish( &md5state, md5pword );

    char tmp[33];
    strncpy( tmp, "\0", sizeof(tmp) );
    for ( int j = 0; j < 16; j++ )
    {
        char a[3];
        sprintf( a, "%02x", md5pword[j] );
        tmp[2*j] = a[0];
        tmp[2*j+1] = a[1];
    }

    return QString::fromAscii( tmp );
}


bool
Settings::mergeSettings()
{
//     QSettings s( QSettings::IniFormat, QSettings::UserScope, "last.fm", "player", this );
//     s.setFallbacksEnabled( false );

    return true;
}


bool
Settings::resumePlayback()
{
    return ( config->value( "Radio/resumeplayback" ).toInt() == 1 );
}


void
Settings::setResumePlayback( bool enabled )
{
    config->setValue( "Radio/resumeplayback", enabled ? "1" : "0" );
}


QString
Settings::resumeStation()
{
    return config->value( "Radio/resumestation" ).toString();
}


void
Settings::setResumeStation( QString station )
{
    config->setValue( "Radio/resumestation", station );
}


int
Settings::volume( )
{
   return config->value( "Radio/volume", 50 ).toInt();
}


void
Settings::setVolume( int volume )
{
   config->setValue( "Radio/volume", volume );
}


int
Settings::positionX()
{
    return config->value( "Radio/posx", 0 ).toInt();
}


void
Settings::setPositionX( int x )
{
    config->setValue( "Radio/posx", x );
}


int
Settings::positionY()
{
    return config->value( "Radio/posy", 0 ).toInt();
}


void
Settings::setPositionY( int y )
{
    config->setValue( "Radio/posy", y );
}


int
Settings::soundSystem()
{
    return config->value( "Radio/soundsystem", 0 ).toInt();
}


void
Settings::setSoundSystem( int system )
{
    config->setValue( "Radio/soundsystem", system );
}


int
Settings::soundCard()
{
    return config->value( "Radio/soundcard", 0 ).toInt();
}


void
Settings::setSoundCard( int card )
{
    if ( card < 0 )
        card = 0;

    config->setValue( "Radio/soundcard", card );
}


int
Settings::externalSoundSystem()
{
    int externalSystem = -1;
    #ifdef WIN32
    externalSystem = 1;
    #endif
    #ifdef Q_WS_X11
    externalSystem = 2;
    #endif
    #ifdef Q_WS_MAC
    externalSystem = 1;
    #endif

    return externalSystem;
}


QString
Settings::username()
{
    return config->value( "Users/CurrentUser" ).toString();
}


QString
Settings::password()
{
    QString password = config->value( "Users/" + username() + "/Password" ).toString();
    if ( password.length() < 32 && !password.isEmpty() )
    {
        setPassword( password );
        password = MD5Digest( password.toAscii() );
    }

    return password;
}


QString
Settings::proxyUsername()
{
    return config->value( "ProxyUser" ).toString();
}


QString
Settings::proxyPassword()
{
    return config->value( "ProxyPass" ).toString();
}


QString
Settings::proxyHost()
{
    return config->value( "ProxyHost" ).toString();
}


int
Settings::proxyPort()
{
    return config->value( "ProxyPort" ).toInt();
}


bool
Settings::proxyUsage()
{
    return ( config->value( "ProxyEnabled" ).toInt() == 1 );
}


void
Settings::setUsername( QString username )
{
    config->setValue( "Users/CurrentUser", username );
}


void
Settings::setPassword( QString password )
{
    if ( !password.isEmpty() && password != "********" )
    {
        password = MD5Digest( password.toAscii() );
        config->setValue( "Users/" + username() + "/Password", password );
    }
}


void
Settings::setProxyUsername( QString username )
{
    config->setValue( "ProxyUsername", username );
}


void
Settings::setProxyPassword( QString password )
{
    config->setValue( "ProxyPassword", password );
}


void
Settings::setProxyHost( QString host )
{
    config->setValue( "ProxyHost", host );
}


void
Settings::setProxyPort( int port )
{
    config->setValue( "ProxyPort", port );
}


void
Settings::setProxyUsage( bool enabled )
{
    config->setValue( "ProxyEnabled", enabled ? "1" : "0" );
}


QString
Settings::browser()
{
    return config->value( "browser" ).toString();
}


void
Settings::setBrowser( QString browser )
{
    config->setValue( "browser", browser );
}


int
Settings::stationCounter( QString url )
{
    return config->value( "History/" + url ).toInt();
}


QString
Settings::stationName( QString url )
{
    QString name = config->value( "Users/" + username() + "/Stations/" + url ).toString();
    if ( !name.isEmpty() )
    {
        QStringList parts = name.split( ",", QString::SkipEmptyParts );
        name = "";

        for ( int i = 0; i < parts.count() && i < 5; i++ )
        {
            if ( i ) name += ", ";
            name += parts.at( i );
        }
    }

    return name;
}


void
Settings::incrementStationCounter( QString url, QString stationName )
{
    if ( stationName != 0 )
    {
        config->setValue( "Users/" + username() + "/Stations/" + url, stationName );
    }

    config->setValue( "Users/" + username() + "/History/" + url, stationCounter( url ) + 1 );
}


void
Settings::deleteStationCounter( QString url )
{
    config->remove( "Users/" + username() + "/History/" + url );
}


void
Settings::resetStationCounter( QString url )
{
    config->setValue( "Users/" + username() + "/History/" + url, 0 );
}


void
Settings::clearStationCounters()
{
    config->remove( "Users/" + username() + "/History" );
}


QStringList
Settings::stationItems()
{
    config->sync();

    config->beginGroup( "Users/" + username() + "/History" );
    QStringList items = config->allKeys();
    config->endGroup();

    return items;
}


void
Settings::save( bool restartConnection, bool restartAudio )
{
    config->setValue( "Radio/version", VERSION );
    config->sync();

    if ( restartConnection )
        emit reconnect();

    if ( restartAudio )
        emit resetAudio();
}


void
Settings::startBrowser( QString url )
{
    #ifdef Q_WS_X11
    if ( browser().isEmpty() )
    {
        QMessageBox::information( 0, "Attention", "Please configure a Browser in the Settings-Dialog!", QMessageBox::Ok );
        return;
    }
    #endif

    if ( !url.isEmpty() )
    {
        BrowserThread thread;
        thread.setUrl( url );
        thread.start();
    }
}


void
BrowserThread::run()
{
    qDebug( "Starting Browser!" );

    #ifdef WIN32
    WCHAR val[1024];
    memset( val, 0, 1024 );
    MultiByteToWideChar( CP_UTF8, 0, m_url.toAscii(), m_url.length(), val, 1024 );
    ShellExecute( 0, 0, val, NULL, NULL, SW_SHOW );
    #endif

    #ifdef Q_WS_MAC
    ICInstance icInstance;
    OSType psiSignature = 'Psi ';
    OSStatus error = ICStart( &icInstance, psiSignature );

    ConstStr255Param hint( 0x0 );
    const char* data = m_url.toLocal8Bit().data();
    long length = m_url.length();
    long start( 0 );
    long end( length );

    ICLaunchURL( icInstance, hint, data, length, &start, &end );
    ICStop( icInstance );
    #endif

    #ifdef Q_WS_X11
    QStringList params;
    params.append( QString( QUrl( m_url ).toEncoded() ) );

    QStringList options;
    options = Settings::instance()->browser().split( " " );
    if ( options.size() == 0 )
        options.append( Settings::instance()->browser() );

    for ( int i = 1; i < options.size(); i++ )
    {
        qDebug( QString( "param: " + options.at( i ) ).toLocal8Bit() );
        params.append( "\"" + options.at( i ) + "\"" );
    }

    QProcess::execute( options.at( 0 ), params );
    #endif
}

