/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include "CrashReporter.h"
#include "containerutils.h"
#include "http.h"
#include "Settings.h"

#define ZLIB_WINAPI
#include "zip.h"

#include <time.h>

#include <QPixmap>
#include <QIcon>
#include <QDebug>
#include <QTimer>
#include <QDir>

static const QString k_host = "crashreport.last.fm";
//static const QString k_host = "moses.last.fm";

static float k_taskRanges[2][2] = {
    { 0.0f, 0.1f }, // zip files
    { 0.1f, 1.0f }  // send files
};

CrashReporter::CrashReporter(
    int argc,
    char* argv[],
    QWidget* parent ) :
        QDialog( parent ),
        m_http( NULL )
{
    #ifndef WIN32
    QIcon icon( dataPath( "icons/as.ico" ) );
    setWindowIcon( icon );
    #endif

    ui.setupUi( this );

    // Want etched, not flat
    ui.line1->setFrameShadow( QFrame::Sunken );
    ui.line2->setFrameShadow( QFrame::Sunken );

    ui.logoLabel->setPixmap( QPixmap(dataPath( "app_55.png" )) );
    ui.topLabel->setText( tr( "Sending crash info to Last.fm..." ) );
    ui.progressBar->setRange( 0, 100 );
    ui.okButton->setEnabled( false );

    m_http = new Http( k_host, 80, this );
    connect(m_http, SIGNAL(requestFinished(int, bool)),
            this,   SLOT  (sendFinished(int, bool)), Qt::QueuedConnection);
    connect(m_http, SIGNAL(dataSendProgress(int, int)),
            this,   SLOT  (sendProgressMade(int, int)));

    for ( int i = 1; i < argc; i += 2 )
    {
        if ( strncmp( argv[i], "-d", 2 ) == 0 )
        {
            m_pathMinidump = argc > (i + 1) ? argv[i + 1] : "";
        }
        else if ( strncmp( argv[i], "-u", 2 ) == 0 )
        {
            m_username = argc > (i + 1) ? argv[i + 1] : "";
        }
        else if ( strncmp( argv[i], "-l", 2 ) == 0 )
        {
            m_pathLogDir = argc > (i + 1) ? argv[i + 1] : "";
        }
    }
}

// Didn't need this in the end, seems the system strips the quotes before
// sticking the args into argv.
QString
CrashReporter::parseArg(
    const char* arg )
{
    // Args must be quoted otherwise all bets are off

    QString qsArg( arg );
    int start = qsArg.indexOf( '\"' );
    int end = qsArg.lastIndexOf( '\"' );

    QString unquoted = qsArg.mid( start + 1, end - start - 2 );

    return unquoted;
}    

int
CrashReporter::exec()
{
    QTimer::singleShot( 0, this, SLOT( go() ) );
    return QDialog::exec();
}

void
CrashReporter::go()
{
    gatherData();
    if ( zipFiles() )
    {
        sendZip();
    }
}

void
CrashReporter::reject()
{
    // Cancel on-going ops
    m_http->abort();

    QDialog::reject();
}

void
CrashReporter::gatherData()
{
    UserSettings& user = The::settings().currentUser();
    
    if (user.isNull()) 
    {
        m_username = The::settings().allUsers().value( 0 );
        m_passMd5 = The::user( m_username ).password();
        
        if (m_passMd5 == "")
            m_passMd5 = m_username = "unknown";
    }
    else {
        m_username = user.username();
        m_passMd5 = user.password();
    }
    
    if (m_pathLogDir.isEmpty())
        m_pathLogDir = savePath( "" );
    
    if ( !m_pathLogDir.isEmpty() )
    {
        QDir logDir( m_pathLogDir );
        foreach( QString fileName, logDir.entryList( QStringList( "*.log" ), QDir::Files ) )
        {
            m_filesToZip.append(
                QDir::cleanPath( logDir.absoluteFilePath( fileName ) ) );
        }
    }

    if ( m_pathMinidump.isEmpty() )
    {
        m_pathMinidump = savePath( "minidump.dmp" );
    }

    m_filesToZip.append( m_pathMinidump );
}

bool
CrashReporter::zipFiles()
{
    QString zipFileName = "LastFMCrashData.zip";

    QDir tempDir = QDir::temp();
    tempDir.remove( zipFileName );

    QString zipFilePathOrig = tempDir.absolutePath() + "/" + zipFileName;
    m_zipFile.setFileName( zipFilePathOrig );
    
    QByteArray zipFilePath = QFile::encodeName( zipFilePathOrig );
        
    #ifdef WIN32
        // Minizip uses DOS encoding
        char buf[MAX_PATH];
        CharToOemA( zipFilePath.data(), buf );
        zipFilePath = buf;
    #endif        

    zipFile zipHandle = zipOpen( zipFilePath.data(), APPEND_STATUS_CREATE );

    if ( zipHandle == NULL )
    {
        return false;
    }

    int numFiles = m_filesToZip.size();
    int i = 1;
    foreach( QString fileName, m_filesToZip )
    {
        // Update display
        QString msg = tr( "Zipping file %1 of %2." ).arg( i ).arg( numFiles );
        ui.bottomLabel->setText( msg );
        setProgress( 0, (float)i++ / numFiles );
        QApplication::processEvents();

        // Zip a file
        QFile file( fileName );

        bool success = file.open( QIODevice::ReadOnly );
        if ( !success )
        {
            // Couldn't open source file, move on to the next one
            continue;
        }
        
        // Check that file isn't bigger than 1 MB to prevent abuse
        if ( file.size() > 1000000 )
        {
            continue;
        }

        QString name = QFileInfo( fileName ).fileName();
        
        // Create entry for file in zip
        zip_fileinfo dummy;
        memset( &dummy, 0, sizeof( dummy ) );
        int res = zipOpenNewFileInZip(
            zipHandle,
            qPrintable( QFileInfo( fileName ).fileName() ),
            &dummy,
            NULL, 0,
            NULL, 0,
            "",
            Z_DEFLATED,
            Z_DEFAULT_COMPRESSION );
            
        if ( res != ZIP_OK )
        {
            // Creation failed, move on to the next one
            zipCloseFileInZip( zipHandle );
            continue;
        }

        QByteArray fileData = file.readAll();
        res = zipWriteInFileInZip(
            zipHandle,
            reinterpret_cast<const void*>( fileData.constData() ),
            fileData.size() - 1 ); // because QByteArray adds a \0
            
        zipCloseFileInZip( zipHandle );
    }

    zipClose( zipHandle, "" );

    return true;
}

void
CrashReporter::sendZip()
{
    // Get Unix time
    time_t now; 
    time(&now);
    QString time = QString::number( now );

    // Concatenate pw hash with time
    QString auth = m_passMd5 + time;
    QString authLower = m_passMd5.toLower() + time;
    // Hash the concatenated string to create auth code
    QString authMd5 = MD5Digest( auth.toUtf8() );
    QString authMd5Lower = MD5Digest( authLower.toUtf8() );

    QString path = QString( "/?user=%1&time=%2&auth=%3&authlower=%4" )
        .arg( m_username, time, authMd5, authMd5Lower );

    qDebug() << "Sending " << path;

    m_reqId = m_http->post( path, &m_zipFile );
}

void
CrashReporter::sendFinished(
    int id,
    bool error )
{
    if ( id != m_reqId ) { return; }
    
    if ( error )
    {
        setProgress( 1, 1 );
        ui.topLabel->setText( tr( "Failed to send crash info." ) );
        ui.bottomLabel->setText( "" );
    }
    else
    {
        ui.topLabel->setText( tr( "Crash info sent." ) );
        qDebug() << "Response: " << m_http->readAll();
    }
    
    ui.cancelButton->setEnabled( false );
    ui.okButton->setEnabled( true );
}    

void
CrashReporter::sendProgressMade(
    int done,
    int total )
{
    if ( total > 0 )
    {
        setProgress( 1, (float)done / total );

        int kbDownloaded = done / 1024;
        int kbTotal = total / 1024;
        QString msg = tr( "Uploaded %L1k of %L2k." )
            .arg( kbDownloaded )
            .arg( kbTotal );
        ui.bottomLabel->setText( msg );
    }
    else
    {
        // Can't find how to get the progress bar to just bounce back and forth
    }
}    

void
CrashReporter::setProgress(
    int task,
    float percent )
{
    float start = k_taskRanges[task][0];
    float stop = k_taskRanges[task][1];
    float range = stop - start;
    float offset = percent * range;
    float pos = start + offset;

    ui.progressBar->setValue( pos * 100 );
}
