/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include <QtCore>
#include <QSqlQuery>
#include <QSqlResult>
#include <QSqlError>
#include <QFileSystemWatcher>

#include "mediadevicewatcher.h"
#include "containerutils.h"
#include "logger.h"
#include "Settings.h"

#ifdef WIN32
    #include "windows.h"
    #include "shfolder.h"
#endif

#define ESC( token ) QString( token ).replace( "\'", "''" )

QString XML_VERSION = "1.0";

#ifdef Q_WS_MAC
ComponentInstance theComponent = NULL;
OSAID m_OSA_CheckITunes = kOSANullScript;
#endif


void
sendToInstance( const QString& data )
{
    if ( The::settings().isFirstRun() )
        return;

    #ifdef QT_NO_DEBUG
        #ifdef WIN32
            QString appExe = "LastFM.exe";
        #endif
        #ifdef Q_WS_MAC
            QString appExe = "Last.fm";
        #endif
        #ifdef Q_WS_X11
            QString appExe = "last.fm.linux.sh";
        #endif
    #else
        #ifdef WIN32
            QString appExe = "LastFMd.exe";
        #endif
        #ifdef Q_WS_MAC
            QString appExe = "Last.fm_debug";
        #endif
        #ifdef Q_WS_X11
            QString appExe = "last.fm.linux.debug.sh";
        #endif
    #endif

    QString app = QCoreApplication::applicationDirPath() + "/" + appExe;
    QStringList params( "-tray" );
    params << data;

    qDebug() << "Starting new instance of" << app << params;

    bool isVista = false;
  #ifdef WIN32
    if ( ( QSysInfo::WindowsVersion & QSysInfo::WV_VISTA ) != 0 )
    {
        isVista = true;
    }
  #endif

    if ( !isVista )
    {
        QProcess::startDetached( app, params );
    }
    else
    {
      #ifdef WIN32
        // On Vista, use ShellExecute because otherwise it fails to launch the
        // client when we're running as a non-admin (which is pretty much always)
        SHELLEXECUTEINFOW sei;
        memset(&sei, 0, sizeof(sei));

        QString paramString = params.join(" ");
        qDebug() << "Param string for Vista: " << paramString;

        sei.cbSize = sizeof(sei);
        sei.fMask  = 0;
        sei.hwnd   = GetForegroundWindow();
        sei.lpVerb = L"open";
        sei.lpFile = reinterpret_cast<LPCWSTR>(app.utf16());
        sei.lpParameters = reinterpret_cast<LPCWSTR>(paramString.utf16());
        sei.nShow  = SW_SHOWNORMAL;

        BOOL bOK = ShellExecuteExW(&sei);
        if (!bOK)
        {
            LOG(1, "Couldn't ShellExecuteEx " << app << " " <<
                paramString << ". GetLastError: " << GetLastError() << "\n");
        }
      #endif
    }
}


MediaDeviceWatcher::MediaDeviceWatcher()
{
    qDebug() << "MediaDeviceWatcher init";

    #ifdef Q_WS_MAC
        #ifndef QT_NO_DEBUG
        m_runPath = QDir( QCoreApplication::applicationDirPath() ).canonicalPath() + "/LastFMHelper_debug";
        #else
        m_runPath = QDir( QCoreApplication::applicationDirPath() ).canonicalPath() + "/LastFMHelper";
        #endif
    #endif

    m_gpod = iTunesDevice();
    if ( !m_gpod )
        qDebug() << "Loading mediadevice plugin failed";
    else
    {
        qDebug() << "Loading mediadevice succeeded";
        m_gpod->setupWatchers();

        connect( m_gpod, SIGNAL( deviceAdded( QString ) ),
                 this,     SLOT( deviceAdded( QString ) ) );
        connect( m_gpod, SIGNAL( deviceChangeStart( QString, QDateTime ) ),
                 this,     SLOT( deviceChangeStart( QString, QDateTime ) ) );
        connect( m_gpod, SIGNAL( deviceChangeEnd( QString ) ),
                 this,     SLOT( deviceChangeEnd( QString ) ) );
        connect( m_gpod, SIGNAL( trackChanged( TrackInfo, int ) ),
                 this,     SLOT( trackChanged( TrackInfo, int ) ) );
    }

    #ifdef Q_WS_MAC
    QFileSystemWatcher* fs = new QFileSystemWatcher( this );
    QDir dir = QDir( QCoreApplication::applicationDirPath() );
    dir.cdUp();
    dir.cdUp();

    fs->addPath( dir.canonicalPath() );
    fs->addPath( QCoreApplication::applicationDirPath() );

    #ifndef QT_NO_DEBUG
        fs->addPath( QCoreApplication::applicationDirPath() + "/Last.fm_debug" );
    #else
        fs->addPath( QCoreApplication::applicationDirPath() + "/Last.fm" );
    #endif

    connect( fs, SIGNAL( fileChanged( QString ) ),
             this, SLOT( shutdownHelper( QString ) ) );
    connect( fs, SIGNAL( directoryChanged( QString ) ),
             this, SLOT( shutdownHelper( QString ) ) );
    #endif

    #ifdef Q_WS_MAC
    m_cthread = new CocoaThread( this );

/*    m_cocoa = CocoaUtils::getInstance();

    connect( m_cocoa, SIGNAL( iTunesLaunched() ),
             m_parent, SLOT( startWithiTunes() ) );

    m_cocoa->watchOutForiTunes();*/
    #endif
}


MediaDeviceWatcher::~MediaDeviceWatcher()
{
    #ifdef Q_WS_MAC
    delete m_cthread;
    #endif
}


QList<TrackInfo>
MediaDeviceWatcher::readQueue( const QString& uid )
{
    QString user = The::settings().mediaDeviceUser( uid );

    QList<TrackInfo> queue;
    QString path = savePath( user + "_mediadevice.xml" );

    QFile file( path );
    if ( !file.open( QIODevice::ReadOnly | QIODevice::Text ) )
    {
        qDebug() << "Could not open cache file to read submit queue: " << path;
        return queue;
    }

    QTextStream stream( &file );
    stream.setCodec( "UTF-8" );

    QDomDocument d;
    QString contents = stream.readAll();

    if ( !d.setContent( contents ) )
    {
        qDebug() << "Couldn't parse file: " << path;
        return queue;
    }

    const QString ITEM( "item" ); //so we don't construct these QStrings all the time

    for( QDomNode n = d.namedItem( "submissions" ).firstChild(); !n.isNull() && n.nodeName() == ITEM; n = n.nextSibling() )
        queue << TrackInfo( n.toElement() );

    return queue;
}


void
MediaDeviceWatcher::trackChanged( const TrackInfo& track, int playCounter )
{
	// since we subtract playcounts in iTunesDevice due to iTunes weirdness
	// this is indeed a necessary check. DO NOT REMOVE! --mxcl
	if (track.playCount() > 0)
	{
		qDebug() << "Scrobbling" << playCounter << "plays for" << track.artist() << "-" << track.track();

		QDomElement i = track.toDomElement( m_newsubdoc );
		m_submitQueue.appendChild( i );
	}
}


void
MediaDeviceWatcher::deviceAdded( const QString& uid )
{
    QString user = The::settings().mediaDeviceUser( uid );
    qDebug() << "A new mediadevice found" << uid << "for user" << user;

    // If there was no user found, tell the main app.
    if ( user.isEmpty() )
        sendToInstance( "container://addMediaDevice/" + uid );
}


void
MediaDeviceWatcher::deviceChangeStart( const QString& uid, QDateTime lastItunesUpdateTime )
{
    QString user = The::settings().mediaDeviceUser( uid );

    if ( !user.isEmpty() && user != "<disabled>" )
    {
        QList<TrackInfo> tl;

        m_newsubdoc = QDomDocument();
        m_submitQueue = m_newsubdoc.createElement( "submissions" );
        m_submitQueue.setAttribute( "product", "Audioscrobbler" );
        m_submitQueue.setAttribute( "version", XML_VERSION );
        m_submitQueue.setAttribute( "lastSubmission", QDateTime::currentDateTime().toTime_t() );
        m_submitQueue.setAttribute( "lastItunesUpdate", lastItunesUpdateTime.toTime_t() );

        tl = readQueue( uid );
        foreach( TrackInfo track, tl )
        {
            QDomElement i = track.toDomElement( m_newsubdoc );
            m_submitQueue.appendChild( i );
        }
    }
}


void
MediaDeviceWatcher::deviceChangeEnd( const QString& uid )
{
    QString user = The::settings().mediaDeviceUser( uid );

    if ( !user.isEmpty() && user != "<disabled>" )
    {
        QFile subfile( savePath( user + "_mediadevice.xml" ) );
        if( !subfile.open( QIODevice::WriteOnly | QIODevice::Text ) )
        {
            qDebug() << "Could not open cache file to write submit queue";
            return;
        }

        QDomNode submitNode = m_newsubdoc.importNode( m_submitQueue, true );
        m_newsubdoc.appendChild( submitNode );

        QTextStream stream( &subfile );
        stream.setCodec( "UTF-8" );
        stream << "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n";
        stream << m_newsubdoc.toString();
        subfile.close();

        if ( m_submitQueue.childNodes().count() > 0 )
            sendToInstance( "container://checkScrobblerCache/" + user );
    }
}


void
MediaDeviceWatcher::forceDetection( const QString& path )
{
    m_gpod->forceDetection( path );
}


void
MediaDeviceWatcher::shutdownHelper( const QString& path )
{
    // Get the path again to check if we've been moved to the trash
    #ifndef QT_NO_DEBUG
        QString mypath = QDir( QCoreApplication::applicationDirPath() ).canonicalPath() + "/LastFMHelper_debug";
    #else
        QString mypath = QDir( QCoreApplication::applicationDirPath() ).canonicalPath() + "/LastFMHelper";
    #endif

    if ( !QFile::exists( m_runPath ) || m_runPath != mypath )
    {
        qDebug() << "Shutting down cause of:" << path;
        qDebug() << "Path to me:" << mypath;
        qDebug() << "Runpath:" << m_runPath;
        qDebug() << "Do I still exist?" << QFile::exists( m_runPath );

        QCoreApplication::quit();
    }
}


#ifdef Q_WS_MAC

/* AppleScriptAvailable returns true if AppleScript is available
and the routines defined herein can be called. */
bool AppleScriptAvailable()
{
//     qDebug() << "Getting available";

    long response;
    if ( Gestalt( gestaltAppleScriptAttr, &response ) != noErr )
        response = 0;

    return ( ( response & ( 1 << gestaltAppleScriptPresent ) ) != 0 );
}


OSAID LowCompileAppleScript( const void* text, long textLength )
{
//     qDebug() << "Compiling script";

    QString result;
    OSStatus err;
    AEDesc scriptTextDesc;
    OSAID scriptID = kOSANullScript;

    AECreateDesc( typeNull, NULL, 0, &scriptTextDesc );

    /* put the script text into an aedesc */
    err = AECreateDesc( typeChar, text, textLength, &scriptTextDesc );
    if ( err != noErr )
        goto bail;

    /* compile the script */
    err = OSACompile( theComponent, &scriptTextDesc, kOSAModeNull, &scriptID );
    if ( err != noErr )
    {
        qDebug() << "Compiling err";    
        goto bail;
    }

bail:
    return scriptID;
}


/* LowRunAppleScript compiles and runs an AppleScript
provided as text in the buffer pointed to by text.  textLength
bytes will be compiled from this buffer and run as an AppleScript
using all of the default environment and execution settings.  If
resultData is not NULL, then the result returned by the execution
command will be returned as typeChar in this descriptor record
(or typeNull if there is no result information).  If the function
returns errOSAScriptError, then resultData will be set to a
descriptive error message describing the error (if one is
available). */
bool LowExecAppleScript( OSAID scriptID, QString& resultToken )
{
//     qDebug() << "Executing script";

    QString s;
    char result[4096] = "\0";

    OSStatus err;
    AEDesc resultData;
    OSAID resultID = kOSANullScript;

//     qDebug() << "Running script";
    /* run the script/get the result */
    err = OSAExecute( theComponent, scriptID, kOSANullScript, kOSAModeAlwaysInteract, &resultID );

//     qDebug() << "Running script done" << err;
    if ( err == errOSAScriptError )
    {
        OSAScriptError( theComponent, kOSAErrorMessage, typeUTF8Text, &resultData );
        int length = AEGetDescDataSize( &resultData );
        {
            AEGetDescData( &resultData, result, length < 4096 ? length : 4096 );
            result[ length ] = '\0';
            s = QString::fromUtf8( (char *)&result, length );
            LOG( 1, "AppleScript error: " << s << "\n" );
        }

        return false;
    }
    else if ( err == noErr && resultID != kOSANullScript )
    {
//         qDebug() << "Getting script result";

        OSADisplay( theComponent, resultID, typeUTF8Text, kOSAModeNull, &resultData );

        int length = AEGetDescDataSize( &resultData );
        {
            AEGetDescData( &resultData, result, length < 4096 ? length : 4096 );
            s = QString::fromUtf8( (char*)&result, length );

            // Strip surrounding quotes
            if ( s.startsWith( "\"" ) && s.endsWith( "\"" ) )
            {
                s = s.mid( 1, s.length() - 2 );
            }

            // iTunes sometimes gives us strings with null terminators which
            // fucks things up if we don't remove them.
            while ( s.endsWith( QChar( QChar::Null ) ) )
            {
                s.chop( 1 );
            }

            // It also escapes quotes so convert those too
            s.replace( "\\\"", "\"" );
        }
    }
    else
    {
        // AppleScript not responding
        return false;
    }

//     qDebug() << "Executing script done";
    if ( resultID != kOSANullScript )
        OSADispose( theComponent, resultID );

    resultToken = s;
    return true;
}


void
MediaDeviceWatcher::startWithiTunes()
{
    sendToInstance( "" );
}


bool
CocoaThread::isITunesRunning()
{
    if ( m_OSA_CheckITunes == kOSANullScript )
    {
        QString script =
            "set itunes_active to false\n"
            "with timeout of 30 seconds\n"
            "tell application \"Finder\"\n"
                "if (get name of every process) contains \"iTunes\" then set itunes_active to true\n"
            "end tell\n"
            "end timeout\n"
            "return itunes_active\n";

        m_OSA_CheckITunes = LowCompileAppleScript( script.toUtf8(), script.toUtf8().length() );
    }

    QString r;
    LowExecAppleScript( m_OSA_CheckITunes, r );

    return ( r == "true" );
}


void
CocoaThread::run()
{
    m_run = true;
    theComponent = OpenDefaultComponent( kOSAComponentType, typeAppleScript );

    while ( m_run )
    {
        bool ir = isITunesRunning();

        if ( !m_itunesRunning && ir )
        {
            m_itunesRunning = true;
            m_parent->startWithiTunes();
        }
        else if ( m_itunesRunning && !ir )
        {
            m_itunesRunning = false;
        }

        usleep( 3000000 );
    }
}


#endif
