/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Max Howell, Last.fm Ltd <max@last.fm>                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include <QtGui>
#include "containerutils.h"
#include "Radio.h"
#include "RestStateWidget.h"
#include "Settings.h"

#include "winutils.h"

// These are pixel sizes
#ifdef WIN32
    // Small config
    static const int k_helloFontSize = 18;
    static const int k_standardFontSize = 11;
#else
    static const int k_helloFontSize = 24;
    static const int k_standardFontSize = 18;
#endif


static void setFontPixelSize( QWidget* widget, int const size )
{
    QFont f = widget->font();
    f.setPixelSize( size );
    widget->setFont( f );
}


RestStateWidget::RestStateWidget( QWidget* parent )
        : WatermarkWidget( parent )
{
    //TODO give title tooltips for the tag keyword
    
    ui.setupUi( this );
    
    setFontPixelSize( ui.hello, k_helloFontSize );
    setFontPixelSize( ui.label1, k_standardFontSize );
    setFontPixelSize( ui.label2, k_standardFontSize );
    
    #ifdef Q_WS_MAC
        QSize s = QPixmap(":/mac/RestStateWidgetCombo.png").size();

        ui.label2->setText( tr("or listen to your music in %1.", "%1 is a media player").arg( "iTunes" ) );

        if (The::settings().appLanguage() == "en")
        {
            // we only show this graphic for english as other languages are too wide
            ui.combo->setFixedSize( s );
            
//          ui.play->setFixedWidth( s.width() );
            ui.combo->installEventFilter( this );
            ui.hboxLayout1->setSpacing( 6 );
        
            // this stuff is because for some reason the custom drawing isn't aligned
            // unless we add a few pixels here and there. Bizarre.
            ui.combo->setFixedHeight( ui.combo->height() + 2 );
            ui.edit->setFixedHeight( ui.combo->height() - 2 );

            ui.hboxLayout1->setAlignment( ui.play, Qt::AlignBottom );
        }

    #elif defined WIN32

        updatePlayerNames();
            
        // the long line is too wide and makes this->sizeHint huge unless we wordwrap
        ui.label2->setWordWrap( true );
        
        // Dirty hack to get heights looking the same on Windows
        ui.combo->setFixedHeight( 20 );
        ui.edit->setFixedHeight( 20 );
        ui.play->setFixedHeight( 22 );
        
    #elif defined LINUX

        ui.label2->hide(); //no plugins on Linux
        
        QList<QWidget*> widgets; widgets << ui.edit << ui.combo << ui.play;
        int h = 0;
        foreach (QWidget* w, widgets)
            h = qMax( w->height(), h );
        foreach (QWidget* w, widgets)
            w->setFixedHeight( h );    

    #endif
        
    connect( ui.edit, SIGNAL(returnPressed()), ui.play, SLOT(animateClick()) );
    connect( ui.edit, SIGNAL(textChanged( QString )), SLOT(onEditTextChanged( QString )) );
    connect( ui.play, SIGNAL(clicked()), SLOT(onPlayClicked()) );
    
    connect( &The::settings(), SIGNAL(userSwitched( UserSettings& )), SLOT(onUserChanged( UserSettings& )) );
    
    setFocusProxy( ui.edit );
    setWatermark( dataPath("watermark.png") );
    onUserChanged( The::currentUser() );
    
    ui.edit->installEventFilter( this );
}


RestStateWidget::~RestStateWidget()
{
    CurrentUserSettings().setValue( "RestStateWidgetComboIndex", ui.combo->currentIndex() );
}


void
RestStateWidget::updatePlayerNames()
{
#ifdef WIN32
    QStringList plugins = The::settings().allPlugins( false );
    plugins.removeAll( "" );

    if (plugins.count())
    {
        QString last_plugin = plugins.takeLast();
        
        if (plugins.count())
        {
            QString text = tr("or listen to your music in %1 or %2.",
                    "%1 is a list of plugins, eg. 'Foobar, Winamp, Windows Media Player'" );
            
            ui.label2->setText( text.arg( plugins.join( ", " ) ).arg( last_plugin ) );
        }
        else
            ui.label2->setText( tr("or listen to your music in %1.", "%1 is a media player").arg( last_plugin ) );
    }
    else
        ui.label2->setText( tr("or install a player plugin to scrobble music from your media player." ) );
#endif
}


void
RestStateWidget::onUserChanged( UserSettings& user )
{
    ui.hello->setText( tr("Hello %1,").arg( user.username() ) );
    ui.combo->setCurrentIndex( CurrentUserSettings().value( "RestStateWidgetComboIndex", 0 ).toInt() );
}


void
RestStateWidget::onEditTextChanged( const QString& new_text )
{
    ui.play->setDisabled( new_text.isEmpty() );
}


void
RestStateWidget::onPlayClicked()
{
    QString url;
    
    if (ui.edit->text().startsWith( "lastfm://" ))
        url = "%1";

    else if (ui.combo->currentIndex() == 0)
        url = "lastfm://artist/%1/similarartists";
    
    else
        url = "lastfm://globaltags/%1";
            
    The::radio().playStation( StationUrl( url.arg(ui.edit->text()) ) );
}


void
RestStateWidget::clear()
{
    ui.edit->clear();
}


bool
RestStateWidget::eventFilter( QObject* o, QEvent* e )
{
    if (o == ui.edit)
    {
        int const key = static_cast<QKeyEvent*>(e)->key();
        if (key == Qt::Key_Up || key == Qt::Key_Down)
            //send these to the combo as a convenience feature
            ui.combo->event( e );
    }
    
    #ifdef Q_WS_MAC
    // can only do with english as the others have different width texts
    if (o == ui.combo && e->type() == QEvent::Paint && The::settings().appLanguage() == "en")
    {
		QRect r = ui.combo->rect();
        QPainter p( ui.combo );

        p.drawPixmap( QPoint( 0, 1 ), QPixmap(":/mac/RestStateWidgetCombo.png") );
        r.adjust( 0, 2, -20, -2 );
        p.drawText( r, ui.combo->currentText(), Qt::AlignVCenter | Qt::AlignRight );
        return true;
    }
    #endif
    
    return false;
}
