/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include "configwizard.h"
#include "logger.h"
#include "Settings.h"
#include "winutils.h"
#include "WebService.h"
#include "LastMessageBox.h"

#include <QtGui>

using namespace std;

bool ConfigWizard::s_wizardRunning = false;


/******************************************************************************
    ConfigWizard
******************************************************************************/
ConfigWizard::ConfigWizard( QWidget* parent, Mode mode,  QString uid )
        : BaseWizard( parent )
        , m_mode( mode )
        , m_bootstrapAllowed( false )
        , m_didBootstrap( false )
{

    LOGL(3, "Launching ConfigWizard");

    m_uid = uid;

    // Init strings here as they don't get translated if they're global
    m_introHeader = tr("Last.fm Setup");

    m_introInfo =
        tr("Now that you've installed Last.fm, it needs to be "
        "set up for your computer. Don't worry, it won't take long and only "
        "needs to be done once.\n\nThe social music revolution awaits! ");
    #ifdef Q_WS_MAC
        m_introInfo += tr("Click Continue to begin.");
    #else
        m_introInfo += tr("Click Next to begin.");
    #endif

    m_notAllowedInfo =
        tr("As this wizard installs file on your computer, you must be "
        "logged in as an Administrator to complete it. Please get an Administrator "
        "to run this for you.");

    m_loginHeader =
        tr("Log in");

    m_detectExplainHeader =
        tr("Music Player Detection");

    m_detectExplainInfo =
        tr("Last.fm will now look for music players on your computer "
        "and then download the plugins you need to get scrobbling.\n\n"
        "Before continuing, make sure all your music player software is closed.\n\n"
        "Click Next to continue.");

    m_detectHeader =
        tr("Detecting Music Players");

    m_detectInfo =
        tr("Downloading plugin information from Last.fm.");

    m_selectHeader =
        tr("Select Plugins");

    m_downloadHeader =
        tr("Downloading Plugins");

    m_doneHeader =
        tr("Finally...");

    m_bootstrapHeader =
        tr("Import your iTunes listening history");

    m_bootstrapQuestion =
        tr("Do you want to import your iTunes listening history?\n\n"
        "This will add charts to your profile based on what you've listened to in the past.");

    m_bootstrapInfo =
        tr("Last.fm is now importing your iTunes listening history.");

    m_mediaDeviceHeader =
        tr("Connect your iPod with Last.fm");

    m_mediaDeviceQuestion =
        tr("You've connected your iPod with Last.fm running for the first time. "
           "Would you like to scrobble the tracks played on your iPod to a profile from now on?");

    #ifdef Q_WS_MAC
        m_doneInfoFirstRun =
            tr("Last.fm is set up and you're ready to start scrobbling.\n\n"
            "If you close your Last.fm window, you can easily access it from the "  
            "icon in the menu bar.");
    #else
        m_doneInfoFirstRun =
            tr("Last.fm is set up and you're ready to start scrobbling.\n\n"
            "You can access Last.fm at any time by double-clicking the Last.fm user "
            "icon in the system tray.");
    #endif

    m_doneInfoBootstrapExtra = 
        tr("\n\nYour imported iTunes library will show up on your profile page "
        "within a few minutes.");

    m_doneInfoPlugin =
        tr("The plugin(s) you selected have now been installed.");

    m_doneInfoMediaDeviceYes = 
        tr("Your iPod has now been connected to your Last.fm account");
    m_doneInfoMediaDeviceNo = 
        tr("Your iPod has not been connected to your Last.fm account");

    #ifdef Q_WS_MAC
        setTitle(tr("Set up Last.fm"));
    #else 
        setTitle(tr("Setup Wizard"));
    #endif

    if ( mode == Login )
    {
        m_pageOffset = 0;
#ifdef WIN32
        setNumPages(10);
#endif
#ifdef Q_WS_MAC
        setNumPages(6);
#endif
#ifdef Q_WS_X11
        setNumPages(3);
#endif
    }
    else if ( mode == MediaDevice )
    {
        #ifdef Q_WS_X11
            Q_ASSERT( !"MediaDevice mode not available on Linux" );
        #endif

        setTitle(tr("Set up Last.fm for your iPod"));

        m_pageOffset = 7;
        setNumPages(3);
    }
    else
    {
        #ifndef WIN32
            Q_ASSERT( !"Plugin mode only available on Win" );
        #endif

        m_pageOffset = 2;
        setNumPages(8);
    }

    BaseWizard::nextButtonClicked();


    if ( mode != MediaDevice )
    {
        connect(&m_infoGetter, SIGNAL(updateInfoDone(bool, QString)),
                this,          SLOT  (pluginInfoDone(bool, QString)), Qt::QueuedConnection);

        connect(&m_updater, SIGNAL(updateDownloadDone(bool, QString)),
                this,       SLOT  (pluginDownloadDone(bool, QString)), Qt::QueuedConnection);

        connect( The::webService(), SIGNAL(handshakeResult( Handshake* )), SLOT(handshakeFinished()) );
    }
}

/******************************************************************************
    createPage
******************************************************************************/
QWidget* ConfigWizard::createPage( int index )
{
    index += m_pageOffset;

    switch ( index )
    {
        case 0:
        {
            if ( canUserRunWizard() )
            {
                m_page1 = new WizardInfoPage( this, m_introInfo );
            }
            else
            {
                m_page1 = new WizardInfoPage( this, m_notAllowedInfo );
                enableNext( false );
            }

            return m_page1;
        }
        break;

        case 1:
        {
            m_page2 = new WizardLoginPage( this );
            connect( m_page2, SIGNAL( verifyResult( bool, bool ) ),
                     this,    SLOT( loginVerified( bool, bool ) ) );
            return m_page2;
        }
        break;

        case 2:
        {
            if ( canUserRunWizard() )
            {
                m_page3 = new WizardInfoPage( this, m_detectExplainInfo );
            }
            else
            {
                m_page3 = new WizardInfoPage( this, m_notAllowedInfo );
                enableNext( false );
            }
            return m_page3;
        }
        break;

        case 3:
        {
            m_page4 = new WizardProgressPage( this, "", "" );

            // Hook up infoGetter signals to progress page
            connect( &m_infoGetter, SIGNAL( progressMade( int, int ) ),
                     m_page4,       SLOT( setProgress( int, int ) ) );
            connect( &m_infoGetter, SIGNAL( statusChange( QString ) ),
                     m_page4,       SIGNAL( detailedInfoChanged( QString ) ) );
            m_page4->setInfo( m_detectInfo );

            return m_page4;
        }
        break;

        case 4:
        {
            m_page5 = new WizardSelectPluginPage( this );
            return m_page5;
        }
        break;

        case 5:
        {
            m_page6 = new WizardProgressPage(this, "", "");

            // Hook up downloader signals to progress page
            connect(&m_updater, SIGNAL(progressMade(int, int)),
                    m_page6,    SLOT  (setProgress(int, int)));
            connect(&m_updater, SIGNAL(statusChange(QString)),
                    m_page6,    SIGNAL(detailedInfoChanged(QString)));
            connect(&m_updater, SIGNAL(newFile(QString)),
                    m_page6,    SLOT  (setInfo(QString)));

            return m_page6;
        }
        break;

        case 6:
        {
            const QString& info = m_bootstrapQuestion;
            m_page7 = new WizardBootstrapAskPage(this, info);

            return m_page7;
        }
        break;

        case 7:
        {
            const QString& info = m_mediaDeviceQuestion;
            m_page8 = new WizardMediaDeviceAskPage(this, info);
            adjustSize();

            return m_page8;
        }
        break;

        case 8:
        {
            // Create a progress page
            const QString& info = m_bootstrapInfo;
            m_page9 = new WizardBootstrapPage(this, info);

            connect( &m_mediaDevices, SIGNAL( bootstrapProgress( int, TrackInfo ) ),
                     m_page9,         SLOT  ( setProgress( int, TrackInfo ) ) );

            connect( &m_mediaDevices, SIGNAL( bootstrapDone() ),
                     this,            SLOT  ( bootstrapDone() ) );

            return m_page9;
        }
        break;


        case 9:
        {
            QString info;
            if ( m_mode == Login )
            {
                LOG(3, "Setting first run done\n");
                The::settings().setFirstRunDone();
                info = m_doneInfoFirstRun;
                if ( m_didBootstrap )
                {
                    info += m_doneInfoBootstrapExtra;
                }
            }
            else if ( m_mode == Plugin )
            {
                info = m_doneInfoPlugin;
            }
            else if ( m_mode == MediaDevice )
            {
                if ( m_page8 && m_page8->uiWidget.yesButton->isChecked() )
                    info = m_doneInfoMediaDeviceYes;
                else
                    info = m_doneInfoMediaDeviceNo;
            }

            m_page10 = new WizardInfoPage(this, info);
            enableBack(false);
            return m_page10;
        }
        break;

    }

    return NULL;
}

/******************************************************************************
    headerForPage
******************************************************************************/
QString
ConfigWizard::headerForPage(
    int index)
{
    index += m_pageOffset;

    switch (index)
    {
        case 0: return m_introHeader;         break;
        case 1: return m_loginHeader;         break;
        case 2: return m_detectExplainHeader; break;
        case 3: return m_detectHeader;        break;
        case 4: return m_selectHeader;        break;
        case 5: return m_downloadHeader;      break;
        case 6: return m_bootstrapHeader;     break;
        case 7: return m_mediaDeviceHeader;   break;
        case 8: return m_bootstrapHeader;     break;
        case 9: return m_doneHeader;          break;
    }

    return "";
}

/******************************************************************************
    backButtonClicked
******************************************************************************/
void
ConfigWizard::backButtonClicked()
{
    BaseWizard::backButtonClicked();
}

/******************************************************************************
    nextButtonClicked
******************************************************************************/
void
ConfigWizard::nextButtonClicked()
{
    // When we get in here, currentPage holds the page number we're leaving

    switch (currentPage() + m_pageOffset)
    {
        case 0:
        {
            // Do nothing
        }
        break;

        case 1:
        {
            // Leaving login page, verify login details
            QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
            m_page2->verify();
            return;
        }
        break;

        case 2:
        {
            // Leaving detect explain page, download plugin info
            mAvailPlugins.clear();
            m_infoGetter.GetUpdateInfo(NULL, &mAvailPlugins, true);
        }
        break;

        case 3:
        {
            // Leaving progress page, display available plugins
            BaseWizard::nextButtonClicked();
            m_page5->Populate(mAvailPlugins);
            return;
        }
        break;

        case 4:
        {
            // Made plugin selection, move to progress screen
            BaseWizard::nextButtonClicked();
            if (!downloadPlugins())
            {
                BaseWizard::backButtonClicked();
            }
            return;
        }
        break;

        case 5:
        {
            // Progress screen for plugin download
        }
        break;

        case 6:
        {
            // We have just answered the bootstrap question, which means we
            // need to inc the page offset to skip the iPod question
            if ( m_page7 && m_page7->uiProgress.yesButton->isChecked() )
            {
                setNumPages( numPages() - 1 );
                m_pageOffset++;

                BaseWizard::nextButtonClicked();
                m_mediaDevices.bootStrap();

                return;
            }
            else
            {
                setNumPages( numPages() - 2 );
                m_pageOffset += 2;
            }
        }
        break;

        case 7:
        {
            // This is where the iPod connect wizard starts
            if ( m_page8 )
            {
                // If m_page8 exists, we have been asked the iPod question
                if ( m_page8->uiWidget.yesButton->isChecked() )
                {
                    The::settings().addMediaDevice( m_uid,
                                                       m_page8->uiWidget.userComboBox->itemText( m_page8->uiWidget.userComboBox->currentIndex() ) );

/*                    if ( !The::settings().bootstrap() )
                    {
                        BaseWizard::nextButtonClicked();
                        return;
                    }
                    else*/
                    {
                        setNumPages( numPages() - 1 );
                        m_pageOffset++;
                    }
                }
                else
                {
                    The::settings().addMediaDevice( m_uid, "<disabled>" );
                    setNumPages( numPages() - 1 );
                    m_pageOffset++;
                }
            }
        }
        break;

        case 8:
        {
            // Leaving the bootstrap progress screen
        }
        break;

        case 9:
        {
            // Done page
        }
        break;

        default:
            Q_ASSERT( !"duh" );

    }

    BaseWizard::nextButtonClicked();

}

/******************************************************************************
    downloadPlugins

    A lot of this is duplicate code from UpdateWizard, should really
    consolidate this.
******************************************************************************/
bool
ConfigWizard::downloadPlugins()
{
    mDownloadTasks.clear();

    // Get plugin selections
    vector<int> indicesToInstall;
    m_page5->GetChecked(indicesToInstall);
    for (size_t i = 0; i < indicesToInstall.size(); ++i)
    {
        CPluginInfo& current = mAvailPlugins.at(indicesToInstall.at(i));
        mDownloadTasks.push_back(&current);
    }

    // Check if any are running and ask to shut them down
    vector<QString> vecRunning;
    if (m_updater.CheckIfRunning(mDownloadTasks, vecRunning))
    {
        // Ask user if we should shut these running apps down
        QString sPrompt(tr("The following music players seem to be running at the moment:\n\n"));
        for (size_t j = 0; j < vecRunning.size(); ++j)
        {
            sPrompt += vecRunning.at(j) + "\n";
        }
        sPrompt += tr("\nThey need to be shut down before plugins can be installed.\n"
            "Do you want Last.fm to close them?");

        QString sCaption(tr("Detected Running Player(s)"));

        int answer = LastMessageBox::question(sCaption, sPrompt,
            QMessageBox::Yes, QMessageBox::No);
        if (answer == QMessageBox::No)
        {
            return false;
        }

        update();

        // Go ahead and kill the poor bastards
        QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        bool bSuccess = m_updater.KillRunning();

        if (!bSuccess)
        {
            QApplication::restoreOverrideCursor();

            // Let user shut them down if it didn't work
            LastMessageBox::warning(tr("Shutdown Failed"),
                tr("Some of the running programs couldn't be shut down. Please close them manually."),
                QMessageBox::Ok, QMessageBox::NoButton);

            return false;
        }

        QApplication::restoreOverrideCursor();
    }

    // Let the download commence
    m_updater.downloadUpdates(mDownloadTasks);

    return true;
}


/******************************************************************************
    loginVerified
******************************************************************************/
void
ConfigWizard::loginVerified(
    bool valid,
    bool bootstrap)
{
    m_bootstrapAllowed = bootstrap;
    qDebug() << "Login verified" << m_bootstrapAllowed;

    #ifdef Q_WS_X11
    m_bootstrapAllowed = false;
    #endif

    if ( valid )
    {
        // This will cause the web service to do a handshake. We do not want
        // to advance to the next page until the handshake has finished so
        // we'll wait until that signal comes back before advancing.
        m_page2->save();
    }
    else
    {
        // Do nothing, remain on login page
        QApplication::restoreOverrideCursor();
    }
}

/******************************************************************************
    handshakeFinished
******************************************************************************/
void
ConfigWizard::handshakeFinished()
{
    QApplication::restoreOverrideCursor();

    #ifndef WIN32
        #ifdef Q_WS_MAC
            if ( m_bootstrapAllowed )
                m_pageOffset += 4;
            else
            {
                setNumPages( numPages() - 3 );
                m_pageOffset += 7;
            }

            nextButtonClicked();
        #else
            m_pageOffset += 7;
            BaseWizard::nextButtonClicked();
        #endif

    #else
        BaseWizard::nextButtonClicked();
    #endif
}

/******************************************************************************
    pluginInfoDone
******************************************************************************/
void
ConfigWizard::pluginInfoDone(
    bool    error,
    QString errorMsg)
{
    if (error)
    {
        LOG(2, "Connection problem: " << errorMsg << "\n");

        LastMessageBox::critical(tr("Connection Problem"),
            tr("Last.fm couldn't connect to the Internet to download "
               "plugin information.\n\nError: %1").arg(errorMsg));

        BaseWizard::backButtonClicked();
    }
    else
    {
        // mAvailPlugins should now have been filled
        nextButtonClicked();
    }
}

/******************************************************************************
    pluginDownloadDone
******************************************************************************/
void
ConfigWizard::pluginDownloadDone(
    bool    error,
    QString errorMsg)
{
    if (error)
    {
        LOG(2, "Download Error: " << errorMsg << "\n");

        LastMessageBox::critical(tr("Download Error"),
            tr("Last.fm failed to download and install the selected "
               "plugins.\n\nError: %1").arg(errorMsg));

        BaseWizard::backButtonClicked();
    }
    else
    {
        // Leaving select plugin page, start download
        QStringList l = The::settings().allPlugins();
        qDebug() << l;

        bool gotITunes = false;
        for ( int x = 0; x < l.count(); x++ )
        {
            if ( l.at( x ).toLower().contains( "itunes" ) )
                gotITunes = true;
        }

        // Skip all iPod/bootstrap screens
        if ( !gotITunes || !m_bootstrapAllowed )
        {
            setNumPages( numPages() - 3 );
            m_pageOffset += 3;
        }

        /*
        else if ( m_bootstrapAllowed )
        {
            setNumPages( numPages() - 1 );
            m_pageOffset += 1;
            nextButtonClicked();
            return;
        }
        */

        BaseWizard::nextButtonClicked();
    }
}

/******************************************************************************
    reject
******************************************************************************/
void
ConfigWizard::reject()
{
    m_infoGetter.Cancel();
    m_updater.Cancel();

    QDialog::reject();
}

/******************************************************************************
    canUserRunWizard
******************************************************************************/
bool
ConfigWizard::canUserRunWizard()
{
#ifdef WIN32
    if ( ( QSysInfo::WindowsVersion & QSysInfo::WV_VISTA ) > 0 )
	{
		// On Vista, we always return true as UAC will request admin privileges
		// to perform the install
		return true;
	}
	else
	{
		return !CWinUtils::IsLimitedUser();
	}
#else
    return true;
#endif
}


void
ConfigWizard::bootstrapDone()
{
    m_didBootstrap = true;

    BaseWizard::nextButtonClicked();
}


int
ConfigWizard::exec()
{
    s_wizardRunning = true;
    activateWindow();

    #ifdef Q_WS_MAC
    if ( m_mode == MediaDevice )
        setMinimumSize( 520, 392 );
    #endif

    int r = QDialog::exec();
    s_wizardRunning = false;

    return r;
}
