/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include "containerutils.h"
#include "http.h"
#include "Settings.h"
#include "WebService.h"

#include <QDir>
#include <QTimer>

#ifdef WIN32
    #include <windows.h>
#endif

#undef LOG
#undef LOGL
#define LOG(x, y)
#define LOGL(x, y)

Http::Http( QObject* parent ) :
    QHttp( parent ),
    m_dataID( -1 ),
    m_nextId( 0 ),
    m_inProgress( false )
{
    init();
}


Http::Http( const QString& hostName, int port, QObject* parent ) :
    QHttp( parent ),
    m_dataID( -1 ),
    m_hostname( hostName ),
    m_nextId( 0 ),
    m_inProgress( false )
{
    init();
    setHost( hostName, port );
}


Http::~Http()
{
    // EJ: Added this warning just to prevent stupidity like forgetting to
    // create the Http object on the stack instead of the heap.
    if ( m_inProgress )
    {
        qDebug() << "Http object destroyed while in progress:\n" <<
            m_hostname + currentRequest().path();
    }
}


void
Http::init()
{
    QDir( cachePath() ).mkdir( cachePath() ); //rofl@Qt.com
    
    applyProxy();

    connect( this, SIGNAL(requestFinished( int, bool )), this, SLOT(dataFinished( int, bool )) );
    connect( this, SIGNAL(done( bool )), this, SLOT(requestDone( bool )) );
}


void
Http::applyProxy()
{
    //TODO really this should be determined by an settingsservice key like autoDetectProxy()
    
    if (The::settings().isUseProxy()) {
        setProxy( The::settings().getProxyHost(),
                  The::settings().getProxyPort(),
                  The::settings().getProxyUser(),
                  The::settings().getProxyPassword() );
    }
    else if (The::webService()->isAutoDetectedProxy()) {
        setProxy( The::webService()->proxyHost(),
                  The::webService()->proxyPort(),
                  QString(),
                  QString() );        
    }
}


void
Http::applyUserAgent( QHttpRequestHeader& header )
{
    // NEVER CHANGE THIS STRING!
    // martin says we can append stuff if we like, just never change the first bit

    QString userAgent = "Last.fm Client " + The::settings().version();
  #ifdef WIN32
    userAgent += " (Windows)";
  #elif defined (Q_WS_MAC)
    userAgent += " (OS X)";
  #elif defined (Q_WS_X11)
    userAgent += " (X11)";
  #endif

    header.setValue( "User-Agent", userAgent );
}


int
Http::get( const QString& path, bool useCache )
{
    applyProxy();
    m_buffer.clear();
    QString url = m_hostname + path;

    if ( useCache && haveCachedCopy( url ) )
    {
        // Using a singleshot so that we can return an ID and have
        // the operation proceed asynchronously.
        m_cacheStack.push( CachedRequestData( ++m_nextId, url ) );
        QTimer::singleShot( 0, this, SLOT( getFromCache() ) );
        return m_nextId;
    }

    QHttpRequestHeader header( "GET", path );
    header.setValue( "Host", m_hostname );
    applyUserAgent( header );

    m_dataID = request( header );
    if ( useCache )
        m_requestStack.insert( m_dataID, CachedRequestData( ++m_nextId, url ) );

    m_inProgress = true;
    
    return m_dataID;
}


int
Http::post( const QString& path, QIODevice* data )
{
    applyProxy();
    m_buffer.clear();

    m_dataID = QHttp::post( path, data );

    m_inProgress = true;

    return m_dataID;
}


int
Http::post( const QString& path, const QByteArray& data )
{
    applyProxy();
    m_buffer.clear();

    m_dataID = QHttp::post( path, data );

    m_inProgress = true;

    return m_dataID;
}


int
Http::request( const QHttpRequestHeader& header, QIODevice* data, QIODevice* to )
{
    QHttpRequestHeader h( header );
    applyProxy();
    applyUserAgent( h );

    m_buffer.clear();
    m_dataID = QHttp::request( h, data, to );

    m_inProgress = true;

    return m_dataID;
}


int
Http::request( const QHttpRequestHeader& header, const QByteArray& data, QIODevice* to, bool useCache )
{
    QHttpRequestHeader h( header );
    applyProxy();
    applyUserAgent( h );

    m_buffer.clear();
    QString key( data );
    if ( useCache && haveCachedCopy( key ) )
    {
        // Using a singleshot so that we can return an ID and have
        // the operation proceed asynchronously.
        m_cacheStack.push( CachedRequestData( ++m_nextId, key ) );
        QTimer::singleShot( 0, this, SLOT( getFromCache() ) );
        return m_nextId;
    }

    m_dataID = QHttp::request( h, data, to );

    m_inProgress = true;

    if ( useCache )
        m_requestStack.insert( m_dataID, CachedRequestData( ++m_nextId, key ) );

    return m_dataID;
}


void
Http::dataFinished( int id, bool error )
{
    if ( error )
        emit errorOccured( QHttp::error(), QHttp::errorString() );
    else
        if ( id == m_dataID )
        {
            checkBuffer();

            CachedRequestData req = m_requestStack.take( id );
            if ( !req.m_cacheKey.isEmpty() )
            {
                putCachedCopy( req.m_cacheKey, m_buffer );
            }

            emit dataAvailable( m_buffer );
        }
}


bool
Http::haveCachedCopy( QString url ) const
{
    return QFile::exists( pathToCachedCopy( url ) ) && QFileInfo( pathToCachedCopy( url ) ).isReadable();
}


void
Http::putCachedCopy( QString url, const QByteArray& data )
{
    #ifdef WIN32
        if ( url.size() > MAX_PATH )
        {
            url.chop( url.size() - MAX_PATH );
        }
    #endif

    QFile f( pathToCachedCopy( url ) );

    if ( !f.open( QIODevice::WriteOnly ) )
    {
        //LOGL( 1, "Failed to open file " << url << " for writing to cache" );
        return;
    }

    f.write( data );
}


void Http::getFromCache()
{
    Q_ASSERT( !m_cacheStack.isEmpty() );

    // Pick next request off cache stack
    CachedRequestData req = m_cacheStack.pop();

    QFile f( pathToCachedCopy( req.m_cacheKey ) );
    if ( !f.open( QIODevice::ReadOnly ) )
    {
        //LOGL( 1, "Failed to open cached file, returning error" );

        // TODO: emit error
        emit done( true );
        return;
    }

    // Keeping it as UTF-8, conversion will the done in parse function
    QByteArray result = f.readAll();
    emit dataAvailable( result );
    emit done( false );
}

void Http::abort()
{
    m_inProgress = false;
    QHttp::abort();
}

void Http::requestDone( bool /*error*/ )
{
    m_inProgress = false;
}
