/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include "loginwidget.h"
#include "containerutils.h"
#include "Settings.h"
#include "URLLabel.h"
#include "failedlogindialog.h"
#include "logger.h"
#include "LastMessageBox.h"

#include "WebService/Request.h"


// These are functions so that the translator will have had time to get
// initialised before tr is called.
inline static QString badUserErrorString()
{
    return LoginWidget::tr( "No Last.fm user with that username was found.\n"
        "Please enter the username you used to sign up at Last.fm." );
}

inline static QString badPassErrorString()
{
    return LoginWidget::tr( "The password isn't correct.\n"
        "Please enter the password you used to sign up at Last.fm." );
}

/******************************************************************************
    LoginWidget
******************************************************************************/
LoginWidget::LoginWidget( QWidget* parent,
                          Mode mode,
                          QString defaultUser )
        : QWidget( parent )
        , m_Mode( mode )
        , m_saveLowerPass( false )
{
    ui.setupUi( this );
    QPalette palette;

    ui.signUpLink->setText( tr( "Sign up for a Last.fm account" ) );
    ui.signUpLink->setURL( "http://www.last.fm/join" );
    ui.signUpLink->setFloat( true );
    ui.signUpLink->setUnderline( true );
    ui.signUpLink->setHighlightedColor( QColor( 0, 0, 0 ) );
    ui.signUpLink->setSelectedColor( QColor( 100, 100, 100 ) );

    ui.forgotLink->setText( tr( "Forgot your password?" ) );
    ui.forgotLink->setURL( "https://www.last.fm/settings/lostpassword/" );
    ui.forgotLink->setFloat( true );
    ui.forgotLink->setUnderline( true );
    ui.forgotLink->setHighlightedColor( QColor( 0, 0, 0 ) );
    ui.forgotLink->setSelectedColor( QColor( 100, 100, 100 ) );

    QString username;
    Qt::CheckState check = Qt::Checked;
    if ( !defaultUser.isEmpty() )
    {
        UserSettings &user = The::settings().user( defaultUser );
        username = user.username();
        check = user.rememberPass() ? Qt::Checked : Qt::Unchecked;
    }    

    switch ( mode )
    {
        // Fill drop down with all users
        case LOGIN:
        {
            LOGL( 3, "Launching login dialog mode LOGIN" );

            ui.stack->setCurrentIndex( 0 );

            // Populate dropdown
            int index = -1;
            QStringList allUsers = The::settings().allUsers();
            for ( int i = 0; i < allUsers.size(); ++i )
            {
                ui.userCombo->addItem( allUsers.at( i ) );
                if ( allUsers.at( i ) == username )
                {
                    index = i;
                    break;
                }
            }

            ui.userCombo->setCurrentIndex( index );
            ui.rememberCheck->setCheckState( check );
            if ( index != -1 )
            {
                ui.passwordEdit->setFocus();
            }
            else
            {
                ui.userCombo->setFocus();
            }

        }
        break;

        // Display empty line edit
        case ADD_USER:
        {
            LOGL( 3, "Launching login dialog mode ADD_USER" );

            ui.stack->setCurrentIndex( 1 );

            // AARGH, none of these work
            ui.userEdit->setFocus();
            //ui.userCombo->setFocus();
            //ui.stack->setFocus();

            ui.rememberCheck->setCheckState( Qt::Checked );
        }
        break;

        // Display only current user in edit
        case CHANGE_PASS:
        {
            LOGL( 3, "Launching login dialog mode CHANGE_PASS" );

            ui.stack->setCurrentIndex( 1 );
            ui.userEdit->setText( username );
            ui.userEdit->setEnabled( false ); // looks ugly
            ui.passwordEdit->setFocus();
            ui.rememberCheck->setCheckState( check );
        }
        break;

    } // end switch

    connect( ui.userCombo, SIGNAL( currentIndexChanged( QString ) ),
             this,         SLOT  ( userComboChanged( QString ) ) );

}

/******************************************************************************
    verify
******************************************************************************/
void
LoginWidget::verify()
{
    LOGL(4, "");

    QString username = m_Mode == LOGIN 
            ? ui.userCombo->currentText()
            : ui.userEdit->text();

    if (!The::settings().isFirstRun() &&
            The::settings().isExistingUser( username ) &&
            The::settings().user( username ).rememberPass())
    {
        // This happens when flipping the user in the drop-down to one
        // with remembered pass and we fill in the password for them.
        emit verifyResult( true, false ); //TODO mxcl check old code FIXME
        return;
    }

    QString password = ui.passwordEdit->text();
    QString passwordLower = password.toLower();
    QString pwMD5 = MD5Digest( password.toUtf8() );
    QString pwMD5Lower = MD5Digest( passwordLower.toUtf8() );

    VerifyUserRequest *verify = new VerifyUserRequest;
    verify->setUsername( username );
    verify->setPasswordMd5( pwMD5 );
    verify->setPasswordMd5Lower( pwMD5Lower );
    
    connect( verify, SIGNAL(result( Request* )), SLOT(verifyResult( Request* )), Qt::QueuedConnection );
    
    verify->start();
}

/******************************************************************************
    save
******************************************************************************/
void
LoginWidget::save()
{
    QString username = m_Mode == LOGIN 
            ? ui.userCombo->currentText()
            : ui.userEdit->text();

    // This allows a login screen to be used as an add user screen transparently
    if ( !The::settings().isExistingUser( username ) )
    {
        m_Mode = ADD_USER;
    }

    // This creates a new user if the user is not found
    UserSettings& userSettings = The::settings().user( username );

    QString pw = m_saveLowerPass ? ui.passwordEdit->text().toLower() :
                                   ui.passwordEdit->text();
    userSettings.setPassword( pw );
    userSettings.setRememberPass( ui.rememberCheck->checkState() == Qt::Checked );
    
    if ( m_Mode == ADD_USER )
    {
        userSettings.setIcon( The::settings().getFreeColour() );
    }

    userSettings.save();

    The::settings().setCurrentUsername(username);
    The::settings().save(true, false);
}

/******************************************************************************
    verifyResult
******************************************************************************/
void
LoginWidget::verifyResult( Request *request )
{
    VerifyUserRequest *verify = static_cast<VerifyUserRequest*>(request);

    UserAuthCode result = verify->userAuthCode();

    LOGL( 4, "result: " << result );

    bool bootstrap = (verify->bootStrapCode() == BOOTSTRAP_ALLOWED);

    if (result == AUTH_OK)
    {
        m_saveLowerPass = false;
        emit verifyResult( true, bootstrap );
    }
    else if ( result == AUTH_OK_LOWER )
    {
        m_saveLowerPass = true;
        emit verifyResult( true, bootstrap );
    }
    else
    {
        QString msg;

        QApplication::restoreOverrideCursor();

        if ( result == AUTH_BADUSER )
        {
            LastMessageBox::critical( tr( "Login Failed" ), badUserErrorString() );
        }
        else if ( result == AUTH_BADPASS )
        {
            LastMessageBox::critical( tr( "Login Failed" ), badPassErrorString() );
        }
        else if ( result == AUTH_ERROR )
        {
            // If we're adding a user or changing password, we must verify
            // against the internet so we fire up FailedLoginDialog.
            // If it's just a login, we can fall back on verifying against
            // the password in the registry instead.
            if ( m_Mode == LOGIN )
            {
                LOGL( 3, "Couldn't authenticate via web, falling back on local" );

                if ( verifyLocally() )
                {
                    emit verifyResult( true, false );
                    return;
                }
                else
                {
                    LastMessageBox::critical( tr("Login Failed"), badPassErrorString() );
                }
            }
            else
            {
                LOGL( 4, "Got an AUTH_ERROR, will launch FailedDialog" );

                FailedLoginDialog( this ).exec();
            }
        }

        emit verifyResult( false, false );
    }

}

/******************************************************************************
    verifyLocally
******************************************************************************/
bool
LoginWidget::verifyLocally()
{
    QString username = m_Mode == LOGIN 
            ? ui.userCombo->currentText()
            : ui.userEdit->text();
    QString enteredPass = ui.passwordEdit->text();
    enteredPass = MD5Digest(enteredPass.toUtf8());

    if (!The::settings().isExistingUser( username ))
    {
        return false;
    }

    QString storedPass = The::settings().user( username ).password();

    return enteredPass == storedPass;
}

/******************************************************************************
    userComboChanged
******************************************************************************/
void
LoginWidget::userComboChanged( QString username )
{
    if (The::settings().user(username).rememberPass())
    {
        ui.rememberCheck->setCheckState( Qt::Checked );
        ui.passwordEdit->setText( "********" );
    }
    else
    {
        ui.rememberCheck->setCheckState(Qt::Unchecked);
        ui.passwordEdit->clear();
    }
}
