/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Max Howell, Last.fm Ltd <max@last.fm>                              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/
     
#include "container.h"
#include "last.fm.h"
#include "http.h"
#include "Radio.h"
#include "Settings.h"
#include "SideBarModel.h"
#include "WebService.h"
#include "WebService/Request.h"
#include <QtGui>
#include <QtCore>


using namespace SideBar;

      
SideBarModel::SideBarModel() : m_track_is_playing( false )
{
    // sets up m_change_station_vars
    onTrackStartedOrStopped( MetaData() );
    
    connect( &Container::instance(), SIGNAL(newSong( MetaData )), SLOT(onTrackStartedOrStopped( MetaData )) );
    connect( The::webService(), SIGNAL(result( Request* )), SLOT(onResult( Request* )) );
    connect( &The::radio(), SIGNAL(stateChanged( RadioState )), SLOT(onRadioStateChanged( RadioState )) );
}


void
SideBarModel::addRecentlyPlayedTrack( Track track )
{
    QModelIndex const parent = index( SideBar::RecentlyPlayed, 0 );

    beginInsertRows( parent, 0, 1 );
    m_played.prepend( track );
    endInsertRows();
}


QVariant
SideBarModel::data( const QModelIndex &index, int role ) const
{
    if (!index.isValid())
        return QVariant();

    SideBarItem i( index );

    if (role == Qt::DisplayRole)
        switch (i.type()) {
            case MyProfile:           return The::currentUsername();
            case StartAStation:       return m_change_station_text;
            case NowPlaying:          return tr("Now Playing");
            case MyRecommendations:   return tr("My Recommendations");
            case PersonalRadio:       return tr("My Radio Station");
            case LovedTracksRadio:    return tr("My Loved Tracks");
            case NeighbourhoodRadio:  return tr("My Neighbourhood");
            case RecentlyPlayed:      return tr("Recently Played");
            case RecentlyLoved:       return tr("Recently Loved");
            case RecentlyBanned:      return tr("Recently Banned");
            case MyTags:              return tr("My Tags");
            case Friends:             return tr("Friends");
            case Neighbours:          return tr("Neighbours");
            case History:             return tr("History");

            case RecentlyPlayedTrack: return m_played.value( index.row() );
            case RecentlyLovedTrack:  return m_loved.value( index.row() );
            case RecentlyBannedTrack: return m_banned.value( index.row() );
            case MyTagsChild:         return m_tags.value( index.row() );
            case FriendsChild:        return m_friends.value( index.row() );
            case NeighboursChild:     return m_neighbours.value( index.row() );
            case HistoryStation:      return m_history.value( index.row() ).name();
            
            default: break; //gcc warning--            
        }
        
    if (role == Qt::ToolTipRole)
        switch ((int)i.type()) {
            case RecentlyPlayedTrack:
            case RecentlyLovedTrack:
            case RecentlyBannedTrack:
            case MyTagsChild:
            case FriendsChild:
            case NeighboursChild:
            case HistoryStation:
                return data( index, Qt::DisplayRole );
                //return data( index, UrlRole );
        }
    
    if (role == Qt::DecorationRole)
        switch (i.type()) {
            case MyProfile:           return m_avatar;
            case StartAStation:       return LastFm::icon( "icon_radio" );
            case NowPlaying:          return QIcon( ":/SideBarNowPlaying.png" );
            case MyRecommendations:   return LastFm::icon( "recommended_radio" );
            case PersonalRadio:       return LastFm::icon( "personal_radio" );
            case LovedTracksRadio:    return LastFm::icon( "loved_radio" );
            case NeighbourhoodRadio:  return LastFm::icon( "neighbour_radio" );
            case RecentlyPlayed:      return LastFm::icon( "recent_tracks" );
            case RecentlyLoved:       return LastFm::icon( "recently_loved" );
            case RecentlyBanned:      return LastFm::icon( "recently_banned" );
            case MyTags:              return LastFm::icon( "my_tags" );
            case Friends:             return LastFm::icon( "my_friends" );
            case Neighbours:          return LastFm::icon( "my_neighbours" );
            #ifdef Q_WS_MAC
            case History:             return LastFm::icon( "history32" );
            #else
            case History:             return LastFm::icon( "history16" );
            #endif
            
            case RecentlyPlayedTrack: //FALL THROUGH
            case RecentlyLovedTrack:  //FALL THROUGH
            case RecentlyBannedTrack: return LastFm::icon( "icon_track" );
            case MyTagsChild:         return LastFm::icon( "icon_tag" );
            case FriendsChild:        return LastFm::icon( "user_blue" );
            case NeighboursChild:     return LastFm::icon( "user_purple" );
            case HistoryStation:      return LastFm::icon( "icon_radio" );
            
            default: break; //gcc warning--
        }
    
    QString const encoded_username = QUrl::toPercentEncoding( The::currentUsername() );
    
    if (role == StationUrlRole)
        switch (i.type()) 
        {
            #define encode( x ) QUrl::toPercentEncoding( x.value( index.row() ) )                

            case MyProfile:
            case MyRecommendations:   return "lastfm://user/" + encoded_username + "/recommended";
            case PersonalRadio:       return "lastfm://user/" + encoded_username + "/personal";
            case LovedTracksRadio:    return "lastfm://user/" + encoded_username + "/loved";
            case NeighbourhoodRadio:  return "lastfm://user/" + encoded_username + "/neighbours";
            
            case RecentlyPlayedTrack:
            case RecentlyLovedTrack: 
            case RecentlyBannedTrack:
                //NOTE we can't know this without calling the TrackToId WebService request :(
                return QVariant();
                
            case MyTagsChild:
            {
                QString tagToEncode = m_tags.value( index.row() );
                return "lastfm://globaltags/" + QUrl::toPercentEncoding( 
                    tagToEncode.remove( tagToEncode.lastIndexOf( " (" ), tagToEncode.length() ) );
            }
            case FriendsChild:        return "lastfm://user/" + encode( m_friends ) + "/personal";
            case Neighbours:          return "lastfm://user/" + encode( m_neighbours ) + "/personal";
            case HistoryStation:      return m_history.value( index.row() ).url();
        
            default: break; //gcc warning--
            
            #undef encode
        }
    
    if (role == TrackRole) {
        Track track;
        switch (i.type()) {
            case RecentlyPlayedTrack: track = m_played.value( index.row() ); break;
            case RecentlyLovedTrack:  track = m_loved.value( index.row() ); break;
            case RecentlyBannedTrack: track = m_banned.value( index.row() ); break;
            
            default: break; //gcc warning--
        }
        
        if (!track.isEmpty()) {
            QVariantMap map;
            map["artist"] = track.artist();
            map["title"] = track.title();
            return map;
        }
    }
	
  #ifdef Q_WS_MAC   
	if (role == Qt::FontRole)
	{
		QFont font;
		font.setPixelSize( 11 );
		return font;
	}
  #endif
    
    if (role == UrlRole)
        if (i.type() == MyProfile)
            return "http://www.last.fm/user/" + encoded_username;
        else
            return data( index, StationUrlRole );
        
    return QVariant();
}


Qt::ItemFlags
SideBarModel::flags( const QModelIndex& index ) const
{
    SideBarItem i( index );
    Qt::ItemFlags flags = Qt::ItemIsEnabled | Qt::ItemIsDropEnabled;

    switch (i.type())
    {
        case MyProfile:
        case StartAStation:
        case NowPlaying:
//        case MyRecommendations:
//        case PersonalRadio:
//        case LovedTracksRadio:
//        case NeighbourhoodRadio:
        case RecentlyPlayedTrack:
        case RecentlyLovedTrack:
        case RecentlyBannedTrack:
        case MyTagsChild:
        case FriendsChild:
        case NeighboursChild:
        case HistoryStation:
            flags |= Qt::ItemIsSelectable;
            break;
            
        default: break; //gcc warning--
    }
    
    if (i.type() == NowPlaying && !m_track_is_playing)
        flags &= ~(Qt::ItemIsEnabled | Qt::ItemIsSelectable); //disable    
    
    switch (SideBarItem(i).classification())
    {
        case SideBarItem::User:
        case SideBarItem::Track:
        case SideBarItem::Tag:
            flags |= Qt::ItemIsDragEnabled;
            break;
            
        default: 
            break; //gcc warning--
    }
    
    return flags;
}

     
QModelIndex
SideBarModel::index( int row, int column, const QModelIndex& parent ) const
{
    return createIndex( row, column, parent.isValid() ? parent.row() : 0 );
}


QModelIndex
SideBarModel::parent( const QModelIndex& i ) const
{
    switch (i.internalId())
    {
        case 0:  return QModelIndex();
        default: return index( i.internalId(), 0 );
    }
}


int
SideBarModel::rowCount( const QModelIndex &parent ) const
{
    if (!parent.isValid())
        return SideBar::RowCount;

    // no second levels pls
    if (parent.internalId() > 0)
        return 0;

    switch (parent.row())
    {
        case RecentlyPlayed:
            return m_played.count();
        case RecentlyLoved:
            return m_loved.count();
        case RecentlyBanned:
            return m_banned.count();
        case MyTags:
            return m_tags.count();
        case Friends:
            return m_friends.count();
        case Neighbours:
            return m_neighbours.count();
        case History:
            return m_history.count();
    
        default:
            return 0;
    }
}


QStringList
SideBarModel::mimeTypes() const
{
     return QStringList() << "item/tag"
                          << "item/user"
                          << "item/artist"
                          << "item/track"
                          << "item/station"
                          ;
}


QMimeData*
SideBarModel::mimeData( const QModelIndexList& indexes ) const
{
    Q_DEBUG_BLOCK;

    QModelIndex i = indexes.value( 0 );
    SideBarItem item( i );
    QMimeData *m = new QMimeData;
    
    switch (item.type())
    {
        case FriendsChild:
        case NeighboursChild:
        case MyProfile:
            m->setData( "item/user", i.data().toByteArray() );
            break;
        
        case MyRecommendations:
        case PersonalRadio:
        case LovedTracksRadio:
        case NeighbourhoodRadio:
        case HistoryStation:
            m->setData( "item/station", i.data( SideBar::StationUrlRole ).toByteArray() );
            break;
        
        case MyTags:
        case Friends:
        case Neighbours:
        case History:
        case RecentlyPlayed:
        case RecentlyLoved:
        case RecentlyBanned:
            return 0;
            
        case MyTagsChild:
            m->setData( "item/tag", i.data().toString().remove( QRegExp(" \\(\\d*\\)$") ).toUtf8() );
            break;
            
        case RecentlyBannedTrack:
        case RecentlyPlayedTrack:
        case RecentlyLovedTrack:
        {
            Track track = item.track();
            m->setData( "item/track", track.title().toUtf8() );
            m->setData( "item/artist", track.artist().toUtf8() );
            break;
        }
        
        default:
            break; //gcc warning--
    }
    
    return m;
}


template <class T> void
SideBarModel::changeData( int row, T& old_data, const T& new_data )
{
    QModelIndex const parent = index( row, 0 );
    int const n = old_data.count() - new_data.count();
    if (n > 0) beginRemoveRows( parent, 0, n );
    if (n < 0) beginInsertRows( parent, 0, -n );
    old_data = new_data;
    if (n > 0) endRemoveRows();
    if (n < 0) endInsertRows();
    emit dataChanged( index( 0, 0, parent ), index( new_data.count() - 1, 0, parent ) );
}


void
SideBarModel::onResult( Request *r )
{
    switch (r->type())
    {
        case TypeHandshake:
        {
            //TODO don't do again if already handshook with this user!
            //TODO do we even need this anymore?
            //connect( &The::settings().currentUser(), SIGNAL(historyChanged()), SLOT(updateHistory()) );
    
            m_friends.clear();
            m_neighbours.clear();
            m_banned.clear();
            m_loved.clear();
            m_played.clear();
            m_tags.clear();
            
            m_isSubscriber = static_cast<Handshake*>(r)->isSubscriber();
        
            (new FriendsRequest)->start();
            (new NeighboursRequest)->start();
            (new UserTagsRequest)->start();
            (new RecentTracksRequest)->start();
            (new RecentlyLovedTracksRequest)->start();
        
            UserPicturesRequest::fetchCurrentUser();
            
          #ifndef HIDE_RADIO
            (new RecentlyBannedTracksRequest)->start();
            
            //TODO should do this when userswitched
            connect( &The::currentUser(), SIGNAL(historyChanged()), SLOT(updateHistory()) );
            updateHistory();
          #endif
          
            break;
        }
    
        #define CASE( T ) case Type##T: { \
            T##Request *request = dynamic_cast<T##Request*>(r); \
            Q_ASSERT( request );

        #define break } break

        CASE( RecentTracks )
            changeData( SideBar::RecentlyPlayed, m_played, request->tracks() );
            break;
        
        CASE( RecentlyLovedTracks )
            changeData( SideBar::RecentlyLoved, m_loved, request->tracks() );
            break;
        
        CASE( RecentlyBannedTracks )
            changeData( SideBar::RecentlyBanned, m_banned, request->tracks() );
            break;
                    
        CASE( UserTags )
            m_tags.clear();
            sortTags( (WeightedStringList)request->tags(), (SideBar::SortOrder) The::settings().currentUser().sideBarTagsSortOrder() );
            break;

        CASE( Friends )
            changeData( SideBar::Friends, m_friends, request->usernames() );
            break;
        
        CASE( Neighbours )
            sortNeighbours( request->usernames(), (SideBar::SortOrder) The::settings().currentUser().sideBarNeighbourSortOrder() );
            break;
    
    //////        
        CASE( UserPictures )
            downloadAvatar( request->urls().values().value( 0 ) );
            break;

    //////
        CASE( Ban )
            beginInsertRows( index( SideBar::RecentlyBanned, 0 ), 0, 0 );
            m_banned.prepend( request->track() );
            endInsertRows();
            break;
            
        CASE( Love )
            beginInsertRows( index( SideBar::RecentlyLoved, 0 ), 0, 0 );
            m_loved.prepend( request->track() );
            endInsertRows();
            break;
        
        CASE( UnBan )
            int const n = m_banned.indexOf( request->track() );
            if (n != -1) {
                beginRemoveRows( index( SideBar::RecentlyBanned, 0 ), n, n );
                m_banned.removeAt( n );
                endRemoveRows();
            }
            else
                (new RecentlyBannedTracksRequest)->start();
            break;
            
        CASE( UnLove )
            int n = m_loved.indexOf( request->track() );
            if (n != -1) {
                beginRemoveRows( index( SideBar::RecentlyLoved, 0 ), n, n );
                m_loved.removeAt( n );
                endRemoveRows();
            }
            else
                (new RecentlyLovedTracksRequest)->start();
            break;
            
        CASE( UnListen )
            int n = m_played.indexOf( request->track() );
            if (n != -1) {
                beginRemoveRows( index( SideBar::RecentlyPlayed, 0 ), n, n );
                m_played.removeAt( n );
                endRemoveRows();
            }
            else 
                (new RecentTracksRequest)->start();
            break;
    
        #undef CASE
        #undef break
        
        default:
            break; //gcc warnings--
    }
}


void
SideBarModel::updateHistory()
{
  #ifndef HIDE_RADIO
    changeData( SideBar::History, m_history, The::currentUser().recentStations() );
  #endif
}


void
SideBarModel::sortTags( WeightedStringList tagsToSort, SideBar::SortOrder sortOrder )
{
    if ( m_tags.count() == 0 ) 
    {
        for ( int i = 0; i < tagsToSort.count(); i++ )
            tagsToSort[i] += " (" + QVariant( tagsToSort.at( i ).weighting() ).toString() + ")";
    }
    
    if ( sortOrder == SideBar::MostWeightOrder ) 
    {
        tagsToSort.sortWeightingDescending();
    }
    else if ( sortOrder == SideBar::AscendingOrder )
    {
        tagsToSort.sortAscending();
    }
    else if ( sortOrder == SideBar::DescendingOrder )
    {
        tagsToSort.sortDescending();      
    }
    
    changeData( SideBar::MyTags, m_tags, tagsToSort );
}


void
SideBarModel::sortTags( SideBar::SortOrder sortOrder )
{
    sortTags( m_tags, sortOrder );
}


bool caseInsensitiveLessThan(const QString &s1, const QString &s2)
{
    return s1.toLower() < s2.toLower();
}


void
SideBarModel::sortNeighbours( QStringList neighboursToSort, SideBar::SortOrder sortOrder )
{    
    if ( m_neighbours.count() == 0 )
    {
        m_neighboursBySimilarity = neighboursToSort;
    }
    
    if ( sortOrder == SideBar::MostWeightOrder ) 
    {
        neighboursToSort = m_neighboursBySimilarity;
    }
    else if ( sortOrder == SideBar::AscendingOrder )
    {
        qSort( neighboursToSort.begin(), neighboursToSort.end(), caseInsensitiveLessThan );
    }
    else if ( sortOrder == SideBar::DescendingOrder )
    {
        qSort( neighboursToSort.begin(), neighboursToSort.end(), qGreater<QString>() );        
    }
    
    changeData( SideBar::Neighbours, m_neighbours, neighboursToSort );
}


void
SideBarModel::sortNeighbours( SideBar::SortOrder sortOrder )
{
    sortNeighbours( m_neighbours, sortOrder );
}


void
SideBarModel::downloadAvatar( const QUrl& url )
{
    Http* http = new Http();
    http->setHost( url.host() );
    http->get( url.encodedQuery().isEmpty()
            ? url.path()
            : url.path() + "?" + url.encodedQuery(), 
            false );

    connect( http, SIGNAL(dataAvailable( QByteArray )), SLOT(onAvatarDownloaded( QByteArray )) );
}


void
SideBarModel::onAvatarDownloaded( QByteArray const buffer )
{
    m_avatar.loadFromData( buffer );

    if (!m_avatar.isNull()) {
        #ifdef LINUX
        int m = 38;
        #else
        int m = 30;
        #endif
        m_avatar = m_avatar.scaled( m, m, Qt::KeepAspectRatio, Qt::SmoothTransformation );
        emitRowChanged( SideBar::MyProfile );
    }
    else
        LOGL( 1, "Loading of avatar image from QByteArray failed." );
    
    sender()->deleteLater();
}


void
SideBarModel::onTrackStartedOrStopped( const MetaData& track )
{
    // confusingly we are told the track is empty during station transitions
    // don't confuse the user though
    if (track.isEmpty() && The::container().isTuningInOrPlaying())
        return;

    m_track_is_playing = !track.isEmpty();

    if (track.source() != TrackInfo::Radio || track.isEmpty())
        m_change_station_text = tr("Start a Station");
    else
        m_change_station_text = tr("Change Station");

    QModelIndex const i = index( StartAStation, 0 );
    emit dataChanged( i, i.sibling( i.row() + 1, 0 ) );
}


void
SideBarModel::onRadioStateChanged( RadioState state )
{
    switch (state)
    {
        case State_ChangingStation:
        case State_FetchingPlaylist:
        case State_FetchingStream:
            if (!m_track_is_playing) {
                m_track_is_playing = true;
                QModelIndex const i = index( NowPlaying, 0 );
                emit dataChanged( i, i );
            }
            break;
        
        default:
            break;
    }
}


void
SideBarModel::emitRowChanged( int parent_row, int child_row /*= -1*/ )
{
    QModelIndex parent;
    if (child_row != -1)
        parent = index( parent_row, 0 );
    
    QModelIndex i = index( child_row, 0, parent );
        
    emit dataChanged( i, i );
}



/////////////////
// SideBarItem //
/////////////////

SideBarItem::SideBarItem( const QModelIndex& i ) :
        m_classification( ClassNone ),
        m_type( TypeUnknown ),
        m_index( i )
{
    if (i.parent().isValid())
        switch (i.parent().row())
        {
            case RecentlyPlayed: m_type = RecentlyPlayedTrack; break;
            case RecentlyLoved:  m_type = RecentlyLovedTrack; break;
            case RecentlyBanned: m_type = RecentlyBannedTrack; break;
            case MyTags:         m_type = MyTagsChild; break;
            case Friends:        m_type = FriendsChild; break;
            case Neighbours:     m_type = NeighboursChild; break;
            case History:        m_type = HistoryStation; break;
            
            default: break; //gcc warning--            
        }
    else
        m_type = (SideBar::Type)i.row();

//////
    switch (m_type)
    {
        case RecentlyPlayedTrack:
        case RecentlyLovedTrack:
        case RecentlyBannedTrack:
            m_classification = Track;
            break;
            
        case MyTagsChild:
            m_classification = Tag;
            break;
            
        case MyProfile:
        case FriendsChild:
        case NeighboursChild:
            m_classification = User;
            break;
        
        case MyRecommendations:
        case PersonalRadio:
        case LovedTracksRadio:
        case NeighbourhoodRadio:
        case HistoryStation:
            m_classification = Station;
            break;
            
        default: break; //gcc warning--            
    }
};


QIcon
SideBarItem::icon() const
{
    return m_index.data( Qt::DecorationRole ).value<QIcon>();
}


Track
SideBarItem::track() const
{
    QVariantMap map = m_index.data( SideBar::TrackRole ).toMap();

    ::Track track;
    track.setArtist( map["artist"].toString() );
    track.setTitle( map["title"].toString() );
    return track;
}


Station
SideBarItem::station() const
{
    ::Station station;
    station.setName( m_index.data().toString() );
    station.setUrl( StationUrl( m_index.data( StationUrlRole ).toString() ) );
    return station;
}
