// -*- coding:iso-8859-15 -*-
/** \file server/drivers/imonlcd_font.h
 * This is a definition of a font for use with the iMON LCD.
 * 
 * The iMON LCD doesn't have a "text mode" - everthing is pixel-based.
 * So we need to define our own font, basically.
 */

/*-
 * Created March 2009 by Jonathan Kyler for inclusion in lcdproc
 * Structure based on codeka's imonlcd patch by Dean Harding
 * Font based on Toshiba T6963 based LCD display driver font by Manuel Stahl
 *
 * Copyright (c) 2009, Jonathan Kyler <jkyler (at) users (dot) sourceforge (dot) net>
 *
 * This file is released under the GNU General Public License. Refer to the
 * COPYING file distributed with this package.
 */

#ifndef IMONLCD_EXTENDED_CHARACTERS
#define IMONLCD_EXTENDED_CHARACTERS

#define IMONLCD_FONT_FULL_HEART		3
#define IMONLCD_FONT_SMALL_HEART	4
#define IMONLCD_FONT_FULL_BLOCK		141
#define IMONLCD_FONT_LEFT_ARROW		155
#define IMONLCD_FONT_UP_ARROW		156
#define IMONLCD_FONT_RIGHT_ARROW	157
#define IMONLCD_FONT_DOWN_ARROW		158

#define IMONLCD_FONT_STOP		19
#define IMONLCD_FONT_PLAY		16
#define IMONLCD_FONT_PAUSE		18
#define IMONLCD_FONT_REVERSE		17
#define IMONLCD_FONT_FWD		21
#define IMONLCD_FONT_RWD		22
#define IMONLCD_FONT_RECORD		15

#define IMONLCD_FONT_START_VBAR_FULL	23
#define IMONLCD_FONT_START_VBAR_NARROW	127
#define IMONLCD_FONT_START_HBAR_NARROW	135


/**
 * This structure holds the definition of the font.
 *
 * The characters we define here are 6x8 pixels in size, each byte in the
 * 'pixels' array represents one column of pixels. The most significant bit
 * is the top row, the least significant bit is the bottom row.
 */
typedef struct _imon_font {
    unsigned char pixels[LCD_DEFAULT_CELL_WIDTH];
} imon_font;

/*
 * Based on ISO 8859-15: http://en.wikipedia.org/wiki/ISO_8859-15
 *  Characters 0-14 are from the t6963 font, which is based on the
 *   CP437 character set.  The diamond is replaced with a small heart
 *   for use in a heartbeat display.
 *  Characters 15-22 provide common play, record, stop, etc icons.
 *  Characters 23-29 provide a vertical progress bar display - using all
 *   6 pixels in width
 *  Characters 30 & 31 are currently blank
 *  Characters 32-126 match ISO_8859-15 characters
 *  Characters 127-134 provide a vertical progress bar display, but only
 *   use 4 of the 6 characters in width
 *  Characters 135-140 provide a horizontal progress bar, using only
 *   4 of the 8 characters in height
 *  Character 141 is a full block (all pixels used)
 *  Characters 142-152 provide some greek symbols
 *  Characters 153 & 154 make up a mail envelope icon
 *  Characters 155-159 are arrows
 *  Character 160 is currently blank
 *  Characters 161-255 match ISO_8859-15 characters
 */
static imon_font font[] = {
	{ {0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } }, //   0 \u0020 space
	{ {0x00, 0x7c, 0xa2, 0x8a, 0xa2, 0x7c } }, //   1 \u263a light smiley face
	{ {0x00, 0x7c, 0xd6, 0xf6, 0xd6, 0x7c } }, //   2 \u263b dark smiley face
	{ {0x00, 0x38, 0x7c, 0x3e, 0x7c, 0x38 } }, //   3 \u2665 full heart
	{ {0x00, 0x00, 0x38, 0x1c, 0x38, 0x00 } }, //   4        small heart
	{ {0x00, 0x0c, 0x6c, 0xfe, 0x6c, 0x0c } }, //   5 \u2660 club
	{ {0x00, 0x18, 0x3a, 0x7e, 0x3a, 0x18 } }, //   6 \u2660 spade
	{ {0x00, 0x00, 0x18, 0x18, 0x00, 0x00 } }, //   7 \u2022 bullet
	{ {0xff, 0xff, 0xe7, 0xe7, 0xff, 0xff } }, //   8 \u25d8 big rectangle
	{ {0x00, 0x3c, 0x24, 0x24, 0x3c, 0x00 } }, //   9 \u25cb small rectangle
	{ {0xff, 0xc3, 0xdb, 0xdb, 0xc3, 0xff } }, //  10 \u25d9 filled rectangle
	{ {0x00, 0x0c, 0x12, 0x52, 0x6c, 0x70 } }, //  11 \u2642 man symbol
	{ {0x00, 0x60, 0x94, 0x9e, 0x94, 0x60 } }, //  12 \u2640 woman symbol
	{ {0x00, 0x06, 0x0e, 0xfc, 0x40, 0x20 } }, //  13 \u266a musical note
	{ {0x00, 0x06, 0x7e, 0x50, 0xac, 0xfc } }, //  14 \u266b double music note
	{ {0x00, 0x18, 0x24, 0x24, 0x24, 0x18 } }, //  15 \u25cf record
	{ {0x00, 0x00, 0xfe, 0x7c, 0x38, 0x10 } }, //  16 \u25b6 play
	{ {0x00, 0x10, 0x38, 0x7c, 0xfe, 0x00 } }, //  17 \u25c0 play backwards
	{ {0x00, 0x7e, 0x7e, 0x00, 0x7e, 0x7e } }, //  18        pause
	{ {0x00, 0x3c, 0x3c, 0x3c, 0x3c, 0x00 } }, //  19 \u25fe stop
	{ {0x00, 0x0a, 0x3a, 0xfa, 0x3a, 0x0a } }, //  20 \u23cf eject
	{ {0x00, 0xfe, 0x7c, 0x38, 0x10, 0xfe } }, //  21        fwd
	{ {0x00, 0xfe, 0x10, 0x38, 0x7c, 0xfe } }, //  22        rev
	{ {0x01, 0x01, 0x01, 0x01, 0x01, 0x01 } }, //  23 \u2581 lower 1/8 block (full)
	{ {0x03, 0x03, 0x03, 0x03, 0x03, 0x03 } }, //  24 \u2582 lower 1/4 block (full)
	{ {0x07, 0x07, 0x07, 0x07, 0x07, 0x07 } }, //  25 \u2583 lower 3/8 block (full)
	{ {0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f } }, //  26 \u2584 lower 1/2 block (full)
	{ {0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f } }, //  27 \u2585 lower 5/8 block (full)
	{ {0x3f, 0x3f, 0x3f, 0x3f, 0x3f, 0x3f } }, //  28 \u2586 lower 3/4 block (full)
	{ {0x7f, 0x7f, 0x7f, 0x7f, 0x7f, 0x7f } }, //  29 \u2587 lower 7/8 block (full)
	{ {0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } }, //  30
	{ {0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } }, //  31
	{ {0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } }, //  32 \u0020 space
	{ {0x00, 0x00, 0x60, 0xfa, 0x60, 0x00 } }, //  33 \u0021 !
	{ {0x00, 0xe0, 0xc0, 0x00, 0xe0, 0xc0 } }, //  34 \u0022 "
	{ {0x00, 0x24, 0x7e, 0x24, 0x7e, 0x24 } }, //  35 \u0023 #
	{ {0x00, 0x24, 0x54, 0xd6, 0x48, 0x00 } }, //  36 \u0024 $
	{ {0x00, 0xc6, 0xc8, 0x10, 0x26, 0xc6 } }, //  37 \u0025 %
	{ {0x00, 0x6c, 0x92, 0x6a, 0x04, 0x0a } }, //  38 \u0026 &
	{ {0x00, 0x00, 0xe0, 0xc0, 0x00, 0x00 } }, //  39 \u0027 '
	{ {0x00, 0x00, 0x7c, 0x82, 0x00, 0x00 } }, //  40 \u0028 (
	{ {0x00, 0x00, 0x82, 0x7c, 0x00, 0x00 } }, //  41 \u0029 )
	{ {0x00, 0x10, 0x7c, 0x38, 0x7c, 0x10 } }, //  42 \u002a *
	{ {0x00, 0x10, 0x10, 0x7c, 0x10, 0x10 } }, //  43 \u002b +
	{ {0x00, 0x00, 0x07, 0x06, 0x00, 0x00 } }, //  44 \u002c ,
	{ {0x00, 0x10, 0x10, 0x10, 0x10, 0x10 } }, //  45 \u002d -
	{ {0x00, 0x00, 0x06, 0x06, 0x00, 0x00 } }, //  46 \u002e .
	{ {0x00, 0x04, 0x08, 0x10, 0x20, 0x40 } }, //  47 \u002f /
	{ {0x00, 0x7c, 0x8a, 0x92, 0xa2, 0x7c } }, //  48 \u0030 0
	{ {0x00, 0x00, 0x42, 0xfe, 0x02, 0x00 } }, //  49 \u0031 1
	{ {0x00, 0x46, 0x8a, 0x92, 0x92, 0x62 } }, //  50 \u0032 2
	{ {0x00, 0x44, 0x92, 0x92, 0x92, 0x6c } }, //  51 \u0033 3
	{ {0x00, 0x18, 0x28, 0x48, 0xfe, 0x08 } }, //  52 \u0034 4
	{ {0x00, 0xf4, 0x92, 0x92, 0x92, 0x8c } }, //  53 \u0035 5
	{ {0x00, 0x3c, 0x52, 0x92, 0x92, 0x0c } }, //  54 \u0036 6
	{ {0x00, 0x80, 0x8e, 0x90, 0xa0, 0xc0 } }, //  55 \u0037 7
	{ {0x00, 0x6c, 0x92, 0x92, 0x92, 0x6c } }, //  56 \u0038 8
	{ {0x00, 0x60, 0x92, 0x92, 0x94, 0x78 } }, //  57 \u0039 9
	{ {0x00, 0x00, 0x6c, 0x6c, 0x00, 0x00 } }, //  58 \u003a :
	{ {0x00, 0x00, 0x37, 0x36, 0x00, 0x00 } }, //  59 \u003b ;
	{ {0x00, 0x10, 0x28, 0x44, 0x82, 0x00 } }, //  60 \u003c <
	{ {0x00, 0x24, 0x24, 0x24, 0x24, 0x24 } }, //  61 \u003d =
	{ {0x00, 0x00, 0x82, 0x44, 0x28, 0x10 } }, //  62 \u003e >
	{ {0x00, 0x40, 0x80, 0x9a, 0x90, 0x60 } }, //  63 \u003f ?
	{ {0x00, 0x7c, 0x82, 0xba, 0xaa, 0x78 } }, //  64 \u0040 @
	{ {0x00, 0x7e, 0x88, 0x88, 0x88, 0x7e } }, //  65 \u0041 A
	{ {0x00, 0xfe, 0x92, 0x92, 0x92, 0x6c } }, //  66 \u0042 B
	{ {0x00, 0x7c, 0x82, 0x82, 0x82, 0x44 } }, //  67 \u0043 C
	{ {0x00, 0xfe, 0x82, 0x82, 0x82, 0x7c } }, //  68 \u0044 D
	{ {0x00, 0xfe, 0x92, 0x92, 0x92, 0x82 } }, //  69 \u0045 E
	{ {0x00, 0xfe, 0x90, 0x90, 0x90, 0x80 } }, //  70 \u0046 F
	{ {0x00, 0x7c, 0x82, 0x92, 0x92, 0x5e } }, //  71 \u0047 G
	{ {0x00, 0xfe, 0x10, 0x10, 0x10, 0xfe } }, //  72 \u0048 H
	{ {0x00, 0x00, 0x82, 0xfe, 0x82, 0x00 } }, //  73 \u0049 I
	{ {0x00, 0x0c, 0x02, 0x02, 0x02, 0xfc } }, //  74 \u004a J
	{ {0x00, 0xfe, 0x10, 0x28, 0x44, 0x82 } }, //  75 \u004b K
	{ {0x00, 0xfe, 0x02, 0x02, 0x02, 0x02 } }, //  76 \u004c L
	{ {0x00, 0xfe, 0x40, 0x20, 0x40, 0xfe } }, //  77 \u004d M
	{ {0x00, 0xfe, 0x40, 0x20, 0x10, 0xfe } }, //  78 \u004e N
	{ {0x00, 0x7c, 0x82, 0x82, 0x82, 0x7c } }, //  79 \u004f O
	{ {0x00, 0xfe, 0x90, 0x90, 0x90, 0x60 } }, //  80 \u0050 P
	{ {0x00, 0x7c, 0x82, 0x8a, 0x84, 0x7a } }, //  81 \u0051 Q
	{ {0x00, 0xfe, 0x90, 0x90, 0x98, 0x66 } }, //  82 \u0052 R
	{ {0x00, 0x64, 0x92, 0x92, 0x92, 0x4c } }, //  83 \u0053 S
	{ {0x00, 0x80, 0x80, 0xfe, 0x80, 0x80 } }, //  84 \u0054 T
	{ {0x00, 0xfc, 0x02, 0x02, 0x02, 0xfc } }, //  85 \u0055 U
	{ {0x00, 0xf8, 0x04, 0x02, 0x04, 0xf8 } }, //  86 \u0056 V
	{ {0x00, 0xfc, 0x02, 0x3c, 0x02, 0xfc } }, //  87 \u0057 W
	{ {0x00, 0xc6, 0x28, 0x10, 0x28, 0xc6 } }, //  88 \u0058 X
	{ {0x00, 0xe0, 0x10, 0x0e, 0x10, 0xe0 } }, //  89 \u0059 Y
	{ {0x00, 0x8e, 0x92, 0xa2, 0xc2, 0x00 } }, //  90 \u005a Z
	{ {0x00, 0x00, 0xfe, 0x82, 0x82, 0x00 } }, //  91 \u005b [
	{ {0x00, 0x40, 0x20, 0x10, 0x08, 0x04 } }, //  92 \u005c slash
	{ {0x00, 0x00, 0x82, 0x82, 0xfe, 0x00 } }, //  93 \u005d ]
	{ {0x00, 0x20, 0x40, 0x80, 0x40, 0x20 } }, //  94 \u005e ^
	{ {0x01, 0x01, 0x01, 0x01, 0x01, 0x01 } }, //  95 \u005f _
	{ {0x00, 0x00, 0xc0, 0xe0, 0x00, 0x00 } }, //  96 \u0060 `
	{ {0x00, 0x04, 0x2a, 0x2a, 0x2a, 0x1e } }, //  97 \u0061 a
	{ {0x00, 0xfe, 0x22, 0x22, 0x22, 0x1c } }, //  98 \u0062 b
	{ {0x00, 0x1c, 0x22, 0x22, 0x22, 0x14 } }, //  99 \u0063 c
	{ {0x00, 0x1c, 0x22, 0x22, 0x22, 0xfe } }, // 100 \u0064 d
	{ {0x00, 0x1c, 0x2a, 0x2a, 0x2a, 0x10 } }, // 101 \u0065 e
	{ {0x00, 0x10, 0x7e, 0x90, 0x90, 0x00 } }, // 102 \u0066 f
	{ {0x00, 0x18, 0x25, 0x25, 0x25, 0x3e } }, // 103 \u0067 g
	{ {0x00, 0xfe, 0x20, 0x20, 0x1e, 0x00 } }, // 104 \u0068 h
	{ {0x00, 0x00, 0x00, 0xbe, 0x02, 0x00 } }, // 105 \u0069 i
	{ {0x00, 0x02, 0x01, 0x21, 0xbe, 0x00 } }, // 106 \u006a j
	{ {0x00, 0xfe, 0x08, 0x14, 0x22, 0x00 } }, // 107 \u006b k
	{ {0x00, 0x00, 0x00, 0xfe, 0x02, 0x00 } }, // 108 \u006c l
	{ {0x00, 0x3e, 0x20, 0x18, 0x20, 0x1e } }, // 109 \u006d m
	{ {0x00, 0x3e, 0x20, 0x20, 0x1e, 0x00 } }, // 110 \u006e n
	{ {0x00, 0x1c, 0x22, 0x22, 0x22, 0x1c } }, // 111 \u006f o
	{ {0x00, 0x3f, 0x22, 0x22, 0x22, 0x1c } }, // 112 \u0070 p
	{ {0x00, 0x1c, 0x22, 0x22, 0x22, 0x3f } }, // 113 \u0071 q
	{ {0x00, 0x22, 0x1e, 0x22, 0x20, 0x10 } }, // 114 \u0072 r
	{ {0x00, 0x10, 0x2a, 0x2a, 0x2a, 0x04 } }, // 115 \u0073 s
	{ {0x00, 0x20, 0x7c, 0x22, 0x24, 0x00 } }, // 116 \u0074 t
	{ {0x00, 0x3c, 0x02, 0x04, 0x3e, 0x00 } }, // 117 \u0075 u
	{ {0x00, 0x38, 0x04, 0x02, 0x04, 0x38 } }, // 118 \u0076 v
	{ {0x00, 0x3c, 0x06, 0x0c, 0x06, 0x3c } }, // 119 \u0077 w
	{ {0x00, 0x36, 0x08, 0x08, 0x36, 0x00 } }, // 120 \u0078 x
	{ {0x00, 0x39, 0x05, 0x06, 0x3c, 0x00 } }, // 121 \u0079 y
	{ {0x00, 0x26, 0x2a, 0x2a, 0x32, 0x00 } }, // 122 \u007a z
	{ {0x00, 0x10, 0x7c, 0x82, 0x82, 0x00 } }, // 123 \u007b {
	{ {0x00, 0x00, 0x00, 0xff, 0x00, 0x00 } }, // 124 \u007c |
	{ {0x00, 0x00, 0x82, 0x82, 0x7c, 0x10 } }, // 125 \u007d }
	{ {0x00, 0x40, 0x80, 0x40, 0x80, 0x00 } }, // 126 \u007e ~
	{ {0x00, 0x01, 0x01, 0x01, 0x01, 0x00 } }, // 127 \u2581 lower 1/8 block (trim)
	{ {0x00, 0x03, 0x03, 0x03, 0x03, 0x00 } }, // 128 \u2582 lower 1/4 block (trim)
	{ {0x00, 0x07, 0x07, 0x07, 0x07, 0x00 } }, // 129 \u2583 lower 3/8 block (trim)
	{ {0x00, 0x0f, 0x0f, 0x0f, 0x0f, 0x00 } }, // 130 \u2584 lower 1/2 block (trim)
	{ {0x00, 0x1f, 0x1f, 0x1f, 0x1f, 0x00 } }, // 131 \u2585 lower 5/8 block (trim)
	{ {0x00, 0x3f, 0x3f, 0x3f, 0x3f, 0x00 } }, // 132 \u2586 lower 3/4 block (trim)
	{ {0x00, 0x7f, 0x7f, 0x7f, 0x7f, 0x00 } }, // 133 \u2587 lower 7/8 block (trim)
	{ {0x00, 0xff, 0xff, 0xff, 0xff, 0x00 } }, // 134 \u2588 full block (vert trim)
	{ {0x3c, 0x00, 0x00, 0x00, 0x00, 0x00 } }, // 135 \u258f left 1/6 block (trim)
	{ {0x3c, 0x3c, 0x00, 0x00, 0x00, 0x00 } }, // 136 \u258e left 1/3 block (trim)
	{ {0x3c, 0x3c, 0x3c, 0x00, 0x00, 0x00 } }, // 137 \u258c left 1/2 block (trim)
	{ {0x3c, 0x3c, 0x3c, 0x3c, 0x00, 0x00 } }, // 138 \u258b left 2/3 block (trim)
	{ {0x3c, 0x3c, 0x3c, 0x3c, 0x3c, 0x00 } }, // 139 \u258a left 5/6 block (trim)
	{ {0x3c, 0x3c, 0x3c, 0x3c, 0x3c, 0x3c } }, // 140 \u2588 full block (horiz trim)
	{ {0xff, 0xff, 0xff, 0xff, 0xff, 0xff } }, // 141 \u2588 really full block
	{ {0x00, 0x44, 0xaa, 0x9a, 0x0c, 0x00 } }, // 142 \u03b4 delta
	{ {0x00, 0x38, 0x54, 0x54, 0x54, 0x00 } }, // 143 \u03b5 epsilon
	{ {0x00, 0x40, 0x7e, 0x40, 0x7e, 0x40 } }, // 144 \u03c0 pi
	{ {0x00, 0x18, 0x24, 0x24, 0x38, 0x20 } }, // 145 \u03c3 sigma
	{ {0x00, 0x10, 0x20, 0x1e, 0x20, 0x00 } }, // 146 \u03c4 tau
	{ {0x00, 0x18, 0x24, 0x7e, 0x24, 0x18 } }, // 147 \u03c6 phi
	{ {0x00, 0x18, 0x24, 0x18, 0x24, 0x18 } }, // 148 \u221e infinity
	{ {0x00, 0xfe, 0x80, 0x80, 0xc0, 0x00 } }, // 149 \u0393 Gamma
	{ {0x00, 0x7c, 0x92, 0x92, 0x7c, 0x00 } }, // 150 \u0398 Theta
	{ {0x00, 0x10, 0xaa, 0xee, 0xaa, 0x10 } }, // 151 \u03a6 Phi
	{ {0x00, 0x32, 0x4e, 0x40, 0x4e, 0x32 } }, // 152 \u03a9 Omega
	{ {0x3f, 0x31, 0x31, 0x29, 0x29, 0x25 } }, // 153        left half of mail icon
	{ {0x25, 0x29, 0x29, 0x31, 0x31, 0x3f } }, // 154        right half of mail icon
	{ {0x00, 0x10, 0x38, 0x54, 0x10, 0x10 } }, // 155 \u2190 left arrow
	{ {0x00, 0x20, 0x40, 0xfe, 0x40, 0x20 } }, // 156 \u2191 up arrow
	{ {0x00, 0x10, 0x10, 0x54, 0x38, 0x10 } }, // 157 \u2192 right arrow
	{ {0x00, 0x08, 0x04, 0xfe, 0x04, 0x08 } }, // 158 \u2193 down arrow
	{ {0x10, 0x38, 0x54, 0x54, 0x38, 0x10 } }, // 159 \u2194 left right arrrow
	{ {0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } }, // 160
	{ {0x00, 0x00, 0x0c, 0xbe, 0x0c, 0x00 } }, // 161 \u00a1  inverted exclamation mark
	{ {0x00, 0x18, 0x24, 0x7e, 0x24, 0x00 } }, // 162 \u00a2  cent
	{ {0x00, 0x12, 0x7c, 0x92, 0x92, 0x46 } }, // 163 \u00a3  pound
	{ {0x00, 0x1c, 0x2a, 0x2a, 0x22, 0x14 } }, // 164 \u20ac  euro
	{ {0x00, 0x94, 0x54, 0x3e, 0x54, 0x94 } }, // 165 \u00a5  yen
	{ {0x00, 0x12, 0xaa, 0x6a, 0xaa, 0x24 } }, // 166 \u0160  capital S with caron
	{ {0x00, 0x44, 0xb2, 0xaa, 0x9a, 0x44 } }, // 167 \u00a7  section
	{ {0x00, 0x10, 0xaa, 0x6a, 0xaa, 0x04 } }, // 168 \u0161  small s with caron
	{ {0x00, 0x1c, 0x3e, 0x36, 0x22, 0x1c } }, // 169 \u00a9  copyright
	{ {0x00, 0x10, 0xaa, 0xaa, 0xaa, 0x7a } }, // 170 \u00aa  feminine ordinal
	{ {0x00, 0x10, 0x28, 0x00, 0x10, 0x28 } }, // 171 \u00ab  left double angle
	{ {0x20, 0x20, 0x20, 0x20, 0x20, 0x38 } }, // 172 \u00ac  not
	{ {0x00, 0x10, 0x10, 0x10, 0x10, 0x10 } }, // 173 \u00ad  hyphen
	{ {0x00, 0x1c, 0x3e, 0x3a, 0x2e, 0x1c } }, // 174 \u00ae  registered
	{ {0x00, 0x80, 0x80, 0x80, 0x80, 0x80 } }, // 175 \u00af  macron
	{ {0x00, 0x60, 0x90, 0x90, 0x90, 0x60 } }, // 176 \u00b0  degree
	{ {0x00, 0x24, 0x24, 0xfc, 0x24, 0x24 } }, // 177 \u00b1  plus-minus
	{ {0x00, 0x90, 0xb0, 0x50, 0x00, 0x00 } }, // 178 \u00b2  superscript 2
	{ {0x00, 0xa8, 0xa8, 0x50, 0x00, 0x00 } }, // 179 \u00b3  superscript 3
	{ {0x00, 0xa6, 0x6a, 0x72, 0xa2, 0x00 } }, // 180 \u017d  Z caron
	{ {0x00, 0x3f, 0x04, 0x04, 0x38, 0x00 } }, // 181 \u00b5  micro
	{ {0x00, 0x60, 0x90, 0xfe, 0x80, 0xfe } }, // 182 \u00b6  paragraph
	{ {0x00, 0x00, 0x18, 0x18, 0x00, 0x00 } }, // 183 \u00b7  middle dot
	{ {0x00, 0xa6, 0x6a, 0x6a, 0xb2, 0x00 } }, // 184 \u017e  z caron
	{ {0x00, 0x48, 0xf8, 0x08, 0x00, 0x00 } }, // 185 \u00b9  superscript 1
	{ {0x00, 0x72, 0x8a, 0x8a, 0x72, 0x00 } }, // 186 \u00ba  masculine ordinal
	{ {0x00, 0x28, 0x10, 0x00, 0x28, 0x10 } }, // 187 \u00bb  right double angle
	{ {0x00, 0xfe, 0x82, 0xfe, 0x92, 0x92 } }, // 188 \u0152  OE
	{ {0x00, 0x3e, 0x22, 0x3e, 0x2a, 0x12 } }, // 189 \u0153  oe
	{ {0x00, 0x20, 0x90, 0x0e, 0x90, 0x20 } }, // 190 \u0178  Y diaeresis
	{ {0x00, 0x0c, 0x12, 0xb2, 0x02, 0x04 } }, // 191 \u00bf  inverted question mark
	{ {0x00, 0x0e, 0x94, 0xa4, 0x54, 0x0e } }, // 192 \u00c0  A grave
	{ {0x00, 0x0e, 0x54, 0xa4, 0x94, 0x0e } }, // 193 \u00c1  A acute
	{ {0x00, 0x0e, 0x54, 0xa4, 0x54, 0x0e } }, // 194 \u00c2  A circumflex
	{ {0x00, 0x4e, 0x94, 0x64, 0x94, 0x0e } }, // 195 \u00c3  A tilde
	{ {0x00, 0x0e, 0x94, 0x24, 0x94, 0x0e } }, // 196 \u00c4  A diaeresis
	{ {0x00, 0x1e, 0xf4, 0xa4, 0xf4, 0x1e } }, // 197 \u00c5  A ring
	{ {0x00, 0x7e, 0x90, 0xfe, 0x92, 0x92 } }, // 198 \u00c6  AE
	{ {0x00, 0x78, 0x85, 0x87, 0x84, 0x48 } }, // 199 \u00c7  C cedilla
	{ {0x00, 0x3e, 0xaa, 0xaa, 0x6a, 0x62 } }, // 200 \u00c8  E grave
	{ {0x00, 0x3e, 0x6a, 0x6a, 0xaa, 0xa2 } }, // 201 \u00c9  E acute
	{ {0x00, 0x7e, 0xaa, 0xaa, 0x6a, 0x22 } }, // 202 \u00ca  E circumflex
	{ {0x00, 0x3e, 0xaa, 0x2a, 0x2a, 0xa2 } }, // 203 \u00cb  E diaeresis
	{ {0x00, 0x00, 0xa2, 0xbe, 0x62, 0x00 } }, // 204 \u00cc  I grave
	{ {0x00, 0x00, 0x62, 0xbe, 0xa2, 0x00 } }, // 205 \u00cd  I acute
	{ {0x00, 0x00, 0x62, 0xbe, 0x62, 0x00 } }, // 206 \u00ce  I circumflex
	{ {0x00, 0x00, 0xa2, 0x3e, 0xa2, 0x00 } }, // 207 \u00cf  I diaeresis
	{ {0x00, 0xfe, 0x92, 0x92, 0x82, 0x7c } }, // 208 \u00d0  Eth
	{ {0x00, 0x5e, 0x88, 0x44, 0x9e, 0x00 } }, // 209 \u00d1  N tilde
	{ {0x00, 0x9c, 0xa2, 0x62, 0x1c, 0x00 } }, // 210 \u00d2  O grave
	{ {0x00, 0x1c, 0x62, 0xa2, 0x9c, 0x00 } }, // 211 \u00d3  O acute
	{ {0x00, 0x1c, 0x62, 0xa2, 0x5c, 0x00 } }, // 212 \u00d4  O circumflex
	{ {0x00, 0x5c, 0xa2, 0x62, 0x9c, 0x00 } }, // 213 \u00d5  O tilde
	{ {0x00, 0x9c, 0x22, 0x22, 0x9c, 0x00 } }, // 214 \u00d6  O diaeresis
	{ {0x00, 0x14, 0x08, 0x08, 0x14, 0x00 } }, // 215 \u00d7  multiplication
	{ {0x00, 0x7c, 0x86, 0xba, 0xc2, 0x7c } }, // 216 \u00d8  O slash
	{ {0x00, 0x3c, 0x82, 0x82, 0x7c, 0x00 } }, // 217 \u00d9  U grave
	{ {0x00, 0x3c, 0x42, 0x82, 0xbc, 0x00 } }, // 218 \u00da  U acute
	{ {0x00, 0x1c, 0x42, 0x82, 0x5c, 0x00 } }, // 219 \u00db  U circumflex
	{ {0x00, 0x3c, 0x82, 0x02, 0xbc, 0x00 } }, // 220 \u00dc  U diaeresis
	{ {0x00, 0x20, 0x50, 0x8e, 0x90, 0x20 } }, // 221 \u00dd  Y acute
	{ {0x00, 0xff, 0x22, 0x22, 0x22, 0x1c } }, // 222 \u00de  Thorn
	{ {0x00, 0x7e, 0x50, 0x52, 0x2c, 0x00 } }, // 223 \u00df  Eszett
	{ {0x00, 0x04, 0xaa, 0xaa, 0x6a, 0x1e } }, // 224 \u00e0  a grave
	{ {0x00, 0x04, 0x6a, 0xaa, 0xaa, 0x1e } }, // 225 \u00e1  a acute
	{ {0x00, 0x04, 0x6a, 0xaa, 0x6a, 0x1e } }, // 226 \u00e2  a circumplex
	{ {0x00, 0x04, 0x6a, 0xaa, 0x6a, 0x9e } }, // 227 \u00e3  a tilde
	{ {0x00, 0x04, 0xaa, 0x2a, 0xaa, 0x1e } }, // 228 \u00e4  a diaeresis
	{ {0x00, 0x04, 0xea, 0xaa, 0xea, 0x1e } }, // 229 \u00e5  a ring
	{ {0x00, 0x2c, 0x2a, 0x3e, 0x2a, 0x1a } }, // 230 \u00e6  ae
	{ {0x00, 0x38, 0x45, 0x47, 0x44, 0x28 } }, // 231 \u00e7  c cedilla
	{ {0x00, 0x1c, 0xaa, 0xaa, 0x6a, 0x10 } }, // 232 \u00e8  e grave
	{ {0x00, 0x1c, 0x2a, 0x6a, 0xaa, 0x90 } }, // 233 \u00e9  e acute
	{ {0x00, 0x1c, 0x6a, 0xaa, 0x6a, 0x10 } }, // 234 \u00ea  e circumflex
	{ {0x00, 0x1c, 0xaa, 0x2a, 0xaa, 0x10 } }, // 235 \u00eb  e diaeresis
	{ {0x00, 0x00, 0x80, 0xbe, 0x42, 0x00 } }, // 236 \u00ec  i grave
	{ {0x00, 0x00, 0x40, 0xbe, 0x82, 0x00 } }, // 237 \u00ed  i acute
	{ {0x00, 0x00, 0x40, 0x9e, 0x42, 0x00 } }, // 238 \u00ee  i circumflex
	{ {0x00, 0x00, 0x80, 0x3e, 0x82, 0x00 } }, // 239 \u00ef  i diaeresis
	{ {0x00, 0x9c, 0xe2, 0xa2, 0xa2, 0x7c } }, // 240 \u00f0  eth
	{ {0x00, 0x5e, 0x90, 0x50, 0x8e, 0x00 } }, // 241 \u00f1  n tilde
	{ {0x00, 0x9c, 0xa2, 0x62, 0x1c, 0x00 } }, // 242 \u00f2  o grave
	{ {0x00, 0x1c, 0x62, 0xa2, 0x9c, 0x00 } }, // 243 \u00f3  o acute
	{ {0x00, 0x1c, 0x62, 0xa2, 0x5c, 0x00 } }, // 244 \u00f4  o circumflex
	{ {0x00, 0x5c, 0xa2, 0x62, 0x9c, 0x00 } }, // 245 \u00f5  o tilde
	{ {0x00, 0x1c, 0xa2, 0x22, 0x9c, 0x00 } }, // 246 \u00f6  o diaeresis
	{ {0x00, 0x10, 0x10, 0x54, 0x10, 0x10 } }, // 247 \u00f7  division
	{ {0x00, 0x1d, 0x26, 0x2a, 0x32, 0x5c } }, // 248 \u00f8  o slash
	{ {0x00, 0xbc, 0x82, 0x44, 0x3e, 0x00 } }, // 249 \u00f9  u grave
	{ {0x00, 0x3c, 0x42, 0x84, 0xbe, 0x00 } }, // 250 \u00fa  u acute
	{ {0x00, 0x1c, 0x42, 0x84, 0x5e, 0x00 } }, // 251 \u00fb  u circumflex
	{ {0x00, 0x3c, 0x82, 0x04, 0xbe, 0x00 } }, // 252 \u00fc  u diaeresis
	{ {0x00, 0x39, 0x45, 0x86, 0xbc, 0x00 } }, // 253 \u00fd  y acute
	{ {0x00, 0xff, 0x12, 0x12, 0x12, 0x0c } }, // 254 \u00fe  thorn
	{ {0x00, 0x39, 0x85, 0x06, 0xbc, 0x00 } }  // 255 \u00ff  y diaeresis
};
#endif

#ifdef IMONLCD_FONT_DESC
/* This char array is available for use when converting to and from the
 * structure.  It is not needed for normal LCD use.
 */
static char *desc[] = {
	"  0 \\u0020 space",
	"  1 \\u263a light smiley face",
	"  2 \\u263b dark smiley face",
	"  3 \\u2665 full heart",
	"  4         small heart",
	"  5 \\u2660 club",
	"  6 \\u2660 spade",
	"  7 \\u2022 bullet",
	"  8 \\u25d8 big rectangle",
	"  9 \\u25cb small rectangle",
	" 10 \\u25d9 filled rectangle",
	" 11 \\u2642 man symbol",
	" 12 \\u2640 woman symbol",
	" 13 \\u266a musical note",
	" 14 \\u266b double music note",
	" 15 \\u25cf record",
	" 16 \\u25b6 play",
	" 17 \\u25c0 play backwards",
	" 18         pause",
	" 19 \\u25fe stop",
	" 20 \\u23cf eject",
	" 21         fwd",
	" 22         rev",
	" 23 \\u2581 lower 1/8 block (full)",
	" 24 \\u2582 lower 1/4 block (full)",
	" 25 \\u2583 lower 3/8 block (full)",
	" 26 \\u2584 lower 1/2 block (full)",
	" 27 \\u2585 lower 5/8 block (full)",
	" 28 \\u2586 lower 3/4 block (full)",
	" 29 \\u2587 lower 7/8 block (full)",
	" 30",
	" 31",
	" 32 \\u0020 space",
	" 33 \\u0021 !",
	" 34 \\u0022 \"",
	" 35 \\u0023 #",
	" 36 \\u0024 $",
	" 37 \\u0025 %",
	" 38 \\u0026 &",
	" 39 \\u0027 '",
	" 40 \\u0028 (",
	" 41 \\u0029 )",
	" 42 \\u002a *",
	" 43 \\u002b +",
	" 44 \\u002c ,",
	" 45 \\u002d -",
	" 46 \\u002e .",
	" 47 \\u002f /",
	" 48 \\u0030 0",
	" 49 \\u0031 1",
	" 50 \\u0032 2",
	" 51 \\u0033 3",
	" 52 \\u0034 4",
	" 53 \\u0035 5",
	" 54 \\u0036 6",
	" 55 \\u0037 7",
	" 56 \\u0038 8",
	" 57 \\u0039 9",
	" 58 \\u003a :",
	" 59 \\u003b ;",
	" 60 \\u003c <",
	" 61 \\u003d =",
	" 62 \\u003e >",
	" 63 \\u003f ?",
	" 64 \\u0040 @",
	" 65 \\u0041 A",
	" 66 \\u0042 B",
	" 67 \\u0043 C",
	" 68 \\u0044 D",
	" 69 \\u0045 E",
	" 70 \\u0046 F",
	" 71 \\u0047 G",
	" 72 \\u0048 H",
	" 73 \\u0049 I",
	" 74 \\u004a J",
	" 75 \\u004b K",
	" 76 \\u004c L",
	" 77 \\u004d M",
	" 78 \\u004e N",
	" 79 \\u004f O",
	" 80 \\u0050 P",
	" 81 \\u0051 Q",
	" 82 \\u0052 R",
	" 83 \\u0053 S",
	" 84 \\u0054 T",
	" 85 \\u0055 U",
	" 86 \\u0056 V",
	" 87 \\u0057 W",
	" 88 \\u0058 X",
	" 89 \\u0059 Y",
	" 90 \\u005a Z",
	" 91 \\u005b [",
	" 92 \\u005c \\",
	" 93 \\u005d ]",
	" 94 \\u005e ^",
	" 95 \\u005f _",
	" 96 \\u0060 `",
	" 97 \\u0061 a",
	" 98 \\u0062 b",
	" 99 \\u0063 c",
	"100 \\u0064 d",
	"101 \\u0065 e",
	"102 \\u0066 f",
	"103 \\u0067 g",
	"104 \\u0068 h",
	"105 \\u0069 i",
	"106 \\u006a j",
	"107 \\u006b k",
	"108 \\u006c l",
	"109 \\u006d m",
	"110 \\u006e n",
	"111 \\u006f o",
	"112 \\u0070 p",
	"113 \\u0071 q",
	"114 \\u0072 r",
	"115 \\u0073 s",
	"116 \\u0074 t",
	"117 \\u0075 u",
	"118 \\u0076 v",
	"119 \\u0077 w",
	"120 \\u0078 x",
	"121 \\u0079 y",
	"122 \\u007a z",
	"123 \\u007b {",
	"124 \\u007c |",
	"125 \\u007d }",
	"126 \\u007e ~",
	"127 \\u2581 lower 1/8 block (trim)",
	"128 \\u2582 lower 1/4 block (trim)",
	"129 \\u2583 lower 3/8 block (trim)",
	"130 \\u2584 lower 1/2 block (trim)",
	"131 \\u2585 lower 5/8 block (trim)",
	"132 \\u2586 lower 3/4 block (trim)",
	"133 \\u2587 lower 7/8 block (trim)",
	"134 \\u2588 full block (vert trim)",
	"135 \\u258f left 1/6 block (trim)",
	"136 \\u258e left 1/3 block (trim)",
	"137 \\u258c left 1/2 block (trim)",
	"138 \\u258b left 2/3 block (trim)",
	"139 \\u258a left 5/6 block (trim)",
	"140 \\u2588 full block (horiz trim)",
	"141 \\u2588 really full block",
	"142 \\u03b4 delta",
	"143 \\u03b5 epsilon",
	"144 \\u03c0 pi",
	"145 \\u03c3 sigma",
	"146 \\u03c4 tau",
	"147 \\u03c6 phi",
	"148 \\u221e infinity",
	"149 \\u0393 Gamma",
	"150 \\u0398 Theta",
	"151 \\u03a6 Phi",
	"152 \\u03a9 Omega",
	"153         left half of mail icon",
	"154         right half of mail icon",
	"155 \\u2190 left arrow",
	"156 \\u2191 up arrow",
	"157 \\u2192 right arrow",
	"158 \\u2193 down arrow",
	"159 \\u2194 left right arrrow",
	"160",
	"161 \\u00a1  inverted exclamation mark",
	"162 \\u00a2  cent",
	"163 \\u00a3  pound",
	"164 \\u20ac  euro",
	"165 \\u00a5  yen",
	"166 \\u0160  capital S with caron",
	"167 \\u00a7  section",
	"168 \\u0161  small s with caron",
	"169 \\u00a9  copyright",
	"170 \\u00aa  feminine ordinal",
	"171 \\u00ab  left double angle",
	"172 \\u00ac  not",
	"173 \\u00ad  hyphen",
	"174 \\u00ae  registered",
	"175 \\u00af  macron",
	"176 \\u00b0  degree",
	"177 \\u00b1  plus-minus",
	"178 \\u00b2  superscript 2",
	"179 \\u00b3  superscript 3",
	"180 \\u017d  Z caron",
	"181 \\u00b5  micro",
	"182 \\u00b6  paragraph",
	"183 \\u00b7  middle dot",
	"184 \\u017e  z caron",
	"185 \\u00b9  superscript 1",
	"186 \\u00ba  masculine ordinal",
	"187 \\u00bb  right double angle",
	"188 \\u0152  OE",
	"189 \\u0153  oe",
	"190 \\u0178  Y diaeresis",
	"191 \\u00bf  inverted question mark",
	"192 \\u00c0  A grave",
	"193 \\u00c1  A acute",
	"194 \\u00c2  A circumflex",
	"195 \\u00c3  A tilde",
	"196 \\u00c4  A diaeresis",
	"197 \\u00c5  A ring",
	"198 \\u00c6  AE",
	"199 \\u00c7  C cedilla",
	"200 \\u00c8  E grave",
	"201 \\u00c9  E acute",
	"202 \\u00ca  E circumflex",
	"203 \\u00cb  E diaeresis",
	"204 \\u00cc  I grave",
	"205 \\u00cd  I acute",
	"206 \\u00ce  I circumflex",
	"207 \\u00cf  I diaeresis",
	"208 \\u00d0  Eth",
	"209 \\u00d1  N tilde",
	"210 \\u00d2  O grave",
	"211 \\u00d3  O acute",
	"212 \\u00d4  O circumflex",
	"213 \\u00d5  O tilde",
	"214 \\u00d6  O diaeresis",
	"215 \\u00d7  multiplication",
	"216 \\u00d8  O slash",
	"217 \\u00d9  U grave",
	"218 \\u00da  U acute",
	"219 \\u00db  U circumflex",
	"220 \\u00dc  U diaeresis",
	"221 \\u00dd  Y acute",
	"222 \\u00de  Thorn",
	"223 \\u00df  Eszett",
	"224 \\u00e0  a grave",
	"225 \\u00e1  a acute",
	"226 \\u00e2  a circumplex",
	"227 \\u00e3  a tilde",
	"228 \\u00e4  a diaeresis",
	"229 \\u00e5  a ring",
	"230 \\u00e6  ae",
	"231 \\u00e7  c cedilla",
	"232 \\u00e8  e grave",
	"233 \\u00e9  e acute",
	"234 \\u00ea  e circumflex",
	"235 \\u00eb  e diaeresis",
	"236 \\u00ec  i grave",
	"237 \\u00ed  i acute",
	"238 \\u00ee  i circumflex",
	"239 \\u00ef  i diaeresis",
	"240 \\u00f0  eth",
	"241 \\u00f1  n tilde",
	"242 \\u00f2  o grave",
	"243 \\u00f3  o acute",
	"244 \\u00f4  o circumflex",
	"245 \\u00f5  o tilde",
	"246 \\u00f6  o diaeresis",
	"247 \\u00f7  division",
	"248 \\u00f8  o slash",
	"249 \\u00f9  u grave",
	"250 \\u00fa  u acute",
	"251 \\u00fb  u circumflex",
	"252 \\u00fc  u diaeresis",
	"253 \\u00fd  y acute",
	"254 \\u00fe  thorn",
	"255 \\u00ff  y diaeresis"
};

#endif

/**
 * This is the definition for a "big" font, which is a font that
 * simply takes up twice as many pixels as the normal font.
 * (12 col x 18 row pixels).  It is only used for drawing numbers.
 */
typedef struct _imon_bigfont {
	int ch;
	unsigned short pixels[12];
} imon_bigfont;

/** Character data for the big font */
static imon_bigfont bigfont[] = {
	{'0', {0x0000, 0x07E0, 0x1FF8, 0x3FFC, 0x7FFE, 0x4002, 0x4002, 0x4002, 0x3FFC, 0x3FFC, 0x1FF8, 0x07E0}},
	{'1', {0x0000, 0x0000, 0x0000, 0x4002, 0x7FFE, 0x7FFE, 0x7FFE, 0x7FFE, 0x0002, 0x0000, 0x0000, 0x0000}},
	{'2', {0x0000, 0x1806, 0x3C2C, 0x7C7C, 0x5C5C, 0x40DE, 0x7F9E, 0x7F8E, 0x3F0E, 0x1E0C, 0x0018, 0x0000}},
	{'3', {0x0000, 0x001C, 0x3C3C, 0x7C3E, 0x7C1A, 0x0080, 0x4182, 0x7FFE, 0x7FFE, 0x3E7C, 0x1C38, 0x0000}},
	{'4', {0x0000, 0x0030, 0x0050, 0x0190, 0x0610, 0x0002, 0x1FFE, 0x3FFE, 0x7FFE, 0x7FFE, 0x0012, 0x0002}},
	{'5', {0x0000, 0x0018, 0x7FBC, 0x793E, 0x3B1A, 0x3800, 0x3B02, 0x3BFE, 0x31FE, 0x61FC, 0x00F8, 0x0000}},
	{'6', {0x0000, 0x07E0, 0x1FF8, 0x3FFC, 0x7FFE, 0x4002, 0x0180, 0x5982, 0x7DFE, 0x3DFC, 0x18FC, 0x0078}},
	{'7', {0x0000, 0x0800, 0x7000, 0x3000, 0x703C, 0x787E, 0x79FE, 0x7BFC, 0x3E00, 0x3000, 0x6000, 0x0000}},
	{'8', {0x0000, 0x1C3C, 0x3E7E, 0x7FFE, 0x7FFE, 0x4182, 0x4182, 0x7FFE, 0x7FFE, 0x3E7E, 0x1C3C, 0x0000}},
	{'9', {0x0000, 0x1E18, 0x3F3C, 0x7FBE, 0x7F9A, 0x0180, 0x4002, 0x7FFE, 0x3FFC, 0x1FF8, 0x07E0, 0x0000}},
	{':', {0x0000, 0x030C, 0x079E, 0x079E, 0x030C, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000}},

	/*
	 * Marks the end of the array, but also serves as the character that
	 * unknown inputs are mapped to (essentially, a "space")
	 */
	{'\0', {0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0}}
};
