/*
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 *
 * Added support for X86_64 architecture Mar 2004
 *      Prashanth Tamraparni (prasht@in.ibm.com)
 *      Sachin Sant (sachinp@in.ibm.com)
 */

#include <lcrash.h>

int defcpu_x86_64 = -1;

/*
 * defcpu_x86_64_cmd() -- Run the 'defcpu' command.
 */
int
defcpu_x86_64_cmd(command_t *cmd)
{
	kl_dump_header_x86_64_t dha;
	int cpu;

	if (cmd->nargs == 0) {
		if (defcpu_x86_64 == -1) {
			fprintf(cmd->efp, "No default cpu set\n");
		} else {
			fprintf(cmd->ofp, "Default cpu is %d\n", defcpu_x86_64);
		}
		return(0);
	} 

	if (CORE_OS_KMEM) {
		fprintf(cmd->efp, "Can't use this command on live system\n");
		return (1);
	}
	if (kl_get_dump_header_x86_64(&dha))
		return (1);

	cpu = strtol(cmd->args[0], NULL, 10);

	if (cpu >= KL_GET_UINT32(&dha.smp_num_cpus)) {
		fprintf(cmd->efp, "Error setting defcpu to %s\n", cmd->args[0]);
		return (1);
	}
	defcpu_x86_64 = cpu;
	fprintf(cmd->ofp, "Default cpu is %d\n", defcpu_x86_64);

	if (KL_GET_UINT64(&dha.stack[defcpu_x86_64])) {
		deftask = (kaddr_t)KL_GET_UINT64(&dha.smp_current_task[defcpu_x86_64]);
		fprintf(cmd->ofp, "Default task is 0x%"FMTPTR"x\n", deftask);
	}
	return (0);
}

#define _DEFCPU_USAGE	"[-w outfile] [cpu]"

/*
 * defcpu_x86_64_usage() -- Print the usage string for the 'defcpu' command.
 */
void
defcpu_x86_64_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _DEFCPU_USAGE);
}

/*
 * defcpu_x86_64_help() -- Print the help information for the 'defcpu' command.
 */
void
defcpu_x86_64_help(command_t *cmd)
{
	CMD_HELP(cmd, _DEFCPU_USAGE,
	"Set the default cpu if one is indicated. Otherwise print the "
	"value of default cpu." 
        "When 'lcrash' is run on a live system, defcpu has no "
        "meaning.\n\n"
	"This command also sets the default task to the task running "
	"on the default cpu at the time the dump is taken. "
	"The rd command will display the registers on the default cpu "
	"at the time the dump is taken. "
        "The trace command will display a trace wrt the task "
        "running on the default cpu at the time the dump is taken. ");
}

/*
 * defcpu_x86_64_parse() -- Parse the command line arguments for 'defcpu'.
 */
int
defcpu_x86_64_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_WRITE), 0)) {
		return(1);
	}
	return(0);
}
